/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/mixin/orderdndmixin',
    ['io.ox/office/tk/utils'], function (Utils) {

    'use strict';

    var BUTTON_SELECTOR = Utils.BUTTON_SELECTOR + '[data-value]';

    // mix-in class OrderDnDMixin ===================================

    function OrderDnDMixin(app, options) {
        var self = this,
            model = app.getModel(),
            cursor = Utils.getStringOption(options, 'cursor', 'move'),
            inserter = Utils.getStringOption(options, 'inserter', 'icon-exclamation'),
            trackingOptions = {selector: BUTTON_SELECTOR, cursor: cursor},
            trackingStart = null,
            trackingIndex = null,
            trackingInfo = null;

        // private methods ----------------------------------------------------

        function getSheetIndex(button) {
            return model.getSheetModel(button.attr('title')).getIndex();
        }

        function isAboutToMove(startIndex) {
            if (startIndex === trackingIndex || startIndex + 1 === trackingIndex) {
                // object landed directly right or left of itself, it wont be moved
            } else {
                return true;
            }

        }

        function startTracking(evt) {
            var target = $(evt.target);

            var start = target.closest(BUTTON_SELECTOR);
            if (!start) {
                $.cancelTracking();
                app.getView().grabFocus();
                return;
            }

            trackingStart = start;
            if (!trackingInfo) {
                trackingInfo = $('<i>');
                trackingInfo.addClass(inserter);
                trackingInfo.css({
                    position: 'absolute',
                    zIndex: 10000,
                    fontSize: '200%',
                    color: 'rgba(51, 51, 51, 0.8)'
                });
            }
            trackingInfo.hide();
            trackingInfo.appendTo('body');

        }

        function moveTracking(evt) {
            var currentX = evt.pageX;

            var node = self.getNode();
            var buttons = node.find(BUTTON_SELECTOR);
            var posX = null;
            trackingIndex = null;

            var lastBtn = null;

            var res = _.find(buttons, function (button) {
                var btn = $(button);

                if (!btn.is('.hidden')) {
                    var mid = btn.offset().left + btn.outerWidth() / 2;
                    if (currentX < mid + 1) {
                        if (!lastBtn) {
                            posX = btn.offset().left;
                            trackingIndex = getSheetIndex(btn);
                            return true;
                        } else {
                            mid = lastBtn.offset().left + lastBtn.outerWidth() / 2;
                            if (currentX > mid - 1) {
                                posX = lastBtn.offset().left + lastBtn.outerWidth();
                                trackingIndex = getSheetIndex(btn);
                                return true;
                            }
                        }
                    }
                    lastBtn = btn;
                }
            });
            if (!res) {
                posX = lastBtn.offset().left + lastBtn.outerWidth();
                trackingIndex = getSheetIndex(lastBtn) + 1;
            }

            var startIndex =  getSheetIndex(trackingStart);

            if (isAboutToMove(startIndex)) {
                var parentPos = node.offset();
                trackingInfo.show();
                trackingInfo.css({
                    left: (posX - 6) + 'px',
                    top: (parentPos.top - 15) + 'px'
                });
            } else {
                trackingInfo.hide();
            }
        }

        function endTracking() {
            trackingInfo.remove();
            app.getView().grabFocus();

            if (trackingIndex === null) {
                //nothing was moved, so there is no dnd event
                return;
            }

            var startIndex = getSheetIndex(trackingStart);

            if (isAboutToMove(startIndex)) {
                if (trackingIndex > startIndex) {
                    trackingIndex = trackingIndex - 1;
                }
                self.trigger('change:order', startIndex, trackingIndex);
            }
            trackingIndex = null;
        }

        function cancelTracking() {
            trackingInfo.remove();
        }

        function initHandler() {
            var rootNode = self.getNode();
            rootNode.on('tracking:start', startTracking);
            rootNode.on('tracking:move', moveTracking);
            rootNode.on('tracking:end', endTracking);
            rootNode.on('tracking:cancel', cancelTracking);
        }

        function changeEditModeHandler(evt, editMode) {
            var rootNode = self.getNode();
            if (editMode) {
                rootNode.enableTracking(trackingOptions);
            } else {
                rootNode.disableTracking();
            }
        }

        // initialization -----------------------------------------------------

        initHandler();
        app.getModel().on('change:editmode', changeEditModeHandler);


    } // class OrderDnDMixin

    // exports ================================================================

    return OrderDnDMixin;

});
