/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/tokenarray',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/model/modelobject',
     'io.ox/office/spreadsheet/utils/sheetutils'
    ], function (Utils, ModelObject, SheetUtils) {

    'use strict';

    /**
     * Invokes the passed callback function, after initializing or updating the
     * global cache of visited names.
     */
    var guardVisitedNames = (function () {

        var // the cache of all visited names while recursively processing token arrays
            // (prevent endless loop in cyclic name references)
            visitedNamesCache = null;

        // a new callback function that will be passed to the passed callback function
        // returns whether the name referred by the passed name token has not been visited
        // yet, and marks it as visited
        function visitNameTokenOnce(token) {

            var // build a unique key for the passed name token
                nameKey = token.getSheet() + '!' + token.getName(),
                // whether the name token has been visited already
                visited = visitedNamesCache[nameKey];

            // mark the name as 'visited'
            visitedNamesCache[nameKey] = true;
            return !visited;
        }

        // return the actual guardVisitedNames() function
        return function (callback, context) {

            var // whether this is the initial call (initialize cache of visited names)
                initialCall = !_.isObject(visitedNamesCache);

            // Initialize cache of visited names. Using a real module-global variable
            // allows to use the same cache for invocations of this method on multiple
            // instances of the TokenArray class (for each visited defined name).
            if (initialCall) { visitedNamesCache = {}; }

            try {
                // invoke the passed callback function
                return callback.call(context, visitNameTokenOnce);
            } finally {
                // deinitialize cache of visited names
                if (initialCall) { visitedNamesCache = null; }
            }
        };
    }());

    // class TokenArray =======================================================

    /**
     * Providing support for parsing localized formula strings and handling of
     * the resulting formula token arrays.
     *
     * Instances of this class trigger the following events:
     * - 'change:tokens'
     *      After the token array has been changed (manually, or by parsing a
     *      formula), or cleared.
     * - 'change:token'
     *      After a single token in the token array has been changed. The event
     *      handlers receive a reference to the changed token, and the array
     *      index of the token.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetApplication} app
     *  The application containing this formula parser.
     *
     * @param {SheetModel} [sheetModel]
     *  The model of the sheet that contains the object with this token array.
     *  If omitted, the token array is not associated to a specific sheet.
     *
     * @param {Object} [initOptions]
     *  A map with additional options for this instance. Supports all options
     *  that are supported by the base class ModelObject.
     */
    function TokenArray(app, sheetModel, initOptions) {

        var // self reference
            tokenArray = this,

            // the spreadsheet document model
            model = app.getModel(),

            // the parser configuration (initialized once per application)
            config = app.getOrCreateConstant('tokenarray:config', createParserConfig),

            // the array of all formula tokens
            tokens = [];

        // base constructor ---------------------------------------------------

        ModelObject.call(this, app, initOptions);

        // class Token --------------------------------------------------------

        /**
         * Base class for all formula tokens.
         *
         * @constructor
         *
         * @param {String} text
         *  The text representation of the token. Used for example while the
         *  formula is edited and the same token value can be represented by
         *  different text strings, e.g. '1' vs. '1.000'.
         */
        var Token = _.makeExtendable(function (type, text) {

            /**
             * Returns the type identifier of this token.
             */
            this.getType = function () { return type; };

            /**
             * Returns the current text representation of this token.
             */
            this.getText = function () { return text; };

            /**
             * Changes the current text representation of this token, and
             * triggers a 'change'token' event at the token array instance.
             */
            this.setText = function (newText, options) {
                text = newText;
                if (!Utils.getBooleanOption(options, 'silent', false)) {
                    tokenArray.trigger('change:token', this, _(tokens).indexOf(this));
                }
                return this;
            };

        }); // class Token

        // class ValueToken ---------------------------------------------------

        /**
         * This formula token represents a constant value (a number, a string,
         * a Boolean value, or an error code) in a formula.
         *
         * @constructor
         *
         * @extends Token
         */
        var ValueToken = Token.extend({ constructor: function (text, value) {

            // base constructor
            Token.call(this, 'value', text);

            /**
             * Returns the current value of this token.
             */
            this.getValue = function () { return value; };

            /**
             * Changes the current value of this token.
             */
            this.setValue = function (newValue, options) {
                value = newValue;
                if (_.isNumber(value)) {
                    this.setText(String(value).replace('.', config.DEC), options);
                } else if (_.isString(value)) {
                    if (value[0] === '#') {
                        this.setText(app.getErrorCodeLiteral(value), options);
                    } else {
                        this.setText('"' + value.replace(/"/g, '""') + '"', options);
                    }
                } else if (_.isBoolean(value)) {
                    this.setText(app.getBooleanLiteral(value), options);
                } else {
                    Utils.error('ValueToken.setValue(): unsupported value type');
                    value = '#VALUE!';
                    this.setText(config.VAL_ERROR, options);
                }
                return this;
            };

        }}); // class ValueToken

        // class FunctionToken ------------------------------------------------

        /**
         * This formula token represents a function name in a formula.
         *
         * @constructor
         *
         * @extends Token
         */
        var FunctionToken = Token.extend({ constructor: function (text) {

            // base constructor
            Token.call(this, 'func', text);

            /**
             * Returns the current function name with upper-case characters.
             */
            this.getName = function () { return text.toUpperCase(); };

        }}); // class FunctionToken

        // class RefToken -----------------------------------------------------

        /**
         * This formula token represents a cell reference in a formula.
         *
         * @constructor
         *
         * @extends Token
         */
        var RefToken = Token.extend({ constructor: function (text, cell1Ref, cell2Ref, sheet1Ref, sheet2Ref) {

            var // self reference
                self = this,
                // whether the range address is invalid
                rangeError = false;

            // base constructor
            Token.call(this, 'ref', text);

            /**
             * Updates the string representation of this reference token,
             * according to the current reference settings.
             */
            function updateTokenText(options) {
                self.setText(generateReference(cell1Ref, cell2Ref, sheet1Ref, sheet2Ref, rangeError), options);
            }

            /**
             * Relocates the relative column/row components in the passed cell
             * reference in-place.
             */
            function relocateCellRef(cellRef, refAddress, targetAddress, wrapReferences) {

                // relocates a single column/row index
                function relocate(addrIndex, maxIndex) {
                    var index = cellRef.address[addrIndex] + (targetAddress[addrIndex] - refAddress[addrIndex]);
                    if (wrapReferences) {
                        if (index < 0) {
                            index += (maxIndex + 1);
                        } else if (index > maxIndex) {
                            index -= (maxIndex + 1);
                        }
                    }
                    cellRef.address[addrIndex] = index;
                }

                // relocate relative column and row
                if (!cellRef.absCol) { relocate(0, model.getMaxCol()); }
                if (!cellRef.absRow) { relocate(1, model.getMaxRow()); }
            }

            /**
             * Returns copies of the own cell reference objects that have been
             * relocated according to the passed settings.
             */
            function makeRelocatedCellRefs(refAddress, targetAddress, wrapReferences) {

                var // the result object returned by this method
                    result = {
                        cell1Ref: rangeError ? null : _.copy(cell1Ref, true),
                        cell2Ref: rangeError ? null : _.copy(cell2Ref, true),
                        rangeError: rangeError
                    };

                if (!result.rangeError && result.cell1Ref) {
                    relocateCellRef(result.cell1Ref, refAddress, targetAddress, wrapReferences);
                    result.rangeError = !model.isValidAddress(result.cell1Ref.address);
                }
                if (!result.rangeError && result.cell2Ref) {
                    relocateCellRef(result.cell2Ref, refAddress, targetAddress, wrapReferences);
                    result.rangeError = !model.isValidAddress(result.cell1Ref.address);
                }

                return result;
            }

            /**
             * Returns the sheet range referred by this token, in the object
             * properties 'sheet1' and 'sheet2'. If the token does not contain
             * a valid sheet range, returns null.
             */
            this.getSheetRange = function () {
                var sheet1 = _.isObject(sheet1Ref) ? sheet1Ref.sheet : sheetModel ? sheetModel.getIndex() : -1,
                    sheet2 = _.isObject(sheet2Ref) ? sheet2Ref.sheet : sheet1;
                return (_.isNumber(sheet1) && (sheet1 >= 0) && _.isNumber(sheet2) && (sheet2 >= 0)) ? { sheet1: Math.min(sheet1, sheet2), sheet2: Math.max(sheet1, sheet2) } : null;
            };

            /**
             * Returns whether this reference token refers to the specified
             * sheet.
             */
            this.isSheet = function (sheet) {
                var sheetRange = this.getSheetRange();
                return _.isObject(sheetRange) && (sheetRange.sheet1 === sheet) && (sheetRange.sheet2 === sheet);
            };

            /**
             * Returns whether this reference token refers to the specified
             * sheet, or refers to a sheet range that contains the specified
             * sheet.
             */
            this.containsSheet = function (sheet) {
                var sheetRange = this.getSheetRange();
                return _.isObject(sheetRange) && (sheetRange.sheet1 <= sheet) && (sheet <= sheetRange.sheet2);
            };

            /**
             * Returns the reference expression of the relocated cell range for
             * this token.
             */
            this.generateReference = function (refAddress, targetAddress, wrapReferences) {
                var result = makeRelocatedCellRefs(refAddress, targetAddress, wrapReferences);
                return generateReference(result.cell1Ref, result.cell2Ref, sheet1Ref, sheet2Ref, result.rangeError);
            };

            /**
             * Returns the logical address of the original cell range
             * represented by this token, or null if the range address is
             * invalid.
             */
            this.getRange = function () {
                return rangeError ? null : SheetUtils.getAdjustedRange({ start: cell1Ref.address, end: (cell2Ref || cell1Ref).address });
            };

            /**
             * Extracts the logical address of the relocated cell range from
             * this token, or null if the resulting range address is invalid.
             */
            this.getRelocatedRange = function (refAddress, targetAddress, wrapReferences) {
                var result = makeRelocatedCellRefs(refAddress, targetAddress, wrapReferences);
                return result.rangeError ? null : SheetUtils.getAdjustedRange({ start: result.cell1Ref.address, end: (result.cell2Ref || result.cell1Ref).address });
            };

            /**
             * Changes the current cell range of this token, keeps all absolute
             * flags and sheet references intact.
             */
            this.setRange = function (range, options) {
                cell1Ref.address = _.clone(range.start);
                if (!cell2Ref && !_.isEqual(range.start, range.end)) { cell2Ref = _.copy(cell1Ref, true); }
                if (cell2Ref) { cell2Ref.address = _.clone(range.end); }
                rangeError = false;
                updateTokenText(options);
                return this;
            };

            /**
             * Transforms this reference token after columns or rows have been
             * inserted into or deleted from a sheet in the document.
             *
             * @returns {Boolean}
             *  Whether the token has been changed.
             */
            this.transformToken = function (sheet, interval, insert, columns) {

                // nothing to do, if this token contains invalid cell references,
                // or points to another sheet, or sheet reference is invalid
                if (rangeError || !this.isSheet(sheet)) { return false; }

                var // the original range
                    oldRange = this.getRange(),
                    // the transformed range address
                    newRange = model.transformRange(oldRange, interval, insert, columns);

                // nothing to do, if the range does not change
                if (_.isEqual(oldRange, newRange)) { return false; }

                // check that the range has been transformed successfully
                rangeError = !_.isObject(newRange);
                if (!rangeError) { this.setRange(newRange, { silent: true }); }
                return true;
            };

            /**
             * Relocates this reference token to a new cell position.
             *
             * @returns {Boolean}
             *  Whether the token has been changed.
             */
            this.relocateToken = function (refAddress, targetAddress, wrapReferences) {

                // do not touch tokens with invalid cell range
                if (rangeError) { return false; }

                // relocate both cell references
                relocateCellRef(cell1Ref, refAddress, targetAddress, wrapReferences);
                if (cell2Ref) { relocateCellRef(cell2Ref, refAddress, targetAddress, wrapReferences); }

                // check validity of the new references
                rangeError = !model.isValidAddress(cell1Ref.address) || (_.isObject(cell2Ref) && !model.isValidAddress(cell2Ref.address));

                // recalculate and set the new text representation
                updateTokenText({ silent: true });
                return true;
            };

            /**
             * Refreshes the sheet references after a sheet has been inserted,
             * deleted, or moved in the document.
             *
             * @returns {Boolean}
             *  Whether the token has been changed.
             */
            this.transformSheet = function (to, from) {
                var changed1 = transformSheetRef(sheet1Ref, to, from),
                    changed2 = transformSheetRef(sheet2Ref, to, from);
                return changed1 || changed2;
            };

            /**
             * Changes all sheet references containing the specified old sheet
             * index to the new sheet index. Used for example when copying
             * existing sheets, and adjusting all references and defined names
             * that point to the original sheet.
             *
             * @param {Number} oldSheet
             *  The zero-based index of the original sheet.
             *
             * @param {Number} newSheet
             *  The zero-based index of the target sheet.
             *
             * @returns {Boolean}
             *  Whether the token has been changed.
             */
            this.relocateSheet = function (oldSheet, newSheet) {
                var changed1 = relocateSheetRef(sheet1Ref, oldSheet, newSheet),
                    changed2 = relocateSheetRef(sheet2Ref, oldSheet, newSheet);
                if (changed1 || changed2) {
                    updateTokenText({ silent: true });
                    return true;
                }
                return false;
            };

            /**
             * Refreshes the text representation after a sheet has been renamed
             * in the document.
             *
             * @returns {Boolean}
             *  Whether the token has been changed.
             */
            this.renameSheet = function (sheet) {
                var sheetRange = this.getSheetRange();
                if (sheetRange && ((sheetRange.sheet1 === sheet) || (sheetRange.sheet2 === sheet))) {
                    updateTokenText({ silent: true });
                    return true;
                }
                return false;
            };

        }}); // class RefToken

        // class NameToken ----------------------------------------------------

        /**
         * This formula token represents a defined name in a formula.
         *
         * @constructor
         *
         * @extends Token
         */
        var NameToken = Token.extend({ constructor: function (text, name, sheetRef) {

            var self = this;

            // base constructor
            Token.call(this, 'name', text);

            /**
             * Updates the string representation of this name token, according
             * to the current reference settings.
             */
            function updateTokenText(options) {
                self.setText(generateSheetName(sheetRef) + name, options);
            }

            /**
             * Returns the current name.
             */
            this.getName = function () {
                return name;
            };

            /**
             * Returns the sheet reference descriptor.
             */
            this.getSheetRef = function () {
                return _.clone(sheetRef);
            };

            /**
             * Returns the index of the sheet referred by this token. If no
             * sheet index can be resolved (no sheet reference in the token,
             * and no sheet model in the token array), returns -1.
             */
            this.getSheet = function () {
                var sheet = _.isObject(sheetRef) ? sheetRef.sheet : sheetModel ? sheetModel.getIndex() : -1;
                return (_.isNumber(sheet) && (sheet >= 0)) ? sheet : -1;
            };

            /**
             * Returns whether this name token refers to the specified sheet.
             */
            this.isSheet = function (sheet) {
                return (sheet >= 0) && (this.getSheet() === sheet);
            };

            /**
             * Returns the token array with the definition of this name.
             */
            this.getTokenArray = function () {

                // sheet reference exists (e.g.: Sheet2!name): get name from specified sheet
                if (sheetRef) {
                    var refSheetModel = model.getSheetModel(this.getSheet());
                    return refSheetModel ? refSheetModel.getNameCollection().getTokenArray(name) : null;
                }

                // no sheet reference: try sheet names (if token array in sheet context), then global names
                return (sheetModel && sheetModel.getNameCollection().getTokenArray(name)) || model.getNameCollection().getTokenArray(name);
            };

            /**
             * Refreshes the sheet reference after a sheet has been inserted,
             * deleted, or moved in the document.
             *
             * @returns {Boolean}
             *  Whether the token has been changed.
             */
            this.transformSheet = function (to, from) {
                return transformSheetRef(sheetRef, to, from);
            };

            /**
             * Changes the sheet reference to the new sheet index, if it
             * contains the specified old sheet index. Used for example when
             * copying existing sheets, and adjusting all references and
             * defined names that point to the original sheet.
             *
             * @param {Number} oldSheet
             *  The zero-based index of the original sheet.
             *
             * @param {Number} newSheet
             *  The zero-based index of the target sheet.
             *
             * @returns {Boolean}
             *  Whether the token has been changed.
             */
            this.relocateSheet = function (oldSheet, newSheet) {
                if (relocateSheetRef(sheetRef, oldSheet, newSheet)) {
                    updateTokenText({ silent: true });
                    return true;
                }
                return false;
            };

            /**
             * Refreshes the text representation after a sheet has been renamed
             * in the document.
             *
             * @returns {Boolean}
             *  Whether the token has been changed.
             */
            this.renameSheet = function (sheet) {
                if (sheetRef && (sheetRef.sheet === sheet)) {
                    updateTokenText({ silent: true });
                    return true;
                }
                return false;
            };

        }}); // class NameToken

        // private methods ----------------------------------------------------

        /**
         * Lazy initialization of the token array parser configuration.
         */
        function createParserConfig() {

            var // RE pattern for the leading character of a simple sheet name (without apostrophes).
                SHEET_LEADING_CHAR_PATTERN = '[a-z_\\xa1-\\u2027\\u202a-\\uffff]',

                // RE pattern for other characters of a simple sheet name (without apostrophes).
                SHEET_SIMPLE_CHAR_PATTERN = '[\\w.\\xa1-\\u2027\\u202a-\\uffff]',

                // RE pattern for any character of a complex sheet name (with apostrophes).
                SHEET_COMPLEX_CHAR_PATTERN = '[^\\x00-\\x1f\\x80-\\x9f\\[\\]*?:/\\\\]',

                // RE pattern for a simple sheet name.
                SHEET_SIMPLE_PATTERN = SHEET_LEADING_CHAR_PATTERN + SHEET_SIMPLE_CHAR_PATTERN + '*',

                // RE pattern for a complex sheet name.
                SHEET_COMPLEX_PATTERN = SHEET_COMPLEX_CHAR_PATTERN + '+(?:\'\'' + SHEET_COMPLEX_CHAR_PATTERN + '*)*' + SHEET_COMPLEX_CHAR_PATTERN,

                // RE pattern for a complete sheet name used in formulas, e.g. 'Sheet1' in the formula =Sheet1!A1.
                // - Group 1: The name of the sheet (simple sheet name).
                // - Group 2: The name of the sheet (complex sheet name).
                SHEET_REF_PATTERN = '(?:(' + SHEET_SIMPLE_PATTERN + ')|\'(' + SHEET_COMPLEX_PATTERN + ')\')',

                // RE pattern for a sheet range used in formulas, e.g. 'Sheet1:Sheet2' in the formula =SUM(Sheet1:Sheet2!A1).
                // - Group 1: The name of the first sheet (simple sheet names).
                // - Group 2: The name of the second sheet (simple sheet names).
                // - Group 3: The name of the first sheet (complex sheet names).
                // - Group 4: The name of the second sheet (complex sheet names).
                SHEET_RANGE_PATTERN = '(?:(' + SHEET_SIMPLE_PATTERN + '):(' + SHEET_SIMPLE_PATTERN + ')|\'(' + SHEET_COMPLEX_PATTERN + '):(' + SHEET_COMPLEX_PATTERN + ')\')',

                // RE pattern for a column reference (absolute or relative).
                // - Group 1: The absolute marker.
                // - Group 2: The column name.
                COL_REF_PATTERN = '(\\$)?([a-z]+)',

                // RE pattern for a row reference (absolute or relative).
                // - Group 1: The absolute marker.
                // - Group 2: The row name.
                ROW_REF_PATTERN = '(\\$)?([0-9]+)',

                // RE pattern for a 2D cell reference (no leading sheet name).
                // - Group 1: The absolute marker of the column.
                // - Group 2: The column name.
                // - Group 3: The absolute marker of the row.
                // - Group 4: The row name.
                CELL_REF_PATTERN = COL_REF_PATTERN + ROW_REF_PATTERN,

                // RE pattern for a 2D cell range (no leading sheet name).
                // - Group 1: The absolute marker for the first column.
                // - Group 2: The name of the first column.
                // - Group 3: The absolute marker for the first row.
                // - Group 4: The name of the first row.
                // - Group 5: The absolute marker for the second column.
                // - Group 6: The name of the second column.
                // - Group 7: The absolute marker for the second row.
                // - Group 8: The name of the second row.
                CELL_RANGE_PATTERN = CELL_REF_PATTERN + ':' + CELL_REF_PATTERN,

                // RE pattern for a 2D column interval (no leading sheet name).
                // - Group 1: The absolute marker for the first column.
                // - Group 2: The name of the first column.
                // - Group 3: The absolute marker for the second column.
                // - Group 4: The name of the second column.
                COL_RANGE_PATTERN = COL_REF_PATTERN + ':' + COL_REF_PATTERN,

                // RE pattern for a 2D row interval (no leading sheet name).
                // - Group 1: The absolute marker for the first row.
                // - Group 2: The name of the first row.
                // - Group 3: The absolute marker for the second row.
                // - Group 4: The name of the second row.
                ROW_RANGE_PATTERN = ROW_REF_PATTERN + ':' + ROW_REF_PATTERN,

                // RE pattern for the leading character of a name or identifier.
                NAME_LEADING_CHAR_PATTERN = '[a-z_\\xa1-\\u2027\\u202a-\\uffff]',

                // RE pattern for other characters of a name or identifier.
                NAME_INNER_CHAR_PATTERN = '[\\w.\\xa1-\\u2027\\u202a-\\uffff]',

                // RE pattern for a complete name or identifier used in formulas.
                // - Group 1: The entire name.
                NAME_PATTERN = '(' + NAME_LEADING_CHAR_PATTERN + NAME_INNER_CHAR_PATTERN + '*)',

                // RE look-ahead pattern to exclude an opening parenthesis after a name
                // all valid inner characters of names must be excluded too, otherwise
                // the previous groups would match less characters than they should
                NO_PARETHESIS_PATTERN = '(?!\\(|' + NAME_INNER_CHAR_PATTERN + ')',

                // RE pattern for any invalid characters
                INVALID_CHAR_PATTERN = '[!§$?°~#;,.|]',

                // the configuration object to be returned
                config = {
                    // the decimal separator character
                    DEC: app.getDecimalSeparator(),
                    // the list separator character
                    SEP: app.getListSeparator(),
                    // the translated error code for reference errors
                    REF_ERROR: app.getErrorCodeLiteral('#REF!'),
                    // the translated error code for value errors
                    VAL_ERROR: app.getErrorCodeLiteral('#VALUE!'),
                    // a collection of regular expressions for various formula tokens
                    RE: {}
                };

            // Number literals: With or without decimal separator, digits preceding
            // and/or following the separator, optional scientific prefix. No
            // leading minus sign, will be parsed as operator, otherwise a formula
            // such as =1-1 becomes an invalid sequence of two numbers.
            config.RE.NUMBER = new RegExp('^(\\d*\\' + config.DEC + '\\d+|\\d+\\' + config.DEC + '?)(e[-+]\\d+)?', 'i');

            // String literals: Enclosed in double quotes, embedded double quotes
            // are represented by a sequence of two double quote characters.
            config.RE.STRING = /^("[^"]*")+/;

            // Boolean literals: The localized literals received from server, but
            // not followed by any character valid inside an identifier. This
            // prevents to match 'trueA' or 'false_1' or similar name tokens.
            config.RE.BOOLEAN = new RegExp('^(' + app.getBooleanLiteral(true) + '|' + app.getBooleanLiteral(false) + ')(?!' + NAME_INNER_CHAR_PATTERN + ')', 'i');

            // Error code literals: The localized literals received from server.
            config.RE.ERROR = new RegExp('^(' + app.getErrorCodeLiterals().join('|').replace(/([?#])/g, '\\$1') + ')(?!' + NAME_INNER_CHAR_PATTERN + ')', 'i');

            // Array literals: For now, simply match the entire literal, not any contents.
            config.RE.ARRAY = /^{[^{]+}/;

            // All unary and binary operators except the list separator.
            config.RE.OPERATOR = /^(<=|>=|<>|[-+*\/^%&:<>=])/;

            // The list/parameter separator character.
            config.RE.SEPARATOR = new RegExp('^' + config.SEP);

            // The opening parenthesis.
            config.RE.OPEN = /^\(/;

            // The closing parenthesis.
            config.RE.CLOSE = /^\)/;

            // A cell reference, e.g. A1 or Sheet1!$A$1.
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The absolute marker for the column.
            // - Group 4: The column name.
            // - Group 5: The absolute marker for the row.
            // - Group 6: The row name.
            config.RE.CELL_REF = new RegExp('^(?:' + SHEET_REF_PATTERN + '!)?' + CELL_REF_PATTERN + NO_PARETHESIS_PATTERN, 'i');

            // A range reference, e.g. A1:B2 or Sheet1!$A$1:$A$1.
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The absolute marker for the first column.
            // - Group 4: The name of the first column.
            // - Group 5: The absolute marker for the first row.
            // - Group 6: The name of the first row.
            // - Group 7: The absolute marker for the second column.
            // - Group 8: The name of the second column.
            // - Group 9: The absolute marker for the second row.
            // - Group 10: The name of the second row.
            config.RE.RANGE_REF = new RegExp('^(?:' + SHEET_REF_PATTERN + '!)?' + CELL_RANGE_PATTERN + NO_PARETHESIS_PATTERN, 'i');

            // A column interval, e.g. A:B or Sheet1!$C:$C (always with colon).
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The absolute marker for the first column.
            // - Group 4: The name of the first column.
            // - Group 5: The absolute marker for the second column.
            // - Group 6: The name of the second column.
            config.RE.COL_INTERVAL = new RegExp('^(?:' + SHEET_REF_PATTERN + '!)?' + COL_RANGE_PATTERN + NO_PARETHESIS_PATTERN, 'i');

            // A row interval, e.g. 1:2 or Sheet1!$3:$3 (always with colon).
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The absolute marker for the first row.
            // - Group 4: The name of the first row.
            // - Group 5: The absolute marker for the second row.
            // - Group 6: The name of the second row.
            config.RE.ROW_INTERVAL = new RegExp('^(?:' + SHEET_REF_PATTERN + '!)?' + ROW_RANGE_PATTERN + NO_PARETHESIS_PATTERN, 'i');

            // A 3D cell reference, e.g. Sheet1:Sheet2!$A$1.
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The absolute marker for the column.
            // - Group 6: The column name.
            // - Group 7: The absolute marker for the row.
            // - Group 8: The row name.
            config.RE.CELL_3D_REF = new RegExp('^' + SHEET_RANGE_PATTERN + '!' + CELL_REF_PATTERN + NO_PARETHESIS_PATTERN, 'i');

            // A 3D range reference, e.g. Sheet1:Sheet2!$A$1:$A$1.
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The absolute marker for the first column.
            // - Group 6: The name of the first column.
            // - Group 7: The absolute marker for the first row.
            // - Group 8: The name of the first row.
            // - Group 9: The absolute marker for the second column.
            // - Group 10: The name of the second column.
            // - Group 11: The absolute marker for the second row.
            // - Group 12: The name of the second row.
            config.RE.RANGE_3D_REF = new RegExp('^' + SHEET_RANGE_PATTERN + '!' + CELL_RANGE_PATTERN + NO_PARETHESIS_PATTERN, 'i');

            // A 3D column interval, Sheet1:Sheet2!$C:$C (always with colon).
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The absolute marker for the first column.
            // - Group 6: The name of the first column.
            // - Group 7: The absolute marker for the second column.
            // - Group 8: The name of the second column.
            config.RE.COL_3D_INTERVAL = new RegExp('^' + SHEET_RANGE_PATTERN + '!' + COL_RANGE_PATTERN + NO_PARETHESIS_PATTERN, 'i');

            // A 3D row interval, e.g. Sheet1:Sheet2!$3:$3 (always with colon).
            // - Group 1: The simple name of the first sheet.
            // - Group 2: The simple name of the second sheet.
            // - Group 3: The complex name of the first sheet.
            // - Group 4: The complex name of the second sheet.
            // - Group 5: The absolute marker for the first row.
            // - Group 6: The name of the first row.
            // - Group 7: The absolute marker for the second row.
            // - Group 8: The name of the second row.
            config.RE.ROW_3D_INTERVAL = new RegExp('^' + SHEET_RANGE_PATTERN + '!' + ROW_RANGE_PATTERN + NO_PARETHESIS_PATTERN, 'i');

            // Global names with sheet name (defined names, macros, etc.), e.g. Sheet1!my_range.
            // - Group 1: The simple sheet name.
            // - Group 2: The complex sheet name.
            // - Group 3: The name or identifier following the sheet name.
            config.RE.GLOBAL_NAME = new RegExp('^' + SHEET_REF_PATTERN + '!' + NAME_PATTERN + NO_PARETHESIS_PATTERN, 'i');

            // Simple names without sheet name (defined names, function names, macros, etc.), e.g. my_range.
            config.RE.NAME = new RegExp('^' + NAME_PATTERN + NO_PARETHESIS_PATTERN, 'i');

            // Any function name (defined names followed by an opening parenthesis).
            config.RE.FUNCTION = new RegExp('^(' + NAME_PATTERN + ')(?=\\()', 'i');

            // Any whitespace characters.
            config.RE.WHITESPACE = /^[\s\x00-\x1f\x80-\x9f]+/;

            // Matches a simple sheet name that does not need to be enclosed in apostrophes.
            config.RE.SIMPLE_SHEET_NAME = new RegExp('^' + SHEET_SIMPLE_PATTERN + '$', 'i');

            // A single unsupported character.
            config.RE.INVALID_CHAR = new RegExp('^' + INVALID_CHAR_PATTERN);

            return config;
        }

        /**
         * Appends a new token to this token array, and triggers a
         * 'change:token' event for this token.
         *
         * @param {Token} token
         *  The new formula token. This token array will take ownership.
         */
        function appendToken(token) {
            tokens.push(token);
            tokenArray.trigger('change:token', token, tokens.length - 1);
        }

         /**
         * Transforms the passed sheet reference, after a sheet has been
         * inserted, deleted, or moved in the document.
         */
        function transformSheetRef(sheetRef, to, from) {

            // no sheet reference passed, or invalid sheet name in reference
            if (!sheetRef || !_.isNumber(sheetRef.sheet)) { return false; }

            // sheet inserted
            if (!_.isNumber(from)) {
                // referenced sheet follows inserted sheet
                if (to <= sheetRef.sheet) { sheetRef.sheet += 1; return true; }
            }

            // sheet deleted
            else if (!_.isNumber(to)) {
                // referenced sheet follows deleted sheet
                if (from < sheetRef.sheet) { sheetRef.sheet -= 1; return true; }
                // referenced sheet has been deleted
                if (from === sheetRef.sheet) { sheetRef.sheet = -1; return true; }
            }

            // sheet moved
            else {
                // referenced sheet moved to another position
                if (from === sheetRef.sheet) { sheetRef.sheet = to; return true; }
                // sheet moved backwards, referenced sheet follows
                if ((to < from) && (to <= sheetRef.sheet) && (sheetRef.sheet < from)) { sheetRef.sheet += 1; return true; }
                // sheet moved forwards, referenced sheet follows
                if ((from < to) && (from < sheetRef.sheet) && (sheetRef.sheet <= to)) { sheetRef.sheet -= 1; return true; }
            }

            return false;
        }

        /**
         * Changes the passed sheet reference to the new sheet index, if it
         * refers to the old sheet index.
         */
        function relocateSheetRef(sheetRef, oldSheet, newSheet) {
            // TODO: always for Excel formulas, but only relative sheet references for ODF
            if (sheetRef && _.isNumber(sheetRef.sheet) && (sheetRef.sheet === oldSheet)) {
                sheetRef.sheet = newSheet;
                return true;
            }
            return false;
        }

       /**
         * Returns the sheet prefix used by cell references and defined names,
         * including the trailing exclamation mark.
         */
        function generateSheetName(sheet1Ref, sheet2Ref) {

            var sheet1Name = null,
                sheet2Name = null,
                simpleNames = false;

            // convert first sheet name
            if (!_.isObject(sheet1Ref)) { return ''; }
            sheet1Name = _.isNumber(sheet1Ref.sheet) ? model.getSheetName(sheet1Ref.sheet) : sheet1Ref.sheet;
            if (!_.isString(sheet1Name)) { return config.REF_ERROR + '!'; }
            simpleNames = config.RE.SIMPLE_SHEET_NAME.test(sheet1Name);

            // convert second sheet name
            if (_.isObject(sheet2Ref) && !_.isEqual(sheet1Ref, sheet2Ref)) {
                sheet2Name = _.isNumber(sheet2Ref.sheet) ? model.getSheetName(sheet2Ref.sheet) : sheet2Ref.sheet;
                if (!_.isString(sheet2Name)) { return config.REF_ERROR + '!'; }
                simpleNames = simpleNames && config.RE.SIMPLE_SHEET_NAME.test(sheet2Name);
                sheet1Name += ':' + sheet2Name;
            }

            // enclose complex names in apostrophes
            return simpleNames ? (sheet1Name + '!') : ('\'' + sheet1Name.replace(/'/g, '\'\'') + '\'!');
        }

        /**
         * Returns the string representation of the specified reference.
         */
        function generateReference(cell1Ref, cell2Ref, sheet1Ref, sheet2Ref, rangeError) {

            var range = rangeError ? null : SheetUtils.getAdjustedRange({ start: cell1Ref.address, end: (cell2Ref || cell1Ref).address }),
                text = generateSheetName(sheet1Ref, sheet2Ref);

            // returns the complete name of the column described by the passed cell reference
            function generateColName(cellRef) {
                return (cellRef.absCol ? '$' : '') + SheetUtils.getColName(cellRef.address[0]);
            }

            // returns the complete name of the row described by the passed cell reference
            function generateRowName(cellRef) {
                return (cellRef.absRow ? '$' : '') + SheetUtils.getRowName(cellRef.address[1]);
            }

            // returns the complete name of the cell described by the passed cell reference
            function generateCellName(cellRef) {
                return generateColName(cellRef) + generateRowName(cellRef);
            }

            if (!model.isValidRange(range)) {
                text += config.REF_ERROR;
            }

            // generate row interval (preferred over column interval, when entire sheet is selected)
            else if (cell1Ref.absCol && cell2Ref && cell2Ref.absCol && model.isRowRange(range)) {
                text += generateRowName(cell1Ref) + ':' + generateRowName(cell2Ref);
            }

            // generate column interval
            else if (cell1Ref.absRow && cell2Ref && cell2Ref.absRow && model.isColRange(range)) {
                text += generateColName(cell1Ref) + ':' + generateColName(cell2Ref);
            }

            // generate range or cell address
            else {
                text += generateCellName(cell1Ref);
                if (cell2Ref) { text += ':' + generateCellName(cell2Ref); }
            }

            return text;
        }

        /**
         * Invokes the specified method at all tokens, and triggers a
         * 'change:tokens' event, if any of the methods has returned true.
         */
        function invokeTokenMethodAndTrigger(methodName) {

            var // whether any token has returned true
                changed = false,
                // the arguments to be passed to the token method
                args = _.toArray(arguments).slice(1);

            // invoke method for all tokens, and collect changed state
            _(tokens).each(function (token) {
                if (_.isFunction(token[methodName]) && token[methodName].apply(token, args)) { changed = true; }
            });

            // notify listeners
            if (changed) { tokenArray.trigger('change:tokens'); }
            return changed;
        }

        // methods ------------------------------------------------------------

        /**
         * Removes all tokens from this token array.
         *
         * @returns {TokenArray}
         *  A reference to this instance.
         */
        this.clear = function () {
            if (tokens.length > 0) {
                tokens = [];
                this.trigger('change:tokens');
            }
            return this;
        };

        /**
         * Appends a list operator token to this token array.
         *
         * @returns {TokenArray}
         *  A reference to this instance.
         */
        this.appendListOperator = function () {
            appendToken(new Token('sep', config.SEP));
            return this;
        };

        /**
         * Appends a new reference token to this token array.
         *
         * @param {Object} cell1Ref
         *  The first cell reference. Must contain the following properties:
         *  @param {Number[]} cell1Ref.address
         *      The logical address for the cell reference.
         *  @param {Boolean} [cell1Ref.absCol=false]
         *      Whether the column reference will be marked as absolute
         *      (leading dollar sign).
         *  @param {Boolean} [cell1Ref.absRow=false]
         *      Whether the row reference will be marked as absolute (leading
         *      dollar sign).
         *
         * @param {Object} [cell2Ref]
         *  The second cell reference. Must contain the same properties as the
         *  parameter 'cell1Ref'. If omitted, the reference token will refer to
         *  a single cell specified by 'cell1Ref'.
         *
         * @param {Object} [sheet1Ref]
         *  The first sheet reference. Must contain the following properties:
         *  @param {Number|String} sheet1Ref.sheet
         *      The zero-based index of the sheet the reference token will
         *      refer to, or the explicit name of a non-existing sheet. If
         *      omitted, the reference token will not contain a sheet name.
         *  @param {Boolean} [sheet1Ref.abs=false]
         *      Whether the sheet reference will be marked as absolute.
         *
         * @param {Object} [sheet2Ref]
         *  The second sheet reference, if the reference token refers to a
         *  range of sheets. Must contain the same properties as the parameter
         *  'sheet1Ref'. If omitted, the reference token will refer to a single
         *  sheet specified by 'sheet1Ref'.
         *
         * @returns {TokenArray}
         *  A reference to this instance.
         */
        this.appendReference = function (cell1Ref, cell2Ref, sheet1Ref, sheet2Ref) {
            var text = generateReference(cell1Ref, cell2Ref, sheet1Ref, sheet2Ref);
            appendToken(new RefToken(text, cell1Ref, cell2Ref, sheet1Ref, sheet2Ref));
            return this;
        };

        /**
         * Appends a new reference token to this token array, that will be
         * built from the passed cell range address.
         *
         * @param {Object} range
         *  The logical address of the cell range to be added as reference
         *  token to this token array.
         *
         * @param {Object} [options]
         *  A map with additional options. The following options are supported:
         *  @param {Number} [options.abs=false]
         *      If set to true, all column and row references will be marked as
         *      absolute (leading dollar signs).
         *  @param {Number} [options.sheet]
         *      The zero-based index of the sheet the reference token will
         *      point to. If omitted, no sheet reference will be inserted into
         *      the reference token (sheet-local reference).
         *
         * @returns {TokenArray}
         *  A reference to this instance.
         */
        this.appendRange = function (range, options) {

            var // whether the range spans entire columns or rows
                colRange = model.isColRange(range),
                rowRange = model.isRowRange(range),
                // the cell references
                abs = Utils.getBooleanOption(options, 'abs', false),
                cell1Ref = { address: range.start, absCol: abs || rowRange, absRow: abs || (colRange && !rowRange) },
                cell2Ref = _.isEqual(range.start, range.end) ? null : { address: range.end, absCol: abs || rowRange, absRow: abs || (colRange && !rowRange) },
                // the sheet reference
                sheet = Utils.getIntegerOption(options, 'sheet'),
                sheetRef = _.isNumber(sheet) ? { sheet: sheet, abs: true } : null;

            return this.appendReference(cell1Ref, cell2Ref, sheetRef);
        };

        /**
         * Appends new reference tokens to this token array, that will be built
         * from the passed cell range addresses. If the passed range array
         * contains more than one range address, the reference tokens will be
         * separated by list operator tokens.
         *
         * @param {Array} ranges
         *  The logical addresses of the cell ranges to be added as reference
         *  tokens to this token array.
         *
         * @param {Object} [options]
         *  A map with additional options. The following options are supported:
         *  @param {Number} [options.abs=false]
         *      If set to true, all column and row references will be marked as
         *      absolute (leading dollar signs).
         *  @param {Number} [options.sheet]
         *      The zero-based index of the sheet the reference tokens will
         *      point to. If omitted, no sheet references will be inserted into
         *      the reference tokens (sheet-local references).
         *  @param {Boolean} [options.parentheses=false]
         *      If set to true, and the passed array contains more than one
         *      range, the entire range list will be enclosed into parentheses.
         *
         * @returns {TokenArray}
         *  A reference to this instance.
         */
        this.appendRangeList = function (ranges, options) {

            var // whether to enclose the range list into parentheses
                parentheses = (ranges.length > 1) && Utils.getBooleanOption(options, 'parentheses', false);

            if (parentheses) { appendToken(new Token('open', '(')); }
            _(ranges).each(function (range, index) {
                if (index > 0) { this.appendListOperator(); }
                this.appendRange(range, options);
            }, this);
            if (parentheses) { appendToken(new Token('close', ')')); }

            return this;
        };

        /**
         * Parses the specified formula string.
         *
         * @param {String} formula
         *  The formula string, without the leading equality sign.
         *
         * @returns {TokenArray}
         *  A reference to this instance.
         */
        this.parseFormula = function (formula) {

            var // the matches of the regular expressions
                matches = null,
                // sheet references, and start and end cell references of cell ranges
                sheet1Ref = null, sheet2Ref = null, cell1Ref = null, cell2Ref = null;

            // returns the sheet index from the passed simple or complex sheet name
            function getSheetRef(simpleSheet, complexSheet) {
                var sheetRef = null,
                    sheetIndex = null;
                if (_.isString(simpleSheet)) {
                    sheetIndex = model.getSheetIndex(simpleSheet);
                    sheetRef = { sheet: (sheetIndex >= 0) ? sheetIndex : simpleSheet, abs: true };
                } else if (_.isString(complexSheet)) {
                    complexSheet = complexSheet.replace(/''/g, '\'');
                    sheetIndex = model.getSheetIndex(complexSheet);
                    sheetRef = { sheet: (sheetIndex >= 0) ? sheetIndex : complexSheet, abs: true };
                }
                return _.isObject(sheetRef) ? sheetRef : null;
            }

            // returns a cell reference descriptor
            function getCellRef(absCol, col, absRow, row) {
                var address = [_.isNumber(col) ? col : SheetUtils.parseColName(col), _.isNumber(row) ? row : SheetUtils.parseRowName(row)];
                return model.isValidAddress(address) ? {
                    address: address,
                    absCol: _.isBoolean(absCol) ? absCol : (absCol === '$'),
                    absRow: _.isBoolean(absRow) ? absRow : (absRow === '$')
                } : null;
            }

            // do not parse if document is still importing (referenced sheets may be missing)
            // store formula text in a bad token which will be reparsed after import is done
            if (!app.isImportFinished()) {
                tokens = [new Token('bad', formula)];
                return this;
            }

            // extract all supported tokens from start of formula string
            tokens = [];
            while (formula.length > 0) {

                // try/finally to be able to continue the while-loop at any place
                try {

                    // unary and binary operators
                    if ((matches = config.RE.OPERATOR.exec(formula))) {
                        tokens.push(new Token('op', matches[0]));
                        continue;
                    }

                    // list/parameter separator
                    if ((matches = config.RE.SEPARATOR.exec(formula))) {
                        tokens.push(new Token('sep', matches[0]));
                        continue;
                    }

                    // opening parenthesis
                    if ((matches = config.RE.OPEN.exec(formula))) {
                        tokens.push(new Token('open', matches[0]));
                        continue;
                    }

                    // closing parenthesis
                    if ((matches = config.RE.CLOSE.exec(formula))) {
                        tokens.push(new Token('close', matches[0]));
                        continue;
                    }

                    // 3D column intervals (before cell references)
                    if ((matches = config.RE.COL_3D_INTERVAL.exec(formula))) {
                        sheet1Ref = getSheetRef(matches[1], matches[3]);
                        sheet2Ref = getSheetRef(matches[2], matches[4]);
                        cell1Ref = getCellRef(matches[5], matches[6], true, 0);
                        cell2Ref = getCellRef(matches[7], matches[8], true, model.getMaxRow());
                        if (_.isNumber(sheet1Ref.sheet) && _.isObject(cell1Ref) && _.isObject(cell2Ref)) {
                            tokens.push(new RefToken(matches[0], cell1Ref, cell2Ref, sheet1Ref, sheet2Ref));
                            continue;
                        }
                        // Otherwise: continue parsing with other RE's, it may be a
                        // combination of name and reference, such as name42:Sheet2!$A:$A,
                        // which has been matched by the current RE for 3D range references.
                    }

                    // 3D row intervals
                    if ((matches = config.RE.ROW_3D_INTERVAL.exec(formula))) {
                        sheet1Ref = getSheetRef(matches[1], matches[3]);
                        sheet2Ref = getSheetRef(matches[2], matches[4]);
                        cell1Ref = getCellRef(true, 0, matches[5], matches[6]);
                        cell2Ref = getCellRef(true, model.getMaxCol(), matches[7], matches[8]);
                        if (_.isNumber(sheet1Ref.sheet) && _.isObject(cell1Ref) && _.isObject(cell2Ref)) {
                            tokens.push(new RefToken(matches[0], cell1Ref, cell2Ref, sheet1Ref, sheet2Ref));
                            continue;
                        }
                        // Otherwise: continue parsing with other RE's, it may be a
                        // combination of name and reference, such as name42:Sheet2!$1:$1,
                        // which has been matched by the current RE for 3D range references.
                    }

                    // 3D range references
                    if ((matches = config.RE.RANGE_3D_REF.exec(formula))) {
                        sheet1Ref = getSheetRef(matches[1], matches[3]);
                        sheet2Ref = getSheetRef(matches[2], matches[4]);
                        cell1Ref = getCellRef(matches[5], matches[6], matches[7], matches[8]);
                        cell2Ref = getCellRef(matches[9], matches[10], matches[11], matches[12]);
                        if (_.isNumber(sheet1Ref.sheet) && _.isObject(cell1Ref) && _.isObject(cell2Ref)) {
                            tokens.push(new RefToken(matches[0], cell1Ref, cell2Ref, sheet1Ref, sheet2Ref));
                            continue;
                        }
                        // Otherwise: continue parsing with other RE's, it may be a
                        // combination of cell reference and name, such as Sheet1:Sheet2!$A$1:name42,
                        // or of name and reference, such as name42:Sheet2!$A1:$A1,
                        // which has been matched by the current RE for 3D range references.
                    }

                    // 3D cell references
                    else if ((matches = config.RE.CELL_3D_REF.exec(formula))) {
                        sheet1Ref = getSheetRef(matches[1], matches[3]);
                        sheet2Ref = getSheetRef(matches[2], matches[4]);
                        cell1Ref = getCellRef(matches[5], matches[6], matches[7], matches[8]);
                        if (_.isNumber(sheet1Ref.sheet) && _.isObject(cell1Ref)) {
                            tokens.push(new RefToken(matches[0], cell1Ref, null, sheet1Ref, sheet2Ref));
                            continue;
                        }
                        // Otherwise: continue parsing with other RE's, it may be the
                        // combination of global name and local name, such as name41:Sheet2!name42,
                        // which has been matched by the current RE for cell references.
                    }

                    // column intervals (before cell references)
                    if ((matches = config.RE.COL_INTERVAL.exec(formula))) {
                        sheet1Ref = getSheetRef(matches[1], matches[2]);
                        cell1Ref = getCellRef(matches[3], matches[4], true, 0);
                        cell2Ref = getCellRef(matches[5], matches[6], true, model.getMaxRow());
                        if (_.isObject(cell1Ref) && _.isObject(cell2Ref)) {
                            tokens.push(new RefToken(matches[0], cell1Ref, cell2Ref, sheet1Ref));
                        } else {
                            tokens.push(new Token('bad', matches[0]));
                        }
                        continue;
                    }

                    // row intervals
                    if ((matches = config.RE.ROW_INTERVAL.exec(formula))) {
                        sheet1Ref = getSheetRef(matches[1], matches[2]);
                        cell1Ref = getCellRef(true, 0, matches[3], matches[4]);
                        cell2Ref = getCellRef(true, model.getMaxCol(), matches[5], matches[6]);
                        if (_.isObject(cell1Ref) && _.isObject(cell2Ref)) {
                            tokens.push(new RefToken(matches[0], cell1Ref, cell2Ref, sheet1Ref));
                        } else {
                            tokens.push(new Token('bad', matches[0]));
                        }
                        continue;
                    }

                    // range references
                    if ((matches = config.RE.RANGE_REF.exec(formula))) {
                        sheet1Ref = getSheetRef(matches[1], matches[2]);
                        cell1Ref = getCellRef(matches[3], matches[4], matches[5], matches[6]);
                        cell2Ref = getCellRef(matches[7], matches[8], matches[9], matches[10]);
                        if (_.isObject(cell1Ref) && _.isObject(cell2Ref)) {
                            tokens.push(new RefToken(matches[0], cell1Ref, cell2Ref, sheet1Ref));
                            continue;
                        }
                        // Otherwise: continue parsing with other RE's, it may be a
                        // combination of cell reference and name, such as Sheet1!$A$1:name42,
                        // which has been matched by the current RE for range references.
                    }

                    // cell references
                    else if ((matches = config.RE.CELL_REF.exec(formula))) {
                        sheet1Ref = getSheetRef(matches[1], matches[2]);
                        cell1Ref = getCellRef(matches[3], matches[4], matches[5], matches[6]);
                        if (_.isObject(cell1Ref)) {
                            tokens.push(new RefToken(matches[0], cell1Ref, null, sheet1Ref));
                            continue;
                        }
                        // Otherwise: continue parsing with other RE's, it may be
                        // a local name, such as Sheet1!name42, which has been
                        // matched by the current RE for cell references.
                    }

                    // number literals
                    if ((matches = config.RE.NUMBER.exec(formula))) {
                        tokens.push(new ValueToken(matches[0], parseFloat(matches[0].replace(config.DEC, '.'))));
                        continue;
                    }

                    // string literals
                    if ((matches = config.RE.STRING.exec(formula))) {
                        tokens.push(new ValueToken(matches[0], matches[0].slice(1, matches[0].length - 1).replace(/""/g, '"')));
                        continue;
                    }

                    // Boolean literals
                    if ((matches = config.RE.BOOLEAN.exec(formula))) {
                        tokens.push(new ValueToken(matches[0], matches[0].toUpperCase() === app.getBooleanLiteral(true)));
                        continue;
                    }

                    // error code literals
                    if ((matches = config.RE.ERROR.exec(formula))) {
                        tokens.push(new ValueToken(matches[0], app.getErrorCodeValue(matches[0])));
                        continue;
                    }

                    // array literals
                    if ((matches = config.RE.ARRAY.exec(formula))) {
                        tokens.push(new Token('array', matches[0]));
                        continue;
                    }

                    // global names and other identifiers
                    if ((matches = config.RE.GLOBAL_NAME.exec(formula))) {
                        tokens.push(new NameToken(matches[0], matches[3], getSheetRef(matches[1], matches[2])));
                        continue;
                    }

                    // function names
                    if ((matches = config.RE.FUNCTION.exec(formula))) {
                        tokens.push(new FunctionToken(matches[0]));
                        continue;
                    }

                    // local names and other identifiers
                    if ((matches = config.RE.NAME.exec(formula))) {
                        tokens.push(new NameToken(matches[0], matches[0]));
                        continue;
                    }

                    // whitespace
                    if ((matches = config.RE.WHITESPACE.exec(formula))) {
                        tokens.push(new Token('ws', matches[0]));
                        continue;
                    }

                    // whitespace
                    if ((matches = config.RE.INVALID_CHAR.exec(formula))) {
                        Utils.warn('TokenArray.parseFormula(): invalid character: "' + matches[0] + '"');
                        tokens.push(new Token('bad', matches[0]));
                        continue;
                    }

                    // error, or other unexpected/unsupported tokens:
                    // push entire remaining formula text as 'bad' token
                    Utils.warn('TokenArray.parseFormula(): invalid formula data: "' + formula + '"');
                    matches = [formula];
                    tokens.push(new Token('bad', matches[0]));
                }

                // remove the current match from the formula string
                finally {
                    formula = formula.slice(matches[0].length);
                }
            }

            // post-process the token array
            Utils.iterateArray(tokens, function (token, index) {

                var // a specific token following the current token
                    nextToken = 0,
                    // a specific token preceding the current token
                    prevToken = 0,
                    // text and character index of a space character
                    spaceText = '', spaceIndex = 0;

                // replace combination of <MINUS> <NUMBER> with a negative number, if
                // the minus is the leading token or preceded by an operator or opening parenthesis
                if ((token.getType() === 'op') && (token.getText() === '-') &&
                    (nextToken = tokenArray.getToken(index + 1, 'value')) && _.isNumber(nextToken.getValue()) && (nextToken.getValue() > 0) &&
                    (!(prevToken = tokenArray.getPrevToken(index)) || (/^(op|open)$/.test(prevToken.getType())))
                ) {
                    tokens[index + 1] = new ValueToken('-' + nextToken.getText(), -nextToken.getValue());
                    tokens.splice(index, 1);
                }

                // insert a range intersection operator for a white-space token
                // containing a SPACE character and surrounded by reference/name
                // tokens or other subexpressions in parentheses
                else if ((token.getType() === 'ws') && ((spaceIndex = (spaceText = token.getText()).indexOf(' ')) >= 0) &&
                    (prevToken = tokenArray.getToken(index - 1)) && (/^(ref|name|close)$/.test(prevToken.getType())) &&
                    (nextToken = tokenArray.getToken(index + 1)) && (/^(ref|name|open|func)$/.test(nextToken.getType()))
                ) {
                    // insert the remaining white-space following the first SPACE character
                    if (spaceIndex + 1 < spaceText.length) {
                        tokens.splice(index + 1, 0, new Token('ws', spaceText.slice(spaceIndex + 1)));
                    }
                    // insert an intersection operator for the SPACE character
                    tokens.splice(index + 1, 0, new Token('op', ' '));
                    // shorten the current white-space token to the text preceding the first SPACE character, or remove it
                    if (spaceIndex > 0) {
                        tokens[index] = new Token('ws', spaceText.slice(0, spaceIndex));
                    } else {
                        tokens.splice(index, 1);
                    }
                }

            }, { reverse: true });

            // notify listeners
            this.trigger('change:tokens');
            return this;
        };

        /**
         * Transforms the passed formula expression according to the specified
         * column or row operation.
         *
         * @param {Object} sourceRange
         *  The logical address of the cell range to be transformed.
         *
         * @param {Object} interval
         *  The column/row interval of the operation, with the index properties
         *  'first' and 'last'.
         *
         * @param {Boolean} insert
         *  Whether the specified column/row interval has been inserted into
         *  the sheet (true), or deleted from the sheet (false).
         *
         * @param {Boolean} columns
         *  Whether the specified interval is a column interval (true), or a
         *  row interval (false).
         *
         * @returns {Boolean}
         *  Whether any token in this token array has been changed.
         */
        this.transformFormula = function (sheet, interval, insert, columns) {
            return invokeTokenMethodAndTrigger('transformToken', sheet, interval, insert, columns);
        };

        /**
         * Relocates this formula to a new reference address by adjusting the
         * relative column and row components in all reference tokens. The
         * string representation of all reference tokens will be adjusted
         * accordingly.
         *
         * @param {Number[]} refAddress
         *  The logical address of the original reference cell the formula is
         *  related to.
         *
         * @param {Number[]} targetAddress
         *  The logical address of the new reference cell.
         *
         * @param {Boolean} [wrapReferences=false]
         *  If set to true, relocated range addresses that are located outside
         *  the sheet range will be wrapped at the sheet borders.
         *
         * @returns {Boolean}
         *  Whether any token in this token array has been changed.
         */
        this.relocateFormula = function (refAddress, targetAddress, wrapReferences) {
            return !_.isEqual(refAddress, targetAddress) && invokeTokenMethodAndTrigger('relocateToken', refAddress, targetAddress, wrapReferences);
        };

        /**
         * Changes all tokens with sheet references containing the specified
         * old sheet index to the new sheet index. Used for example when
         * copying existing sheets, and adjusting all references and defined
         * names that point to the original sheet.
         *
         * @param {Number} oldSheet
         *  The zero-based index of the original sheet.
         *
         * @param {Number} newSheet
         *  The zero-based index of the target sheet.
         *
         * @returns {Boolean}
         *  Whether any token in this token array has been changed.
         */
        this.relocateSheet = function (oldSheet, newSheet) {
            return (oldSheet !== newSheet) && invokeTokenMethodAndTrigger('relocateSheet', oldSheet, newSheet);
        };

        /**
         * Returns the string representation of the formula represented by this
         * token array.
         *
         * @returns {String}
         *  The string representation of the formula.
         */
        this.getFormula = function () {
            return _(tokens).reduce(function (formula, token) { return formula + token.getText(); }, '');
        };

        /**
         * Returns the string representation of the formula represented by this
         * token array, if it would have been relocated according to the passed
         * settings. This token array will not be modified.
         *
         * @param {Number[]} refAddress
         *  The logical address of the original reference cell the formula is
         *  related to.
         *
         * @param {Number[]} targetAddress
         *  The logical address of the new reference cell.
         *
         * @param {Boolean} [wrapReferences=false]
         *  If set to true, relocated range addresses that are located outside
         *  the sheet range will be wrapped at the sheet borders.
         *
         * @returns {String}
         *  The string representation of the relocated formula.
         */
        this.getRelocatedFormula = function (refAddress, targetAddress, wrapReferences) {
            return _(tokens).reduce(function (formula, token) {
                return formula + (_.isFunction(token.generateReference) ? token.generateReference(refAddress, targetAddress, wrapReferences) : token.getText());
            }, '');
        };

        /**
         * Returns the token at the specified array index.
         *
         * @param {Number} index
         *  The array index of the token to be returned.
         *
         * @param {String} [options.type]
         *  If passed, the token must be of the specified type, otherwise this
         *  method returns null.
         *
         * @returns {Token|Null}
         *  The token instance at the specified index; or null if the index is
         *  invalid.
         */
        this.getToken = function (index, type) {
            var token = tokens[index];
            return (token && (!_.isString(type) || (token.getType() === type))) ? token : null;
        };

        /**
         * Returns the token following the token at the specified array index.
         * If that token is a white-space token, it will be skipped, and the
         * token following that white-space token will be returned.
         *
         * @param {Number} index
         *  The array index of the token whose successor will be returned.
         *
         * @param {String} [type]
         *  If passed, the token must be of the specified type, otherwise this
         *  method returns null.
         *
         * @returns {Token|Null}
         *  The formula token next to the token at the specified index; or null
         *  if no matching token could be found.
         */
        this.getNextToken = function (index, type) {
            // adjust index, skip a white-space token
            index += this.getToken(index + 1, 'ws') ? 2 : 1;
            // return the token if it matches the specified type
            return this.getToken(index, type);
        };

        /**
         * Returns the token preceding the token at the specified array index.
         * If that token is a white-space token, it will be skipped, and the
         * token preceding that white-space token will be returned.
         *
         * @param {Number} index
         *  The array index of the token whose predecessor will be returned.
         *
         * @param {String} [type]
         *  If passed, the token must be of the specified type, otherwise this
         *  method returns null.
         *
         * @returns {Token|Null}
         *  The formula token preceding the token at the specified index; or
         *  null if no matching token could be found.
         */
        this.getPrevToken = function (index, type) {
            // adjust index, skip a white-space token
            index -= this.getToken(index - 1, 'ws') ? 2 : 1;
            // return the token if it matches the specified type
            return this.getToken(index, type);
        };

        /**
         * Invokes the passed iterator function for all tokens contained in
         * this token array.
         *
         * @param {Function} iterator
         *  The iterator function invoked for all tokens. Receives the
         *  following parameters:
         *  (1) {Token} token
         *      The current token.
         *  (2) {Number} index
         *      The zero-based token index.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the tokens will be visited in reversed order.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateTokens = function (iterator, options) {
            return Utils.iterateArray(tokens, function (token, index) {
                return iterator.call(this, token, index);
            }, options);
        };

        /**
         * Returns the text cursor position of the specified token.
         *
         * @param {Number} index
         *  The array index of the formula token.
         *
         * @returns {Object|Null}
         *  The text cursor position of the token, in the integer properties
         *  'start' and 'end'; or null, if the passed index is invalid.
         */
        this.getTokenPosition = function (index) {

            var // the text cursor position of the token
                position = { start: 0, end: 0 };

            // validate index
            if ((index < 0) || (index >= tokens.length)) { return null; }

            // iterate to the specified token and calculate text position
            for (var currIndex = 0; currIndex <= index; currIndex += 1) {
                position.start = position.end;
                position.end = position.start + tokens[currIndex].getText().length;
            }

            return position;
        };

        /**
         * Returns information about the token containing or preceding the
         * passed text cursor position in the formula string.
         *
         * @param {Number} pos
         *  The text cursor position in the formula string.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the behavior of this
         *  method. The following options are supported:
         *  @param {Boolean} [options.preferOperand=true]
         *      Specifies how to resolve the situation that the text cursor
         *      position is located between two tokens. If set to true, and one
         *      of the tokens is an operand token (constant value tokens, array
         *      literal tokens, reference tokens, name tokens, or function name
         *      tokens), this token will be returned. Otherwise, always returns
         *      the token preceding the text position.
         *
         * @returns {Object|Null}
         *  A descriptor for the token whose text representation contains the
         *  specified text cursor position, in the following properties:
         *  - {Token} token
         *      The token instance located at the text cursor position.
         *  - {Number} index
         *      The array index of the token.
         *  - {Number} start
         *      The text position of the first character of the token in the
         *      string representation of this token array.
         *  - {Number} end
         *      The text position behind (!) the last character of the token in
         *      the string representation of this token array.
         *  If the text cursor position is located outside of the formula text,
         *  or at its beginning (without the option 'preferOperand'), this
         *  method will return the value null.
         */
        this.getTokenAtPosition = function (pos, options) {

            var // whether to prefer operand tokens
                preferOperand = Utils.getBooleanOption(options, 'preferOperand', false),
                // the length of the formula text already consumed (including the resulting token)
                length = 0,
                // the array index of the token containing or preceding the text cursor position
                index = 0;

            // returns whether the passed token is an operand token
            function isOperand(token) {
                return (/^(value|array|ref|name|func)$/).test(token.getType());
            }

            // cursor is located at the beginning of the formula, and operands are preferred:
            // return the first token (following the text cursor position)
            if (preferOperand && (pos === 0) && (tokens.length > 0) && isOperand(tokens[0])) {
                return { token: tokens[0], index: 0, start: 0, end: tokens[0].getText().length };
            }

            // invalid text position passed
            if (pos <= 0) { return null; }

            // find the token containing or preceding the passed text position
            index = Utils.findFirstIndex(tokens, function (token) {
                length += token.getText().length;
                return pos <= length;
            });
            if (index < 0) { return null; }

            // check if the cursor is located between two tokens, and the trailing is the preferred operand
            if (preferOperand && (pos === length) && (index + 1 < tokens.length) && !isOperand(tokens[index]) && isOperand(tokens[index + 1])) {
                return { token: tokens[index + 1], index: index + 1, start: length, end: length  + tokens[index + 1].getText().length };
            }

            // build and return the complete descriptor for the token
            return { token: tokens[index], index: index, start: length - tokens[index].getText().length, end: length };
        };

        /**
         * Returns information about the innermost function containing the
         * passed text cursor position in the formula string.
         *
         * @param {Number} pos
         *  The text cursor position in the formula string.
         *
         * @returns {Object|Null}
         *  A descriptor for the innermost function containing the specified
         *  text cursor position, in the following properties:
         *  - {String} name
         *      The translated function name.
         *  - {Number} paramIndex
         *      The zero-based index of the function parameter covering the
         *      specified text cursor position.
         *  - {Number} paramCount
         *      The total number of parameters found in the function parameter
         *      list. If the formula is incomplete or contains errors, this
         *      number may be wrong, but it will always be greater than the
         *      value returned in the property 'paramIndex'.
         *  If the text cursor position is outside of any function, this method
         *  will return the value null.
         */
        this.getFunctionAtPosition = function (pos) {

            var // information about the token at the passed position
                tokenInfo = this.getTokenAtPosition(pos, { preferOperand: true }),
                // current index, current token
                index = 0, token = null,
                // the parameter index and count
                param = 0, count = 0;

            // seek to the preceding opening parenthesis, skip embedded pairs of
            // parentheses, return number of skipped separator characters
            function seekToOpeningParenthesis() {
                var seps = 0;
                while (index >= 0) {
                    switch (tokens[index].getType()) {
                    case 'open':
                        return seps;
                    case 'close':
                        index -= 1;
                        seekToOpeningParenthesis();
                        break;
                    case 'sep':
                        seps += 1;
                        break;
                    }
                    index -= 1;
                }
                return 0;
            }

            // seek to the following closing parenthesis, skip embedded pairs of
            // parentheses, return number of skipped separator characters
            function seekToClosingParenthesis() {
                var seps = 0;
                while (index < tokens.length) {
                    switch (tokens[index].getType()) {
                    case 'open':
                        index += 1;
                        seekToClosingParenthesis();
                        break;
                    case 'close':
                        return seps;
                    case 'sep':
                        seps += 1;
                        break;
                    }
                    index += 1;
                }
                return seps;
            }

            // invalid text cursor position passed
            if (!tokenInfo) { return null; }

            // try as long as a function has been found, or the beginning of the formula is reached
            index = tokenInfo.index;
            while (index >= 0) {

                // seek to the preceding opening parenthesis
                param = seekToOpeningParenthesis();

                // check if a function name precedes the parenthesis
                if ((token = this.getToken(index - 1, 'func'))) {
                    index += 1;
                    count = seekToClosingParenthesis() + 1;
                    return { name: token.getName(), paramIndex: param, paramCount: count };
                }

                // continue seeking to preceding opening parenthesis
                index -= 1;
            }

            // cursor is not located inside a function, but it may point to a top-level
            // function name, in that case return this function with invalid parameter index
            if ((tokenInfo.token.getType() === 'func') && (tokens[tokenInfo.index + 1].getType() === 'open')) {
                index = tokenInfo.index + 2;
                count = seekToClosingParenthesis() + 1;
                return { name: tokenInfo.token.getName(), paramIndex: -1, paramCount: count };
            }

            return null;
        };

        /**
         * Resolves the token array to a list of range addresses. The formula
         * represented by this token array must be completely resolvable to a
         * range list (single reference, a list of references separated by the
         * list operator, optionally enclosed in parentheses, a defined name
         * that resolves to a range list, also recursively, or a mixed list of
         * defined names and references). To extract the range addresses
         * contained in an arbitrary formula, the method
         * TokenArray.extractRanges() can be used.
         *
         * @param {Object} [options]
         *  A map with options to control the behavior of this method. The
         *  following options are supported:
         *  @param {Number[]} [options.refAddress]
         *      The source reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as source reference cell.
         *  @param {Number[]} [options.targetAddress]
         *      The target reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as target reference cell.
         *  @param {Boolean} [options.wrapReferences=false]
         *      If set to true, relocated range addresses that are located
         *      outside the sheet range will be wrapped at the sheet borders.
         *
         * @returns {Array|Null}
         *  The logical addresses of the resulting cell ranges. Each range
         *  address object contains the additional property 'sheet' with the
         *  zero-based sheet index of the range. If the formula cannot be
         *  resolved successfully to a range list, this method returns null.
         */
        this.resolveRangeList = function (options) {

            var // the resulting range addresses
                ranges = [];

            // helper guard to prevent endless loops for circular references in defined names
            guardVisitedNames(function (visitNameTokenOnce) {

                var // source reference address for relative references
                    refAddress = Utils.getArrayOption(options, 'refAddress', [0, 0]),
                    // target reference address for relative references
                    targetAddress = Utils.getArrayOption(options, 'targetAddress', [0, 0]),
                    // whether to wrap range addresses at sheet borders
                    wrapReferences = Utils.getBooleanOption(options, 'wrapReferences', false),
                    // a string with a single character reflecting the token types in the token array
                    tokenTypes = '',
                    // whether all tokens were valid
                    validTokens = false;

                // extracts the cell range address from the passed reference token
                function resolveReferenceRange(refToken) {

                    var // the sheet range (different start/end for multi-sheet references)
                        sheetRange = refToken.getSheetRange(),
                        // the cell range address, will be null for invalid ranges (#REF! errors)
                        range = refToken.getRelocatedRange(refAddress, targetAddress, wrapReferences),
                        // whether the token contains a valid range (formula must not contain multi-sheet references)
                        valid = _.isObject(sheetRange) && (sheetRange.sheet1 === sheetRange.sheet2) && _.isObject(range);

                    if (valid) {
                        ranges.push(_(range).extend({ sheet: sheetRange.sheet1 }));
                    }
                    return valid;
                }

                // extracts all cell range addresses from the passed name token
                function resolveNamedRanges(nameToken) {

                    var // the token array with the definition of the name
                        tokenArray = visitNameTokenOnce(nameToken) ? nameToken.getTokenArray() : null,
                        // the resulting ranges referred by the name
                        namedRanges = null;

                    // names are always defined relative to cell A1, and relative references will warp at sheet borders
                    if (tokenArray) {
                        namedRanges = tokenArray.resolveRangeList(Utils.extendOptions(options, { refAddress: [0, 0], wrapReferences: true }));
                    }

                    if (namedRanges) {
                        ranges = ranges.concat(namedRanges);
                        return true;
                    }
                    return false;
                }

                // process all tokens as long as they are considered valid, collect token types
                validTokens = _(tokens).all(function (token) {
                    switch (token.getType()) {
                    case 'ref':
                        tokenTypes += 'R';
                        return resolveReferenceRange(token);
                    case 'name':
                        tokenTypes += 'R';
                        return resolveNamedRanges(token);
                    case 'open':
                        tokenTypes += '<';
                        return true;
                    case 'close':
                        tokenTypes += '>';
                        return true;
                    case 'sep':
                        tokenTypes += ',';
                        return true;
                    case 'ws':
                        return true;
                    default:
                        return false;
                    }
                });

                // on success, check the formula structure
                validTokens = validTokens && /^(R(,R)*|<R(,R)*>)$/.test(tokenTypes);

                // reset resulting ranges if formula structure is invalid
                if (!validTokens) { ranges = null; }

            }, this);

            return ranges;
        };

        /**
         * Returns the addresses of all cell ranges this token array refers to.
         *
         * @param {Object} [options]
         *  A map with options to control the behavior of this method. The
         *  following options are supported:
         *  @param {Number} [options.filterSheet]
         *      If specified, the result will contain only ranges contained in
         *      the sheet with this index.
         *  @param {Number[]} [options.refAddress]
         *      The source reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as source reference cell.
         *  @param {Number[]} [options.targetAddress]
         *      The target reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as target reference cell.
         *  @param {Boolean} [options.wrapReferences=false]
         *      If set to true, relocated range addresses that are located
         *      outside the sheet range will be wrapped at the sheet borders.
         *  @param {Boolean} [options.resolveNames=false]
         *      If set to true, will return the range addresses referred by all
         *      defined names contained in this token array.
         *
         * @returns {Array}
         *  The logical addresses of all cell ranges contained in this token
         *  array with additional information, in an array of objects with the
         *  following properties:
         *  - {Object} sheet1
         *      The zero-based index of the first sheet in the sheet range
         *      contained by the token.
         *  - {Object} sheet2
         *      The zero-based index of the last sheet in the sheet range
         *      contained by the token.
         *  - {Object} range
         *      The range address represented by a formula token.
         *  - {Number} index
         *      The internal array index of the token containing the range
         *      (may be equal for different objects in the array, for example
         *      for a single defined name referring to multiple cell ranges).
         *  - {String} type
         *      The type of the token containing the range. Will be either
         *      'ref' for a regular reference token, or 'name' for a defined
         *      name that evaluates to one or more ranges in the specified
         *      sheet (only if option 'resolveNames' is set, see above).
         */
        this.extractRanges = function (options) {

            var // the resulting ranges
                ranges = [];

            // helper guard to prevent endless loops for circular references in defined names
            guardVisitedNames(function (visitNameTokenOnce) {

                var // the index of the sheet to filter the result
                    filterSheet = Utils.getIntegerOption(options, 'filterSheet'),
                    // source reference address for relative references
                    refAddress = Utils.getArrayOption(options, 'refAddress', [0, 0]),
                    // target reference address for relative references
                    targetAddress = Utils.getArrayOption(options, 'targetAddress', [0, 0]),
                    // whether to wrap range addresses at sheet borders
                    wrapReferences = Utils.getBooleanOption(options, 'wrapReferences', false),
                    // whether to resolve defined name tokens
                    resolveNames = Utils.getBooleanOption(options, 'resolveNames', false);

                // extracts the cell range address from the passed reference token
                function extractReferenceRange(refToken, tokenIndex) {
                    if (!_.isNumber(filterSheet) || refToken.containsSheet(filterSheet)) {
                        var range = refToken.getRelocatedRange(refAddress, targetAddress, wrapReferences); // returns null for invalid ranges (#REF! errors)
                        if (range) { ranges.push(_({ range: range, index: tokenIndex, type: 'ref' }).extend(refToken.getSheetRange())); }
                    }
                }

                // extracts all cell range addresses from the passed name token
                function extractNamedRanges(nameToken, tokenIndex) {
                    var tokenArray = (resolveNames && visitNameTokenOnce(nameToken)) ? nameToken.getTokenArray() : null,
                        rangeInfos = null;
                    if (tokenArray) {
                        // defined names are always defined relative to cell A1, and relative references will warp at sheet borders
                        rangeInfos = tokenArray.extractRanges(Utils.extendOptions(options, { refAddress: [0, 0], wrapReferences: true }));
                        _(rangeInfos).each(function (rangeInfo) { _(rangeInfo).extend({ index: tokenIndex, type: 'name' }); });
                        ranges = ranges.concat(rangeInfos);
                    }
                }

                _(tokens).each(function (token, index) {
                    switch (token.getType()) {
                    case 'ref':
                        extractReferenceRange(token, index);
                        break;
                    case 'name':
                        extractNamedRanges(token, index);
                        break;
                    }
                });
            }, this);

            return ranges;
        };

        // initialization -----------------------------------------------------

        // reparse formula after the document has been imported and all sheets exist
        if (!app.isImportFinished()) {
            this.listenTo(app, 'docs:import:success', function () {
                if ((tokens.length === 1) && (tokens[0].getType() === 'bad')) {
                    tokenArray.parseFormula(tokens[0].getText());
                }
            });
        }

        // refresh all sheet references after the collection of sheets has been changed
        this.listenTo(model, 'insert:sheet:after', function (event, sheet) { invokeTokenMethodAndTrigger('transformSheet', sheet, null); });
        this.listenTo(model, 'delete:sheet:after', function (event, sheet) { invokeTokenMethodAndTrigger('transformSheet', null, sheet); });
        this.listenTo(model, 'move:sheet:after', function (event, to, from) { invokeTokenMethodAndTrigger('transformSheet', to, from); });
        this.listenTo(model, 'rename:sheet', function (event, sheet) { invokeTokenMethodAndTrigger('renameSheet', sheet); });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            model = sheetModel = config = tokens = null;
        });

    } // class TokenArray

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: TokenArray });

});
