/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/app/application',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/io',
     'io.ox/office/editframework/app/editapplication',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/model/model',
     'io.ox/office/spreadsheet/view/view',
     'io.ox/office/spreadsheet/app/controller',
     'gettext!io.ox/office/spreadsheet'
    ], function (Utils, IO, EditApplication, SheetUtils, SpreadsheetModel, SpreadsheetView, SpreadsheetController, gt) {

    'use strict';

    // class SpreadsheetApplication ===========================================

    /**
     * The OX Spreadsheet application.
     *
     * @constructor
     *
     * @extends EditApplication
     *
     * @param {Object} [appOptions]
     *  A map of static application options, that have been passed to the
     *  static method BaseApplication.createLauncher().
     *
     * @param {Object} [launchOptions]
     *  A map of options to control the properties of the application. Supports
     *  all options supported by the base class EditApplication.
     */
    var SpreadsheetApplication = EditApplication.extend({ constructor: function (appOptions, launchOptions) {

        var // self reference
            self = this,

            // the locale data (translated function names etc.)
            localeData = null,

            // the function help resource
            functionHelp = {};

        // base constructor ---------------------------------------------------

        EditApplication.call(this, SpreadsheetModel, SpreadsheetView, SpreadsheetController, appOptions, launchOptions, {
            newDocumentParams: { initial_sheetname: SheetUtils.generateSheetName(1) },
            preProcessHandler: preProcessDocument,
            preProcessProgressSize: 0.1,
            postProcessHandler: postProcessDocument,
            postProcessProgressSize: 0,
            importFailedHandler: prepareInvalidDocument,
            flushHandler: flushHandler,
            realTimeDelay: 10,
            prepareLosingEditRightsHandler: prepareLosingEditRights
        });

        // private methods ----------------------------------------------------

        /**
         * Inserts additional information into the locale data object, after it
         * has been modified.
         */
        function postProcessLocaleData() {
            localeData.invErrorCodes = _.invert(localeData.errorCodes);
            localeData.functionNames = _.chain(localeData.functions).values().sort().value();
        }

        /**
         * Preparation of the document, before its operations will be applied.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  document has been prepared successfully; or rejected on any error.
         */
        function preProcessDocument() {

            var // load function help data asynchronously
                functionHelpPromise = IO.loadResource('io.ox/office/spreadsheet/resource/functions');

            // store function help data, but ignore the success/failure result
            // (application may run even without function help)
            functionHelpPromise.done(function (data) { functionHelp = data; });

            // load the locale data for the current UI language from the server
            return self.sendFilterRequest({
                method: 'POST',
                params: {
                    action: 'query',
                    requestdata: JSON.stringify({
                        type: 'localeData',
                        locale: Utils.LOCALE
                    })
                }
            })
            .then(function (result) {

                // extract locale data settings from response data object
                _(localeData.functions).extend(Utils.getObjectOption(result, 'functions'));
                _(localeData.errorCodes).extend(Utils.getObjectOption(result, 'errorCodes'));
                localeData.decSeparator = Utils.getStringOption(result, 'decSeparator', localeData.decSeparator)[0];
                localeData.listSeparator = Utils.getStringOption(result, 'listSeparator', (localeData.decSeparator === ',') ? ';' : ',');
                localeData.trueLiteral = Utils.getStringOption(result, 'trueLiteral', localeData.trueLiteral).toUpperCase();
                localeData.falseLiteral = Utils.getStringOption(result, 'falseLiteral', localeData.falseLiteral).toUpperCase();
                localeData.engineVersion = Utils.getStringOption(result, 'engineVersion');
                postProcessLocaleData();
                Utils.info('SpreadsheetApplication.preProcessDocument(): new locale data:', localeData);

                // check that the translated function names exist in the function help
                functionHelpPromise.done(function (data) {
                    var supportedFuncs = _(localeData.functions).values(),
                        annotatedFuncs = _(data).keys(),
                        missingFuncs = _(supportedFuncs).difference(annotatedFuncs),
                        unknownFuncs = _(annotatedFuncs).difference(supportedFuncs);
                    if (missingFuncs.length > 0) {
                        missingFuncs.sort();
                        Utils.warn('SpreadsheetApplication.preProcessDocument(): missing help texts for functions:', missingFuncs);
                    }
                    if (unknownFuncs.length > 0) {
                        unknownFuncs.sort();
                        Utils.info('SpreadsheetApplication.preProcessDocument(): help available for unknown functions:', unknownFuncs);
                    }
                });

            }, function () {
                Utils.error('SpreadsheetApplication.preProcessDocument(): could not download locale data for "' + Utils.LOCALE + '"');
                return $.Deferred().reject({ headline: gt('Server Error'), message: gt('The engine used for calculating formulas is not available.') });
            });
        }

        /**
         * Post-processing of the document, after all its import operations
         * have been applied successfully.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  document has been post-processed successfully; or rejected when the
         *  document is invalid, or an error has occurred.
         */
        function postProcessDocument() {
            // document must contain at least one sheet (returning a rejected Deferred
            // object will cause invocation of the prepareInvalidDocument() method)
            return (self.getModel().getSheetCount() === 0) ? $.Deferred().reject() : self.getModel().postProcessDocument();
        }

        /**
         * Will be called when importing the document fails for any reason.
         */
        function prepareInvalidDocument() {
            // document must contain at least one sheet, to prevent if-else
            // statements for empty documents at thousand places in the view code
            self.getModel().prepareInvalidDocument();
        }

        /**
         * Preprocessing before the document will be flushed for downloading,
         * printing, or sending as mail.
         */
        function flushHandler() {
            return self.getView().prepareFlush();
        }

        /**
         * Prepare that the edit mode will be switched to read-only.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  application is prepared to lose the edit rights (e.g. edit mode
         *  will be switched to false).
         */
        function prepareLosingEditRights() {
            // call leaveCellEditMode with cell to ensure that the last data
            // entered is stored before we leave the edit mode.
            self.getView().leaveCellEditMode('cell');
            return $.when();
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the sorted list with the names of all supported functions
         * used in formulas, translated for the current UI language.
         *
         * @returns {Array}
         *  The sorted list with the names of all supported functions.
         */
        this.getFunctionNames = function () {
            return localeData.functionNames;
        };

        /**
         * Returns the name of the specified function, translated for the
         * current UI language.
         *
         * @param {String} englishFuncName
         *  The English name of the function.
         *
         * @returns {String}
         *  The translated name of the specified function if available,
         *  otherwise the passed function name.
         */
        this.getFunctionName = function (englishFuncName) {
            englishFuncName = englishFuncName.toUpperCase();
            return (englishFuncName in localeData.functions) ? localeData.functions[englishFuncName] : englishFuncName;
        };

        /**
         * Returns a help text descriptor for the specified translated function
         * name.
         *
         * @param {String} localFuncName
         *  The translated name of the function.
         *
         * @returns {Object|Null}
         *  A help text descriptor if available, with the following properties:
         *  - {String} description
         *      A text describing the entire formula.
         *  - {String[]} params
         *      An array with the names of all function parameters. Functions
         *      with trailing repeating parameters will be mentioned only once
         *      in this array.
         *  - {String[]} paramshelp
         *      An array describing each parameter more detailed. This array
         *      MUST have the same length as the 'params' array.
         *  - {Number} [varargs]
         *      If contained, the zero-based index of the first parameter that
         *      can be used repeatedly. MUST be less than the length of the
         *      'params' array. If this index does not point to the last array
         *      element, the function expects pairs or even larger tuples of
         *      repeating parameters.
         */
        this.getFunctionHelp = function (localFuncName) {
            localFuncName = localFuncName.toUpperCase();
            return (localFuncName in functionHelp) ? functionHelp[localFuncName] : null;
        };

        /**
         * Returns the sorted list with the names of all supported error codes
         * used in formulas, translated for the current UI language.
         *
         * @returns {Array}
         *  The sorted list with the names of all supported error codes.
         */
        this.getErrorCodeLiterals = function () {
            return _.chain(localeData.errorCodes).values().sort().value();
        };

        /**
         * Returns the name of the specified error code, translated for the
         * current UI language.
         *
         * @param {String} errorCodeValue
         *  The English name of the error code, e.g. '#REF!', '#NUM!' etc.
         *
         * @returns {String}
         *  The translated name of the specified error code if available,
         *  otherwise the unmodified passed string.
         */
        this.getErrorCodeLiteral = function (errorCodeValue) {
            errorCodeValue = errorCodeValue.toUpperCase();
            return (errorCodeValue in localeData.errorCodes) ? localeData.errorCodes[errorCodeValue] : errorCodeValue;
        };

        /**
         * Returns the internal value of the specified translated error code
         * literal.
         *
         * @param {String} errorCode
         *  The translated name of the error code.
         *
         * @returns {String}
         *  The internal English value of the specified error code literal if
         *  available, otherwise the unmodified passed string.
         */
        this.getErrorCodeValue = function (errorCodeLiteral) {
            errorCodeLiteral = errorCodeLiteral.toUpperCase();
            return (errorCodeLiteral in localeData.invErrorCodes) ? localeData.invErrorCodes[errorCodeLiteral] : errorCodeLiteral;
        };

        /**
         * Returns the decimal separator for the current UI language.
         *
         * @returns {String}
         *  The decimal separator for the current UI language.
         */
        this.getDecimalSeparator = function () {
            return localeData.decSeparator;
        };

        /**
         * Returns the list separator for the current UI language. The list
         * separator is used to separate parameters of functions, and for the
         * range list operator.
         *
         * @returns {String}
         *  The list separator for the current UI language.
         */
        this.getListSeparator = function () {
            return localeData.listSeparator;
        };

        /**
         * Returns the string literal of the specified Boolean value for the
         * current UI language.
         *
         * @param {Boolean} value
         *  The Boolean value.
         *
         * @returns {String}
         *  The string literal of the specified Boolean value.
         */
        this.getBooleanLiteral = function (value) {
            return localeData[value ? 'trueLiteral' : 'falseLiteral'];
        };

        /**
         * Returns the version of the server-side Calc Engine.
         *
         * @returns {String}
         *  The version of the Calc Engine.
         */
        this.getEngineVersion = function () {
            return localeData.engineVersion;
        };

        /**
         * Queries the contents of cells referred by the specified cell ranges.
         * A single server request will be sent debounced after collecting
         * multiple queries.
         *
         * @param {Array} ranges
         *  An array of arrays (!) with logical range addresses. Each inner
         *  array of ranges will be resolved to a single list of cell contents.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the behavior of this
         *  method. The following options are supported:
         *  @param {Number} [options.maxCount=1000]
         *      The maximum number of cell values allowed per result list
         *      (originating from an inner array of range addresses, see
         *      parameter 'ranges' above).
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved with the
         *  cell contents referred by the specified ranges. The result will be
         *  an array with the same number of elements as range lists are
         *  contained in the parameter 'ranges', and in the same order. Each
         *  array element is an array of cell content objects, each object will
         *  contain the following properties:
         *  - {String} display
         *      The display string of the cell result.
         *  - {Number|String|Boolean|Null} result
         *      The typed result value. Error codes are represented by strings
         *      starting with the hash character.
         */
        this.queryRangeContents = (function () {

            var // all pending queries (one element per call of this method)
                pendingQueries = [],
                // all range lists to be queried from the server
                sourceRanges = [],
                // maps range list names to array index in 'sourceRanges' to prevent duplicates
                sourceRangesMap = {};

            // returns the name of the passed range with sheet index
            function getRangeName(range) {
                return range.sheet + '!' + SheetUtils.getCellName(range.start) + ':' + SheetUtils.getCellName(range.end || range.start);
            }

            // returns the names of the passed range list with sheet indexes
            function getRangesName(ranges) {
                return _(ranges).map(getRangeName).join(',');
            }

            // registers a new source data query
            function registerQuery(ranges, options) {

                var // the entry in the 'pendingQueries' array for this invocation
                    pendingQuery = {
                        def: $.Deferred(),
                        maxCount: Utils.getIntegerOption(options, 'maxCount', 1000, 1, 1000),
                        indexes: []
                    };

                // add all range lists to 'sourceRanges' and update 'sourceRangesMap'
                Utils.info('SpreadsheetApplication.queryRangeContents(): registering ranges:', '[' + _(ranges).map(getRangesName).join('] [') + ']');
                _(ranges).each(function (rangeList) {

                    var // string representation of the range list, used as map key
                        rangesName = getRangesName(rangeList);

                    // if the range list exists already in 'sourceRanges', store array
                    // index; otherwise store the new range list in the array
                    if (!(rangesName in sourceRangesMap)) {
                        sourceRangesMap[rangesName] = sourceRanges.length;
                        sourceRanges.push(_.copy(rangeList, true));
                    }
                    pendingQuery.indexes.push(sourceRangesMap[rangesName]);
                });

                // store the new query entry, and return a promise that will be resolved with the cell contents
                pendingQueries.push(pendingQuery);

                // add an abort() method to the returned Promise
                return _(pendingQuery.def.promise()).extend({ abort: function () {
                    pendingQuery.def.reject();
                }});
            }

            // sends a single server request for all cached source data queries
            function executeQuery() {

                var // local copy of the queries
                    localQueries = pendingQueries,
                    // the request data to be sent to the server
                    requestData = { type: 'sourceData', locale: Utils.LOCALE, ranges: sourceRanges },
                    // the server request
                    request = null;

                // reset cache variables (collect new queries while server request is running)
                pendingQueries = [];
                sourceRanges = [];
                sourceRangesMap = {};

                // send the server request
                Utils.info('SpreadsheetApplication.queryRangeContents(): sending server request:', requestData);
                request = self.sendFilterRequest({
                    method: 'POST',
                    params: { action: 'query', requestdata: JSON.stringify(requestData) },
                    resultFilter: function (data) {
                        Utils.info('SpreadsheetApplication.queryRangeContents(): server response received:', data);
                        var contents = Utils.getArrayOption(data, 'contents');
                        return (_.isArray(contents) && (contents.length === requestData.ranges.length)) ? contents : undefined;
                    }
                });

                // process response data on success
                request.done(function (allContents) {

                    // resolve each registered query with the correct result arrays
                    _(localQueries).each(function (localQuery) {

                        var // the complete result (array of arrays) to be passed to the Deferred object of this query
                            queryResults = [];

                        // check whether the request has been aborted already
                        if (localQuery.def.state() !== 'pending') { return; }

                        // localQuery.indexes contains the array indexes into the 'allContents' array
                        _(localQuery.indexes).each(function (index) {

                            var // the unpacked result array for a single range list
                                result = [];

                            // unpack a result array from the server result into 'result'
                            _(allContents[index]).any(function (cellContents) {
                                var entry = { display: cellContents.display, result: cellContents.result },
                                    repeat = Math.min(Utils.getIntegerOption(cellContents, 'count', 1), localQuery.maxCount - result.length);
                                _(repeat).times(function () { result.push(entry); });
                                return result.length === localQuery.maxCount;
                            });

                            queryResults.push(result);
                        });

                        // resolve the Deferred object of the current query
                        localQuery.def.resolve(queryResults);
                    });
                });

                // reject all Deferred objects on error
                request.fail(function (response) {
                    Utils.warn('SpreadsheetApplication.queryRangeContents(): server request failed:', response);
                    _(localQueries).each(function (query) {
                        query.def.reject(response);
                    });
                });
            }

            // return the actual debounced queryRangeContents() method
            return self.createDebouncedMethod(registerQuery, executeQuery);
        }());

        // initialization -----------------------------------------------------

        // default settings for the locale data object
        localeData = {
            functions: {},
            errorCodes: { '#DIV/0!': '#DIV/0!', '#N/A': '#N/A', '#NAME?': '#NAME?', '#NULL!': '#NULL!', '#NUM!': '#NUM!', '#REF!': '#REF!', '#VALUE!': '#VALUE!' },
            decSeparator: '.',
            listSeparator: ',',
            trueLiteral: 'TRUE',
            falseLiteral: 'FALSE'
        };
        postProcessLocaleData();

    }}); // class SpreadsheetApplication

    // static methods ---------------------------------------------------------

    /**
     * Replacement for the generic method EditApplication.createLauncher()
     * without parameters, to launch spreadsheet applications.
     */
    SpreadsheetApplication.createLauncher = function () {
        return EditApplication.createLauncher('io.ox/office/spreadsheet', SpreadsheetApplication, { icon: 'icon-table', search: true });
    };

    // exports ================================================================

    return SpreadsheetApplication;

});
