/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Malte Timmermann <malte.timmermann@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/editframework/app/editapplication',
    ['io.ox/files/api',
     'io.ox/office/tk/utils',
     'io.ox/office/tk/config',
     'io.ox/office/tk/dialogs',
     'io.ox/office/baseframework/app/baseapplication',
     'io.ox/office/baseframework/app/extensionregistry',
     'io.ox/office/baseframework/app/toolbaractions',
     'io.ox/office/editframework/app/rtconnection',
     'gettext!io.ox/office/editframework'
    ], function (FilesAPI, Utils, Config, Dialogs, BaseApplication, ExtensionRegistry, ToolBarActions, RTConnection, gt) {

    'use strict';

    // private static functions ===============================================

    /**
     * Extracts and validates the actions array from the passed data object of
     * a server response.
     *
     * @param {Object} data
     *  The response data object of the server request.
     *
     * @returns {Array|Undefined}
     *  The actions array, if existing, otherwise undefined.
     */
    function getActionsFromData(data) {

        var // the operations array (may be missing completely)
            actions = Utils.getArrayOption(data, 'actions', []);

        // check that all array elements are objects
        return _(actions).all(_.isObject) ? actions : undefined;
    }

    // class EditApplication ==================================================

    /**
     * The base class for all OX Documents applications allowing to edit a
     * document.
     *
     * Triggers the events supported by the base class BaseApplication, and the
     * following additional events:
     * - 'docs:state': When the state of the application has changed. The event
     *      handlers receive the current application state, as also returned by
     *      the method 'EditApplication.getState()'.
     * - 'docs:state:<STATE_ID>': When the state of the application has changed
     *      to the specified identifier <STATE_ID>. Will be triggered after the
     *      generic 'docs:state' event. All identifiers returned by the method
     *      'EditApplication.getState()' are supported. Example: when the state
     *      'offline' is reached, the application will trigger a 'docs:state'
     *      event, and a 'docs:state:offline' event.
     * - 'docs:update': When the application receives an update push message
     *      from the real-time framework. The event handlers receive the
     *      complete data object of the update message.
     *
     * @constructor
     *
     * @extends BaseApplication
     *
     * @param {Function} ModelClass
     *  The constructor function of the document model class. See the
     *  BaseApplication base class constructor for details.
     *
     * @param {Function} ViewClass
     *  The constructor function of the view class. See the BaseApplication
     *  base class constructor for details.
     *
     * @param {Function} ControllerClass
     *  The constructor function of the controller class. See the
     *  BaseApplication base class constructor for details.
     *
     * @param {Object} [appOptions]
     *  A map of static application options, that have been passed to the
     *  static method BaseApplication.createLauncher().
     *
     * @param {Object} [launchOptions]
     *  A map of options passed to the core launcher (the ox.launch() method).
     *  The following options are supported:
     *  @param {String} [launchOptions.action]
     *      Controls how to connect the application to a document file. The
     *      following actions are supported:
     *      - 'load': Tries to load the file described in the launch option
     *          'launchOptions.file'.
     *      - 'new': Tries to create a new document. Optionally, the new file
     *          will be a clone of an existing file described in the launch
     *          option 'launchOptions.templateFile'.
     *      - 'convert': Tries to convert the file described in the launch
     *          option 'launchOptions.templateFile' to a native file format.
     *  @param {Object} [launchOptions.file]
     *      The descriptor of the file to be loaded. Used by the action 'load'
     *      only.
     *  @param {String} [launchOptions.folderId]
     *      The identifier of the folder a new file will be created in. Used by
     *      the actions 'new' and 'convert'.
     *  @param {Object} [launchOptions.templateFile]
     *      The descriptor of an existing file used as template for the new
     *      file. Used by the actions 'new' and 'convert'.
     *  @param {Boolean} [launchOptions.preserveFileName=false]
     *      If set to true, the name of the template file will be preserved
     *      when converting a file to a native file format. Otherwise, a
     *      default file name will be generated. Used by the action 'convert'
     *      only.
     *
     * @param {Object} [initOptions]
     *  A map of options to control the properties of the application. Supports
     *  all properties that are supported by the base class BaseApplication.
     *  Additionally, the following options are supported:
     *  @param {Object} [initOptions.newDocumentParams]
     *      Additional parameters that will be inserted into the AJAX request
     *      sent to create a new empty document on the server.
     *  @param {Function} [initOptions.preProcessHandler]
     *      A function that will be called before the document operations will
     *      be downloaded and applied, with the progress bar already being
     *      visible. Will be called in the context of this application. Must
     *      return a Deferred object that will be resolved or rejected after
     *      the document has been prepared, and that may notify the progress of
     *      the processing operation.
     *  @param {Number} [preProcessProgressSize=0]
     *      The size on the import progress bar used to prepare the document.
     *      Must be non-negative, must be less than or equal to 0.9 (90% of the
     *      progress bar used for preparation). Will be ignored, if no
     *      preprocess callback handler has been specified with the option
     *      'preProcessHandler'.
     *  @param {Function} [initOptions.postProcessHandler]
     *      A function that will be called after the document operations have
     *      been downloaded and applied successfully, and that post-processes
     *      the document contents while the progress bar is still visible. Will
     *      be called in the context of this application. Must return a
     *      Deferred object that will be resolved or rejected after the
     *      document has been post-processed, and that may notify the progress
     *      of the processing operation.
     *  @param {Function} [initOptions.postProcessHandlerStorage]
     *      Load performance: Similar to postProcessHandler. A function that
     *      will be called after the document has been loaded from the local
     *      storage. Will be called in the context of this application. Must
     *      return a Deferred object that will be resolved or rejected after
     *      the document has been post-processed, and that may notify the
     *      progress of the processing operation.
     *  @param {Number} [postProcessProgressSize=0]
     *      The size on the import progress bar used for post-processing the
     *      document. Must be non-negative, must be less than or equal to 0.9
     *      (90% of the progress bar used for post-processing). Will be
     *      ignored, if no post-processing callback handler has been specified
     *      with the option 'postProcessHandler'.
     *  @param {Function} [initOptions.importFailedHandler]
     *      A function that will be called when importing the document has
     *      failed for any reason (either connection problems, or while
     *      applying the operations). Will be called in the context of this
     *      application.
     *  @param {Function} [initOptions.optimizeOperationsHandler]
     *      A function that can be used to optimize the operations before they
     *      are sent to the server. Receives an operations array as first
     *      parameter, and must return an operations array. Will be called in
     *      the context of this application instance.
     *  @param {Number} [initOptions.sendActionsDelay=0]
     *      The duration in milliseconds used to debounce passing new pending
     *      actions to the realtime connection.
     *  @param {Number} [initOptions.realTimeDelay=700]
     *      The duration in milliseconds the realtime framework will collect
     *      messages before sending them to the server.
     *  @param {Boolean} [initOptions.useStorage=false]
     *      Load performance: Whether the local storage shall be used for
     *      storing the document.
     *  @param {Function} [initOptions.prepareLosingEditRightHandler]
     *      A function that will be called if the server sends a request to
     *      end the edit mode. Will be called in the context of this application.
     *      Must return a Deferred object that will be resolved/rejected
     *      as soon as the application can savely switch to read-only mode.
     */
    var EditApplication = BaseApplication.extend({ constructor: function (ModelClass, ViewClass, ControllerClass, appOptions, launchOptions, initOptions) {

        var // self reference
            self = this,

            // the document model instance
            model = null,

            // application view: contains panes, tool bars, etc.
            view = null,

            // the unique identifier of this application
            clientId = null,

            // connection to the realtime framework
            rtConnection = null,

            // buffer for actions not yet sent to the server
            actionsBuffer = [],

            // new action collecting multiple operations in an undo group
            pendingAction = null,

            // whether actions are currently sent to the server
            sendingActions = false,

            // additional parameters for the AJAX request to create a new document
            newDocumentParams = Utils.getObjectOption(initOptions, 'newDocumentParams'),

            // preprocessing of the document before applying import operations
            preProcessHandler = Utils.getFunctionOption(initOptions, 'preProcessHandler'),

            // post-processing of the document after successfully applying import operations
            postProcessHandler = Utils.getFunctionOption(initOptions, 'postProcessHandler'),

            // Load performance: A simplified post process handler, that is called after using document from local storage
            postProcessHandlerStorage = Utils.getFunctionOption(initOptions, 'postProcessHandlerStorage'),

            // post-processing of the document when import has failed
            importFailedHandler = Utils.getFunctionOption(initOptions, 'importFailedHandler', $.noop),

            // callback to flush the document before accessing its source
            flushHandler = Utils.getFunctionOption(initOptions, 'flushHandler', $.noop),

            // callback function that implements merging of actions to reduce data transfer to the server
            optimizeOperationsHandler = Utils.getFunctionOption(initOptions, 'optimizeOperationsHandler'),

            // Load Performance: Only selected operations need to be executed, if document can be loaded from local storage
            operationsFilter = Utils.getFunctionOption(initOptions, 'operationFilter'),

            // application specific delay in milliseconds, until operations are sent to the server
            sendActionsDelay = Utils.getNumberOption(initOptions, 'sendActionsDelay', 0),

            // current application state (see method EditApplication.getState())
            state = null,

            // whether the application is currently connected to the server
            connected = false,

            // cached name of the user currently editing the document
            oldEditUser = null,

            // if any operations where generated by the own document model and sent to the server
            locallyModified = false,

            // if any operations where generated by a remote application
            remotelyModified = false,

            // whether the file has been renamed locally
            locallyRenamed = false,

            // whether application is locked globally (no file permissions)
            locked = false,

            // document file is locked by another user (lock set via OX Drive)
            documentFileIsLocked = false,

            // document file is locked by the user
            documentFileLockedByUser = null,

            // whether application is locked globally (internal unrecoverable error)
            internalError = false,

            // whether the user had edit rights before the offline handler was called
            onlineEditRights = false,

            // whether the connection was at least once offline
            offlineHandlerTriggered = false,

            // the update attributes used by server responses
            updateDataAttributes = ['editUserId', 'editUser', 'hasErrors', 'writeProtected', 'activeClients', 'locked', 'lockedByUser', 'failSafeSaveDone', 'wantsToEditUser', 'wantsToEditUserId'],

            // old edit user id
            oldEditUserId = null,

            // whether we are in a prepare losing edit rights state
            inPrepareLosingEditRights = false,

            // the user who gets the edit rights
            wantsToEditUser = null,

            // whether the file shall be saved in local storage
            saveFileInLocalStorage = Utils.getBooleanOption(initOptions, 'useStorage', false),

            // load document failed
            loadDocumentFailed = false;

        // base constructor ---------------------------------------------------

        BaseApplication.call(this, ModelClass, ViewClass, ControllerClass, importHandler, appOptions, launchOptions, {
            flushHandler: flushDocument
        });

        // private methods ----------------------------------------------------

        /**
         * Extracts the update data from the passed data object of a
         * server response.
         *
         * @param {Object} data
         *  The response data object of a server request.
         *
         * @returns {Object}
         *  An object which contains only the update attributes, if
         *  existing, otherwise undefined.
         */
        function getUpdateAttributesFromData(data) {
            var // the return object
                updateData = {},
                // the found attributes
                attrs = _.intersection(_.keys(data), updateDataAttributes);

            // copy only the known update attributes
            _(attrs).each(function (key) {
                updateData[key] = data[key];
            });

            return updateData;
        }

        /**
         * Recalculates the application state. Triggers a 'docs:state' event,
         * if the application state has been changed.
         */
        function updateState() {

            // calculate the new application state
            var newState = internalError ? 'error' :
                    !connected ? 'offline' :
                    (locked || !model.getEditMode()) ? 'readonly' :
                    self.hasUnsavedChanges() ? 'sending' :
                    locallyModified ? 'ready' :
                    'initial';

            // trigger event if state has been changed
            if (state !== newState) {
                state = newState;
                self.getWindow().nodes.outer.attr('data-app-state', state);
                self.trigger('docs:state', state);
                self.trigger('docs:state:' + state);
            }
        }

        /**
         * Sets this application into the internal error mode.
         */
        function setInternalError() {
            internalError = true;
            connected = false;
            locked = true;
            model.setEditMode(false);
            updateState();
        }

        /**
         * Shows a read-only alert banner according to the current state of the
         * application. If the document is locked due to missing editing
         * permissions, shows an appropriate alert, otherwise shows the name of
         * the current editing user.
         */
        function showReadOnlyAlert(message) {

            var // alert type
                type = 'info',
                // the alert title
                headline = gt('Read-Only Mode');

            // build message text
            if (!message) {
                if (locked) {
                    if (offlineHandlerTriggered) {
                        // Messages due to offline detection
                        if (connected) {
                            message = gt('Connection to server was lost. Please reopen the document to acquire edit rights.');
                        } else {
                            type = 'warning';
                            message = gt('Connection to server was lost. Switched to read-only mode.');
                        }
                    } else if (documentFileIsLocked) {
                        // Messages due to file locking
                        if (documentFileLockedByUser && documentFileLockedByUser.length > 0) {
                            //#. %1$s is the name of the user who has locked the document
                            //#, c-format
                            message = gt('The document has been locked by %1$s.', _.noI18n(documentFileLockedByUser));
                        } else {
                            message = gt('The document has been locked by another user.');
                        }
                    }  else {
                        // General message due to missing permissions
                        message = gt('You do not have permissions to change this document.');
                    }
                } else {
                    if (inPrepareLosingEditRights) {
                        // message if we are in the process to lose the edit rights, the editor is in read-only
                        // mode to prevent endless editing
                        if (wantsToEditUser) {
                            message =
                                //#. %1$s is the name of the user who will get the edit rights
                                //#, c-format
                                gt('Transferring edit rights to %1$s. Your latest changes will be saved now.', _.noI18n(wantsToEditUser));
                        } else {
                            message = gt('Transferring edit rights to another user. Your latest changes will be saved now.');
                        }
                    } else if (oldEditUser) {
                        // message if we lost the edit rights
                        message =
                            //#. %1$s is the name of the user who is currently editing the document
                            //#, c-format
                            gt('%1$s is currently editing this document.', _.noI18n(oldEditUser));
                    } else if (loadDocumentFailed) {
                        message =
                            gt('The document could not be loaded correctly. Editing is not allowed.');
                    } else {
                        message = gt('Another user is currently editing this document.');
                    }
                }
            }

            // show the alert banner
            view.yell(type, message, headline);
        }

        /**
         * Initialization after construction. Will be called once after
         * construction of the application is finished.
         */
        function initHandler() {

            // get references to MVC instances after construction
            model = self.getModel();
            view = self.getView();

            // calculate initial application state
            updateState();

            // update application state when model changes read-only state
            // (this includes all kinds of internal errors)
            model.on('readonly', updateState);

            // check whether editing is globally disabled for this application type
            if (!ExtensionRegistry.supportsEditMode(self.getDocumentType())) {
                locked = true;
                model.setEditMode(false);
                // TODO: replace with generic message, or get specific message from configuration
                showReadOnlyAlert(gt('Editing on Android not supported yet.'));
            }
        }

        /**
         * Handles a lost network connection. Switches the document model to
         * read-only mode.
         */
        function connectionOfflineHandler() {
            offlineHandlerTriggered = true;
            if (connected && !internalError && !self.isInQuit()) {
                onlineEditRights = model.getEditMode();
                connected = false;
                // Currently we don't have a synchronization process for clients
                // therefore being offline means that we have to switch to
                // read-only mode permanently.
                // TODO: must be changed if synchronization for on/offline is available
                locked = true;
                model.setEditMode(false);
                updateState();
                showReadOnlyAlert();
                // more logging for RT to better see who has the edit rights
                RTConnection.log('Connection offline handler called by Realtime!');
            }
        }

        /**
         * Handles if we get a network connection again. Sets the internal
         * connected flag to true if our internal error state is not set.
         */
        function connectionOnlineHandler() {
            // RT-Library sometimes calls online although the state has not changed,
            // therefore we also check the offlineHandlerTriggered flag. See
            // #27831 for more information.
            if (offlineHandlerTriggered && !connected && !internalError && !self.isInQuit()) {
                connected = true;
                updateState();
                showReadOnlyAlert();
                // more logging for RT to better see who has the edit rights
                RTConnection.log('Connection online handler called by Realtime!');
            }
        }

        /**
         * Handles if a connection has been timed out and was reset. The
         * document is set to read-only and the user has to reopen it to
         * edit it again.
         */
        function connectionResetHandler() {
            if (!self.isLocked() && !self.isInQuit()) {
                setInternalError();
                view.yell('warning', gt('Due to a server connection error the editor switched to read-only mode. Please close and reopen the document.'), gt('Connection Error'));
            }
            // more logging for RT to better see who has the edit rights
            RTConnection.log('Reset for connection called by Realtime!');
        }

        /**
         * Handles communication with server failed state.
         */
        function connectionTimeoutHandler() {
            if (!self.isLocked() && !self.isInQuit()) {
                setInternalError();
                view.yell('error', gt('Connection to the server has been lost. Please close and reopen the document.'), gt('Connection Error'));
            }
            // more logging for RT to better see who has the edit rights
            RTConnection.log('Connection time out called by Realtime!');
        }

        /**
         * Handles the situation that we have sent actions to the server but
         * we are not the editor registered at the server. Therefore the server
         * doesn't accept our actions and sends us a hangup notification for
         * reloading. Our document contains changes that are not part of the
         * shared document on the server!
         */
        function connectionHangupHandler() {
            if (!self.isInQuit()) {
                setInternalError();
                view.yell('error', gt('Your document changed on the server. Please close and reopen the document.'), gt('Synchronization Error'));
            }
            // more logging for RT to better see who has the edit rights
            RTConnection.log('Server sent a hangup message due to inconsistency between client/server, client osn = ' + model.getOperationStateNumber());
        }

        /**
         * Handles the situation that the server notifies us that another
         * client wants to receive the edit rights. Therefore we are forced
         * to switch to read-only mode and flush our buffer. If that happend
         * we have to notify the server that the edit rights switch can be
         * accomplished.
         *
         * @param {jQuery.Event} event
         *  The jQuery event object.
         *
         * @param {Object} data
         *  The data sent with the notification.
         */
        function handlePrepareLosingEditRights(event, data) {
            var prepareLosingEditRightsHandler = null, def = $.when();

            function checkForPendingOperations() {

                if (!self.hasUnsavedChanges()) {

                    // just to be sure that nothing is left
                    sendActions().then(function () {
                        // After sending the actions we can acknowledge that
                        // we are ready to lose our edit rights.
                        rtConnection.canLoseEditRights();
                        oldEditUser = wantsToEditUser || oldEditUser;
                        wantsToEditUser = '';
                        inPrepareLosingEditRights = false;
                        // more logging for RT to better see who has the edit rights
                        RTConnection.log('sent canLoseEditRights');
                    });
                } else {
                    // wait again some time and try again
                    self.executeDelayed(checkForPendingOperations, { delay: 2000 });
                }
            }

            if (!self.isInQuit()) {
                inPrepareLosingEditRights = true;
                wantsToEditUser = Utils.getStringOption(data, 'wantsToEditUser', '');
                prepareLosingEditRightsHandler = Utils.getFunctionOption(initOptions, 'prepareLosingEditRightsHandler', null);
                if (_.isFunction(prepareLosingEditRightsHandler)) {
                    // call optional handler that can handle losing the edit mode more application specific
                    def = prepareLosingEditRightsHandler.call(self);
                }

                def.always(function () {
                    model.setEditMode(false);
                    updateState();

                    RTConnection.log('handle PrepareLosingEditRights');
                    showReadOnlyAlert((wantsToEditUser.length > 0) ?
                        //#. %1$s is the user name who has received the edit rights
                        //#, c-format
                        gt('Transferring edit rights to %1$s. Your latest changes will be saved now.', _.noI18n(wantsToEditUser)) :
                        gt('Transferring edit rights to another user. Your latest changes will be saved now.')
                    );

                    self.executeDelayed(checkForPendingOperations, { delay: 1000 });
                });
            }
        }

        /**
         * Handles the situation that this client will receive the
         * edit rights as soon as the transfer from the editor
         * has been completed.
         */
        function handleAcceptedAcquireEditRights() {
            if (!self.isInQuit()) {
                view.yell('info', gt('Edit rights are being transferred to you. Please wait a moment.'));
            }
            // more logging for RT to better see who has the edit rights
            RTConnection.log('handle AccepteAcquireEditRights');
        }

        /**
         * Handles the situation that this client tries to acquire the
         * edit rights but someone else already did and the process is
         * still in progress.
         */
        function handleDeclinedAcquireEditRights() {
            if (!self.isInQuit()) {
                view.yell('warning', gt('Someone else already wants to receive the edit rights for this document. Please try again later.'));
            }
            // more logging for RT to better see who has the edit rights
            RTConnection.log('handle DeclinedAcquireEditRights');
        }

        /**
         * Handles the situation that this client detects that no update
         * has been received since a certain amount of actions were sent
         * to the server.
         * This can happen if the server side code is not able to reply
         * on our requests. Currently we just show a warning to the
         * user.
         */
        function handleNoUpdateFromServer() {
            if (!self.isInQuit()) {
                view.yell('warning', gt('The server does not answer our action requests. May be the connection is unstable or the server side has problems.'));
            }
            // more logging for RT to better see that the server doesn't send updates
            RTConnection.log('handle NoUpdateFromServer');
        }

        /**
         * Updates the edit state of the application according to the passed data of
         * a server notification message.
         *
         * @param {Object} data
         *  The data of a server notification message.
         */
        function applyEditState(data) {

            var // unique identifier of current user with edit rights
                editUserId = Utils.getStringOption(data, 'editUserId', ''),
                // display name of current user with edit rights
                editUser = Utils.getStringOption(data, 'editUser', ''),
                // current state of document edit mode
                oldEditMode = model.getEditMode(),
                // new state of document edit mode
                newEditMode = clientId === editUserId,
                // the file descriptor of this application
                file = self.getFileDescriptor(),
                // client osn
                clientOSN = model.getOperationStateNumber(),
                // server osn
                serverOSN = Utils.getIntegerOption(data, 'serverOSN', -1);

            // no state information or internal error may have been set while request was running
            if (internalError || (_.intersection(_.keys(data), updateDataAttributes).length === 0)) {
                return;
            }

            if (editUserId !== oldEditUserId) {
                // more logging for RT to better see who has the edit rights
                RTConnection.log('Edit rights changed, new editing user id: ' + editUserId);
            }

            // check if we have to update the files view (fail safe save done on server)
            if (Utils.getBooleanOption(data, 'failSafeSaveDone', false)) {
                FilesAPI.propagate('change', file);
            }

            // internal server error: stop editing completely, show permanent error banner
            if (Utils.getBooleanOption(data, 'hasErrors', false)) {
                setInternalError();
                view.yell('error', gt('Synchronization to the server has been lost.'), gt('Server Error'));
                RTConnection.log('Error, synchronization to the server has been lost.');
                return;
            }

            if (!locked) {
                // check if file lacks write permissions on server
                if (Utils.getBooleanOption(data, 'writeProtected', false)) {
                    locked = true;
                    model.setEditMode(false);
                    showReadOnlyAlert();
                } else if (Utils.getBooleanOption(data, 'locked', false)) {
                    documentFileIsLocked = true;
                    documentFileLockedByUser = Utils.getStringOption(data, 'lockedByUser', '');
                    locked = true;
                    model.setEditMode(false);
                    showReadOnlyAlert();
                }
            }

            // update edit mode (not if document is locked due to missing write permissions)
            if (!locked) {

                // set edit mode at document model
                model.setEditMode(newEditMode);

                // Remember old edit user id to check for edit rights instead of using editMode
                // which can be misleading if we in the process of switching the edit rights.
                // There is a time where the client still has edit rights but is in read-only
                // mode to prevent the user from changing the document further. All changes which
                // have been done should be sent to the server in time!
                if (connected && editUserId && (editUserId !== oldEditUserId)) {
                    oldEditUserId = editUserId;
                    oldEditUser = editUser;
                }

                // Show success banner, if edit rights gained after read-only
                // mode, or after connection re-established, but not directly
                // after loading the document (oldEditMode === null).
                if (newEditMode && (oldEditMode === false)) {
                    view.yell('success', gt('You have edit rights.'));
                }

                // Make consistency check if we just have lost the edit rights
                // and our OSN is not same as the server OSN. The editor must
                // have the same OSN as the server, otherwise we lost messages
                // or the sequence of messages was messed up!
                if (connected && !newEditMode && oldEditMode && serverOSN !== -1 && clientOSN !== serverOSN) {
                    setInternalError();
                    view.yell('error', gt('Synchronization between client and server is broken. Switching editor into read-only mode.'), gt('Client Error'));
                    RTConnection.log('Error, synchronization between client and server is broken. Switching editor into read-only mode.');
                }

                // Show warning if edit rights have been lost, or are still missing
                // after connection has been re-established, or if the edit user has changed.
                if (!newEditMode && (!connected || oldEditMode || (editUser !== oldEditUser))) {
                    oldEditUser = editUser;
                    showReadOnlyAlert();
                }
            }

            // application is now in connected state
            connected = true;
            updateState();
        }

        /**
         * Updates the file state of the application according to the passed
         * data of a server notification message.
         *
         * @param {Object} data
         *  The data of a server notification message.
         */
        function applyFileState(data) {

            var // current file descriptor
                file = self.getFileDescriptor(),
                // the new file name
                newFileName = Utils.getStringOption(data, 'fileName', ''),
                // the new file version
                newFileVersion = Utils.getStringOption(data, 'fileVersion', ''),
                // the changed components of the file descriptor
                changedFile = {};

            // file name changed after a 'rename' operation
            if ((newFileName.length > 0) && (newFileName !== Utils.getStringOption(file, 'filename'))) {
                changedFile.filename = newFileName;
            }

            // file version changed after remote client closes the modified file
            if ((newFileVersion.length > 0) && (newFileVersion !== Utils.getStringOption(file, 'version'))) {
                changedFile.version = newFileVersion;
            }

            if (!internalError && !_.isEmpty(changedFile)) {
                self.updateFileDescriptor(changedFile);
                self.getController().update();
                updateState();
            }
        }

        /**
         * Applies actions and their operations passed with an 'update' message
         * received from a remote editor of the current document in a
         * background task.
         *
         * @param {Object} data
         *  The data of a server notification message.
         *
         * @return {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved after all
         *  actions have been applied.
         */
        function applyRemoteActions(data) {

            var // all actions contained in the passed update message data
                actions = getActionsFromData(data);

            if (internalError) {
                return $.Deferred().reject();
            }

            // do nothing, if the update message does not contain any actions
            if (!_.isArray(actions) || (actions.length === 0)) {
                return $.when();
            }

            // more rt debug logging to better find problems
            RTConnection.log('Last remote action for call model.applyActions: ' + JSON.stringify(_.last(actions)));

            return model.applyActions(actions, { external: true, async: true, delay: 10 }).done(function () {
                // remember that actions have been received from the server
                remotelyModified = true;
            });
        }

        /**
         * Applies the passed application state and actions passed with an
         * 'update' push message received from the real-time framework. Actions
         * will be applied in a background loop to prevent unresponsive browser
         * and warning messages from the browser. If actions from older
         * 'update' messages are still being applied, the new message data
         * passed to this method will be cached, and will be processed after
         * the background task for the old actions is finished.
         *
         * @param {Object} data
         *  The message data received from 'update' messages sent by the
         *  real-time framework.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Boolean} [options.notify=false]
         *      If set to true, the application will trigger a 'docs:update'
         *      event with the message data, after the actions have been
         *      applied.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved or rejected
         *  after the message data has been processed.
         */
        var applyUpdateMessageData = this.createSynchronizedMethod(function (data, options) {

            // apply all actions asynchronously
            return applyRemoteActions(data).done(function () {

                // update application state (read-only mode, etc.)
                applyEditState(data);

                // update application state (fileName, fileVersion)
                applyFileState(data);

                // notify all listeners if specified
                if (Utils.getBooleanOption(options, 'notify', false)) {
                    self.trigger('docs:update', data);
                }
            });
        });

        /**
         * Tries to create a new document in the Files application according to
         * the launch options passed to the constructor.
         *
         * @param {Object} [options]
         *  A map with additional options for this method. The following
         *  options are supported:
         *  @param {Boolean} [options.convert=false]
         *      Whether to convert the template file to a native file format.
         *  @param {Object} [options.params]
         *      Additional parameters to be inserted into the AJAX request.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the new
         *  document has been created successfully. Otherwise, the Promise will
         *  be rejected.
         */
        function createNewDocument(options) {

            var // parameters passed to the server request
                requestParams = Utils.getObjectOption(options, 'params', {}),
                // whether to convert the file to a native format
                convert = Utils.getBooleanOption(options, 'convert', false),

                // file options
                folderId = Utils.getStringOption(launchOptions, 'folderId', ''),
                templateFile = Utils.getObjectOption(launchOptions, 'templateFile'),
                preserveFileName = Utils.getBooleanOption(launchOptions, 'preserveFileName', false);

            // folder identifier must exist in launch options
            if (folderId.length === 0) {
                Utils.error('EditApplication.createNewDocument(): cannot create new document without folder identifier');
                return $.Deferred().reject();
            }

            // create the URL options
            _(requestParams).extend({
                action: 'createdefaultdocument',
                folder_id: folderId,
                document_type: self.getDocumentType(),
                initial_filename: /*#. default base file name (without extension) for new OX Documents files (will be extended to e.g. 'unnamed(1).docx') */ gt('unnamed'),
                preserve_filename: preserveFileName,
                convert: convert
            });

            // additional parameters according to creation mode
            if (_.isObject(templateFile)) {
                _(requestParams).extend({
                    template_id: templateFile.id,
                    version: templateFile.version
                });
            }

            // show alert banners after document has been imported successfully
            self.on('docs:import:success', function () {

                var // the message text of the alert banner
                    message = null,
                    // the target folder, may differ from the folder in the launch options
                    targetFolderId = Utils.getStringOption(self.getFileDescriptor(), 'folder_id'),
                    // whether the file has been created in another folder
                    copied = folderId !== targetFolderId;

                if (convert) {
                    message = copied ?
                        //#. %1$s is the new file name
                        //#, c-format
                        gt('Document was converted and stored in your default folder as "%1$s".', _.noI18n(self.getFullFileName())) :
                        //#. %1$s is the new file name
                        //#, c-format
                        gt('Document was converted and stored as "%1$s".', _.noI18n(self.getFullFileName()));
                } else if (copied) {
                    message =
                        //#. %1$s is the file name
                        //#, c-format
                        gt('Document "%1$s" was created in your default folder to allow editing.', _.noI18n(self.getFullFileName()));
                }

                if (message) {
                    view.yell('info', message);
                }
            });

            // handle response
            return handleCreateDocumentResult(
                // send the request
                self.sendRequest({
                    module: BaseApplication.FILTER_MODULE_NAME,
                    params: requestParams
                })
                .done(function (file) {
                    // creation succeeded: set and propagate file descriptor
                    self.registerFileDescriptor(file);
                })
                .promise());
        }

        /**
         * Handles the response of the createDocument request and
         * creates possible error messages.
         *
         * @param {jQuery.Deferred|jQuery.Promise} orgDeferred
         *  The original jQuery.Deferred or jQuery.Promise that receives
         *  the response from the server.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the new
         *  document has been created successfully. Otherwise, the Promise will
         *  be rejected.
         */
        function handleCreateDocumentResult(orgDeferred, options) {
            var // the Deferred object to be returned
                def = $.Deferred(),
                // convert template file
                convertTemplateFile = Utils.getBooleanOption(options, 'convertTemplateFile', false);

            orgDeferred.then(function (data) {
                var // message regarding errors
                    message = null;

                // check result and possible high-level errors
                if (data && data.error) {
                    if (data.error === 'CREATEDOCUMENT_CONVERSION_FAILED_ERROR') {
                        message = gt('Could not create document because the necessary conversion of the file failed.');
                    } else if (data.error === 'CREATEDOCUMENT_CANNOT_READ_TEMPLATEFILE_ERROR') {
                        message = gt('Could not create document because the necessary template file could not be read.');
                    } else if (data.error === 'CREATEDOCUMENT_CANNOT_READ_DEFAULTTEMPLATEFILE_ERROR') {
                        message = gt('Could not create document because the default template file could not be read.');
                    } else if (data.error === 'CREATEDOCUMENT_PERMISSION_CREATEFILE_MISSING_ERROR')  {
                        message = gt('Could not create document due to missing folder permissions.');
                    } else if (data.error === 'CREATEDOCUMENT_QUOTA_REACHED_ERROR') {
                        message = gt('Could not create document because the allowed quota is reached. Please delete some items in order to create new ones.');
                    } else {
                        message = convertTemplateFile ? gt('Could not convert document.') : gt('Could not create document.');
                    }

                    def.reject({
                        cause: convertTemplateFile ? 'convert' : 'create',
                        headline: gt('Server Error'),
                        message: message
                    });
                } else {
                    def.resolve(data);
                }
            }, function () {
                // low-level errors without any hint are handled here
                def.reject({
                    cause: convertTemplateFile ? 'convert' : 'create',
                    headline: gt('Server Error'),
                    message: convertTemplateFile ? gt('Could not convert document.') : gt('Could not create document.')
                });
            });

            return def.promise();
        }

        /**
         * Loads the document described in the current file descriptor.
         *
         * @param {Object} [point]
         *  The save point, if called from fail-restore.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  document has been loaded, or rejected when an error has occurred.
         */
        function importDocument(point) {

            var // the application window
                win = self.getWindow(),
                // progress size for connecting and downloading operations
                connectProgressSize = 0.05,
                // progress size for pre-processing
                preProcessProgressSize = _.isFunction(preProcessHandler) ? Utils.getNumberOption(initOptions, 'preProcessProgressSize', 0, 0, 0.9) : 0,
                // progress size for post-processing
                postProcessProgressSize = _.isFunction(postProcessHandler) ? Utils.getNumberOption(initOptions, 'postProcessProgressSize', 0, 0, 0.9 - preProcessProgressSize) : 0,
                // progress size for applying operations
                operationsProgressSize = 1 - connectProgressSize - preProcessProgressSize - postProcessProgressSize,
                // the footer node in the window blocker element
                footerNode = null,
                // the container element showing the remaining import time
                remainingTimeNode = null,
                // the timer that updates the remaining time label
                remainingTimer = null,
                // whether to show a progress bar (enabled after a short delay)
                showProgressBar = false,
                // size on the progress bar already used by previous tasks
                progressStartOffset = 0,
                // total size currently used on the progress bar
                currentProgressSize = 0,
                // start time of the import process
                importStartTime = 0,
                // start time for processing the document (operations, formatting)
                processingStartTime = 0,
                // the file descriptor of this application
                file = null,
                // Performance: Whether stored data can be used for loading the document
                useStorageData = false,
                // connect update messages
                connectUpdateMessages = [];

            // initialize additional contents of the busy blocker screen
            function initializeBusyBlocker(header, footer, blocker) {

                // store footer node for outside use
                footerNode = footer;

                // on IE, remove stripes (they are displayed too nervous and flickering)
                if (_.browser.IE) {
                    blocker.find('.progress-striped').removeClass('progress-striped');
                }

                // add a warning banner for large documents that will contain the 'Cancel'
                // button if visible, and a label for remaining minutes/seconds
                footer.append(
                    $('<div>').addClass('size-warning-node').append(
                        $('<div>').addClass('alert alert-warning').append(
                            $('<div>').addClass('title').text(gt('Warning')),
                            $('<div>').text(gt('Sorry, your document is very large. It will take some time to be loaded.'))
                        )
                    ).hide(),
                    remainingTimeNode = $('<div>')
                );
            }

            // updates the 'time remaining' label in the busy window
            function updateRemainingTime() {

                var // the current time stamp
                    now = _.now(),
                    // estimated total import time, in 1/1000 seconds
                    totalTime = (currentProgressSize > 0) ? ((now - importStartTime) / currentProgressSize) : 0,
                    // estimated remaining import time, in 1/1000 seconds
                    remainingTime = (1 - currentProgressSize) * totalTime,
                    // estimated remaining import time, rounded to blocks of 5 seconds
                    remainingSeconds = Math.round(remainingTime / 5000) * 5,
                    // estimated remaining import time, rounded to minutes
                    remainingMinutes = Math.round(remainingSeconds / 60);

                if (remainingMinutes >= 2) {
                    remainingTimeNode.text(gt.format(
                        //#. %1$d is the number of minutes remaining to import a document
                        //#, c-format
                        gt.ngettext('%1$d minute remaining', '%1$d minutes remaining', remainingMinutes),
                        _.noI18n(remainingMinutes)
                    ));
                } else if (remainingSeconds > 0) {
                    remainingTimeNode.text(gt.format(
                        //#. %1$d is the number of seconds remaining to import a document
                        //#, c-format
                        gt.ngettext('%1$d second remaining', '%1$d seconds remaining', remainingSeconds),
                        _.noI18n(remainingSeconds)
                    ));
                } else {
                    remainingTimeNode.empty();
                }

                // after at least 10 seconds of processing, show a warning box if remaining time is too high
                if ((now - processingStartTime > 10000) && (remainingTime > 120000)) {
                    footerNode.find('.size-warning-node').show().find('.alert').append(footerNode.find('.cancel-node').show());
                }
            }

            // invokes the passed callback function that must return a Deferred, and updates the progress bar during invocation
            function invokeCallbackWithProgress(callback, progressSize, message, count) {

                var // profiling: start time of the callbck invocation
                    lapTime = _.now();

                function progressHandler(progress) {
                    currentProgressSize = progressStartOffset + progressSize * progress;
                    if (showProgressBar) { win.busy(currentProgressSize); }
                }

                return (_.isFunction(callback) ? callback.call(self) : $.when())
                    .progress(progressHandler)
                    .always(function () {
                        var totalTime = _.now() - lapTime,
                            timePerElem = (count > 0) ? Utils.round(totalTime / count, 0.001) : 0;
                        Utils.info('EditApplication.importDocument(): ' + message + ' in ' + totalTime + 'ms' + ((count > 0) ? (' (' + count + ' elements, ' + timePerElem + 'ms per element)') : ''));
                        progressHandler(1);
                        progressStartOffset += progressSize;
                    });
            }

            // function to collect update data while connecting to/importing document
            function collectUpdateEvent(event, data) {
                connectUpdateMessages.push(data);
                RTConnection.log('Update received during document import');
            }

            // always load the top-level version of the document when restoring (TODO: really?)
            if (_.isObject(point)) {
                self.updateFileDescriptor({ version: 0 });
            }

            // update debug mode from save point
            view.toggleDebugPane(Utils.getBooleanOption(point, 'debugPane', false));
            view.toggleDebugHighlight(Utils.getBooleanOption(point, 'debugHighlight', false));

            // during opened undo group, collect all operations in the same action
            model.getUndoManager().on('undogroup:open', function () {
                pendingAction = createAction();
            });

            // undo group closed: send the pending action to the server
            model.getUndoManager().on('undogroup:close', function () {
                if (pendingAction.operations.length > 0) {
                    registerAction(pendingAction);
                }
                pendingAction = null;
            });

            // collect all operations generated by the document model, send them to the server
            model.on('operations:success', function (event, operations, external) {
                // ignore external operations
                if (!external) {
                    if (_.isObject(pendingAction)) {
                        // if inside an undo group, push all operations into the pending action
                        pendingAction.operations = pendingAction.operations.concat(operations);
                    } else {
                        // otherwise, generate and send a self-contained action
                        registerAction(createAction(operations));
                    }
                }
            });

            // error occurred while applying operations
            model.on('operations:error', function () {
                setInternalError();
                if (self.isImportFinished()) {
                    view.yell('error', gt('An unrecoverable error occurred while modifying the document.'), gt('Internal Error'));
                    RTConnection.log('Error, An unrecoverable error occurred while modifying the document');
                }
            });

            // disable dropping of images onto the application background area
            self.getWindowNode().on('drop', false);

            // show the progress bar, if import takes more than 1.5 seconds
            self.executeDelayed(function () { showProgressBar = true; }, { delay: 1500 });

            // show the busy blocker screen
            view.enterBusy({ showFileName: true, initHandler: initializeBusyBlocker, cancelHandler: function () { self.quit(); } });

            // start a timer that updates the label showing the remaining import time
            importStartTime = _.now();
            remainingTimer = self.repeatDelayed(updateRemainingTime, { delay: 5000, repeatDelay: 1000 });

            // create the real-time connection to the server, register event listeners
            rtConnection = new RTConnection(self, initOptions);
            rtConnection.on({
                online: connectionOnlineHandler,
                offline: connectionOfflineHandler,
                reset: connectionResetHandler,
                timeout: connectionTimeoutHandler
            });

            // Start to listen for updates before we connect, it's possible
            // that we receive updates before the connect promise is resolved.
            rtConnection.on('update', collectUpdateEvent);

            // Logging the document launch time until now
            Utils.info('EditApplication.importDocument(): preparing load of document in ' + (_.now() - self.getLaunchStarttime()) + 'ms');

            file = self.getFileDescriptor();

            // checking, if the file can be loaded from storage (or another source)
            if ((typeof(Storage) !== 'undefined') && (saveFileInLocalStorage) && (_.isFunction(model.setFullModelNode))) {
                // Support for localStorage and sessionStorage
                if (localStorage[file.id + '_' + file.folder_id + '_' + file.version]) {
                    Utils.info('importDocument, loading document from local storage. Key: ' + file.id + '_' + file.folder_id + '_' + file.version + ' Length: ' + localStorage[file.id + '_' + file.folder_id + '_' + file.version].length);
                    useStorageData = true;
                    model.setFullModelNode(localStorage[file.id + '_' + file.folder_id + '_' + file.version]);
                } else {
                    Utils.info('importDocument, file in local storage not found. Key: ' + file.id + '_' + file.folder_id + '_' + file.version);
                }
            }

            // connect to server, receive initial operations of the document
            return invokeCallbackWithProgress(_.bind(rtConnection.connect, rtConnection), connectProgressSize, 'actions downloaded')
                .then(function (data) {
                    // successfully connected to real-time framework, actions downloaded

                    var // the initial action from the server response
                        actions = getActionsFromData(data),
                        // 'update' messages received by answer from connect
                        docUpdateMessages = [],
                        // update data retrieved from connect message
                        updateData = getUpdateAttributesFromData(data),
                        // an optional error code
                        errorCode = Utils.getStringOption(data, 'error', 'NO_ERROR'),
                        // the result passed with a rejected Deferred object
                        errorResult = null;

                    processingStartTime = _.now();

                    // connect sends us the first update event together with the action data
                    docUpdateMessages.push(updateData);

                    // get the own unique client identifier
                    clientId = Utils.getStringOption(data, 'clientId', '');

                    // check error conditions
                    if (Utils.getBooleanOption(data, 'hasErrors', false)) {
                        Utils.error('EditApplication.importDocument(): server side error');
                        errorResult = { cause: 'server' };
                    } else if (errorCode !== 'NO_ERROR') {
                        Utils.error('EditApplication.importDocument(): server side error, error code = ' + errorCode);
                        errorResult = { cause: 'server', error: errorCode };
                    } else if (clientId.length === 0) {
                        Utils.error('EditApplication.importDocument(): missing client identifier');
                        errorResult = { cause: 'noclientid' };
                    } else if (!_.isArray(actions) || (actions.length === 0)) {
                        Utils.error('EditApplication.importDocument(): missing actions');
                        errorResult = { cause: 'noactions' };
                    }

                    // return rejected Deferred object on any error
                    if (_.isObject(errorResult)) {
                        return $.Deferred().reject(errorResult);
                    }

                    // invoke pre-process callback function passed to the constructor of this application
                    return invokeCallbackWithProgress(preProcessHandler, preProcessProgressSize, 'preprocessing finished')
                        .then(function () {

                            var actionOptions = { external: true, async: true, useStorageData: useStorageData };

                            // Load performance: Execute only specified operations during load of document, if the document can be loaded from local storage
                            if (useStorageData && operationsFilter) { _.extend(actionOptions, {filter: operationsFilter}); }

                            // callback function passed to the method invokeCallbackWithProgress()
                            function applyActions() { return model.applyActions(actions, actionOptions); }

                            var // total number of operations, for logging
                                operationCount = Config.isDebug() ? Utils.getSum(actions, function (action) {
                                    return Utils.getArrayOption(action, 'operations', []).length;
                                }) : 0;

                            // apply operations while the entire application pane is detached from the DOM
                            view.detachAppPane();

                            // apply actions at document model asynchronously, update progress bar
                            return invokeCallbackWithProgress(applyActions, operationsProgressSize, 'operations applied', operationCount)
                                .always(function () {
                                    // insert the application pane into the DOM, needed by post-processing
                                    view.attachAppPane();
                                })
                                .then(function () {
                                    // successfully applied all actions

                                    // After successfully applied all actions we must have a valid operation state number.
                                    // Otherwise there is a synchronization error and we have to stop processing the document
                                    // further showing an error message.
                                    if (model.getOperationStateNumber() < 0) {
                                        return $.Deferred().reject({ cause: 'applyactions', headline: gt('Synchronization Error') });
                                    }

                                    // invoke post-process callback function passed to the constructor of this application
                                    return invokeCallbackWithProgress(useStorageData ? postProcessHandlerStorage : postProcessHandler, postProcessProgressSize, 'postprocessing finished')
                                        .then(function () {

                                            // register handlers for hangup and switching edit rights notifications
                                            rtConnection.on({
                                                hangup: connectionHangupHandler,
                                                preparelosingeditrights: handlePrepareLosingEditRights,
                                                declinedacquireeditrights: handleDeclinedAcquireEditRights,
                                                acceptedacquireeditrights: handleAcceptedAcquireEditRights,
                                                noUpdateFromServer: handleNoUpdateFromServer
                                            });

                                            // update application state (read-only mode, etc.)
                                            applyEditState(data);

                                            // apply all 'update' messages collected during import (wait for the
                                            // 'docs:import:success' event, otherwise, methods that work with the
                                            // method BaseApplication.isImportFinished() will not run correctly)
                                            self.on('docs:import:success', function () {
                                                // first apply actions received on connect() answer
                                                RTConnection.log('Applying document update messages, count = ' + docUpdateMessages.length);
                                                _(docUpdateMessages).each(applyUpdateMessageData);
                                                // second apply actions received during connect() and document import
                                                RTConnection.log('Applying update messages collected during connect/import, count = ' + connectUpdateMessages.length);
                                                _(connectUpdateMessages).each(applyUpdateMessageData);
                                                // reconnect 'update' events for direct message processing, forward event
                                                rtConnection.off('update', collectUpdateEvent).on('update', function (event, data) {
                                                    // public method triggers 'docs:update' event
                                                    self.applyUpdateMessageData(data);
                                                });
                                            });

                                        }, function (response) {
                                            // failure: post-processing failed
                                            return _({ cause: 'postprocess' }).extend(response);
                                        });

                                }, function (response) {
                                    // failure: applying actions failed
                                    rtConnection.off('update', collectUpdateEvent);
                                    return _({ cause: 'applyactions' }).extend(response);
                                });

                        }, function (response) {
                            // failure (pre-processing failed): adjust error response
                            return _({ cause: 'preprocess' }).extend(response);
                        });

                }, function (response) {
                    // RT connection rejected
                    return self.isInQuit() ? { cause: 'quit' } : response;
                })
                .then(null, function (response) {

                    var // specific error code sent by the server
                        error = Utils.getStringOption(response, 'error');

                    internalError = true;
                    loadDocumentFailed = true;
                    importFailedHandler.call(self);

                    response = _({}).extend(response);
                    switch (error) {
                    case 'LOADDOCUMENT_FAILED_ERROR':
//                        response.message = gt('Loading document failed.');
                        delete response.message; // fall-back to default error message
                        break;
                    case 'LOADDOCUMENT_CANNOT_RETRIEVE_OPERATIONS_ERROR':
//                        response.message = gt('Document operations could not be generated.');
                        delete response.message; // fall-back to default error message
                        break;
                    case 'LOADDOCUMENT_CANNOT_READ_PASSWORD_PROTECTED_ERROR':
                        response.message = gt('The document is protected with a password.');
                        break;
                    case 'LOADDOCUMENT_COMPLEXITY_TOO_HIGH':
                        response.message = gt('This document exceeds the spreadsheet size and complexity limits.');
                        break;
                    }

                    return response;
                })
                .always(function () {
                    remainingTimer.abort();
                    view.leaveBusy();
                })
                .promise();
        }

        /**
         * Callback executed by the base class BaseApplication to import the
         * document. Creates a new document or converts an existing document,
         * if specified in the launch options.
         *
         * @param {Object} [point]
         *  The save point, if called from fail-restore.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  document has been loaded, or rejected when an error has occurred.
         */
        function importHandler(point) {

            var // the Deferred object for creation of new document or conversion of existing document
                def = null;

            // create new document in OX Files if required
            switch (Utils.getStringOption(launchOptions, 'action', 'load')) {
            case 'load':
                def = $.when();
                break;
            case 'new':
                def = createNewDocument({ params: newDocumentParams });
                break;
            case 'convert':
                def = createNewDocument({ convert: true });
                break;
            default:
                Utils.error('EditApplication.importHandler(): unsupported launch action');
                def = $.Deferred().reject();
            }

            // import the document, if creation/conversion has succeeded
            return def.then(function () { return importDocument(point); }).promise();
        }

        /**
         * Creates and returns a new action containing operations and
         * additional data.
         *
         * @param {Array} [operations]
         *  The operations contained by the new action. If omitted, an empty
         *  array will be inserted into the action.
         *
         * @returns {Object}
         *  The new action.
         */
        function createAction(operations) {
            return { operations: operations || [] };
        }

        /**
         * Sends all actions that are currently stored in the actions buffer.
         * If called repeatedly while still sending actions, the new actions
         * buffered in the meantime will be stored in another internal buffer,
         * and will be sent afterwards, until no new actions have been inserted
         * in the actions buffer anymore.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  last real-time message containing new actions has been
         *  acknowledged, or rejected if any real-time message has failed.
         */
        var sendActions = (function () {

            var // the Deferred object that will be returned to the caller
                resultDef = null;

            // sends the current actions buffer (calls itself recursively)
            function sendActionsBuffer() {
                var tmp;

                if (RTConnection.debug && inPrepareLosingEditRights && (actionsBuffer && actionsBuffer.length > 0)) {
                    // we need to debug operations while we are in the process of switching edit rights
                    tmp = actionsBuffer[actionsBuffer.length - 1].operations;
                    try {
                        if (tmp && tmp.length > 0) {
                            Utils.log('sendActionsBuffer, last action: ' + JSON.stringify(JSON.stringify(tmp[tmp.length - 1])));
                        }
                    } catch (ex) {
                        // do nothing
                    }
                }

                // try to reduce the number of actions and operations that need to be sent to the server
                // testing optimizations because of problems with second undo/redo
                if (optimizeOperationsHandler) {
                    actionsBuffer = optimizeOperationsHandler.call(self, actionsBuffer);
                }

                // send all existing actions
                rtConnection.sendActions(actionsBuffer)
                .done(function () {
                    if (internalError) {
                        resultDef.reject();
                    } else if (actionsBuffer.length === 0) {
                        resultDef.resolve();
                    } else {
                        sendActionsBuffer();
                    }
                })
                .fail(function (response) {
                    // TODO: error handling
                    resultDef.reject(response);
                });

                actionsBuffer = [];
            }

            return function () {

                // a real-time message is currently running: return the current
                // result Deferred object that will be resolved after all messages
                if (sendingActions) { return resultDef.promise(); }

                // internal error: immediately return with rejected Deferred
                if (internalError) { return $.Deferred().reject(); }

                // no new actions: return immediately
                if (actionsBuffer.length === 0) { return $.when(); }

                // repare the result Deferred object (will be kept unresolved until
                // all actions have been sent, also including subsequent iterations
                resultDef = $.Deferred().always(function () {
                    sendingActions = false;
                    updateState();
                });
                sendingActions = true;
                updateState();

                // start sending the actions (will call itself recursively)
                sendActionsBuffer();
                return resultDef.promise();
            };

        }()); // end of local scope for sendActions() method

        /**
         * Registers the passed action to be sent to the server. All registered
         * actions will be collected and sent automatically in a debounced
         * call to the sendActions() method.
         *
         * @param {Object} action
         *  A new action to be sent to the server.
         */
        var registerAction = (function () {

            // direct callback: called every time registerAction() is called, push action to array
            function storeAction(action) {
                if (RTConnection.debug && (action.operations.length > 0)) {
                    // for debug purposes we log actions
                    RTConnection.log('storeAction, last action: ' + JSON.stringify(_.last(action.operations)));
                }

                if (!internalError && (action.operations.length > 0)) {
                    actionsBuffer.push(action);
                    locallyModified = true;
                    updateState();
                }
            }

            // create and return the debounced registerAction() method
            // Using delay of 1000 ms in OX Text, so that sendActions does not disturb very fast following
            // insertText operations. Measurements showed, that this leads to 10 % faster text
            // insertion, resulting in 10 % faster insert 'feeling' for the user. In OX Spreadsheet this
            // delay would only prolong the server response time, and is therefore set to 0 ms.
            return self.createDebouncedMethod(storeAction, sendActions, { delay: sendActionsDelay, maxDelay: 5 * sendActionsDelay });

        }()); // registerAction()

        /**
         * Sends all pending actions and flushes the document.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  real-time message has been acknowledged.
         */
        function flushDocument() {

            // do nothing if application is already in internal error state
            if (internalError) { return $.Deferred().reject(); }

            // call the custom flush handler passed to the constructor, then send all pending actions
            return $.when(flushHandler.call(self)).then(sendActions).then(_.bind(rtConnection.flushDocument, rtConnection));
        }

        /**
         * The handler function that will be called when the application is
         * asked to be closed. If the edited document has unsaved changes, a
         * dialog will be shown asking whether to save or drop the changes, or
         * to continue editing.
         *
         * @returns {jQuery.Deferred}
         *  A deferred that will be resolved if the application can be closed
         *  (either if it is unchanged, or the user has chosen to save or lose
         *  the changes), or will be rejected if the application will continue
         *  to run (user has cancelled the dialog, save operation failed).
         */
        function beforeQuitHandler() {

            var // the deferred returned to the framework
                def = $.Deferred();

            // don't call sendActions when we are offline or in an error state
            if ((state !== 'offline') && (state !== 'error')) {
                // send pending actions to the server
                $.when(flushHandler.call(self)).then(sendActions)
                .done(function () {
                    // still pending actions: ask user whether to close the application
                    if (self.hasUnsavedChanges() && (clientId === oldEditUserId)) {
                        Dialogs.showYesNoDialog({
                            message: gt('This document contains unsaved changes. Do you really want to close?')
                        }).then(
                            function () { def.resolve(); },
                            function () { def.reject(); }
                        );
                    } else {
                        // all actions saved, close application without dialog
                        def.resolve();
                    }
                })
                .fail(function () {
                    def.resolve();  // reject would keep application alive
                });

                // prevent endless waiting for sendActions to complete
                self.executeDelayed(function () {
                    def.resolve({ cause: 'timeout' });
                }, {delay: 5000});

            } else {
                // in case of offline or error don't wait for the promise
                // to be resolved/rejected. This could mean waiting
                // forever.
                def.resolve(); // reject would keep application alive
            }

            return def.promise();
        }

        /**
         * Called before the application will be really closed.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected if the application
         *  can be savely closed.
         */
        function quitHandler() {

            var // the result deferred
                def = null,
                // the application window
                win = self.getWindow(),
                // the file descriptor of this application
                file = self.getFileDescriptor();

            // saving the file also in the local storage
            function saveFileInStorage(version) {

                // Deleting all older versions in cache
                _.each(_.range(version), function (number) {
                    localStorage.removeItem(file.id + '_' + file.folder_id + '_' + number);
                });
                // and finally saving the new content
                localStorage[file.id + '_' + file.folder_id + '_' + version] = model.getFullModelDescription();
                Utils.info('quitHandler, saving document in local storage with key: ' + file.id + '_' + file.folder_id + '_' + version + ' Length: ' +  localStorage[file.id + '_' + file.folder_id + '_' + version].length);

                // checking content:
                // _.each(_.range(version + 1), function (number) {
                //     Utils.info('checking content. Key: ' + file.filename + '_' + number + ' ::: ' + localStorage[file.filename + '_' + number]);
                // });

            }

            // evaluates the result of the 'closedocument' call
            function checkCloseDocumentResult(data) {

                var // the result Deferred object
                    def = $.Deferred(),
                    // the error code in the response
                    errorCode = Utils.getIntegerOption(data, 'errorCode', 0),
                    // cause of a rejected call
                    cause = Utils.getStringOption(data, 'cause', 'unknown'),
                    // possible error text from the server
                    error = Utils.getStringOption(data, 'error', ''),
                    // the message text
                    message = null,
                    // the headline text
                    headline = gt('Server Error'),
                    // file name
                    fullFileName = self.getFullFileName();

                // no error, forward response data to piped callbacks
                if (errorCode === 0) {
                    // saving file in local storage, using the file version returned from the server (if storage can be used)
                    if (data.fileVersion && _.isNumber(parseInt(data.fileVersion, 10)) && (typeof(Storage) !== 'undefined') && (saveFileInLocalStorage) && (_.isFunction(model.getFullModelDescription))) {
                        saveFileInStorage(data.fileVersion);
                    }

                    return data;
                }

                // create error message depending on error code
                switch (errorCode) {
                case 100:
                    message = gt('The document is inconsistent and the server was not able to create a backup file.');
                    break;
                case 101:
                    fullFileName = ExtensionRegistry.createErrorFileName(fullFileName);
                    message =
                        //#. %1$s is the new file name
                        //#, c-format
                        gt('The document is inconsistent and was saved as "%1$s".', _.noI18n(fullFileName));
                    break;
                case 102:
                    message = gt('The document file cannot be found. Saving the document is not possible.');
                    break;
                case 103:
                    message = gt('The document cannot be saved. The allowed quota is reached. Please delete some items in order to save the document.');
                    break;
                default:
                    if (cause === 'timeout') {
                        headline = gt('Connection Error');
                        message = gt('Connection to the server has been lost.');
                    } else {
                        message = gt('An unknown error occurred on the server.');
                    }
                }

                // show the alert banner
                internalError = true;
                locked = true;
                updateState();
                if (model && view) {
                    model.setEditMode(false);
                    view.yell('error', message, headline);

                    // redefine application's quit() method to reject the Deferred object
                    self.quit = (function () {
                        var origMethod = self.quit;
                        return function () {
                            def.reject();
                            this.quit = origMethod;
                            return this.quit();
                        };
                    }());

                    // back to idle to allow final copy&paste
                    win.idle();
                    def.always(function () { win.busy(); });
                } else {
                    def.reject();
                }

                if (error.length) {
                    Utils.log(error);
                }

                return def.promise();
            }

            // called in case the 'closedocument' call has been rejected
            function closeDocumentResultFailed(data) {

                var // the cause of the failure
                    cause = Utils.getStringOption(data, 'cause', 'unknown'),
                    // a possible explanation for the error
                    error = Utils.getStringOption(data, 'error', '');

                // simulate error with unknown error code, TODO: evaluate response?
                // return new deferred from UI to give user a chance to see the error
                return checkCloseDocumentResult({ errorCode: -1, cause: cause, error: error });
            }

            function propagateChangedFile() {
                return FilesAPI.propagate('change', file);
            }

            win.busy();

            // Don't call closeDocument when we are offline, in an error state or
            // the document has not been modified.
            if ((state !== 'offline') && (state !== 'error')) {
                // notify server to save/close the document file
                def = rtConnection.closeDocument().then(checkCloseDocumentResult, closeDocumentResultFailed);
            } else {
                // error/offline: it's not safe to make remote calls therefore simply close rtConnection
                def = $.when();
            }

            // disconnect real-time connection after document is closed
            // ATTENTION: it be MUST ensured that rtConnection.destroy() is called
            // before model, view and controller are destroyed
            // rtConnection implementation relies on this fact, breaking this will
            // result in problems especially on shutdown/closing documents
            if (rtConnection) {
                def.always(function () {
                    rtConnection.destroy();
                    rtConnection = null;
                });
            }

            if (locallyModified || locallyRenamed || remotelyModified) {
                // propagate changed document file to Files application (also in case of an error)
                def = def.then(propagateChangedFile, propagateChangedFile);
            } else if ((Utils.getStringOption(launchOptions, 'action') === 'new') && !('templateFile' in launchOptions)) {
                // delete new document in Files application if no changes have been made at all
                def = def.then(function () { return FilesAPI.remove([file]); });
            }

            return def;
        }

        // methods ------------------------------------------------------------

        /**
         * Returns whether the edited document is an OpenDocument file.
         */
        this.isODF = function () {
            return this.hasFileDescriptor() && ExtensionRegistry.isOpenDocument(this.getFullFileName());
        };

        /**
         * Returns whether the edited document is a template document file.
         */
        this.isTemplateFile = function () {
            return this.hasFileDescriptor() && ExtensionRegistry.isTemplate(this.getFullFileName());
        };

        /**
         * Returns whether the edited document may contain macro scripts.
         */
        this.isScriptable = function () {
            return this.hasFileDescriptor() && ExtensionRegistry.isScriptable(this.getFullFileName());
        };

        /**
         * Returns the own client identifier that is globally unique.
         *
         * @returns {String}
         *  The own client identifier.
         */
        this.getClientId = function () {
            return clientId;
        };

        /**
         * Returns whether we are in the process of switching edit rights or not.
         */
        this.isPrepareLosingEditRights = function () {
            return inPrepareLosingEditRights;
        };

        /**
         * Sends a real-time request to the server and waits for a response.
         *
         * @param {String} action
         *  The identifier of the action sent to the server.
         *
         * @param {Number} [timeout=20000]
         *  The delay time in milliseconds, after the Promise returned by this
         *  method will be rejected, if the query is still pending.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved with the
         *  answer of the server request, or rejected on error or timeout.
         */
        this.sendRealTimeQuery = function (action, timeout) {
            if (!_.isNumber(timeout)) { timeout = 20000; }
            return rtConnection.sendQuery(action, timeout);
        };

        /**
         * Sends an 'addresource' message to register the passed resource
         * identifier, and returns a Deferred object that waits for the server
         * response.
         *
         * @param {String} resourceId
         *  The identifier of the resource to be added.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved with the
         *  response data object, after the response of the 'addresource'
         *  message has arrived.
         */
        this.addResourceId = function (resourceId) {
            return rtConnection.addResourceId(resourceId);
        };

        /**
         * Applies the passed application state, actions, and other application
         * data. Actions will be applied in a background loop to prevent
         * unresponsive browser and warning messages from the browser. If
         * actions from older 'update' messages are still being applied, the
         * new data passed to this method will be cached, and will be processed
         * after the background task for the old actions is finished. The
         * application will trigger a 'docs:update' event with the passed data,
         * after the actions have been applied.
         *
         * @param {Object} data
         *  The message data to be applied. Must be in the exact same format as
         *  also received from 'update' messages sent by the real-time
         *  framework.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved or rejected
         *  after the application data has been processed.
         */
        this.applyUpdateMessageData = function (data) {
            return applyUpdateMessageData(data, { notify: true });
        };

        /**
         * Renames the file currently edited by this application.
         *
         * @param {String} shortName
         *  The new short file name (without extension).
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  file has been renamed successfully; or that will be rejected, if
         *  renaming the file has failed.
         */
        this.rename = function (shortName) {

            var // the result Deferred object
                def = null,
                // the original file name
                oldFileName = this.getFullFileName(),
                // the new file name (trim NPCs and spaces at beginning and end, replace embedded NPCs)
                newFileName = Utils.trimAndCleanString(shortName);

            // empty name (after trimming)
            if (newFileName.length === 0) {
                return $.Deferred().reject();
            }

            // name does not change
            if (newFileName === this.getShortFileName()) {
                return $.when();
            }

            // Bug 27724: immediately update file descriptor
            newFileName += '.' + this.getFileExtension();
            this.updateFileDescriptor({ filename: newFileName });

            // send the server request
            def = this.sendFilterRequest({
                params: {
                    action: 'renamedocument',
                    filename: newFileName
                }
            });

            // filter response by the 'fileChangeResult' property containing an error code
            def = def.then(function (data) {

                var // the change result containing an error code
                    changeResult = Utils.getObjectOption(data, 'fileChangeResult'),
                    // error code (missing if successful)
                    errorCode = Utils.getStringOption(changeResult, 'error', 'NO_ERROR');

                // forward result data on success, otherwise reject with error code
                return (errorCode === 'NO_ERROR') ? data : $.Deferred().reject({ errorCode: errorCode });
            });

            // renaming succeeded: prevent deletion of new empty file, update file descriptor
            def.done(function (data) {
                locallyRenamed = true;
                self.updateFileDescriptor({ filename: Utils.getStringOption(data, 'fileName', oldFileName) });
            });

            // renaming failed: show an appropriate warning message
            def.fail(function (response) {

                var errorCode = Utils.getStringOption(response, 'errorCode'),
                    type = 'warning',
                    message = null,
                    title = null;

                switch (errorCode) {
                case 'RENAMEDOCUMENT_FAILED_ERROR':
                    message = gt('Renaming the document failed.');
                    break;
                case 'RENAMEDOCUMENT_VALIDATION_FAILED_CHARACTERS_ERROR':
                    message = gt('Renaming the document failed. The file name contains invalid characters.');
                    break;
                default:
                    Utils.warn('EditApplication.rename(): unknown error code "' + errorCode + '"');
                    type = 'error';
                    title = gt('Server Error');
                    message = gt('Renaming the document failed.');
                }
                view.yell(type, message, title);

                // back to old file name
                self.updateFileDescriptor({ filename: oldFileName });
            });

            return def.promise();
        };

        /**
         * Returns whether there are actions left that have not been sent
         * successfully to the server.
         *
         * @attention
         *  Do not rename this method. The OX core uses it to decide whether to
         *  show a warning before the browser refreshes or closes the page.
         *
         * @returns {Boolean}
         *  Whether the application currently caches unsaved actions.
         */
        this.hasUnsavedChanges = function () {
            return sendingActions || (actionsBuffer.length > 0) || _.isObject(pendingAction);
        };

        /**
         * Returns the Promise of a Deferred object that waits until all
         * pending actions with their operations have seen sent to the server.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved after all
         *  pending actions have been saved. If no actions are currently
         *  pending, the returned Promise will be resolved already.
         */
        this.saveChanges = function () {
            return sendActions();
        };

        /**
         * Returns the current application state.
         *
         * @returns {String}
         *  A string representing the application state. Will be one of the
         *  following values:
         *  - 'error': internal error, application locked permanently.
         *  - 'offline': network connection missing.
         *  - 'readonly': the document cannot be changed.
         *  - 'initial': the document has not been changed yet.
         *  - 'ready': all local changes have been sent to the server.
         *  - 'sending': local changes are currently being sent to the server.
         */
        this.getState = function () {
            return state;
        };

        /**
         * Returns whether the application is locked for editing. Reasons for
         * the lock may be missing edit permissions for the document file, or
         * an unrecoverable internal application error.
         *
         * @returns {Boolean}
         *  Whether editing the document is locked.
         */
        this.isLocked = function () {
            return internalError || locked;
        };

        /**
         * Tries to acquire edit rights for the document. Edit rights will be
         * updated automatically while synchronizing operations with the
         * server. Multiple calls of this function will be throttled with a
         * delay of 3 seconds (first call will be executed immediately).
         */
        this.acquireEditRights = _.throttle(function () {
            if (!self.isLocked() && !model.getEditMode() && rtConnection) {
                rtConnection.acquireEditRights();
            }
        }, 3000);

        /**
         * Launches the appropriate OX Document edit applications with
         * current document. The old application instance will be closed.
         */
        this.reloadDocument = function () {
            var // the file descriptor of the current document
                file = this.getFileDescriptor(),
                // full filename
                fullFileName = this.getFullFileName(),
                // the configuration of the file extension of the current document
                extensionSettings = ExtensionRegistry.getExtensionSettings(fullFileName),
                // the edit application module identifier
                editModule = Utils.getStringOption(extensionSettings, 'module', ''),
                // quit deferred
                quitDefer = $.Deferred();

            if (editModule.length > 0) {

                // close this application
                _.defer(function () { self.quit().then(function () { quitDefer.resolve(); }); });

                // Wait for the quit of the current instance and
                // lanuch the correct edit application. Otherwise the
                // launcher will detect that an instance with the current
                // document is open and just doesn't do anything!
                quitDefer.then(function () {
                    if (ExtensionRegistry.isNative(fullFileName)) {
                        _.delay(function () { ox.launch(editModule + '/main', { action: 'load', file: file }); }, 1000);
                    } else {
                        Utils.error('EditApplication.reloadDocument(): unknown document type');
                        return;
                    }

                });
            }
        };

        /**
         * Rejects an attempt to edit the document (e.g. due to received
         * keyboard events in read-only mode). The application will show an
         * appropriate warning alert banner to the user, and will update the
         * internal state according to the cause of the rejection.
         *
         * @param {String} [cause='edit']
         *  The cause of the rejection. If omitted, a simple edit attempt has
         *  been rejected, without any further influence on the application
         *  state. If set to 'image', inserting an image into the document has
         *  failed. If set to 'siri', a Siri event has been captured (the
         *  application goes into the internal error mode and will be locked).
         *
         * @returns {EditApplication}
         *  A reference to this instance.
         */
        this.rejectEditAttempt = function (cause) {

            switch (cause) {
            case 'image':
                view.yell('warning', gt('The image could not be inserted.'));
                break;

            case 'siri':
                setInternalError();
                showReadOnlyAlert(gt('Siri is not supported as input method. Please do not use to input text.'));
                break;

            default:
                showReadOnlyAlert();
            }

            return this;
        };

        // debug --------------------------------------------------------------

        /**
         * Triggers the specified event at the realtime connection instance.
         *
         * @attention
         *  Only available for debug purposes.
         */
        this.debugTriggerRealtimeEvent = Config.isDebug() ? function (type) {
            rtConnection.trigger(type);
            return this;
        } : $.noop;

        /**
         * Quits the application with additional debug actions.
         *
         * @attention
         *  Only available for debug purposes.
         *
         * @param {String} type
         *  The debug action type to be performed. The following actions are
         *  supported:
         *  - 'unsaved': Shows the 'unsaved changes' dialog.
         *  - 'error': Simulates a server error while trying to shut down.
         */
        this.debugQuit = Config.isDebug() ? function (type) {
            switch (type) {
            case 'unsaved':
                var originalMethod = self.hasUnsavedChanges;
                self.hasUnsavedChanges = function () { return true; };
                _.defer(function () { self.quit().fail(function () { self.hasUnsavedChanges = originalMethod; }); });
                break;
            case 'error':
                // simulate server error after 2 seconds
                rtConnection.closeDocument = function () {
                    var def = $.Deferred();
                    _.delay(function () { def.reject(); }, 2000);
                    return def;
                };
                _.defer(function () { self.quit(); });
                break;
            }
        } : $.noop;

        // initialization -----------------------------------------------------
        if (RTConnection.debug) {
            // enable our own console for RT debugging by default
            Utils.enableOwnConsole();
        }

        if (saveFileInLocalStorage) { Utils.info('Saving of document in local storage activated!'); }

        // initialization after construction
        this.on('docs:init', initHandler);

        // set handlers (fail-save handler returns data needed when restoring
        // the application after browser refresh)
        this.registerBeforeQuitHandler(beforeQuitHandler)
            .registerQuitHandler(quitHandler)
            .registerFailSaveHandler(function () {
                return { debugPane: view.isDebugPaneVisible(), debugHighlight: view.isDebugHighlight() };
            });

    }}); // class EditApplication

    // static methods ---------------------------------------------------------

    EditApplication.createLauncher = function (moduleName, ApplicationClass, appOptions) {

        // configure the global window tool bar
        if (Utils.getBooleanOption(appOptions, 'search', false)) {
            ToolBarActions.createSearchIcon(moduleName);
        }
        ToolBarActions.createDownloadIcon(moduleName);
        ToolBarActions.createPrintIcon(moduleName);
        ToolBarActions.createMailIcon(moduleName);

        // create and return the application launcher
        return BaseApplication.createLauncher(moduleName, ApplicationClass, appOptions);
    };

    // exports ================================================================

    return EditApplication;

});
