/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/format/mergedborder',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/model/format/color',
     'io.ox/office/editframework/model/format/border'
    ], function (Utils, Color, Border) {

    'use strict';

    // static class MergedBorder ==============================================

    /**
     * Provides helper function to merge several border attributes from a
     * selection containing multiple elements with borders, and to convert
     * between merged borders and an abstract border mode used in GUI border
     * controls.
     */
    var MergedBorder = {};

    /**
     * The names of all border properties in cells, mapped by border mode
     * properties.
     *
     * @constant
     */
    MergedBorder.CELL_BORDER_ATTRIBUTES = {
        left: 'borderLeft',
        right: 'borderRight',
        top: 'borderTop',
        bottom: 'borderBottom',
        insideh: 'borderInsideHor',
        insidev: 'borderInsideVert'
    };

    /**
     * Returns whether the passed merged border object is valid and visible
     * (line style ambiguous or different to 'none', and line width ambiguous
     * or greater than 0).
     *
     * @param {Object} mergedBorder
     *  The merged border object.
     */
    MergedBorder.isVisibleBorder = function (mergedBorder) {
        return _.isObject(mergedBorder) &&
            // border style visible, if string and not 'none' (including 'mixed'), but not if null (mixed visible/invisible)
            _.isString(mergedBorder.style) && (mergedBorder.style !== 'none') &&
            // border width may be ambiguous, but must not be 0)
            (_.isNull(mergedBorder.width) || (_.isNumber(mergedBorder.width) && (mergedBorder.width > 0)));
    };

    /**
     * Merges the passed border objects, and returns a merged border object
     * representing the states of the single border properties.
     *
     * @param {Object} [mergedBorder]
     *  A merged border, an original border attribute value, an empty object,
     *  or undefined.
     *
     * @param {Object} [border]
     *  A border attribute value. If missing, the object passed in the
     *  'mergedBorder' attribute will be returned unmodified. Otherwise, a new
     *  merged border object will be created from both parameters. Must at
     *  least contain the property 'style'.
     *
     * @returns {Object}
     *  A merged border object representing the states of the properties of
     *  both passed borders. Contains the properties 'style' (line style, or
     *  the special values 'mixed' or null), 'color' (color object or null),
     *  and 'width' (number or null).
     */
    MergedBorder.mergeBorders = function (mergedBorder, border) {

        // no border passed: return merged border unmodified
        if (!_.isObject(border)) { return mergedBorder || {}; }

        // create a clone or an empty object
        mergedBorder = _.extend({}, mergedBorder);

        // update the line style
        if (!('style' in mergedBorder)) {
            mergedBorder.style = border.style;
        } else if (Border.isVisibleBorder(border)) {
            // new border is visible:
            // - if merged style is 'none' (mixed visible/invisible), set to null
            // - otherwise, if line styles differ, set to 'mixed'
            // - otherwise (equal line styles), keep line style
            mergedBorder.style = (mergedBorder.style === 'none') ? null : (mergedBorder.style !== border.style) ? 'mixed' : mergedBorder.style;
        } else {
            // new border is invisible:
            // - if merged style is 'none', keep it
            // - otherwise (mixed visible/invisible), set to null
            mergedBorder.style = (mergedBorder.style === 'none') ? 'none' : null;
        }

        // update line width and color, if the border is visible
        if (Border.isVisibleBorder(border)) {
            if (!('width' in mergedBorder)) {
                mergedBorder.width = border.width;
            } else if (mergedBorder.width !== border.width) {
                mergedBorder.width = null;
            }

            if (!('color' in mergedBorder)) {
                mergedBorder.color = border.color;
            } else if (!Color.isEqual(mergedBorder.color, border.color)) {
                mergedBorder.color = null;
            }
        }

        return mergedBorder;
    };

    /**
     * Merges the passed border objects, and returns a merged border object
     * representing the states of the single border properties. The second
     * parameter can already be a merged border.
     *
     * @param {Object} [mergedBorder]
     *  A merged border, an original border attribute value, an empty object,
     *  or undefined.
     *
     * @param {Object} [border]
     *  A border attribute value. If missing, the object passed in the
     *  'mergedBorder' attribute will be returned unmodified. Otherwise, a new
     *  merged border object will be created from both parameters. This can
     *  already be a merged border, so it can contain the style 'mixed' and
     *  the width 'null' means, that it is merged from borders with more than
     *  one different 'width'.
     *
     * @returns {Object}
     *  A merged border object representing the states of the properties of
     *  both passed borders. Contains the properties 'style' (line style, or
     *  the special values 'mixed' or null), 'color' (color object or null),
     *  and 'width' (number or null).
     */
    MergedBorder.mergeMergedBorders = function (mergedBorder, border) {

        // no border passed: return merged border unmodified
        if (!_.isObject(border)) { return mergedBorder || {}; }

        // create a clone or an empty object
        mergedBorder = _.extend({}, mergedBorder);

        // update line width and color, if the border is visible
        if (MergedBorder.isVisibleBorder(border)) {

            if (!('style' in mergedBorder)) {
                mergedBorder.style = (border.style === 'mixed') ? null : border.style;
            } else if (mergedBorder.style !== border.style) {
                mergedBorder.style = null;
            }

            if (!('width' in mergedBorder)) {
                mergedBorder.width = border.width;
            } else if (mergedBorder.width !== border.width) {
                mergedBorder.width = null;
            }

            if (!('color' in mergedBorder)) {
                mergedBorder.color = border.color;
            } else if (!Color.isEqual(mergedBorder.color, border.color)) {
                mergedBorder.color = null;
            }

        }

        return mergedBorder;
    };

    /**
     * Converts the passed merged border attributes to an object describing the
     * visibility states of all borders.
     *
     * @param {Object} borderAttributes
     *  The attribute map containing merged border attributes.
     *
     * @returns {Object}
     *  An object with position keys ('left', 'top', etc.), and Boolean values
     *  or Null values specifying whether the respective borders are visible.
     *  If a merged border attribute in the passed attribute map is ambiguous
     *  (value null or missing), the map value will be null.
     */
    MergedBorder.getBorderMode = function (borderAttributes) {

        var // result mapping border attribute names to boolean values
            borderMode = {};

        _(MergedBorder.CELL_BORDER_ATTRIBUTES).each(function (attrName, propName) {
            var border = borderAttributes[attrName];
            borderMode[propName] = _.isObject(border) ? MergedBorder.isVisibleBorder(border) : null;
        });

        return borderMode;
    };

    /**
     * Converts the passed object describing the visibility states of cell
     * border attributes to a table attribute map containing all border
     * attributes.
     *
     * @param {Object} borderMode
     *  An object with position keys ('left', 'top', etc.), and Boolean values
     *  specifying whether the respective borders are visible. Properties not
     *  contained in this object will not occur in the returned attribute map.
     *  If the passed object contains a single property set to true, the
     *  current visibility state of the respective border attribute will be
     *  toggled.
     *
     * @param {Object} origAttributes
     *  The original attribute map of the target object (table, cell, ...).
     *
     * @returns {Object}
     *  The table/cell attributes containing all border attributes that are
     *  mentioned in the passed object.
     */
    MergedBorder.getBorderAttributes = function (borderMode, origAttributes) {

        var // the resulting border attributes
            borderAttributes = {},
            // whether to toggle the visibility of a single border
            toggle = _.isObject(origAttributes) && (_.size(borderMode) === 1);

        _(borderMode).each(function (visible, propName) {

            var // the name of the attribute
                borderAttrName = MergedBorder.CELL_BORDER_ATTRIBUTES[propName],
                // the original border attribute value or merged border object
                origBorder = origAttributes[borderAttrName];

            visible = toggle ? !MergedBorder.isVisibleBorder(origBorder) : visible;
            borderAttributes[borderAttrName] = visible ? _.clone(Border.SINGLE) : _.clone(Border.NONE);

            // copy existing (unambiguous) values for style, width, and color
            if (_.isObject(origBorder)) {
                if (visible && _.isString(origBorder.style) && (origBorder.style !== 'none') && (origBorder.style !== 'mixed')) {
                    borderAttributes[borderAttrName].style = origBorder.style;
                }
                if (_.isNumber(origBorder.width)) {
                    borderAttributes[borderAttrName].width = origBorder.width;
                }
                if (_.isObject(origBorder.color)) {
                    borderAttributes[borderAttrName].color = origBorder.color;
                }
            }
        });

        return borderAttributes;
    };


    // exports ================================================================

    return MergedBorder;

});
