/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/format/attributedmodel',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/model/modelobject',
     'io.ox/office/editframework/model/format/documentstyles'
    ], function (Utils, ModelObject, DocumentStyles) {

    'use strict';

    // class AttributedModel ==================================================

    /**
     * An abstract model object containing formatting attributes and the
     * reference to a style sheet of a specific attribute family.
     *
     * Triggers the following events:
     * - 'change:attributes': After the explicit attributes or the style sheet
     *      identifier have been changed, either directly by using the method
     *      'AttributedModel.setAttributes()', or indirectly after the style
     *      sheet has been changed. The event handlers receive the new
     *      resulting merged attribute set, and the old merged attribute set.
     * No events will be generated while the application imports the document.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {EditApplication} app
     *  The application instance containing this attributed model object.
     *
     * @param {String} styleFamily
     *  The main attribute family of this attributed model object.
     *
     * @param {Object} [attributes]
     *  An attribute set with initial formatting attributes for the attributed
     *  model object.
     *
     * @param {Object} [options]
     *  A map with additional options controlling the behavior of this
     *  instance. Supports all attributes also supported by the base class
     *  ModelObject, and the following additional options:
     *  @param {String} [options.additionalFamilies]
     *      If specified, a space-separated list of additional attribute
     *      families supported by instances of this attributed model.
     */
    function AttributedModel(app, styleFamily, attributes, options) {

        var // self reference
            self = this,

            // the global style sheets container of the document model
            documentStyles = app.getModel().getDocumentStyles(),

            // the style sheets of the passed style family
            styleSheets = documentStyles.getStyleSheets(styleFamily),

            // additional attribute families passed in the options (not supported by style sheets)
            additionalFamilies = null,

            // all attribute families supported by instances of this class
            supportedFamilies = null,

            // the explicit attributes of this model object
            explicitAttributes = {},

            // the cached merged attributes of this model object
            mergedAttributes = null;

        // base constructor ---------------------------------------------------

        ModelObject.call(this, app, options);

        // private methods ----------------------------------------------------

        /**
         * Recalculates the merged attribute set.
         */
        function calculateMergedAttributes(options) {

            var // the identifier of the style sheet
                styleId = Utils.getStringOption(explicitAttributes, 'styleId', null),
                // remember the old merged attributes
                oldMergedAttributes = mergedAttributes;

            // start with defaults and style sheet attributes
            mergedAttributes = styleSheets.getStyleSheetAttributes(styleId);

            // add default values of additional attribute families
            _(additionalFamilies).each(function (family) {
                mergedAttributes[family] = documentStyles.getStyleSheets(family).getDefaultAttributeValues();
            });

            // add the explicit attributes (protect resulting style sheet identifier)
            styleId = mergedAttributes.styleId;
            documentStyles.extendAttributes(mergedAttributes, explicitAttributes);
            mergedAttributes.styleId = styleId;

            // notify all listeners
            if (Utils.getBooleanOption(options, 'notify', false)) {
                self.trigger('change:attributes', mergedAttributes, oldMergedAttributes);
            }
        }

        /**
         * Recalculates the effective merged attribute set, after a style sheet
         * has been inserted, removed, or changed in the associated style sheet
         * container.
         */
        function styleSheetsHandler() {
            calculateMergedAttributes({ notify: true });
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the explicit attribute set of this model object.
         *
         * @returns {Object}
         *  An (incomplete) attribute set (a map of attribute maps with
         *  name/value pairs, keyed by the attribute families, optionally
         *  containing the style sheet reference in the 'styleId' property),
         *  containing values for all attributes that have been explicitly set.
         */
        this.getExplicitAttributes = function () {
            return _.copy(explicitAttributes, true);
        };

        /**
         * Returns the merged attribute set of this model object.
         *
         * @returns {Object}
         *  A complete attribute set (a map of attribute maps with name/value
         *  pairs, keyed by the attribute families, additionally containing the
         *  style sheet reference in the 'styleId' property), containing values
         *  for all known attributes, collected from the document default
         *  values, the style sheet currently referenced by this model object,
         *  and the explicit attributes of this model object.
         */
        this.getMergedAttributes = function () {
            return mergedAttributes;
        };

        /**
         * Changes and/or removes specific explicit formatting attributes, or
         * the style sheet reference of this model object.
         *
         * @param {Object} attributes
         *  An (incomplete) attribute set (a map of attribute maps with
         *  name/value pairs, keyed by the attribute families, optionally
         *  containing the style sheet reference in the 'styleId' property)
         *  with all formatting attributes to be changed. To clear an explicit
         *  attribute value (thus defaulting to the current style sheet), the
         *  respective value in this attribute set has to be set explicitly to
         *  the null value. Attributes missing in this attribute set will not
         *  be modified.
         *
         * @returns {Boolean}
         *  Whether any attributes have actually been changed.
         */
        this.setAttributes = function (attributes) {

            var // whether any attribute has been changed
                changed = false;

            // adds or removes an attribute value, updates the changed flag
            function changeValue(values, name, value) {
                if (_.isNull(value)) {
                    if (name in values) {
                        delete values[name];
                        changed = true;
                    }
                } else {
                    if (!_.isEqual(values[name], value)) {
                        values[name] = _.isObject(value) ? _.copy(value, true) : value;
                        changed = true;
                    }
                }
            }

            // add/remove the style sheet identifier
            if (_.isString(attributes.styleId) || _.isNull(attributes.styleId)) {
                changeValue(explicitAttributes, 'styleId', attributes.styleId);
            }

            // add/remove the passed explicit attributes of all supported attribute families
            _(supportedFamilies).each(function (family) {

                var // attribute definitions of the current family
                    definitions = documentStyles.getAttributeDefinitions(family),
                    // target attribute map for the current family
                    values = null;

                // check if new attributes exist in the passed attribute set
                if (_.isObject(attributes[family])) {

                    // add an attribute map for the current family
                    values = explicitAttributes[family] || (explicitAttributes[family] = {});

                    // update the attribute map with the new attribute values
                    _(attributes[family]).each(function (value, name) {
                        if (DocumentStyles.isRegisteredAttribute(definitions, name)) {
                            changeValue(values, name, value);
                        }
                    });

                    // remove empty attribute value maps completely
                    if (_.isEmpty(values)) {
                        delete explicitAttributes[family];
                    }
                }
            });

            // recalculate merged attributes and notify all listeners
            if (changed || !mergedAttributes) {
                calculateMergedAttributes({ notify: changed });
            }

            return changed;
        };

        this.destroy = (function () {
            var baseMethod = self.destroy;
            return function () {
                styleSheets.off('triggered', styleSheetsHandler);
                documentStyles = styleSheets = null;
                baseMethod.call(self);
            };
        }());

        // initialization -----------------------------------------------------

        // get attribute families supported by the style sheets
        supportedFamilies = documentStyles.getSupportedFamilies(styleFamily);

        // get additional attribute families passed in the constructor options
        additionalFamilies = supportedFamilies.concat(Utils.getStringOption(options, 'additionalFamilies', '').split(/\s+/));
        additionalFamilies = _.chain(additionalFamilies).filter(function (family) { return family.length > 0; }).unique().without(supportedFamilies).value();

        // add additional attribute families into array of all supported families
        supportedFamilies = supportedFamilies.concat(additionalFamilies);

        // set the attributes passed to the constructor (filters by supported attributes)
        this.setAttributes(_.isObject(attributes) ? attributes : {});

        // listen to style sheet changes, update the merged attributes and notify own listeners
        styleSheets.on('triggered', styleSheetsHandler);

    } // class AttributedModel

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: AttributedModel });

});
