/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/text/view/controls',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/control/radiolist',
     'io.ox/office/editframework/model/format/fonts',
     'io.ox/office/editframework/view/editcontrols',
     'io.ox/office/text/dom',
     'io.ox/office/text/format/lists',
     'io.ox/office/text/view/paragraphstylenames',
     'io.ox/office/text/view/tablestylenames',
     'gettext!io.ox/office/text'
    ], function (Utils, RadioList, Fonts, EditControls, DOM, Lists, PARAGRAPH_STYLE_NAMES_DB, TABLE_STYLE_NAMES_DB, gt) {

    'use strict';

    // static class TextControls ==============================================

    var TextControls = {};

    // class ParagraphStyleChooser ============================================

    /**
     * A style sheet chooser for paragraph style sheets.
     *
     * @constructor
     *
     * @extends EditControls.StyleSheetChooser
     */
    TextControls.ParagraphStyleChooser = EditControls.StyleSheetChooser.extend({ constructor: function (app, options) {

        var // self reference
            self = this,

            // paragraph style sheets
            paragraphStyles = app.getModel().getStyleSheets('paragraph'),

            // paragraph node inserted into the caption of the menu button
            captionParagraph = createParagraphNode('');

        // base constructor ---------------------------------------------------

        EditControls.StyleSheetChooser.call(this, app, 'paragraph', Utils.extendOptions({
            tooltip: gt('Paragraph style'),
            icon: 'docs-pilcrow',
            itemColumns: 4,
            itemCreateHandler: createItemHandler,
            updateCaptionMode: 'none',
            translationDatabase: PARAGRAPH_STYLE_NAMES_DB
        }, options));

        // private methods ----------------------------------------------------

        /**
         * Creates a new paragraph element with the passed label.
         */
        function createParagraphNode(label) {
            return $('<span>').append(DOM.createTextSpan().text(_.noI18n(label)));
        }

        /**
         * Formats the passed paragraph element according to the specified
         * paragraph style sheet.
         */
        function setParagraphFormatting(paragraph, styleId) {
            paragraphStyles.setElementAttributes(paragraph, { styleId: styleId }, { preview: true });
        }

        /**
         * Initializes a new list item according to the paragraph style sheet.
         */
        function createItemHandler(button, options) {

            var // the user data stored at the button (contains style identifier and name)
                styleId = Utils.getStringOption(options, 'value', ''),
                // the paragraph node for the preview
                paragraph = createParagraphNode(Utils.getControlLabel(button));

            button.empty().append(paragraph);
            setParagraphFormatting(paragraph, styleId);
        }

        /**
         * Updates the drop-down menu button.
         */
        function updateHandler(styleId) {

            var // the span element taking the style sheet name
                span = captionParagraph.children('span').first(),
                // the name of the style sheet
                styleName = paragraphStyles.getName(styleId);

            if (_.isString(styleId) && _.isString(styleName)) {
                setParagraphFormatting(captionParagraph, styleId);
                span.text(self.translateStyleName(styleName)).css('color', 'white');
            } else {
                span.text('');
            }
        }

        // initialization -----------------------------------------------------

        this.getMenuButton().children('div.caption').append(captionParagraph);
        this.registerUpdateHandler(updateHandler);

    }}); // class ParagraphStyleChooser

    // class ListStyleChooser =================================================

    TextControls.ListStyleChooser = RadioList.extend({ constructor: function (app, predefinedListStyles, options) {

        var // the collection of fonts of the edited document
            fonts = app.getModel().getFonts(),
            // font attributes for the drop-down list entries
            labelCss = { fontFamily: fonts.getCssFontFamily('Cambria'), fontSize: '18px', minWidth: '18px' };

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            highlight: function (listStyleId) { return listStyleId !== ''; },
            splitValue: Lists.DEFAULT_VALUE,
            updateCaptionMode: 'none',
            itemDesign: 'framed',
            itemGrid: true,
            sorted: true,
            sortFunctor: Utils.getControlValue // sort by list identifiers
        }, options));

        // initialization -----------------------------------------------------

        this.createMenuSection('default');
        _(predefinedListStyles).each(function (listStyle, listStyleId) {
            if (Utils.trimAndCleanString(listStyle.listLabel).length > 0) {
                this.createOptionButton(listStyleId, {
                        sectionId: 'default',
                        label: _.noI18n(listStyle.listLabel),
                        tooltip: _.noI18n(listStyle.tooltip),
                        css: { textAlign: 'center' },
                        labelCss: labelCss,
                        dataValue: listStyle.listKey
                    });
            } else {
                this.createMenuSection('special', { separator: true })
                    .createOptionButton(listStyleId, {
                        sectionId: 'special',
                        label: _.noI18n(listStyle.tooltip),
                        dataValue: listStyle.listKey
                    });
            }
        }, this);

    }}); // class ListStyleChooser

    // class BulletListStyleChooser ===========================================

    TextControls.BulletListStyleChooser = TextControls.ListStyleChooser.extend({ constructor: function (app, options) {

        // base constructor ---------------------------------------------------

        TextControls.ListStyleChooser.call(this, app, Lists.getPredefinedBulletListStyles(), Utils.extendOptions({
            icon: 'docs-list-bullet',
            tooltip: gt('Bullet list'),
            itemColumns: 4
        }, options));

    }}); // class BulletListStyleChooser

    // class NumberedListStyleChooser =========================================

    TextControls.NumberedListStyleChooser = TextControls.ListStyleChooser.extend({ constructor: function (app, options) {

        // base constructor ---------------------------------------------------

        TextControls.ListStyleChooser.call(this, app, Lists.getPredefinedNumberedListStyles(), Utils.extendOptions({
            icon: 'docs-list-numbered',
            tooltip: gt('Numbered list'),
            itemColumns: 4
        }, options));

    }}); // class NumberedListStyleChooser

    // class TableStyleSheetChooser ===========================================

    /**
     * @constructor
     *
     * @extends EditControls.StyleSheetChooser
     */
    TextControls.TableStyleSheetChooser = EditControls.StyleSheetChooser.extend({ constructor: function (app, options) {

        var // self reference
            self = this,

            // table style sheets
            tableStyles = app.getModel().getStyleSheets('table'),

            /**
             * Registers a new table that will be formatted delayed with a
             * specific table style sheet.
             *
             * @param {jQuery} table
             *  The new table to be formatted.
             *
             * @param {String} styleId
             *  The identifier of the table style sheet.
             */
            updateTableFormatting = (function () {

                var // which tables in list items need formatting
                    pendingTables = [],
                    // background loop processing all pending tables
                    formatTimer = null;

                // direct callback: called every time when updateTableFormatting() has been called
                function registerTable(tableNode, styleId) {
                    pendingTables.push({ tableNode: tableNode, styleId: styleId });
                }

                // deferred callback: called once after the specified timeout
                function formatTables() {

                    // check if the background loop is already running
                    if (formatTimer) { return; }

                    // create a new background loop that processes all pending tables
                    formatTimer = app.repeatDelayed(function () {

                        var // table info with table and style identifier of the next table to be formatted
                            tableInfo = null;

                        // find a pending table that is still in the drop-down menu (menu may have been cleared
                        // in the meantime), abort the background loop, if no more tables are available
                        while ((tableInfo = pendingTables.shift()) && !Utils.containsNode(self.getMenuNode(), tableInfo.tableNode)) {}
                        if (!tableInfo) { return Utils.BREAK; }

                        // update the formatting of the table
                        tableStyles.setElementAttributes(tableInfo.tableNode, {
                            styleId: tableInfo.styleId,
                            table: { exclude: ['lastRow', 'lastCol', 'bandsVert'] }
                        }, { preview: true });
                        tableInfo.tableNode.find(DOM.PARAGRAPH_NODE_SELECTOR).css({ margin: 0 });  // no margins in micro tables allowed
                    });

                    // forget reference to the timer, when all tables have been formatted
                    formatTimer.always(function () { formatTimer = null; });
                }

                // create and return the debounced updateTableFormatting() method (with a delay of 10ms per table)
                return app.createDebouncedMethod(registerTable, formatTables, { delay: 10 });

            }()); // updateTableFormatting()

        // base constructors --------------------------------------------------

        EditControls.StyleSheetChooser.call(this, app, 'table', Utils.extendOptions({
            icon: 'docs-table-style',
            tooltip: gt('Table style'),
            itemColumns: 7,
            itemCreateHandler: createItemHandler,
            updateCaptionMode: 'none',
            translationDatabase: TABLE_STYLE_NAMES_DB
        }, options));

        // private methods ----------------------------------------------------

        function createMicroTable(minParaWidth, minParaHeight, cols, rows) {

            var paragraph = DOM.createParagraphNode().css({ minWidth: minParaWidth, minHeight: minParaHeight }),
                cellNode = DOM.createTableCellNode(paragraph, { plain: true }),
                tableRow = $('<tr>').append(cellNode),
                tableNode = $('<table>').append(tableRow);

            _(cols - 1).times(function () { tableRow.append(cellNode.clone()); });
            _(rows - 1).times(function () { tableNode.append(tableRow.clone()); });
            return tableNode;
        }

        /**
         * Formats the passed table element according to the specified table
         * style sheet.
         */
        function setTableFormatting(tableNode, styleId) {
            tableStyles.setElementAttributes(tableNode, { styleId: styleId, table: { exclude: ['lastRow', 'lastCol', 'bandsVert'] } }, { preview: true });
        }

        /**
         * Initializes a new list item according to the table style sheet.
         */
        function createItemHandler(button, options) {

            var // the preview table element
                tableNode = createMicroTable('8px', '4px', 5, 5);

            button.addClass('mini-caption').prepend(tableNode);
            updateTableFormatting(tableNode, Utils.getStringOption(options, 'value', ''));
        }

    }}); // class TableStyleSheetChooser

    // exports ================================================================

    return TextControls;

});
