/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/view',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/config',
     'io.ox/office/tk/control/button',
     'io.ox/office/tk/control/label',
     'io.ox/office/tk/control/textfield',
     'io.ox/office/tk/control/unitfield',
     'io.ox/office/tk/control/radiogroup',
     'io.ox/office/tk/control/radiolist',
     'io.ox/office/baseframework/view/toolbox',
     'io.ox/office/editframework/view/editview',
     'io.ox/office/editframework/view/editcontrols',
     'io.ox/office/editframework/model/format/color',
     'io.ox/office/editframework/model/format/border',
     'io.ox/office/editframework/model/format/mergedborder',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/model/operations',
     'io.ox/office/spreadsheet/view/cellcollection',
     'io.ox/office/spreadsheet/view/statusbar',
     'io.ox/office/spreadsheet/view/gridpane',
     'io.ox/office/spreadsheet/view/headerpane',
     'io.ox/office/spreadsheet/view/cornerpane',
     'io.ox/office/spreadsheet/view/controls',
     'gettext!io.ox/office/spreadsheet',
     'less!io.ox/office/spreadsheet/view/style.less'
    ], function (Utils, Config, Button, Label, TextField, UnitField, RadioGroup, RadioList, ToolBox, EditView, EditControls, Color, Border, MergedBorder, SheetUtils, Operations, CellCollection, StatusBar, GridPane, HeaderPane, CornerPane, SpreadsheetControls, gt) {

    'use strict';

    var // map pointing to neighbor positions for all grid pane positions
        GRID_PANE_INFOS = {
            topLeft:     { colSide: 'left',  rowSide: 'top',    nextColPos: 'topRight',    nextRowPos: 'bottomLeft'  },
            topRight:    { colSide: 'right', rowSide: 'top',    nextColPos: 'topLeft',     nextRowPos: 'bottomRight' },
            bottomLeft:  { colSide: 'left',  rowSide: 'bottom', nextColPos: 'bottomRight', nextRowPos: 'topLeft'     },
            bottomRight: { colSide: 'right', rowSide: 'bottom', nextColPos: 'bottomLeft',  nextRowPos: 'topRight'    }
        },

        // map pointing to neighbor positions for all header pane sides
        HEADER_PANE_INFOS = {
            left:   { nextSide: 'right',  panePos1: 'topLeft',    panePos2: 'bottomLeft',  columns: true  },
            right:  { nextSide: 'left',   panePos1: 'topRight',   panePos2: 'bottomRight', columns: true  },
            top:    { nextSide: 'bottom', panePos1: 'topLeft',    panePos2: 'topRight',    columns: false },
            bottom: { nextSide: 'top',    panePos1: 'bottomLeft', panePos2: 'bottomRight', columns: false }
        },

        // the size of the freeze separator nodes, in pixels
        FREEZE_SIZE = 1,

        // the size of the split separator nodes, in pixels
        SPLIT_SIZE = 2,

        // the highlighted inner size of the split tracking node, in pixels
        TRACKING_SIZE = 4,

        // the additional margin of the split tracking nodes, in pixels
        TRACKING_MARGIN = 3,

        // the position offset of tracking nodes compared to split lines
        TRACKING_OFFSET = (TRACKING_SIZE - SPLIT_SIZE) / 2 + TRACKING_MARGIN,

        // the minimum size of grid/header panes, in pixels
        MIN_PANE_SIZE = 40,

        // default settings for a pane side
        DEFAULT_PANE_SIDE_SETTINGS = { visible: true, offset: 0, size: 0, scrollable: false, showOppositeScroll: false, hiddenSize: 0 },

        // dummy sheet layout data, used before first view update message for a sheet
        DEFAULT_SHEET_DATA = { sheet: -1, usedCols: 0, usedRows: 0 },

        // default layout data for merged border attributes
        DEFAULT_BORDER_DATA = {
            borderTop: {},
            borderBottom: {},
            borderLeft: {},
            borderRight: {},
            borderInsideHor: {},
            borderInsideVert: {}
        },

        // default layout data for subtotal results
        DEFAULT_SUBTOTAL_DATA = {
            cells: 0,
            numbers: 0,
            sum: 0,
            min: Number.POSITIVE_INFINITY,
            max: Number.NEGATIVE_INFINITY
        },

        // default selection layout data, containing active cell data
        DEFAULT_SELECTION_DATA = {
            active: { display: '', result: null },
            borders: DEFAULT_BORDER_DATA,
            subtotals: DEFAULT_SUBTOTAL_DATA,
            merged: false
        },

        // default options for unit fields (column width and row height)
        DEFAULT_UNIT_FIELD_OPTIONS = {
            width: 70,
            css: { textAlign: 'right' },
            min: 0,
            precision: 10,
            smallStep: 10,
            largeStep: 500,
            roundStep: true
        },

        // the different options for merging and unmerging cells
        MERGE_OPTIONS = [
            { name: 'merge', label: gt('Merge cells')},
            { name: 'horizontal', label: gt('Merge cells horizontally')},
            { name: 'vertical', label: gt('Merge cells vertically')},
            { name: 'unmerge', label: gt('Unmerge cells')}
        ],

        // the maximum number of cells to be filled with one range operation
        MAX_FILL_CELL_COUNT = 1000;

    // private global functions ===============================================

    /**
     * Returns the column pane side identifier of the passed grid pane
     * position.
     *
     * @param {String} panePos
     *  The grid pane position.
     *
     * @returns {String}
     *  The position 'left' for the pane positions 'topLeft' and 'bottomLeft',
     *  or 'right' for the pane positions 'topRight' and 'bottomRight'.
     */
    function getColPaneSide(panePos) {
        return GRID_PANE_INFOS[panePos].colSide;
    }

    /**
     * Returns the row pane side identifier of the passed grid pane position.
     *
     * @param {String} panePos
     *  The grid pane position.
     *
     * @returns {String}
     *  The position 'top' for the pane positions 'topLeft' and 'topRight', or
     *  'bottom' for the pane positions 'bottomLeft' and 'bottomRight'.
     */
    function getRowPaneSide(panePos) {
        return GRID_PANE_INFOS[panePos].rowSide;
    }

    /**
     * Returns the pane position of the horizontal neighbor of the passed grid
     * pane position.
     *
     * @param {String} panePos
     *  The grid pane position.
     *
     * @returns {String}
     *  The grid pane position of the horizontal neighbor (for example, returns
     *  'bottomLeft' for the pane position 'bottomRight').
     */
    function getNextColPanePos(panePos) {
        return GRID_PANE_INFOS[panePos].nextColPos;
    }

    /**
     * Returns the pane position of the vertical neighbor of the passed grid
     * pane position.
     *
     * @param {String} panePos
     *  The grid pane position.
     *
     * @returns {String}
     *  The grid pane position of the vertical neighbor (for example, returns
     *  'topRight' for the pane position 'bottomRight').
     */
    function getNextRowPanePos(panePos) {
        return GRID_PANE_INFOS[panePos].nextRowPos;
    }

    /**
     * Returns the pane position nearest to the specified source pane position,
     * but matching the specified target pane side.
     *
     * @param {String} panePos
     *  The source grid pane position.
     *
     * @param {String} paneSide
     *  The target grid pane side.
     *
     * @returns {String}
     *  The grid pane position of the nearest neighbor, matching the specified
     *  pane side (for example, returns 'topRight' for the source pane position
     *  'bottomRight' and target pane side 'top').
     */
    function getNextPanePos(panePos, paneSide) {
        if ((paneSide === getColPaneSide(panePos)) || (paneSide === getRowPaneSide(panePos))) {
            return panePos;
        }
        return HEADER_PANE_INFOS[paneSide].columns ? getNextColPanePos(panePos) : getNextRowPanePos(panePos);
    }

    /**
     * Generates a localized sheet name with the specified index.
     *
     * @param {Number} index
     *  The sheet index to be inserted into the sheet name.
     *
     * @returns {String}
     *  The generated sheet name (e.g. "Sheet1", "Sheet2", etc.).
     */
    function generateSheetName(index) {
        return (
            //#. %1$d is the index in a sheet name (e.g. "Sheet1", "Sheet2", etc.)
            //#, c-format
            gt('Sheet%1$d', _.noI18n(index)));
    }

    // class SpreadsheetView ==================================================

    /**
     * Represents the entire view of a spreadsheet document. Contains the view
     * panes of the sheet currently shown (the 'active sheet'), which contain
     * the scrollable cell grids (there will be several view panes, if the
     * sheet view is split or frozen); and the selections of all existing
     * sheets.
     *
     * Additionally to the events triggered by the base class EditView, an
     * instance of this class triggers events if global contents of the
     * document, or the contents of the current active sheet have been changed.
     * The following events will be triggered:
     * - 'insert:sheet', 'delete:sheet', 'move:sheet', 'rename:sheet': The
     *      collection of sheets has been changed. See class 'SpreadsheetModel'
     *      for details.
     * - 'before:activesheet': Before another sheet will be activated. Event
     *      handlers receive the index of the current (old) active sheet.
     * - 'change:activesheet': After another sheet has been activated. Event
     *      handlers receive the index of the current (new) active sheet. This
     *      event will also be triggered, after the active sheet has been
     *      deleted, and another sheet is now active at the same sheet index.
     * - 'change:layoutdata': After the selection layout data has been updated,
     *      either directly or from the response data of a server view update.
     * - 'before:selection', 'change:selection': The selection in the active
     *      sheet has been changed. See class 'SheetModel' for details.
     * - 'insert:drawing', 'delete:drawing', 'change:drawing': The collection
     *      of drawing objects in the active sheet has been changed. See class
     *      'DrawingCollection' for details.
     *
     * @constructor
     *
     * @extends EditView
     */
    function SpreadsheetView(app) {

        var // self reference
            self = this,

            // the spreadsheet document model
            model = null,

            // the scrollable root DOM node containing the headers and panes
            rootNode = $('<div>').addClass('abs pane-root'),

            // the top-left corner pane
            cornerPane = null,

            // settings for all pane sides, mapped by pane side identifier
            paneSideSettings = {},

            // the row/column header panes, mapped by position keys
            headerPanes = {},

            // cell collections for all grid panes
            cellCollections = {},

            // the grid panes, mapped by position keys
            gridPanes = {},

            // the status bar below the grid panes
            statusBar = null,

            // the split line separating the left and right panes
            colSplitLine = $('<div>').addClass('split-line columns'),

            // the split line separating the top and bottom panes
            rowSplitLine = $('<div>').addClass('split-line rows'),

            // the split tracking node separating left and right panes
            colSplitTrackingNode = $('<div>')
                .addClass('abs split-tracking columns')
                .css({ width: TRACKING_SIZE + 'px', padding: '0 ' + TRACKING_MARGIN + 'px' }),

            // the split tracking node separating top and bottom panes
            rowSplitTrackingNode = $('<div>')
                .addClass('abs split-tracking rows')
                .css({ height: TRACKING_SIZE + 'px', padding: TRACKING_MARGIN + 'px 0' }),

            // the split tracking node covering the intersection of the other tracking points
            centerSplitTrackingNode = $('<div>')
                .addClass('abs split-tracking columns rows')
                .css({ width: TRACKING_SIZE + 'px', height: TRACKING_SIZE + 'px', padding: TRACKING_MARGIN + 'px' }),

            // all split tracking nodes, as jQuery collection
            allSplitTrackingNodes = colSplitTrackingNode.add(rowSplitTrackingNode).add(centerSplitTrackingNode),

            // tracking overlay nodes for column/row resizing
            resizeOverlayNode = $('<div>')
                .addClass('abs resize-tracking')
                .append($('<div>').addClass('abs leading'), $('<div>').addClass('abs trailing')),

            // the style sheet containers
            documentStyles = app.getModel().getDocumentStyles(),
            cellStyles = app.getModel().getCellStyles(),

            // the column and row collections of the active sheet
            colCollection = null,
            rowCollection = null,

            // the merge collection of the active sheet
            mergeCollection = null,

            // layout data of the active sheet
            sheetLayoutData = _.copy(DEFAULT_SHEET_DATA, true),

            // the contents and formatting of the selection and active cell
            selectionLayoutData = DEFAULT_SELECTION_DATA,

            // view settings of the active sheet
            sheetViewSettings = {
                split: false,
                freeze: false,
                splitLeft: 0,
                splitTop: 0
            },

            // the size of the header corner node (and thus of the row/column header nodes)
            headerWidth = 0,
            headerHeight = 0;

        // base constructor ---------------------------------------------------

        EditView.call(this, app, {
            initHandler: initHandler,
            deferredInitHandler: deferredInitHandler,
            grabFocusHandler: grabFocusHandler,
            overlayMargin: { left: 8, right: Utils.SCROLLBAR_WIDTH + 8, top: 20, bottom: Utils.SCROLLBAR_HEIGHT }
        });

        // private methods ----------------------------------------------------

        /**
         * Builds an attribute set containing all cell formatting attributes
         * set to the null value.
         */
        function buildNullAttributes() {
            var attributes = cellStyles.buildNullAttributes({ supportedFamilies: true });
            delete attributes.styleId;
            return attributes;
        }

        /**
         * Returns the cell ranges covered by all visible grid panes. If the
         * ranges of the grid panes overlap each other, the cell ranges
         * returned by this method will be joined already.
         */
        function getMergedGridPaneRanges() {

            var // the column/row intervals of all header panes
                intervals = {},
                // the resulting cell ranges
                ranges = [];

            function mergeIntervals(interval1, interval2) {

                var intervals = [];

                if (interval1 && interval2 && (interval1.first <= interval2.last + 1) && (interval2.first <= interval1.last + 1)) {
                    intervals.push({
                        first: Math.min(interval1.first, interval2.first),
                        last: Math.max(interval1.last, interval2.last)
                    });
                } else {
                    if (interval1) { intervals.push(interval1); }
                    if (interval2) { intervals.push(interval2); }
                }

                return intervals;
            }

            // get the intervals of all visible header panes
            _(headerPanes).each(function (headerPane, paneSide) {
                intervals[paneSide] = headerPane.isVisible() ? headerPane.getInterval() : null;
            });

            // try to join the intervals
            intervals.cols = mergeIntervals(intervals.left, intervals.right);
            intervals.rows = mergeIntervals(intervals.top, intervals.bottom);

            // build the cell ranges
            _(intervals.cols).each(function (colInterval) {
                _(intervals.rows).each(function (rowInterval) {
                    ranges.push({
                        start: [colInterval.first, rowInterval.first],
                        end: [colInterval.last, rowInterval.last]
                    });
                });
            });
            return ranges;
        }

        /**
         * Returns the layout data of the specified cell. Tries to receive the
         * layout data from all cell collections.
         *
         * @param {Number[]} address
         *  The logical address of the cell.
         *
         * @returns {Object|Null}
         *  The collection entry with the cell layout data, if available;
         *  otherwise null.
         */
        function getCellEntry(address) {
            return cellCollections.bottomRight.getCellEntry(address) ||
                cellCollections.bottomLeft.getCellEntry(address) ||
                cellCollections.topRight.getCellEntry(address) ||
                cellCollections.topLeft.getCellEntry(address);
        }

        /**
         * Tries to find a cell collection that contains the cell range
         * completely.
         *
         * @param {Object} range
         *  The logical address of the cell range.
         *
         * @returns {CellCollection|Null}
         *  The visible cell collection containing the range. If no collection
         *  has been found, null will be returned.
         */
        function getVisibleCellCollection(range) {
            return _(cellCollections).find(function (cellCollection, panePos) {
                return gridPanes[panePos].isVisible() && cellCollection.containsRange(range);
            });
        }

        /**
         * Calls the passed iterator function for all visible cells in the
         * passed cell ranges. For each cell range, a cell collection will be
         * searched containing the range completely. If no cell collection
         * contains a specific range, it will be skipped in the iteration
         * process, or optionally the iteration will be stopped.
         *
         * @param {Object|Array} ranges
         *  The logical address of a single range, or an array of logical
         *  range addresses whose cells will be iterated.
         *
         * @param {Function} iterator
         *  The iterator function called for all visible cells in the passed
         *  ranges. The ranges will be processed independently, cells covered
         *  by several ranges will be visited multiple times. See the method
         *  'CellCollection.iterateCellsInRanges()' for details about the
         *  parameters passed to the iterator.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method.
         *  Supports all options also supported by the method
         *  'CellCollection.iterateCellsInRanges()'. Additionally, the
         *  following options are supported:
         *  @param {Object} [options.skipMissingRanges=false]
         *      If set to true, range not covered by any of the existing cell
         *      collections will be skipped silently. By default, the iteration
         *      process will be stopped in this case (this method returns the
         *      Utils.BREAK object immediately).
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, or if a range has been
         *  found without corresponding cell collection, otherwise undefined.
         */
        function iterateCellsInRanges(ranges, iterator, options) {

            var // whether to skip ranges not contained by any collection
                skipMissingRanges = Utils.getBooleanOption(options, 'skipMissingRanges', false);

            // process each range for its own
            return Utils.iterateArray(_.getArray(ranges), function (range) {
                var cellCollection = getVisibleCellCollection(range);
                return cellCollection ? cellCollection.iterateCellsInRanges(range, iterator, options) : skipMissingRanges ? undefined : Utils.BREAK;
            });
        }

        /**
         * Expands the current selection, if it contains only a part of a merged
         * cell. In this case the complete merged cell must be included into the
         * selection.
         *
         * @param {Object} selection
         *  The cell selection data, in the properties 'ranges' (array of range
         *  addresses), 'activeRange' (array index of the active range in the
         *  'ranges' property), and 'activeCell' (logical cell address).
         */
        function expandMergedSelection(selection) {

            var // handle every merged cell only once
                handledMergedCells = {},
                // whether the iteration must be repeated to look for further expansions
                repeatIteration = false,
                // the currently evaluated selection range
                currentRange = null;

            function expandRange(address, cellEntry, colEntry, rowEntry, originalRange) {

                var key = null,
                    rowSpan = 1, colSpan = 1,
                    topBorder = null, rightBorder = null, bottomBorder = null, leftBorder = null,
                    addr = _.clone(address),
                    refAddress = null,
                    mergedRange = null;

                // resetting handledMergedCells for every original range
                if ((!currentRange) || (originalRange !== currentRange)) {  // really comparing the object references
                    currentRange = originalRange;
                    handledMergedCells = {};
                }

                if (mergeCollection.isMergedCell(address)) {

                    refAddress = mergeCollection.getReferenceAddress(address);  // if this is a hidden cell, the reference address is determined
                    if (refAddress) {
                        addr = _.clone(refAddress);  // saving address of reference cell
                    } else {
                        addr = _.clone(address); // this is already the reference cell
                    }

                    mergedRange = mergeCollection.getMergedRange(addr);

                    key = addr[0] + '_' + addr[1];

                    if (! (key in handledMergedCells)) {
                        rowSpan = SheetUtils.getRowCount(mergedRange);
                        colSpan = SheetUtils.getColCount(mergedRange);

                        topBorder = addr[1];
                        leftBorder = addr[0];
                        bottomBorder = topBorder + rowSpan - 1;
                        rightBorder = leftBorder + colSpan - 1;

                        if (originalRange.start[0] > leftBorder) {
                            originalRange.start[0] = leftBorder;
                            repeatIteration = true;
                        }
                        if (originalRange.start[1] > topBorder) {
                            originalRange.start[1] = topBorder;
                            repeatIteration = true;
                        }
                        if (originalRange.end[0] < rightBorder) {
                            originalRange.end[0] = rightBorder;
                            repeatIteration = true;
                        }
                        if (originalRange.end[1] < bottomBorder) {
                            originalRange.end[1] = bottomBorder;
                            repeatIteration = true;
                        }

                        handledMergedCells[key] = 1;
                    }
                }
            }

            do {
                handledMergedCells = {};
                repeatIteration = false;
                // iterating over all ranges
                iterateCellsInRanges(selection.ranges, expandRange, { outerCellsOnly: true, skipMissingRanges: true });
            } while (repeatIteration);
        }

        /**
         * Updates the selection part of the view layout data from the current
         * row, column, and cell collections.
         */
        var updateSelectionSettings = app.createDebouncedMethod($.noop, function () {

            var // the current cell selection
                selection = self.getSelection(),
                // whether to request a selection update from the server
                requestUpdate = false;

            // update data of active cell (contents, formatting)
            function updateActiveCell() {

                var // the address of the active cell in the passed selection
                    activeCell = selection.activeCell,
                    // collection entry of the active column
                    colEntry = colCollection.getEntry(activeCell[0]),
                    // collection entry of the active row
                    rowEntry = rowCollection.getEntry(activeCell[1]),
                    // collection entry of the active cell
                    cellEntry = null;

                // add data of active cell from cell collection
                if ((cellEntry = getCellEntry(activeCell))) {
                    selectionLayoutData.active = _.clone(cellEntry);
                    selectionLayoutData.active.attrs = cellStyles.getMergedAttributes(selectionLayoutData.active.attrs || {});
                }

                // add column/row attributes of active cell from column collection
                documentStyles.extendAttributes(selectionLayoutData.active.attrs, { column: colEntry.attributes.column });
                documentStyles.extendAttributes(selectionLayoutData.active.attrs, { row: rowEntry.attributes.row });
            }

            // updates data of selected ranges (subtotals, borders)
            function updateSelectionRanges() {

                var // array with names of outer cell borders
                    outerCellBorders = ['borderTop', 'borderBottom', 'borderLeft', 'borderRight'],
                    // object to collect and sort all border types
                    allBorders = {},
                    // result of the iterator (Utils.BREAK if any range is not contained completely in a cell collection)
                    result = null,
                    oppositeBorder = { borderTop: 'borderBottom', borderBottom: 'borderTop', borderLeft: 'borderRight', borderRight: 'borderLeft' };

                 // Calculating the address of a neighbor cell with the information about the
                 // address of the current cell and the border side.
                function getNeighborAddress(address, borderName) {
                    var neighborAddr = null;
                    if ((borderName === 'borderTop') && (address[1] > 0)) { neighborAddr = [address[0], address[1] - 1]; }
                    else if ((borderName === 'borderLeft') && (address[0] > 0)) { neighborAddr = [address[0] - 1, address[1]]; }
                    else if (borderName === 'borderBottom') { neighborAddr = [address[0], address[1] + 1]; }
                    else if (borderName === 'borderRight') { neighborAddr = [address[0] + 1, address[1]]; }
                    return neighborAddr;
                }

                // iterate all passed ranges, bail out if a range not covered by any cell collection has been found
                result = iterateCellsInRanges(selection.ranges, function (address, cellEntry, colEntry, rowEntry, originalRange) {

                    // accumulate borders
                    var useBorders = { borderTop: 'borderInsideHor', borderRight: 'borderInsideVert', borderBottom: 'borderInsideHor', borderLeft: 'borderInsideVert' },
                        neighborBorderName = null,
                        neighborCellEntry = null,
                        neighborAddress = null;

                    if (address[1] === originalRange.start[1]) { useBorders.borderTop = 'borderTop'; }  // first row
                    if (address[0] === originalRange.start[0]) { useBorders.borderLeft = 'borderLeft'; }  // first column
                    if (address[1] === originalRange.end[1]) { useBorders.borderBottom = 'borderBottom'; }  // last row
                    if (address[0] === originalRange.end[0]) { useBorders.borderRight = 'borderRight'; }  // last column

                    _.each(outerCellBorders, function (borderName) {
                        // saving this border in the object 'allBorders' in an array of all borders of the same type
                        if (!(useBorders[borderName] in allBorders)) {
                            allBorders[useBorders[borderName]] = [];  // create a new array for this border type
                        }
                        if (cellEntry && cellEntry.attrs && cellEntry.attrs.cell && (borderName in cellEntry.attrs.cell)) {
                            allBorders[useBorders[borderName]].push(_.clone(cellEntry.attrs.cell[borderName]));
                        } else {
                            // checking the neighbor cell, if no border is set at this cell
                            neighborBorderName = oppositeBorder[borderName];
                            neighborAddress = getNeighborAddress(_.clone(address), borderName);

                            if (neighborAddress) {
                                neighborCellEntry = getCellEntry(neighborAddress);
                            }

                            if (neighborCellEntry && neighborCellEntry.attrs && neighborCellEntry.attrs.cell && (neighborBorderName in neighborCellEntry.attrs.cell)) {
                                allBorders[useBorders[borderName]].push(_.clone(neighborCellEntry.attrs.cell[neighborBorderName]));
                            } else {
                                allBorders[useBorders[borderName]].push({style: 'none'});
                            }
                        }
                    });

                    // accumulate subtotals
                    if (_.isObject(cellEntry) && ('result' in cellEntry) && !_.isNull(cellEntry.result)) {
                        selectionLayoutData.subtotals.cells += 1;
                        if (_.isFinite(cellEntry.result)) {
                            selectionLayoutData.subtotals.numbers += 1;
                            selectionLayoutData.subtotals.sum += cellEntry.result;
                            selectionLayoutData.subtotals.min = Math.min(selectionLayoutData.subtotals.min, cellEntry.result);
                            selectionLayoutData.subtotals.max = Math.max(selectionLayoutData.subtotals.max, cellEntry.result);
                        }
                    }

                });

                // not all cells available in the collections, request new data from server
                if (result === Utils.BREAK) {
                    requestUpdate = true;
                    return;
                }

                // -> all border information collected and sorted in allBorders
                _(MergedBorder.CELL_BORDER_ATTRIBUTES).each(function (borderName) {
                    var mergedBorder = {};
                    if (borderName in allBorders) {
                        if (allBorders[borderName].length === 1) {
                            selectionLayoutData.borders[borderName] = allBorders[borderName][0];
                        } else {
                            _.each(allBorders[borderName], function (oneBorder) {
                                mergedBorder = MergedBorder.mergeBorders(mergedBorder, oneBorder);
                            });
                            selectionLayoutData.borders[borderName] = mergedBorder;
                        }
                    } else {
                        // no information about this border mode
                        selectionLayoutData.borders[borderName] = { style: 'none' };
                    }
                });
            }

            // build the selection layout data (will set the variable 'requestUpdate')
            selectionLayoutData = _.copy(DEFAULT_SELECTION_DATA, true);
            updateActiveCell();
            if (!requestUpdate) { updateSelectionRanges(); }
            self.trigger('change:layoutdata');

            // initialization from known data failed, request new data from server
            if (requestUpdate) {
                requestSelectionUpdate(selection);
            }

        }, { delay: 200 });

        /**
         * Requests an update of the selection part of the view layout data.
         */
        function requestSelectionUpdate(selection) {
            Utils.log('SpreadsheetView.requestSelectionUpdate(): sheet=' + self.getActiveSheet());
            self.requestUpdate({ selection: { ranges: selection.ranges, active: selection.activeCell } });
        }

        /**
         * Triggers a 'before:selection' event, before the selection of the
         * active sheet actually changes.
         */
        function triggerBeforeSelection(selection) {
            self.trigger('before:selection', selection);
        }

        /**
         * Processes a changed sheet selection. Updates the selection part of
         * the view layout data, and triggers a 'change:selection' event.
         */
        function triggerChangeSelection(selection) {
            updateSelectionSettings();
            rootNode.toggleClass('drawing-selection', selection.drawings.length > 0);
            self.trigger('change:selection', selection);
        }

        /**
         * Selects a sheet in this spreadsheet view instance.
         *
         * @param {Number} sheet
         *  The zero-based index of the new current sheet.
         *
         * @param {Boolean} [force]
         *  If set to true, the view will be updated even if the active sheet
         *  index has not changed.
         */
        function setActiveSheet(sheet, force) {

            // do nothing, if sheet index does not change
            if (!force && (sheetLayoutData.sheet === sheet)) { return; }

            // notify selection listeners before the sheet will be deactivated
            if (self.getActiveSheet() >= 0) {
                triggerBeforeSelection(self.getSelection());
                self.trigger('before:activesheet', self.getActiveSheet());
            }

            // initialize the sheet layout data
            sheetLayoutData = Utils.extendOptions(DEFAULT_SHEET_DATA, { sheet: sheet });

            // get column/row collections and mergeCollection from the current active sheet
            colCollection = self.getActiveSheetModel().getColCollection();
            rowCollection = self.getActiveSheetModel().getRowCollection();
            mergeCollection = self.getActiveSheetModel().getMergeCollection();

            // notify listeners
            self.trigger('change:activesheet', sheet);

            // refresh the layout of the header and grid panes, send initial update request
            initializePanes();
            self.requestUpdate({ cells: getMergedGridPaneRanges() });

            // initialize the selection of the new sheet
            triggerChangeSelection(self.getSelection());
        }

        /**
         * Invokes the specified method at all cell collections that are
         * currently contained by visible grid panes.
         */
        function invokeCellCollectionMethod(method) {
            var methodArgs = _.toArray(arguments).slice(1);
            _(cellCollections).each(function (cellCollection, panePos) {
                if (gridPanes[panePos].isVisible()) {
                    cellCollection[method].apply(cellCollection, methodArgs);
                }
            });
        }

        /**
         * Handles a new sheet inserted into the document model.
         */
        function insertSheetHandler(event, sheet) {

            var // the model of the new sheet
                sheetModel = model.getSheetModel(sheet);

            // registers an event handler that will be called while the sheet is active
            function handleActiveSheetEvent(eventSource, eventType, options) {

                var // the local event handler called before the event will be triggered
                    beforeHandlerFunc = Utils.getFunctionOption(options, 'handler'),
                    // the type of the event to be triggered
                    triggerType = Utils.getStringOption(options, 'trigger');

                // handle specified event of the event source object, do nothing if the sheet is not active
                eventSource.on(eventType, function () {
                    if (sheetModel === self.getActiveSheetModel()) {
                        if (_.isFunction(beforeHandlerFunc)) {
                            beforeHandlerFunc.apply(self, _.toArray(arguments));
                        }
                        if (_.isString(triggerType)) {
                            self.trigger.apply(self, [triggerType].concat(_.toArray(arguments).slice(1)));
                        }
                    }
                });
            }

            // listen to selection changes in the sheet, notify listeners of the view
            handleActiveSheetEvent(sheetModel, 'before:selection', { handler: function (event, selection) { triggerBeforeSelection(selection); } });
            handleActiveSheetEvent(sheetModel, 'change:selection', { handler: function (event, selection) { triggerChangeSelection(selection); } });

            // listen to changes in the column collection, notify listeners of the view
            handleActiveSheetEvent(sheetModel.getColCollection(), 'insert:entries', { trigger: 'insert:columns' });
            handleActiveSheetEvent(sheetModel.getColCollection(), 'delete:entries', { trigger: 'delete:columns' });
            handleActiveSheetEvent(sheetModel.getColCollection(), 'change:entries', { trigger: 'change:columns' });
            handleActiveSheetEvent(sheetModel.getColCollection(), 'triggered', { handler: updateSelectionSettings });

            // listen to changes in the row collection, notify listeners of the view
            handleActiveSheetEvent(sheetModel.getRowCollection(), 'insert:entries', { trigger: 'insert:rows' });
            handleActiveSheetEvent(sheetModel.getRowCollection(), 'delete:entries', { trigger: 'delete:rows' });
            handleActiveSheetEvent(sheetModel.getRowCollection(), 'change:entries', { trigger: 'change:rows' });
            handleActiveSheetEvent(sheetModel.getRowCollection(), 'triggered', { handler: updateSelectionSettings });

            // listen to changes in the merge collection, notify listeners of the view
            handleActiveSheetEvent(sheetModel.getMergeCollection(), 'change:mergedCells', { trigger: 'change:mergedCells' });
            handleActiveSheetEvent(sheetModel.getMergeCollection(), 'triggered', { handler: updateSelectionSettings });

            // listen to changes in the drawing collection, notify listeners of the view
            handleActiveSheetEvent(sheetModel.getDrawingCollection(), 'insert:drawing', { trigger: 'insert:drawing' });
            handleActiveSheetEvent(sheetModel.getDrawingCollection(), 'delete:drawing', { trigger: 'delete:drawing' });
            handleActiveSheetEvent(sheetModel.getDrawingCollection(), 'change:drawing', { trigger: 'change:drawing' });
            handleActiveSheetEvent(sheetModel.getDrawingCollection(), 'triggered', { handler: updateSelectionSettings });

            // update active sheet index
            if (sheet <= self.getActiveSheet()) {
                setActiveSheet(self.getActiveSheet() + 1);
            }
        }

        /**
         * Handles a sheet that has been deleted from the document model.
         */
        function deleteSheetHandler(event, sheet) {

            // update active sheet index
            if (sheet < self.getActiveSheet()) {
                setActiveSheet(self.getActiveSheet() - 1);
            } else if (self.getActiveSheet() >= model.getSheetCount()) {
                setActiveSheet(model.getSheetCount() - 1);
            } else if (sheet === self.getActiveSheet()) {
                setActiveSheet(self.getActiveSheet(), true);
            }
        }

        /**
         * Handles 'update' push messages. If the message data contains a list
         * of changed cells for the active sheet, updates the visible cells in
         * all grid panes.
         *
         * @param {Object} data
         *  The message data of the 'update' push message.
         */
        function updateNotificationHandler(data) {

            var // complete changed data
                changedData = Utils.getObjectOption(data, 'changed'),
                // whether to force requesting a view update
                fullUpdate = false,
                // map of changed data per sheet
                sheets = null,
                // map of changed data of the current sheet
                sheetData = null,
                // changed cell ranges
                ranges = null;

            // update message may not contain any change data
            if (!_.isObject(changedData)) { return; }
            Utils.log('SpreadsheetView.updateNotificationHandler(): data=' + JSON.stringify(changedData));

            // process by update type (specific "cells", or "all")
            switch (Utils.getStringOption(changedData, 'type')) {
            case 'cells':
                if ((sheets = Utils.getObjectOption(changedData, 'sheets'))) {
                    if ((sheetData = Utils.getObjectOption(sheets, self.getActiveSheet()))) {

                        // extract the addresses of the changed cell ranges
                        if (_.isArray(sheetData.cells) && (sheetData.cells.length > 0)) {
                            ranges = sheetData.cells;
                            _(ranges).each(function (range) { if (!range.end) { range.end = range.start; } });
                        } else {
                            Utils.warn('SpreadsheetView.updateNotificationHandler(): no changed cells');
                        }
                    }
                } else {
                    Utils.warn('SpreadsheetView.updateNotificationHandler(): no sheets defined');
                }
                break;
            case 'all':
                fullUpdate = true;
                break;
            default:
                Utils.error('SpreadsheetView.updateNotificationHandler(): invalid type in cell update data');
                return;
            }

            // trigger forced update of all grid panes in the active sheet
            if (_.isArray(ranges)) {
                self.requestUpdate({ cells: ranges });
            } else if (fullUpdate) {
                self.requestUpdate({ cells: getMergedGridPaneRanges() });
            }

            // update selection layout data
            // TODO: only, if selection is not completely visible
            requestSelectionUpdate(self.getSelection());
        }

        /**
         * Handles all tracking events for the split separators.
         */
        function splitTrackingHandler(event) {

            var // the event source node
                sourceNode = $(this),
                // whether column split tracking and/or row split tracking is active
                colSplit = sourceNode.hasClass('columns'),
                rowSplit = sourceNode.hasClass('rows'),
                // minimum and maximum position of split lines
                minLeft = headerWidth, maxLeft = rootNode.width() - Utils.SCROLLBAR_WIDTH - SPLIT_SIZE,
                minTop = headerHeight, maxTop = rootNode.height() - Utils.SCROLLBAR_HEIGHT - SPLIT_SIZE;

            // returns the X position of a split line according to the current event
            function getSplitLineLeft() {
                return Utils.minMax(paneSideSettings.left.offset + paneSideSettings.left.size + (colSplit ? event.offsetX : 0), minLeft, maxLeft);
            }

            // returns the Y position of a split line according to the current event
            function getSplitLineTop() {
                return Utils.minMax(paneSideSettings.top.offset + paneSideSettings.top.size + (rowSplit ? event.offsetY : 0), minTop, maxTop);
            }

            switch (event.type) {

            case 'tracking:start':
                colSplitTrackingNode.toggleClass('tracking-active', colSplit);
                rowSplitTrackingNode.toggleClass('tracking-active', rowSplit);
                break;

            case 'tracking:move':
                if (colSplit) { colSplitTrackingNode.add(centerSplitTrackingNode).css({ left: getSplitLineLeft() - TRACKING_OFFSET }); }
                if (rowSplit) { rowSplitTrackingNode.add(centerSplitTrackingNode).css({ top: getSplitLineTop() - TRACKING_OFFSET }); }
                break;

            case 'tracking:end':
                var left = getSplitLineLeft(), top = getSplitLineTop();
                left = ((left <= minLeft) || (left >= maxLeft)) ? 0 : Utils.convertLengthToHmm(left - headerWidth, 'px');
                top = ((top <= minTop) || (top >= maxTop)) ? 0 : Utils.convertLengthToHmm(top - headerHeight, 'px');
                allSplitTrackingNodes.removeClass('tracking-active');
                self.setSplitPosition(left, top);
                app.getController().update();
                self.grabFocus();
                break;

            case 'tracking:cancel':
                allSplitTrackingNodes.removeClass('tracking-active');
                initializePanes();
                self.grabFocus();
                break;
            }
        }

        /**
         * Initialization after construction.
         */
        function initHandler() {

            var // which header pane is currently focused (while selection tracking is active)
                focusPaneSide = null,
                // which grid pane is currently focused
                focusPanePos = null,
                // additional DOM nodes in debug pane
                debugNodes = {};

            // highlights the header panes and grid panes currently focused
            function drawPaneFocus() {

                var // info structure of focused header pane
                    headerPaneInfo = HEADER_PANE_INFOS[focusPaneSide],
                    // which header panes are drawn focused
                    paneSides = [],
                    // which grid panes are drawn focused
                    panePositions = [];

                if (headerPaneInfo) {
                    // header pane is focused (selection tracking): highlight it,
                    // both header panes in the opposite direction, and the two
                    // associated grid panes (ignore focus state of the grid panes)
                    paneSides = _.chain(HEADER_PANE_INFOS).keys().without(headerPaneInfo.nextSide).value();
                    panePositions = [headerPaneInfo.panePos1, headerPaneInfo.panePos2];
                } else if (focusPanePos) {
                    // grid pane is focused: highlight it and the associated header panes
                    paneSides = [getColPaneSide(focusPanePos), getRowPaneSide(focusPanePos)];
                    panePositions = [focusPanePos];
                }

                // highlight header panes and grid panes
                _(headerPanes).each(function (headerPane, paneSide) {
                    headerPane.getNode().toggleClass(Utils.FOCUSED_CLASS, _(paneSides).contains(paneSide));
                });
                _(gridPanes).each(function (gridPane, panePos) {
                    gridPane.getNode().toggleClass(Utils.FOCUSED_CLASS, _(panePositions).contains(panePos));
                });
            }

            // store reference to document model
            model = app.getModel();

            // insert the root node into the application pane
            self.insertContentNode(rootNode);

            // set CSS class for formatting dependent on edit mode (e.g. mouse pointers)
            model.on('change:editmode', function (event, editMode) {
                rootNode.toggleClass('edit-mode', editMode);
            });

            // create the header pane instances, initialize settings for pane sides
            _(HEADER_PANE_INFOS).each(function (hedaderPaneInfo, paneSide) {
                paneSideSettings[paneSide] = _.clone(DEFAULT_PANE_SIDE_SETTINGS);
                headerPanes[paneSide] = new HeaderPane(app, paneSide, HEADER_PANE_INFOS[paneSide].columns);
            });

            // create the cell collections and grid pane instances
            _(GRID_PANE_INFOS).each(function (gridPaneInfo, panePos) {
                var colHeaderPane = self.getColHeaderPane(panePos),
                    rowHeaderPane = self.getRowHeaderPane(panePos);
                cellCollections[panePos] = new CellCollection(app, colHeaderPane, rowHeaderPane);
                cellCollections[panePos].on('triggered', updateSelectionSettings);
                gridPanes[panePos] = new GridPane(app, panePos, colHeaderPane, rowHeaderPane, cellCollections[panePos]);
            });

            // create the corner pane instance
            cornerPane = new CornerPane(app);

            // insert resize overlay node first (will increase its z-index while active)
            rootNode.append(resizeOverlayNode);

            // insert the pane nodes into the DOM
            _(gridPanes).each(function (gridPane) { rootNode.append(gridPane.getNode()); });
            _(headerPanes).each(function (headerPane) { rootNode.append(headerPane.getNode()); });
            rootNode.append(cornerPane.getNode());

            // append the split lines and split tracking nodes
            rootNode.append(colSplitLine, rowSplitLine, colSplitTrackingNode, rowSplitTrackingNode, centerSplitTrackingNode);

            // enable tracking, register event handlers
            allSplitTrackingNodes
                .attr('tabindex', 1) // make focusable to prevent immediate 'tracking:cancel' event
                .append($('<div>'))  // child node for fill color (CSS property 'background-clip' causes paint errors in Firefox)
                .enableTracking()
                .on('tracking:start tracking:move tracking:end tracking:cancel', splitTrackingHandler);

            // listen to sheet events
            model.on({ 'insert:sheet': insertSheetHandler, 'delete:sheet': deleteSheetHandler });

            // application notifies changed contents/results of cells
            app.on('docs:update', updateNotificationHandler);

            // activate a sheet (TODO: use document view settings)
            app.on('docs:import:after', function () {
                setActiveSheet(0);
                self.grabFocus();
            });

            // update position of grid panes, when general view layout changes
            app.on('docs:import:success', function () {
                self.on('refresh:layout', initializePanes);
            });

            // create the status bar
            self.addPane(statusBar = new StatusBar(app));
            app.on('docs:state:error', function () { statusBar.hide(); });

            // disable browser context menu in entire application pane
            rootNode.on('contextmenu', function () {
                self.grabFocus();
                return false;
            });

            // process events triggered by header panes
            _(headerPanes).each(function (headerPane, paneSide) {

                var paneInfo = HEADER_PANE_INFOS[paneSide],
                    offsetAttr = paneInfo.columns ? 'left' : 'top',
                    sizeAttr = paneInfo.columns ? 'width' : 'height',
                    cellCollection1 = cellCollections[paneInfo.panePos1],
                    cellCollection2 = cellCollections[paneInfo.panePos2],
                    setIntervalFunc = paneInfo.columns ? 'setColInterval' : 'setRowInterval';

                function showResizerOverlayNode(offset, size) {
                    rootNode.addClass('active-tracking');
                    resizeOverlayNode.attr('data-orientation', paneInfo.columns ? 'columns' : 'rows');
                    updateResizerOverlayNode(offset, size);
                }

                function updateResizerOverlayNode(offset, size) {
                    var relativeOffset = paneSideSettings[paneSide].offset + offset - headerPane.getVisibleOffset();
                    resizeOverlayNode.toggleClass('collapsed', size === 0);
                    resizeOverlayNode.find('>.leading').css(sizeAttr, (Math.max(0, relativeOffset) + 10) + 'px');
                    resizeOverlayNode.find('>.trailing').css(offsetAttr, (relativeOffset + size) + 'px');
                }

                function hideResizerOverlayNode() {
                    rootNode.removeClass('active-tracking');
                    resizeOverlayNode.find('>.leading').css(sizeAttr, '');
                    resizeOverlayNode.find('>.trailing').css(offsetAttr, '');
                }

                headerPane.on({
                    // changed column/row interval, update cell collections
                    'change:interval': function () {
                        var args = _.toArray(arguments).slice(1);
                        cellCollection1[setIntervalFunc].apply(cellCollection1, args);
                        cellCollection2[setIntervalFunc].apply(cellCollection2, args);
                    },

                    // change focus display while selection tracking in header panes is active
                    'select:start': function () {
                        var activePane = self.getSheetViewProperty('activePane');
                        self.setSheetViewProperty('activePane', getNextPanePos(activePane, paneSide));
                        focusPaneSide = paneSide;
                        drawPaneFocus();
                    },
                    'select:end': function () {
                        focusPaneSide = null;
                        drawPaneFocus();
                    },
                    // visualize resize tracking from header panes
                    'resize:start': function (event, offset, size) {
                        showResizerOverlayNode(offset, size);
                        self.trigger('resize:start', paneSide, offset, size);
                    },
                    'resize:update': function (event, offset, size) {
                        updateResizerOverlayNode(offset, size);
                        self.trigger('resize:update', paneSide, offset, size);
                    },
                    'resize:end': function () {
                        hideResizerOverlayNode();
                        self.trigger('resize:end', paneSide);
                    }
                });
            });

            // process events triggered by grid panes
            _(gridPanes).each(function (gridPane, panePos) {

                // forward cell edit mode in any grid pane to listeners
                gridPane.on('celledit:enter celledit:leave', function (event) {
                    self.trigger(event.type, panePos);
                });

                // update active pane on focus change
                gridPane.getNode().on('focusin', function () {
                    if (focusPanePos !== panePos) {
                        focusPanePos = panePos;
                        self.setSheetViewProperty('activePane', panePos);
                        drawPaneFocus();
                    }
                });
            });

            // initialize merged attribute set for active cell, used until first real update
            selectionLayoutData.active.attrs = cellStyles.getMergedAttributes({});
            documentStyles.extendAttributes(selectionLayoutData.active.attrs, documentStyles.getStyleSheets('column').getMergedAttributes({}));
            documentStyles.extendAttributes(selectionLayoutData.active.attrs, documentStyles.getStyleSheets('row').getMergedAttributes({}));

            // log current selection in debug mode
            if (Config.isDebug()) {

                // create the DOM nodes in the debug pane
                self.addDebugInfoHeader('Selection');
                debugNodes.sheet = self.addDebugInfoNode('sheet');
                debugNodes.ranges = self.addDebugInfoNode('ranges');
                debugNodes.active = self.addDebugInfoNode('active');
                debugNodes.drawing = self.addDebugInfoNode('draw');

                // log information about the active sheet
                self.on('change:layoutdata change:activesheet', function () {
                    var sheet = self.getActiveSheet(),
                        used = ((self.getUsedCols() > 0) && (self.getUsedRows() > 0)) ? SheetUtils.getRangeName({ start: [0, 0], end: [self.getUsedCols() - 1, self.getUsedRows() - 1] }) : '<empty>';
                    debugNodes.sheet.text(_.noI18n('index=' + sheet + ', name="' + model.getSheetName(sheet) + '", used=' + used));
                });

                // log all selection events
                self.on('change:selection', function (event, selection) {
                    debugNodes.ranges.text(_.noI18n('count=' + selection.ranges.length + ', ranges=[' + _(selection.ranges).map(SheetUtils.getRangeName).join() + ']'));
                    debugNodes.active.text(_.noI18n('range=' + SheetUtils.getRangeName(selection.ranges[selection.activeRange]) + ', cell=' + SheetUtils.getCellName(selection.activeCell)));
                    debugNodes.drawing.text(_.noI18n('count=' + selection.drawings.length + ', frames=' + ((selection.drawings.length > 0) ? ('[' + (selection.drawings).map(JSON.stringify).join() + ']') : '<none>')));
                });
            }
        }

        /**
         * Initialization after importing the document. Creates all tool boxes
         * in the side pane and overlay pane. Needed to be executed after
         * import, to be able to hide specific GUI elements depending on the
         * file type.
         */
        function deferredInitHandler() {

            self.createToolBox('format', { label: gt('Format'), visible: 'view/cell/editable' })
                .addGroup('character/fontname', new EditControls.FontFamilyChooser(app, { width: 117 }))
                .addGap(11)
                .addGroup('character/fontsize', new EditControls.FontHeightChooser({ width: 47 }))
                .newLine()
                .addGroup('character/bold',      new Button(EditControls.BOLD_OPTIONS))
                .addGroup('character/italic',    new Button(EditControls.ITALIC_OPTIONS))
                .addGroup('character/underline', new Button(EditControls.UNDERLINE_OPTIONS))
                .addGroup('character/strike',    new Button(EditControls.STRIKEOUT_OPTIONS))
                .newLine()
                .addGroup('character/color', new EditControls.ColorChooser(app, 'text', { icon: 'docs-font-color', tooltip: gt('Text color') }))
                .addGap()
                .addGroup('cell/fillcolor', new EditControls.ColorChooser(app, 'fill', { icon: 'docs-cell-fill-color', tooltip: gt('Fill color') }))
                .addGap()
                .addGroup('cell/border/mode', new EditControls.BorderChooser({ tooltip: gt('Cell borders') }))
                .addRightTab()
                .addGroup('cell/resetAttributes', new Button(EditControls.CLEAR_FORMAT_OPTIONS));

            self.createToolBox('borders', { label: gt('Borders'), visible: 'cell/border/value' })
                .addGroup('cell/border/style', new EditControls.BorderStyleChooser({ width: 96 }))
                .addGap()
                .addGroup('cell/border/width', new EditControls.CellBorderWidthChooser({ width: 35 }))
                .addGap()
                .addGroup('cell/border/color', new EditControls.ColorChooser(app, 'line', { icon: 'docs-cell-fill-color', tooltip: gt('Border color') }));

            self.createToolBox('alignment', { label: gt('Alignment'), visible: 'view/cell/editable' })
                .addGroup('cell/alignhor', new RadioGroup({ toggleValue: 'auto' })
                    .createOptionButton('left',    { icon: 'docs-para-align-left',    tooltip: /*#. alignment of text in paragraphs or cells */ gt('Left') })
                    .createOptionButton('center',  { icon: 'docs-para-align-center',  tooltip: /*#. alignment of text in paragraphs or cells */ gt('Center') })
                    .createOptionButton('right',   { icon: 'docs-para-align-right',   tooltip: /*#. alignment of text in paragraphs or cells */ gt('Right') })
                    .createOptionButton('justify', { icon: 'docs-para-align-justify', tooltip: /*#. alignment of text in paragraphs or cells */ gt('Justify') })
                )
                .addRightTab()
                .addGroup('cell/alignvert', new RadioList({ icon: 'docs-cell-vertical-bottom', tooltip: /*#. text alignment in cells */ gt('Vertical alignment'), highlight: true, updateCaptionMode: 'icon' })
                    .createOptionButton('top',     { icon: 'docs-cell-vertical-top',    label: /*#. vertical text alignment in cells */ gt('Top') })
                    .createOptionButton('middle',  { icon: 'docs-cell-vertical-middle', label: /*#. vertical text alignment in cells */ gt('Middle') })
                    .createOptionButton('bottom',  { icon: 'docs-cell-vertical-bottom', label: /*#. vertical text alignment in cells */ gt('Bottom') })
                    .createOptionButton('justify', { icon: 'docs-cell-vertical-middle', label: /*#. vertical text alignment in cells */ gt('Justify') })
                );

            self.createToolBox('rowscols', { label: gt('Rows and Columns'), visible: 'view/cell/editable' })
                .addGroup('row/insert', new Button({ icon: 'docs-table-insert-row', tooltip: gt('Insert row') }))
                .addGroup('row/delete', new Button({ icon: 'docs-table-delete-row', tooltip: gt('Delete selected rows') }))
                .addRightTab()
                .addPrivateGroup(new Label({ icon: 'docs-table-resize-row', tooltip: gt('Row height'), classes: 'no-borders' }))
                .addGap(1)
                .addGroup('row/height', new UnitField(Utils.extendOptions(DEFAULT_UNIT_FIELD_OPTIONS, { tooltip: gt('Row height'), max: SheetUtils.MAX_ROW_HEIGHT })))
                .newLine()
                .addGroup('column/insert', new Button({ icon: 'docs-table-insert-column', tooltip: gt('Insert column') }))
                .addGroup('column/delete', new Button({ icon: 'docs-table-delete-column', tooltip: gt('Delete selected columns') }))
                .addRightTab()
                .addPrivateGroup(new Label({ icon: 'docs-table-resize-column', tooltip: gt('Column width'), classes: 'no-borders' }))
                .addGap(1)
                .addGroup('column/width',  new UnitField(Utils.extendOptions(DEFAULT_UNIT_FIELD_OPTIONS, { tooltip: gt('Column width'), max: SheetUtils.MAX_COLUMN_WIDTH })));

            self.createToolBox('merge', { label: gt('Merge/Unmerge cells'), visible: 'view/cell/editable' })
                .addGroup('cell/merge',    new SpreadsheetControls.MergeChooser(app, MERGE_OPTIONS, { label: gt('Merge cells'), tooltip: gt('Merge or unmerge cells'), updateCaptionMode: 'label', width: 140 }));

            self.getOverlayToolBox()
                .addGroup('character/bold',   new Button(EditControls.BOLD_OPTIONS))
                .addGroup('character/italic', new Button(EditControls.ITALIC_OPTIONS));

            if (Config.isDebug()) {
                self.createDebugToolBox()
                    .addGroup('sheet/name', new TextField({ tooltip: _.noI18n('Rename sheet'), width: 106 }))
                    .addRightTab()
                    .addGroup('sheet/delete', new Button({ label: _.noI18n('Delete sheet') }))
                    .newLine()
                    .addGroup('view/split', new Button({ label: _.noI18n('Split'), toggle: true }))
                    .addGroup('view/freeze', new Button({ label: _.noI18n('Freeze'), toggle: true }));

                self.registerDebugAction('view', { sectionId: 'update', icon: 'icon-refresh', label: _.noI18n('Request full view update') }, function () {
                        updateNotificationHandler({ changed: { type: 'all' } });
                    })
                    .registerDebugAction('cells', { sectionId: 'update', icon: 'icon-refresh', label: _.noI18n('Request view update of selected cells') }, function () {
                        updateNotificationHandler({ changed: { type: 'cells' }, sheets: Utils.makeSimpleObject(self.getActiveSheet(), { cells: self.getSelectedRanges() }) });
                    })
                    .registerDebugAction('selection', { sectionId: 'update', icon: 'icon-refresh', label: _.noI18n('Request update of selection data') }, function () {
                        requestSelectionUpdate(self.getSelection());
                    })
                    .registerDebugAction('drawing', { sectionId: 'operations', icon: 'icon-sitemap', label: _.noI18n('Apply "insertDrawing" operations') }, function () {
                        var generator = model.createOperationsGenerator(),
                            index = self.getActiveSheetModel().getDrawingCollection().getModelCount(),
                            selection = _(self.getSelection()).extend({ drawings: [] });
                        _(self.getSelectedRanges()).each(function (range) {
                            var attrs = { drawing: {
                                startCol: range.start[0],
                                startColOffset: 100,
                                startRow: range.start[1],
                                startRowOffset: 100,
                                endCol: range.end[0],
                                endColOffset: 2400,
                                endRow: range.end[1],
                                endRowOffset: 400,
                                name: 'Diagram ' + (index + 1),
                                description: 'Very, very, very, very, very, very, very, very, very long description for this diagram.'
                            } };
                            generator.generateSheetOperation(Operations.INSERT_DRAWING, self.getActiveSheet(), [index], { type: 'diagram', attrs: attrs });
                            selection.drawings.push([index]);
                            index += 1;
                        });
                        model.applyOperations(generator.getOperations());
                        self.getActiveSheetModel().setSelection(selection);
                    });
            }
        }

        /**
         * Moves the browser focus into the active sheet pane.
         */
        function grabFocusHandler() {
            var activePane = self.getSheetViewProperty('activePane');
            if (!(activePane in gridPanes)) {
                Utils.error('SpreadsheetView.grabFocusHandler(): invalid active pane');
                activePane = 'bottomRight';
            }
            self.getVisibleGridPane(activePane).grabFocus();
        }

        /**
         * Refreshes the layout of all visible panes and tracking nodes.
         */
        function initializePanes() {

            Utils.takeTime('SpreadsheetView.initializePanes()', function () {

                var // whether freeze mode is active (has higher priority than split mode)
                    freezeActive = self.getFreezeMode(),
                    // whether split mode is active, but not freeze mode
                    splitActive = !freezeActive && self.getSplitMode(),
                    // the size of the split lines
                    splitSize = freezeActive ? FREEZE_SIZE : SPLIT_SIZE,
                    // whether the split lines are visible
                    colSplit = false, rowSplit = false,
                    // start position of the split lines
                    splitLineLeft = 0, splitLineTop = 0;

                // cancel any running mouse tracking
                $.cancelTracking();

                // whether the left and top panes are visible (must have enough room in split mode, always in freeze mode)
                paneSideSettings.left.visible = (splitActive && (sheetViewSettings.splitLeft > MIN_PANE_SIZE)) || (freezeActive && (sheetViewSettings.splitLeft > 0));
                paneSideSettings.top.visible = (splitActive && (sheetViewSettings.splitTop > MIN_PANE_SIZE)) || (freezeActive && (sheetViewSettings.splitTop > 0));

                // calculate current size of header nodes
                // TODO: use current maximum row
                cornerPane.initializePaneLayout(0xFFFFF);
                headerWidth = cornerPane.getNode().width();
                headerHeight = cornerPane.getNode().height();

                // calculate inner width and height of left and top panes, and position of split lines
                // TODO: convert freeze panes from rows/columns to pixels
                if (paneSideSettings.left.visible) {
                    paneSideSettings.left.offset = headerWidth;
                    paneSideSettings.left.size = Utils.convertHmmToLength(sheetViewSettings.splitLeft, 'px', 1);
                    paneSideSettings.left.hiddenSize = 0;
                    splitLineLeft = paneSideSettings.left.offset + paneSideSettings.left.size;
                }
                if (paneSideSettings.top.visible) {
                    paneSideSettings.top.offset = headerHeight;
                    paneSideSettings.top.size = Utils.convertHmmToLength(sheetViewSettings.splitTop, 'px', 1);
                    paneSideSettings.top.hiddenSize = 0;
                    splitLineTop = paneSideSettings.top.offset + paneSideSettings.top.size;
                }

                // determine whether right and bottom panes are visible (must have enough room in split and freeze mode)
                paneSideSettings.right.visible = !paneSideSettings.left.visible || (splitLineLeft + splitSize + MIN_PANE_SIZE + Utils.SCROLLBAR_WIDTH <= rootNode.width());
                paneSideSettings.bottom.visible = !paneSideSettings.top.visible || (splitLineTop + splitSize + MIN_PANE_SIZE + Utils.SCROLLBAR_HEIGHT <= rootNode.height());

                // visibility of the split lines
                colSplit = paneSideSettings.left.visible && paneSideSettings.right.visible;
                rowSplit = paneSideSettings.top.visible && paneSideSettings.bottom.visible;

                // calculate the resulting grid pane positions and sizes
                if (paneSideSettings.right.visible) {
                    paneSideSettings.right.offset = colSplit ? (splitLineLeft + splitSize) : headerWidth;
                    paneSideSettings.right.size = rootNode.width() - paneSideSettings.right.offset;
                    paneSideSettings.right.hiddenSize = freezeActive ? paneSideSettings.left.size : 0;
                } else if (paneSideSettings.left.visible) {
                    paneSideSettings.left.size = rootNode.width() - headerWidth;
                }
                if (paneSideSettings.bottom.visible) {
                    paneSideSettings.bottom.offset = rowSplit ? (splitLineTop + splitSize) : headerHeight;
                    paneSideSettings.bottom.size = rootNode.height() - paneSideSettings.bottom.offset;
                    paneSideSettings.bottom.hiddenSize = freezeActive ? paneSideSettings.top.size : 0;
                } else if (paneSideSettings.top.visible) {
                    paneSideSettings.top.size = rootNode.height() - headerHeight;
                }

                // set scroll mode (left/top panes are not scrollable in frozen mode in their
                // own direction, e.g. left frozen panes are not scrollable to left/right)
                paneSideSettings.left.scrollable = splitActive;
                paneSideSettings.right.scrollable = true;
                paneSideSettings.top.scrollable = splitActive;
                paneSideSettings.bottom.scrollable = true;

                // set up scroll bar visibility in the opposite direction (e.g. left/right
                // scroll bars of top panes are hidden, if bottom panes are visible)
                paneSideSettings.left.showOppositeScroll = !paneSideSettings.right.visible;
                paneSideSettings.right.showOppositeScroll = true;
                paneSideSettings.top.showOppositeScroll = !paneSideSettings.bottom.visible;
                paneSideSettings.bottom.showOppositeScroll = true;

                // initialize the header panes
                _(headerPanes).each(function (headerPane, paneSide) {
                    headerPane.initializePaneLayout(paneSideSettings[paneSide]);
                });

                // initialize the grid panes
                _(gridPanes).each(function (gridPane, panePos) {
                    gridPane.initializePaneLayout(paneSideSettings[getColPaneSide(panePos)], paneSideSettings[getRowPaneSide(panePos)]);
                });

                // visibility and position of the split lines
                colSplitLine.toggle(colSplit).css({ left: splitLineLeft, width: splitSize });
                rowSplitLine.toggle(rowSplit).css({ top: splitLineTop, height: splitSize });

                // visibility and position of the split tracking nodes
                colSplitTrackingNode.toggle(splitActive && colSplit).css({ left: splitLineLeft - TRACKING_OFFSET });
                rowSplitTrackingNode.toggle(splitActive && rowSplit).css({ top: splitLineTop - TRACKING_OFFSET });
                centerSplitTrackingNode.toggle(splitActive && colSplit && rowSplit).css({ left: splitLineLeft - TRACKING_OFFSET, top: splitLineTop - TRACKING_OFFSET });
            });
        }

        /**
         * Changes the cell border attributes in the current selection.
         *
         * @param {Object} borderAttributes
         *  An attribute map containing all border attributes to be changed in
         *  the current selection.
         *
         * @param {Object} [operationOptions]
         *  A map with additional properties for the generated 'fillCellRange'
         *  operation.
         */
        function setBorderAttributes(ranges, borderAttributes, operationOptions) {

            var // the entire sheet range
                sheetRange = model.getSheetRange(),
                // the generator for the 'fillCellRange' operations
                generator = model.createOperationsGenerator();

            // generates a 'fillCellRange' operation for the adjacent cells of one border
            function generateAdjacentOperation(firstCol, firstRow, lastCol, lastRow, border, newBorderName) {
                var range = { start: [firstCol, firstRow], end: [lastCol, lastRow] },
                    options = null;
                if (SheetUtils.rangeContainsRange(sheetRange, range) && _.isObject(border)) {
                    options = _({}).extend(operationOptions, { attrs: { cell: Utils.makeSimpleObject(newBorderName, border) } });
                    generator.generateRangeOperation(Operations.FILL_CELL_RANGE, self.getActiveSheet(), range, options);
                }
            }

            // add the border attributes to the passed operation options, and set it to border mode
            operationOptions = _({}).extend(operationOptions, { attrs: { cell: borderAttributes }, rangeBorders: true });

            // send the 'fillCellRange' operations
            _(ranges).each(function (range) {
                // generate operation for the current range
                generator.generateRangeOperation(Operations.FILL_CELL_RANGE, self.getActiveSheet(), range, operationOptions);

                // Deleting all borders at neighboring cells -> first approach: delete all neighboring borders, so that
                // there is always only one valid border between two cells.
                // Generate operation for the adjacent cells of the range
                if (('attrs' in operationOptions) && ('cell' in operationOptions.attrs)) {
                    if ('borderLeft' in operationOptions.attrs.cell) {
                        generateAdjacentOperation(range.start[0] - 1, range.start[1], range.start[0] - 1, range.end[1], Border.NONE, 'borderRight');
                    }
                    if ('borderRight' in operationOptions.attrs.cell) {
                        generateAdjacentOperation(range.end[0] + 1, range.start[1], range.end[0] + 1, range.end[1], Border.NONE, 'borderLeft');
                    }
                    if ('borderTop' in operationOptions.attrs.cell) {
                        generateAdjacentOperation(range.start[0], range.start[1] - 1, range.end[0], range.start[1] - 1, Border.NONE, 'borderBottom');
                    }
                    if ('borderBottom' in operationOptions.attrs.cell) {
                        generateAdjacentOperation(range.start[0], range.end[1] + 1, range.end[0], range.end[1] + 1, Border.NONE, 'borderTop');
                    }
                }
            });
            model.applyOperations(generator.getOperations());

            // update all cell collections and the grid panes (used until view update has been received)
            invokeCellCollectionMethod('setBorderAttributes', ranges, borderAttributes, operationOptions);
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the specified grid pane.
         *
         * @param {String} panePos
         *  The position identifier of the grid pane.
         *
         * @returns {GridPane}
         *  The grid pane instance at the specified pane position.
         */
        this.getGridPane = function (panePos) {
            return gridPanes[panePos];
        };

        /**
         * Returns the specified grid pane, if it is visible. Otherwise,
         * returns the nearest visible grid pane.
         *
         * @param {String} panePos
         *  The position identifier of the preferred grid pane. If only one
         *  grid pane is visible, it will be returned regardless of this
         *  parameter. If one of the pane sides contained in the passed pane
         *  position is not visible, returns the grid pane from the other
         *  visible pane side (for example, if requesting the top-left pane
         *  while only bottom panes are visible, returns the bottom-left pane).
         *
         * @returns {GridPane}
         *  The grid pane instance at the specified pane position.
         */
        this.getVisibleGridPane = function (panePos) {

            // jump to other column pane, if the pane side is hidden
            if (!paneSideSettings[getColPaneSide(panePos)].visible) {
                panePos = getNextColPanePos(panePos);
            }

            // jump to other row pane, if the pane side is hidden
            if (!paneSideSettings[getRowPaneSide(panePos)].visible) {
                panePos = getNextRowPanePos(panePos);
            }

            // now, panePos points to a visible grid pane (at least one pane is always visible)
            return gridPanes[panePos];
        };

        /**
         * Returns the specified header pane.
         *
         * @param {String} paneSide
         *  The identifier of the pane side ('left', 'right', 'top', 'bottom').
         *
         * @returns {HeaderPane}
         *  The header pane instance at the specified pane side.
         */
        this.getHeaderPane = function (paneSide) {
            return headerPanes[paneSide];
        };

        /**
         * Returns the horizontal header pane (left or right) for the specified
         * pane position.
         *
         * @param {String} panePos
         *  The position identifier of the grid pane.
         *
         * @returns {HeaderPane}
         *  The horizontal header pane instance at the corresponding pane side.
         */
        this.getColHeaderPane = function (panePos) {
            return headerPanes[getColPaneSide(panePos)];
        };

        /**
         * Returns the vertical header pane (top or bottom) for the specified
         * pane position.
         *
         * @param {String} panePos
         *  The position identifier of the grid pane.
         *
         * @returns {HeaderPane}
         *  The vertical header pane instance at the corresponding pane side.
         */
        this.getRowHeaderPane = function (panePos) {
            return headerPanes[getRowPaneSide(panePos)];
        };

        /**
         * Returns the first visible header pane in the specified direction
         * (for example, for columns returns the left header pane, if it is
         * visible, otherwise the right header pane).
         *
         * @param {Boolean} columns
         *  If set to true, returns the first visible horizontal header pane
         *  (left or right); otherwise returns the first visible vertical
         *  header pane (top or bottom).
         *
         * @returns {HeaderPane}
         *  The first visible header pane instance in the specified direction.
         */
        this.getVisibleHeaderPane = function (columns) {
            return columns ?
                (paneSideSettings.left.visible ? headerPanes.left : headerPanes.right) :
                (paneSideSettings.top.visible ? headerPanes.top : headerPanes.bottom);
        };

        /**
         * Returns the top-left corner pane.
         *
         * @returns {CornerPane}
         *  The corner pane instance.
         */
        this.getCornerPane = function () {
            return cornerPane;
        };

        /**
         * Collects several update requests sends a single deferred server
         * request.
         *
         * @param {Object} requestProps
         *  The properties to be inserted into the current request data object.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.requestUpdate = (function () {

            var // the data object collecting all grid pane requests
                requestData = null,
                // the last running request
                runningDef = null,
                // the start time of the initial request (without other requests already running)
                startTime = 0,
                // counter for running requests (next free index)
                nextRequestIndex = 0,
                // minimum request index valid for evaluation (ignore requests after switching to another sheet)
                requiredRequestIndex = 0;

            // direct callback: register a new sheet rectangle to be updated in a grid pane
            function registerUpdateRequest(requestProps) {

                // do not request anything in error mode
                if (app.getState() === 'error') { return self; }

                // initialize request data if missing
                requestData = requestData || { sheet: self.getActiveSheet() };

                // whether to update the cell ranges completely
                if (('cells' in requestData) && ('cells' in requestProps)) {
                    requestProps.cells = getMergedGridPaneRanges();
                }

                // insert the passed request data
                _(requestData).extend(requestProps);

                return self;
            }

            // deferred callback: send single server request and notify all waiting grid panes
            function executeUpdateRequest() {

                var // local reference to the current request dats
                    localRequestData = requestData,
                    // current request index
                    localRequestIndex = nextRequestIndex,
                    // start time of this request
                    localStartTime = _.now(),
                    // the new server request
                    def = null;

                if (!localRequestData) { return; }
                requestData = null;

                // take start time, if no other request is running already
                if (!runningDef) {
                    startTime = _.now();
                }

                // send request and immediately delete the request data (new updates may be requested in the meantime)
                localRequestIndex = nextRequestIndex;
                nextRequestIndex += 1;
                Utils.info('SpreadsheetView.executeUpdateRequest(): sending update request #' + (localRequestIndex + 1) + '...');
                Utils.log(localRequestData);
                def = runningDef = app.sendFilterRequest({
                    method: 'POST',
                    params: {
                        action: 'updateview',
                        requestdata: JSON.stringify(localRequestData)
                    }
                });

                def.fail(function () {
                    Utils.error('SpreadsheetView.executeUpdateRequest(): update request #' + (localRequestIndex + 1) + ' failed');
                })
                .done(function (layoutData) {

                    Utils.info('SpreadsheetView.executeUpdateRequest(): update response #' + (localRequestIndex + 1) + ' received (' + (_.now() - localStartTime) + 'ms)');
                    Utils.log(layoutData);

                    // check validity of response object
                    if (!_.isObject(layoutData) || !_.isObject(layoutData.sheet)) {
                        Utils.error('SpreadsheetView.executeUpdateRequest(): missing required layout data');
                        return;
                    }

                    // ignore this response if the active sheet has changed in the meantime
                    if (localRequestIndex < requiredRequestIndex) { return; }

                    // copy sheet layout data to internal class member
                    sheetLayoutData = _.copy(layoutData.sheet, true);

                    // initialize the cell collections for all grid panes
                    if (_.isArray(layoutData.cells)) {
                        invokeCellCollectionMethod('importCellEntries', localRequestData.cells, layoutData.cells);
                    }

                    // copy layout data for the current selection
                    if (_.isObject(layoutData.selection)) {
                        selectionLayoutData = _.copy(DEFAULT_SELECTION_DATA, true);
                        if (_.isObject(layoutData.selection.active)) {
                            selectionLayoutData.active = _.clone(layoutData.selection.active);
                            selectionLayoutData.active.attrs = cellStyles.getMergedAttributes(selectionLayoutData.active.attrs || {});
                        } else {
                            Utils.warn('SpreadsheetView.executeUpdateRequest(): missing layout data of active cell');
                        }
                        if (_.isObject(layoutData.selection.borders)) {
                            _(selectionLayoutData.borders).extend(layoutData.selection.borders);
                        } else {
                            Utils.warn('SpreadsheetView.executeUpdateRequest(): missing merged border attributes');
                        }
                        if (_.isObject(layoutData.selection.subtotals)) {
                            _(selectionLayoutData.subtotals).extend(layoutData.selection.subtotals);
                        } else {
                            Utils.warn('SpreadsheetView.executeUpdateRequest(): missing subtotal results');
                        }
                    }

                    // Check if this is the last request. Do not notify listeners about response
                    // data of previous requests, but collect the response data and all change
                    // flags of multiple requests, and notify listeners after last response.
                    if (def === runningDef) {
                        runningDef = null;
                        self.trigger('change:layoutdata');
                    } else if (_.now() - startTime >= 3000) {
                        startTime = _.now();
                        self.trigger('change:layoutdata');
                    }
                });
            }

            // delete the current cached request data when the active sheet changes
            self.on('before:activesheet', function () {
                requestData = null;
                requiredRequestIndex = nextRequestIndex;
            });

            // create and return the debounced method SpreadsheetView.requestUpdate()
            return app.createDebouncedMethod(registerUpdateRequest, executeUpdateRequest, { delay: 100, maxDelay: 250 });

        }()); // SpreadsheetView.requestUpdate()

        /**
         * Sets the scroll position of the grid panes and header panes next to
         * the specified grid pane.
         *
         * @internal
         *  Called from the implementation of the GridPane class.
         *
         * @param {String} panePos
         *  The position of the grid pane with the modified scroll position.
         *
         * @param {Number} scrollLeft
         *  The new horizontal scroll position of the specified grid pane.
         *
         * @param {Number} scrollTop
         *  The new vertical scroll position of the specified grid pane.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.syncGridPaneScroll = function (panePos, scrollLeft, scrollTop) {

            // first update scroll position of header panes (updates available column/row interval)
            headerPanes[getColPaneSide(panePos)].scroll(scrollLeft);
            headerPanes[getRowPaneSide(panePos)].scroll(scrollTop);

            // update scroll position of other grid panes
            gridPanes[getNextRowPanePos(panePos)].scrollLeft(scrollLeft);
            gridPanes[getNextColPanePos(panePos)].scrollTop(scrollTop);

            return this;
        };

        // active sheet -------------------------------------------------------

        /**
         * Returns the zero-based index of the active sheet in this spreadsheet
         * view instance.
         *
         * @returns {Number}
         *  The zero-based index of the active sheet.
         */
        this.getActiveSheet = function () {
            return sheetLayoutData.sheet;
        };

        /**
         * Returns the model instance of the active sheet in this spreadsheet
         * view instance.
         *
         * @returns {SheetModel}
         *  The model instance of the active sheet.
         */
        this.getActiveSheetModel = function () {
            return model.getSheetModel(sheetLayoutData.sheet);
        };

        /**
         * Selects a sheet in this spreadsheet view instance.
         *
         * @param {Number} sheet
         *  The zero-based index of the new current sheet.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setActiveSheet = function (sheet) {
            setActiveSheet(sheet);
            return this;
        };

        /**
         * Returns the number of columns in the used area of the active sheet.
         *
         * @returns {Number}
         *  The number of columns in the used area of the active sheet.
         */
        this.getUsedCols = function () {
            return sheetLayoutData.usedCols;
        };

        /**
         * Returns the number of rows in the used area of the active sheet.
         *
         * @returns {Number}
         *  The number of rows in the used area of the active sheet.
         */
        this.getUsedRows = function () {
            return sheetLayoutData.usedRows;
        };

        /**
         * Returns a specific property of the view settings of the active
         * sheet.
         *
         * @param {String} propName
         *  The name of the view property.
         *
         * @returns {Any}
         *  A clone of the property value.
         */
        this.getSheetViewProperty = function (propName) {
            return this.getActiveSheetModel().getViewProperty(propName);
        };

        /**
         * Changes specific view settings of the active sheet.
         *
         * @param {String} propName
         *  The name of the view property.
         *
         * @param {Any} propValue
         *  The new value of the view property. Must neither be undefined nor
         *  null.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setSheetViewProperty = function (propName, propValue) {
            this.getActiveSheetModel().setViewProperty(propName, propValue);
            return this;
        };

        // selection and scrolling --------------------------------------------

        /**
         * Returns an object containing all information about the selection in
         * the active sheet.
         *
         * @returns {Object}
         *  The cell selection data, in the following properties:
         *  - {Array} ranges
         *      Array of range addresses for all selected ranges.
         *  - {Number} activeRange
         *      Element index of the active range in the 'ranges' property.
         *  - {Number[]} activeCell
         *      Logical cell address of the active cell in the active range.
         *  - {Array} drawings
         *      Array with logical positions of all selected drawing frames.
         */
        this.getSelection = function () {
            return this.getActiveSheetModel().getSelection();
        };

        /**
         * Returns all selected cell ranges in the active sheet.
         *
         * @returns {Array}
         *  The selected cell ranges, as array of logical range positions.
         */
        this.getSelectedRanges = function () {
            return this.getSelection().ranges;
        };

        /**
         * Returns the active cell of the selection in the active sheet (the
         * highlighted cell in the active range that will be edited when
         * switching to text edit mode).
         *
         * @returns {Number[]}
         *  The logical position of the active cell.
         */
        this.getActiveCell = function () {
            return this.getSelection().activeCell;
        };

        /**
         * Returns whether the current selection contains at least one range
         * spanning over multiple columns.
         *
         * @returns {Boolean}
         *  Whether the current selection contains at least one range spanning
         *  over multiple columns.
         */
        this.hasMultipleColumnsSelected = function () {
            return _(this.getSelectedRanges()).any(function (range) { return SheetUtils.getColCount(range) > 1; });
        };

        /**
         * Returns whether the current selection contains at least one range
         * spanning over multiple columns.
         *
         * @returns {Boolean}
         *  Whether the current selection contains at least one range spanning
         *  over multiple columns.
         */
        this.hasMultipleRowsSelected = function () {
            return _(this.getSelectedRanges()).any(function (range) { return SheetUtils.getRowCount(range) > 1; });
        };

        /**
         * Returns, whether a drawing object is selected in the current active
         * sheet.
         *
         * @returns {Boolean}
         *  Whether a drawing object is selected in the active sheet.
         */
        this.hasDrawingSelection = function () {
            return this.getActiveSheetModel().hasDrawingSelection();
        };

        /**
         * Returns the logical positions of all selected drawing frames in the
         * active sheet.
         *
         * @returns {Array}
         *  The selected drawng frames, as array of logical drawing positions.
         */
        this.getSelectedDrawings = function () {
            return this.getSelection().drawings;
        };

        /**
         * Changes the cell selection in the active sheet, and deselects all
         * drawing frames.
         *
         * @param {Object} selection
         *  The cell selection data, in the properties 'ranges' (array of range
         *  addresses), 'activeRange' (array index of the active range in the
         *  'ranges' property), and 'activeCell' (logical cell address).
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setCellSelection = function (selection) {

            // deselect all drawing frames
            selection = _({}).extend(selection, { drawings: [] });

            // it might be necessary to expand the selection, if a merged cell
            // is only partly covered by the selection
            expandMergedSelection(selection);

            this.getActiveSheetModel().setSelection(selection);
            return this;
        };

        /**
         * Selects a single cell in the active sheet. The cell will become the
         * active cell of the sheet.
         *
         * @param {Number[]} address
         *  The logical address of the cell to be selected.
         *
         * @param {Object} options
         *  A map of options to control the behavior of this method. The
         *  following options are supported:
         *  @param {Boolean} [options.append=false]
         *      If set to true, the new cell will be appended to the current
         *      selection. Otherwise, the old selection will be removed.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.selectCell = function (address, options) {
            return this.selectRange({ start: address, end: address }, options);
        };

        /**
         * Selects a single cell range in the active sheet.
         *
         * @param {Object} range
         *  The logical address of the cell range to be selected.
         *
         * @param {Object} options
         *  A map of options to control the behavior of this method. The
         *  following options are supported:
         *  @param {Boolean} [options.append=false]
         *      If set to true, the new range will be appended to the current
         *      selection. Otherwise, the old selection will be removed.
         *  @param {Number[]} [options.active]
         *      The logical address of the active cell in the selected range.
         *      If omitted, the top-left cell of the range will be activated.
         *      Must be located inside the passed range address.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.selectRange = function (range, options) {

            var // adjust the passed range address
                adjustedRange = SheetUtils.getAdjustedRange(range),
                // whether to append the range to the current selection
                append = Utils.getBooleanOption(options, 'append', false),
                // the initial selection to append the new range to
                selection = append ? this.getSelection() : { ranges: [] },
                // the new active cell
                activeCell = Utils.getArrayOption(options, 'active', []);

            // initialize and set the new selection
            selection.ranges.push(adjustedRange);
            selection.activeRange = selection.ranges.length - 1;

            // set active cell
            if ((activeCell.length === 2) && SheetUtils.rangeContainsCell(adjustedRange, activeCell)) {
                selection.activeCell = activeCell;
            } else {
                selection.activeCell = adjustedRange.start;
            }

            // commit new selection
            return this.setCellSelection(selection);
        };

        /**
         * Modifies the position of the current active selection range.
         *
         * @param {Object} range
         *  The new logical address of the active selection range. If the
         *  passed range does not contain the current active cell, the top-left
         *  cell of the range will be activated.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.changeActiveRange = function (range) {

            var // adjust the passed range address
                adjustedRange = SheetUtils.getAdjustedRange(range),
                // the initial selection to append the new range to
                selection = this.getSelection();

            // initialize the new active range
            selection.ranges[selection.activeRange] = adjustedRange;
            if (!SheetUtils.rangeContainsCell(adjustedRange, selection.activeCell)) {
                selection.activeCell = adjustedRange.start;
            }

            // commit new selection
            return this.setCellSelection(selection);
        };

        /**
         * Selects a single drawing frame in the active sheet.
         *
         * @param {Number[]} position
         *  The logical position of the drawing frame to be selected.
         *
         * @param {Object} options
         *  A map of options to control the behavior of this method. The
         *  following options are supported:
         *  @param {Boolean} [options.toggle=false]
         *      If set to true, the selection state of the addressed drawing
         *      frame will be toggled. Otherwise, the old drawing selection
         *      will be removed, and the addressed drawing frame will be
         *      selected.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.selectDrawing = function (position, options) {

            var // the current selection to be modified (keep cell selection as it is)
                selection = this.getSelection();

            // construct the new drawing selection
            if (!Utils.getBooleanOption(options, 'toggle', false)) {
                selection.drawings = [position];
            } else if (Utils.spliceValue(selection.drawings, position) === 0) {
                selection.drawings.push(position);
            }

            // commit new selection
            this.getActiveSheetModel().setSelection(selection);
            return this;
        };

        /**
         * Deselects all selected drawings and returns to cell selection mode.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.removeDrawingSelection = function () {
            return this.setCellSelection(this.getSelection());
        };

        /**
         * Sets either the horizontal or vertical scroll position of the grid
         * panes and header pane associated with the specified pane side.
         *
         * @param {String} paneSide
         *  The pane side identifier ('left', 'right', 'top', or 'bottom').
         *
         * @param {Number} scrollPos
         *  The new scroll position of the specified grid and header panes.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.scrollPaneSide = function (paneSide, scrollPos) {

            var // the method name used to scroll the grid pane
                scrollMethod = HEADER_PANE_INFOS[paneSide].columns ? 'scrollLeft' : 'scrollTop';

            // update scroll position of the grid panes and header pane
            gridPanes[HEADER_PANE_INFOS[paneSide].panePos1][scrollMethod](scrollPos);
            gridPanes[HEADER_PANE_INFOS[paneSide].panePos2][scrollMethod](scrollPos);
            headerPanes[paneSide].scroll(scrollPos);

            return this;
        };

        /**
         * Scrolls the grid panes and header pane associated with the specified
         * pane side to make a column or row visible.
         *
         * @param {String} paneSide
         *  The pane side identifier ('left', 'right', 'top', or 'bottom').
         *
         * @param {Number} index
         *  The zero-based column or row index to be made visible.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.scrollPaneSideToEntry = function (paneSide, index) {

            var // the method name used to scroll the grid pane
                scrollMethod = HEADER_PANE_INFOS[paneSide].columns ? 'scrollToColumn' : 'scrollToRow';

            // update scroll position of the grid panes
            gridPanes[HEADER_PANE_INFOS[paneSide].panePos1][scrollMethod](index);
            gridPanes[HEADER_PANE_INFOS[paneSide].panePos2][scrollMethod](index);

            return this;
        };

        /**
         * Returns the value of the specified subtotal in the current
         * selection.
         *
         * @param {String} type
         *  The type identifier of the subtotal value.
         *
         * @returns {Number}
         *  The current value of the specified subtotal.
         */
        this.getSubtotalValue = function (type) {
            return selectionLayoutData.subtotals[type] || 0;
        };

        // split and freeze ---------------------------------------------------

        this.getSplitMode = function () {
            return sheetViewSettings.split;
        };

        this.setSplitMode = function (state) {
            if (self.getSplitMode() !== state) {
                sheetViewSettings.split = state;
                if (state && (sheetViewSettings.splitLeft === 0) && (sheetViewSettings.splitTop === 0)) {
                    sheetViewSettings.splitLeft = 7000;
                    sheetViewSettings.splitTop = 5000;
                }
                initializePanes();
            }
            return this;
        };

        this.setSplitPosition = function (left, top) {
            left = Math.max(0, left);
            top = Math.max(0, top);
            // TODO: calculate split position according to cursor position
            if ((sheetViewSettings.splitLeft !== left) || (sheetViewSettings.splitTop !== top)) {
                sheetViewSettings.splitLeft = left;
                sheetViewSettings.splitTop = top;
                sheetViewSettings.split = sheetViewSettings.split && ((left > 0) || (top > 0));
                initializePanes();
            }
            return this;
        };

        this.getFreezeMode = function () {
            return sheetViewSettings.freeze;
        };

        this.setFreezeMode = function (state) {
            if (self.getFreezeMode() !== state) {
                sheetViewSettings.freeze = state;
                // TODO: calculate split position according to cursor position
                if (state && (sheetViewSettings.splitLeft === 0) && (sheetViewSettings.splitTop === 0)) {
                    sheetViewSettings.splitLeft = 7000;
                    sheetViewSettings.splitTop = 5000;
                }
                initializePanes();
            }
            return this;
        };

        // sheet operations ---------------------------------------------------

        /**
         * Inserts a new sheet in the spreadsheet document, and activates it in
         * the view.
         *
         * @param {String} [sheetName]
         *  The new name of the sheet. If missing, or if this name exists
         *  already, a new unused name will be generated.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.insertSheet = function (sheetName) {

            var insertIndex = this.getActiveSheet() + 1,
                nameIndex = model.getSheetCount();

            // validate sheet name, or generate a valid name
            sheetName = _.isString(sheetName) ? Utils.cleanString(sheetName) : '';
            while ((sheetName.length === 0) || model.hasSheet(sheetName)) {
                sheetName = generateSheetName(nameIndex);
                nameIndex += 1;
            }

            // insert the new sheet into the document, activate the sheet
            if (model.insertSheet(insertIndex, sheetName)) {
                setActiveSheet(insertIndex);
            }

            return this;
        };

        /**
         * Deletes the active sheet from the spreadsheet document, and
         * activates another sheet in the view. Does nothing, if the document
         * contains a single sheet only.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.deleteSheet = function () {
            if (model.getSheetCount() > 1) {
                // handler of the 'delete:sheet' event will update active sheet
                model.deleteSheet(this.getActiveSheet());
            }
            return this;
        };

        /**
         * Renames the active sheet in the spreadsheet document, if possible.
         *
         * @param {String} sheetName
         *  The new name of the active sheet.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.renameSheet = function (sheetName) {
            model.setSheetName(this.getActiveSheet(), sheetName);
            return this;
        };

        /**
         * Returns the merged formatting attributes of the active sheet.
         *
         * @returns {Object}
         *  The merged attribute set of the active sheet.
         */
        this.getSheetAttributes = function () {
            return this.getActiveSheetModel().getMergedAttributes();
        };

        // column operations --------------------------------------------------

        /**
         * Returns, whether additional columns can be inserted into the sheet,
         * without exceeding the maximum number of columns. For this it is
         * neccessary, that the number of used columns plus the number of
         * selected columns is smaller than the maximum number of columns.
         *
         * @returns {Boolean}
         *  Whether additional columns can be inserted corresponding to the
         *  current selection and the number of used columns in the sheet.
         */
        this.columnsInsertable = function () {

            var // the column intervals in the current selection
                intervals = SheetUtils.getColIntervals(this.getSelectedRanges());

            return (this.getUsedCols() + Utils.getSum(intervals, SheetUtils.getIntervalSize)) <= (model.getMaxCol() + 1);
        };

        /**
         * Inserts new columns into the active sheet, according to the current
         * selection.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.insertColumns = function () {

            var // the column intervals in the current selection
                intervals = SheetUtils.getColIntervals(this.getSelectedRanges()),
                // the generator for the operations
                generator = model.createOperationsGenerator();

            // send operations for all intervals in reverse order (!)
            Utils.iterateArray(intervals, function (interval) {
                generator.generateIntervalOperation(Operations.INSERT_COLUMNS, self.getActiveSheet(), interval);
            }, { reverse: true });
            model.applyOperations(generator.getOperations());
            return this;
        };

        /**
         * Deletes existing columns from the active sheet, according to the
         * current selection.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.deleteColumns = function () {

            var // the column intervals in the current selection
                intervals = SheetUtils.getColIntervals(this.getSelectedRanges()),
                // the generator for the operations
                generator = model.createOperationsGenerator();

            // send operations for all intervals in reverse order (!)
            Utils.iterateArray(intervals, function (interval) {
                generator.generateIntervalOperation(Operations.DELETE_COLUMNS, self.getActiveSheet(), interval);
            }, { reverse: true });
            model.applyOperations(generator.getOperations());
            return this;
        };

        /**
         * Changes the width of columns in the active sheet, according to the
         * current selection.
         *
         * @param {Number} width
         *  The column width in 1/100 mm. If this value is less than 1, the
         *  columns will be hidden, and their original width will not be
         *  changed.
         *
         * @param {Number} [targetCol]
         *  The target column to be changed. If specified, the column width
         *  will be set to that column only. If the current selection contains
         *  any ranges covering this column completely, the column width will
         *  be set to all columns contained in these ranges.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setColumnWidth = function (width, targetCol) {

            var // the current selection
                selection = this.getSelection(),
                // the selected ranges
                ranges = selection.ranges,
                // the column intervals
                intervals = null,
                // the generator for the operations
                generator = model.createOperationsGenerator(),
                // the attributes to be inserted into the operations
                attributes = { column: (width < 1) ? { visible: false } : { visible: true, width: width, customWidth: true } };

            // use column ranges in selection, if target column has been passed
            if (_.isNumber(targetCol)) {
                ranges = _(ranges).filter(_.bind(model.isColRange, model));
                // do not use the column ranges, if they do not contain the target column
                if (!_(ranges).any(function (range) { return SheetUtils.rangeContainsCol(range, targetCol); })) {
                    ranges = [model.makeColRange(targetCol)];
                }
            }

            // convert ranges to column intervals
            intervals = SheetUtils.getColIntervals(ranges);

            // send the 'setColumnAttributes' operations
            _(intervals).each(function (interval) {
                generator.generateIntervalOperation(Operations.SET_COLUMN_ATTRIBUTES, self.getActiveSheet(), interval, { attrs: attributes });
            });
            model.applyOperations(generator.getOperations());
            return this;
        };

        // row operations -----------------------------------------------------

        /**
         * Returns, whether additional rows can be inserted into the sheet,
         * without exceeding the maximum number of rows. For this it is
         * neccessary, that the number of used rows plus the number of
         * selected rows is smaller than the maximum number of rows.
         *
         * @returns {Boolean}
         *  Whether additional rows can be inserted corresponding to the
         *  current selection and the number of used rows in the sheet.
         */
        this.rowsInsertable = function () {

            var // the row intervals in the current selection
                intervals = SheetUtils.getRowIntervals(this.getSelectedRanges());

            return (this.getUsedRows() + Utils.getSum(intervals, SheetUtils.getIntervalSize)) <= (model.getMaxRow() + 1);
        };

        /**
         * Inserts new rows into the active sheet, according to the current
         * selection.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.insertRows = function () {

            var // the row intervals in the current selection
                intervals = SheetUtils.getRowIntervals(this.getSelectedRanges()),
                // the generator for the operations
                generator = model.createOperationsGenerator();

            // send operations for all intervals in reverse order (!)
            Utils.iterateArray(intervals, function (interval) {
                generator.generateIntervalOperation(Operations.INSERT_ROWS, self.getActiveSheet(), interval);
            }, { reverse: true });
            model.applyOperations(generator.getOperations());
            return this;
        };

        /**
         * Deletes existing rows from the active sheet, according to the
         * current selection.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.deleteRows = function () {

            var // the row intervals in the current selection
                intervals = SheetUtils.getRowIntervals(this.getSelectedRanges()),
                // the generator for the operations
                generator = model.createOperationsGenerator();

            // send operations for all intervals in reverse order (!)
            Utils.iterateArray(intervals, function (interval) {
                generator.generateIntervalOperation(Operations.DELETE_ROWS, self.getActiveSheet(), interval);
            }, { reverse: true });
            model.applyOperations(generator.getOperations());
            return this;
        };

        /**
         * Changes the height of rows in the active sheet, according to the
         * current selection.
         *
         * @param {Number} height
         *  The row height in 1/100 mm. If this value is less than 1, the rows
         *  will be hidden, and their original height will not be changed.
         *
         * @param {Number} [targetRow]
         *  The target row to be changed. If specified, the row height will be
         *  set to that row only. If the current selection contains any ranges
         *  covering this row completely, the row height will be set to all
         *  rows contained in these ranges.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setRowHeight = function (height, targetRow) {

            var // the current selection
                selection = this.getSelection(),
                // the selected ranges
                ranges = selection.ranges,
                // the row intervals
                intervals = null,
                // the generator for the operations
                generator = model.createOperationsGenerator(),
                // the attributes to be inserted into the operations
                attributes = { row: (height < 1) ? { visible: false } : { visible: true, height: height, customHeight: true } };

            // use row ranges in selection, if target row has been passed
            if (_.isNumber(targetRow)) {
                ranges = _(ranges).filter(_.bind(model.isRowRange, model));
                // do not use the row ranges, if they do not contain the target row
                if (!_(ranges).any(function (range) { return SheetUtils.rangeContainsRow(range, targetRow); })) {
                    ranges = [model.makeRowRange(targetRow)];
                }
            }

            // convert ranges to row intervals
            intervals = SheetUtils.getRowIntervals(ranges);

            // send the 'setRowAttributes' operations
            _(intervals).each(function (interval) {
                generator.generateIntervalOperation(Operations.SET_ROW_ATTRIBUTES, self.getActiveSheet(), interval, { attrs: attributes });
            });
            model.applyOperations(generator.getOperations());
            return this;
        };

        // cell operations ----------------------------------------------------

        /**
         * Returns the value and formatting of the current active cell.
         *
         * @returns {Object}
         *  The cell value and formatting, in the following properties:
         *  - {String} display
         *      The formatted display string.
         *  - {Null|Number|Boolean|String}
         *      The typed result value of the cell, or the formula result.
         *  - {String} [formula]
         *      The formula definition, if the current cell is a formula cell.
         *  - {Object} attrs
         *      All formatting attributes of the cell, mapped by cell attribute
         *      family ('cell' and 'character'), the identifier of the cell
         *      style sheet in the 'styleId' string property, and formatting
         *      attributes of the column and row the active cell is located in,
         *      in the properties 'column' and 'row' respectively.
         */
        this.getCellContents = function () {
            return _.copy(selectionLayoutData.active, true);
        };

        /**
         * Sets the contents of the active cell, and updates the view according
         * to the new cell value and formatting.
         *
         * @param {String} [value]
         *  The new value for the active cell. If omitted, only the cell
         *  formatting will be changed.
         *
         * @param {Object} [attributes]
         *  A cell attribute set, optionally containing the string property
         *  'styleId', and attribute value maps (name/value pairs), keyed by
         *  the supported attribute families 'cell' and 'character'. Omitted
         *  attributes will not be changed. Attributes set to the value null
         *  will be removed from the cell.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the behavior of this
         *  method. The following options are supported:
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the cell while applying the new attributes.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setCellContents = function (value, attributes, options) {

            var // the current selection
                selection = this.getSelection(),
                // the cell contents object (value and attributes)
                contents = { attrs: attributes },
                // additional properties for the operation
                operationOptions = null;

            // add cell value, if passed
            if (_.isString(value)) {
                if (value.length === 0) {
                    contents.value = null;
                } else {
                    contents.value = value;
                    operationOptions = { parse: Utils.LOCALE };
                }
            } else {
                value = undefined;
            }

            // add all attributes to be cleared
            if (Utils.getBooleanOption(options, 'clear', false)) {
                contents.attrs = Utils.extendOptions(buildNullAttributes(), contents.attrs);
            }

            // do nothing, if neither value nor attributes are present
            if (!_.isObject(contents.attrs) || _.isEmpty(contents.attrs)) { delete contents.attrs; }
            if (_.isEmpty(contents)) { return this; }

            // send the 'setCellContents' operation (operation expects two-dimensional array)
            model.applyOperations(_.extend({
                name: Operations.SET_CELL_CONTENTS,
                sheet: this.getActiveSheet(),
                start: selection.activeCell,
                contents: [[contents]]
            }, operationOptions));

            // update all cell collections (used until view update has been received)
            invokeCellCollectionMethod('setCellContents', selection.activeCell, contents.value, contents.attrs);

            return this;
        };

        /**
         * Fills all cell ranges in the current selection with the same value
         * and formatting, and updates the view according to the new cell value
         * and formatting.
         *
         * @param {String} [value]
         *  The new value for all cells in the current selection. If omitted,
         *  only the cell formatting will be changed.
         *
         * @param {Object} [attributes]
         *  A cell attribute set, optionally containing the string property
         *  'styleId', and attribute value maps (name/value pairs), keyed by
         *  the supported attribute families 'cell' and 'character'. Omitted
         *  attributes will not be changed. Attributes set to the value null
         *  will be removed from the cell.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the behavior of this
         *  method. The following options are supported:
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the selection while applying the new attributes.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.fillCellContents = function (value, attributes, options) {

            var // the current selection
                selection = this.getSelection(),
                // the selected ranges (remove duplicates)
                ranges = SheetUtils.getUniqueRanges(selection.ranges),
                // the generator for the 'fillCellRange' operations
                generator = model.createOperationsGenerator(),
                // additional options for the 'fillCellRange' operations
                operationOptions = null,
                // whether the operations change the cell value
                valueChanged = _.isString(value);

            // add value to the operation options (prepare shared formula)
            if (valueChanged) {
                if (value.length === 0) {
                    operationOptions = { value: null };
                } else {
                    // restrict the number of cells filled at the same time
                    if (_(ranges).reduce(function (memo, range) { return memo + SheetUtils.getCellCount(range); }, 0) > MAX_FILL_CELL_COUNT) {
                        self.yell('warning', gt('It is not possible to modify more than %1$d cells at the same time.', _.noI18n(MAX_FILL_CELL_COUNT)));
                        return this;
                    }
                    operationOptions = { value: value, parse: Utils.LOCALE, shared: 0, ref: selection.activeCell };
                }
            } else {
                value = undefined;
            }

            // add all attributes to be cleared
            if (Utils.getBooleanOption(options, 'clear', false)) {
                attributes = Utils.extendOptions(buildNullAttributes(), attributes);
            }

            // add attributes to operation options
            if (_.isObject(attributes) && !_.isEmpty(attributes)) {
                operationOptions = Utils.extendOptions(operationOptions, { attrs: attributes });
            } else {
                attributes = undefined;
            }

            // do nothing, if neither value nor attributes are present
            if (!_.isString(value) && !attributes) { return this; }

            // send the 'fillCellRange' operations
            _(ranges).each(function (range, index) {
                if (!valueChanged && model.isColRange(range)) {
                    generator.generateIntervalOperation(Operations.SET_COLUMN_ATTRIBUTES, self.getActiveSheet(), SheetUtils.getColInterval(range), { attrs: attributes });
                } else if (!valueChanged && model.isRowRange(range)) {
                    generator.generateIntervalOperation(Operations.SET_ROW_ATTRIBUTES, self.getActiveSheet(), SheetUtils.getRowInterval(range), { attrs: attributes });
                } else {
                    generator.generateRangeOperation(Operations.FILL_CELL_RANGE, self.getActiveSheet(), range, operationOptions);
                    if ((index === 0) && valueChanged) {
                        delete operationOptions.value;
                        delete operationOptions.parse;
                        delete operationOptions.ref;
                    }
                }
            });
            model.applyOperations(generator.getOperations());

            // update all cell collections (used until view update has been received)
            invokeCellCollectionMethod('fillCellContents', ranges, value, attributes);

            return this;
        };

        /**
         * Changes a single attribute of type 'cell' for all cells in the
         * current selection.
         *
         * @param {String} name
         *  The name of the cell attribute.
         *
         * @param {Any} value
         *  The new value of the cell attribute. If set to the value null, all
         *  attributes set explicitly will be removed from the cells.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setCellAttribute = function (name, value) {
            return this.fillCellContents(undefined, { cell: Utils.makeSimpleObject(name, value) });
        };

        /**
         * Changes a single attribute of type 'character' for all cells in the
         * current selection.
         *
         * @param {String} name
         *  The name of the character attribute.
         *
         * @param {Any} value
         *  The new value of the character attribute. If set to the value null,
         *  all attributes set explicitly will be removed from the cells.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setCharacterAttribute = function (name, value) {
            return this.fillCellContents(undefined, { character: Utils.makeSimpleObject(name, value) });
        };

        /**
         * removes all explicit formatting attributes from all cells in the
         * current selection.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.clearAttributes = function () {
            return this.fillCellContents(undefined, undefined, { clear: true });
        };

        /**
         * Returns the border mode of the current selection.
         *
         * @returns {Object}
         *  The border mode representing the visibility of the borders in all
         *  ranges of the current selection. See 'MergedBorder.getBorderMode()'
         *  for more details.
         */
        this.getBorderMode = function () {
            return MergedBorder.getBorderMode(selectionLayoutData.borders);
        };

        /**
         * Changes the visibility of the borders in all ranges of the current
         * selection.
         *
         * @param {Object} borderMode
         *  The border mode representing the visibility of the borders in all
         *  ranges of the current selection. See 'MergedBorder.getBorderMode()'
         *  for more details.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setBorderMode = function (borderMode) {

            var // the selected ranges (remove duplicates)
                ranges = SheetUtils.getUniqueRanges(self.getSelectedRanges()),
                // all border attributes to be applied at the selection
                borderAttributes = MergedBorder.getBorderAttributes(borderMode, selectionLayoutData.borders);

            // reduce all ranges to a single column/row, if only a single outer line will be changed
            if (_.size(borderAttributes) === 1) {
                if ('borderLeft' in borderAttributes) {
                    _(ranges).each(function (range) { range.end[0] = range.start[0]; });
                } else if ('borderRight' in borderAttributes) {
                    _(ranges).each(function (range) { range.start[0] = range.end[0]; });
                } else if ('borderTop' in borderAttributes) {
                    _(ranges).each(function (range) { range.end[1] = range.start[1]; });
                } else if ('borderBottom' in borderAttributes) {
                    _(ranges).each(function (range) { range.start[1] = range.end[1]; });
                }
            }

            // send the 'fillCellRange' operations and update the grid panes
            setBorderAttributes(ranges, borderAttributes);
            return this;
        };

        /**
         * Returns the merged border attributes of the current selection.
         *
         * @returns {Object}
         *  An attribute map containing the merged borders of all available
         *  border attributes ('borderTop', 'borderLeft', etc.) in the current
         *  selection. Each border attribute value contains the properties
         *  'style', 'width', and 'color'. If the border property is equal in
         *  all affected cells, it will be set as property value in the border
         *  object, otherwise the property will be set to the value null.
         */
        this.getBorderAttributes = function () {
            return _.copy(selectionLayoutData.borders, true);
        };

        /**
         * Changes certain properties of all visible cell borders in the
         * current selection.
         *
         * @param {Border} border
         *  A border attribute, may be incomplete. All properties contained in
         *  this object will be set for all visible borders in the current
         *  selection. Omitted border properties will not be changed.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setBorderAttributes = function (border) {

            var // the selected ranges (remove duplicates)
                ranges = SheetUtils.getUniqueRanges(self.getSelectedRanges()),
                // all border attributes to be applied at the selection
                borderAttributes = {};

            // insert all partial border attributes into the attribute map
            _(MergedBorder.CELL_BORDER_ATTRIBUTES).each(function (attrName) {
                borderAttributes[attrName] = border;
            });

            // send the 'fillCellRange' operations and update the grid panes
            setBorderAttributes(ranges, borderAttributes, { visibleBorders: true });
            return this;
        };

        /**
         * Merging all cells of the current selection
         *
         * @param {String} type
         *  The merge type. Must be one of: 'merge', 'horizontal', 'vertical',
         *  or 'unmerge'.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.mergeCells = function (type) {

            var // the current selection
                selection = this.getSelection(),
                // the selected ranges (remove duplicates)
                ranges = SheetUtils.getUniqueRanges(selection.ranges),
                // collecting the ranges, that really need to be modified
                modifiedRanges = [],
                // the generator for the 'fillCellRange' operations
                generator = model.createOperationsGenerator();

            // do nothing, if the type is not defined
            if (!_.isString(type)) { return this; }

            // do nothing, if the selection contains overlapping ranges
            if (SheetUtils.anyRangesOverlap(ranges)) { return this; }

            // send the 'mergeCells' operations
            // There might be an 'unmerge' necessary before the merge is send. For example vertically merged cells
            // need to be unmerged, before the cells are merged horizontally.
            _(ranges).each(function (range) {
                var createOperation = true,
                    createAdditionalUnmerge = ((type === 'horizontal') || (type === 'vertical'));

                // do nothing, if there is no merged cell in range to unmerge
                if ((type === 'unmerge') && (! mergeCollection.rangeContainsMergedRange(range))) { createOperation = false; }
                // do nothing, if the range contains only one single cell, that shall be merged
                else if ((type === 'merge') && (this.isSingleCellInRange(range))) { createOperation = false; }
                // do nothing, if the range contains only one single column, that shall be merged horizontally
                else if ((type === 'horizontal') && (SheetUtils.getColCount(range) === 1)) { createOperation = false; }
                // do nothing, if the range contains only one single row, that shall be merged vertically
                else if ((type === 'vertical') && (SheetUtils.getRowCount(range) === 1)) { createOperation = false; }

                // no additional unmerge required, if really only one unmerged cell is selected
                if ((createAdditionalUnmerge) && (SheetUtils.getCellCount(range) === 1)) {
                    createAdditionalUnmerge = false;
                }

                // creating additional unmerge operation (especially required for type 'horizontal' and 'vertical'
                if (createAdditionalUnmerge) {
                    generator.generateRangeOperation(Operations.MERGE_CELLS, self.getActiveSheet(), range, { type: 'unmerge' });
                    modifiedRanges.push(range);
                }

                // TODO: 'end' must be in the 'used range', if complete columns or rows are selected
                // -> in this case the merge has to be expanded to the 'longest' merge in the used range.
                // Limiting the range for vertical or horizontal merges to the used range, it complete row or columns are selected
                // if ((type === 'horizontal') || (type === 'vertical')) {
                    // (self.getUsedCols() > 0) && (self.getUsedRows() > 0)
                // }

                if (createOperation) {
                    generator.generateRangeOperation(Operations.MERGE_CELLS, self.getActiveSheet(), range, { type: type });
                    if (! createAdditionalUnmerge) {
                        modifiedRanges.push(range);
                    }
                }
            }, this);

            if (! _.isEmpty(modifiedRanges)) {
                model.applyOperations(generator.getOperations());
            }

            return this;
        };

        /**
         * If the selection contains merged cells, this is stored in the
         * selectionLayoutData.
         *
         * @returns {Boolean}
         *  Whether the current selection contains merged cells.
         */
        this.selectionHasMergedCells = function () {
            return mergeCollection.rangesContainMergedRange(self.getSelectedRanges());
        };

        /**
         * Checking if a range contains only one cell. The cell might be
         * a merged cell.
         *
         * @param {Object} range
         *  A logical range address.
         *
         * @returns {Boolean}
         *  Whether the cell is the only (merged) cell in the range.
         */
        this.isSingleCellInRange = function (range) {
            return (SheetUtils.getCellCount(range) === 1) || (mergeCollection.isMergedRange(range));
        };

        /**
         * Checking, if the selection contains exactly one cell. This function
         * also returns 'true', if the cell is a merged cell.
         *
         * @param {Object} [sel]
         *  The cell selection data, in the properties 'ranges' (array of range
         *  addresses), 'activeRange' (array index of the active range in the
         *  'ranges' property), and 'activeCell' (logical cell address).
         *
         * @returns {Boolean}
         *  Whether the current selection contains only exact one cell (merged or
         *  unmerged).
         */
        this.isSingleCellSelection = function (sel) {

            var // the current selection
                selection = sel || this.getSelection(),
                // whether only one cell is selected
                isSingleCellSel = false;

            if (selection.ranges.length > 1) {
                isSingleCellSel = false;
            } else if ((selection.ranges.length === 1) && (_.isEqual(selection.ranges[0].start, selection.ranges[0].end))) {
                isSingleCellSel = true;
            } else {
                isSingleCellSel = this.isSingleCellInRange(selection.ranges[0]);
            }

            return isSingleCellSel;
        };

        // drawing operations -------------------------------------------------

        /**
         * Deletes all drawings currently selected in the active sheet.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.deleteDrawings = function () {

            var // the selected drawing objects
                drawings = this.getSelectedDrawings(),
                // the generator for the 'fillCellRange' operations
                generator = model.createOperationsGenerator();

            // back to cell selection
            this.removeDrawingSelection();

            // send the 'deleteDrawing' operations
            _(drawings).each(function (position) {
                generator.generateSheetOperation(Operations.DELETE_DRAWING, self.getActiveSheet(), position);
            });
            model.applyOperations(generator.getOperations());

            return this;
        };

        // --------------------------------------------------------------------

        this.destroy = (function () {
            var baseMethod = self.destroy;
            return function () {
                _(gridPanes).invoke('destroy');
                _(headerPanes).invoke('destroy');
                _(cellCollections).invoke('destroy');
                // call method of base class
                baseMethod.call(self);
            };
        }());

    } // class SpreadsheetView

    // exports ================================================================

    // derive this class from class EditView
    return EditView.extend({ constructor: SpreadsheetView });

});
