/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/cornerpane',
    ['io.ox/office/tk/utils',
     'io.ox/office/spreadsheet/utils/sheetutils'
    ], function (Utils, SheetUtils) {

    'use strict';

    // class CornerPane =======================================================

    /**
     * Represents the top-left corner in the spreadsheet view.
     *
     * @constructor
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this corner pane.
     */
    function CornerPane(app) {

        var // the spreadsheet view
            view = app.getView(),

            // the container node of this corner pane
            rootNode = $('<div>').addClass('corner-pane unselectable'),

            // the cell DOM node used to calculate the size
            cellNode = $('<div>').addClass('cell noI18n');

        // private methods ----------------------------------------------------

        /**
         * Handles mouse clicks and selects the entire sheet.
         */
        function clickHandler(event) {

            var // the top-left cell in the first visible grid pane, if clicked without SHIFT
                topLeftAddress = event.shiftKey ? null : view.getVisibleGridPane('topLeft').getTopLeftAddress();

            if (event.button === 0) {
                view.selectRange(app.getModel().getSheetRange(), { active: topLeftAddress });
            }

            // return focus to active grid pane, after the event has been processed
            _.defer(function () { view.grabFocus(); });
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the root DOM node of this header pane.
         *
         * @returns {jQuery}
         *  The root node of this header pane, as jQuery object.
         */
        this.getNode = function () {
            return rootNode;
        };

        /**
         * Initializes the size of this corner pane.
         *
         * @param {Number} maxRow
         *  The maximum index of the row to be displayed in row header panes.
         *
         * @returns {CornerPane}
         *  A reference to this instance.
         */
        this.initializePaneLayout = function (maxRow) {
            rootNode.append(cellNode.text(SheetUtils.getRowName(maxRow)));
            rootNode.css({ width: cellNode.width() + 2, height: cellNode.height() + 6 });
            cellNode.remove();
            return this;
        };

        // initialization -----------------------------------------------------

        rootNode.on('mousedown touchstart', clickHandler);

    } // class CornerPane

    // exports ================================================================

    return CornerPane;

});
