/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/cellcollection',
    ['io.ox/core/event',
     'io.ox/office/tk/utils',
     'io.ox/office/editframework/model/format/border',
     'io.ox/office/spreadsheet/utils/sheetutils'
    ], function (Events, Utils, Border, SheetUtils) {

    'use strict';

    var // the attribute names of the outer borders of a cell
        OUTER_BORDER_NAMES = ['borderTop', 'borderBottom', 'borderLeft', 'borderRight'];

    // private global functions ===============================================

    /**
     * Returns a unique map key for the passed logical cell address.
     */
    function getKey(address) {
        return address[0] + '/' + address[1];
    }

    /**
     * Tries to detect the type of the passed cell text with a few simple
     * tests, and returns a partial cell entry object containing the typed
     * value and formula properties.
     *
     * @param {String|Null} [value]
     *  If specified, the new unparsed string value for the cell entry, or null
     *  to clear the cell value.
     *
     * @returns {Object|Null}
     *  A cell entry object containing the properties 'display' with the passed
     *  cell text, 'result' with the converted value (null, number, boolean,
     *  error code, or string), optionally 'formula' if the passed value may be
     *  a formula. If no value was passed, returns null.
     */
    function parseCellValue(value) {

        var // the resulting cell data object
            cellEntry = null;

        // parse value if passed
        if (_.isString(value)) {
            cellEntry = { display: value };
            if (value.length === 0) {
                cellEntry.result = null;
            } else if (/^=./.test(value)) {
                cellEntry.result = null; // result of the formula unknown
                cellEntry.formula = value;
            } else if (/^[\-+]?[0-9]/.test(value)) {
                // TODO: remove current group separator, convert decimal separator before parsing
                cellEntry.result = parseFloat(value);
            } else if (/^true$/i.test(value)) { // TODO: use localized name
                cellEntry.result = true;
            } else if (/^false$/i.test(value)) { // TODO: use localized name
                cellEntry.result = false;
            } else {
                // error codes, simple strings, unrecognized formatted values (e.g. dates)
                cellEntry.result = value;
            }
        } else if (_.isNull(value)) {
            cellEntry = { display: '', result: null };
        }

        return cellEntry;
    }

    // class CellCollection ===================================================

    /**
     * Collects layout information about a range of visible cells. Each
     * collection entry is an object with the following properties:
     * - {String} entry.display
     *      The formatted display string of the cell value. Exists always, will
     *      be an empty string for empty cell.
     * - {Number|String|Boolean|Null} entry.result
     *      The cell value, or the result of a formula cell. Exists always,
     *      will be null for empty cells.
     * - {String} [entry.formula]
     *      The formula definition, if the cell is a formula cell.
     * - {Object} [entry.attrs]
     *      The cell attribute set, if the cell is formatted.
     *
     * @constructor
     *
     * @extends Events
     *
     * @param {SpreadsheetApplication} app
     *  The application instance containing this cell collection.
     */
    function CellCollection(app) {

        var // self reference
            self = this,

            // the spreadsheet view
            view = app.getView(),

            // the collections of the active sheet
            colCollection = null,
            rowCollection = null,

            // all cell entries mapped by column/row index
            map = null,

            // the current bounding cell range, intervals, and visible indexes
            boundRange = null,
            colDimension = { interval: null },
            rowDimension = { interval: null },

            // the style sheet containers of the document
            documentStyles = app.getModel().getDocumentStyles(),
            cellStyles = app.getModel().getCellStyles(),

            // merged cell attributes from the default cell style sheet
            defaultAttributes = null;

        // base constructor ---------------------------------------------------

        Events.extend(this);

        // private methods ----------------------------------------------------

        /**
         * Creates a new entry for this collection. An existing entry will be
         * deleted.
         *
         * @param {Number[]} address
         *  The logical address of the cell.
         *
         * @param {Object} [data]
         *  A data object from the server response of a view update request. If
         *  missing, creates a default entry representing an empty unformatted
         *  cell. May contain the properties 'display', 'result', 'formula',
         *  and 'attrs'.
         *
         * @returns {Object}
         *  A new collection entry. Always contains the properties 'display'
         *  and 'result'. Will contain the properties 'formula' and 'attrs', if
         *  contained in the passed data object.
         */
        function createCellEntry(address, data) {

            var // create required properties
                entry = {
                    display: Utils.getStringOption(data, 'display', ''),
                    result: Utils.getOption(data, 'result', null)
                };

            // add optional properties
            if (_.isObject(data)) {
                if (_.isString(data.formula)) { entry.formula = data.formula; }
                if (_.isObject(data.attrs)) { entry.attrs = data.attrs; }
            }

            return map[getKey(address)] = entry;
        }

        /**
         * Returns the cell entry with the specified cell address from the
         * collection for read/write access. The entry will be created on
         * demand if it is missing.
         *
         * @param {Number[]} address
         *  The logical address of the cell.
         *
         * @returns {Object}
         *  The collection entry corresponding to the passed cell address. Can
         *  be modified by the calling code.
         */
        function getOrCreateCellEntry(address) {
            return map[getKey(address)] || createCellEntry(address);
        }

        /**
         * Updates the value and/or formatting attributes of a cell entry.
         *
         * @param {Number[]} address
         *  The logical address of the cell.
         *
         * @param {Object} [parsedValue]
         *  The new data for the cell entry.
         *
         * @param {Object} [attributes]
         *  If specified, the new attributes for the cell.
         */
        function updateCellEntry(address, parsedValue, attributes) {

            var // the cell entry to be updated
                cellEntry = null;

            // update the cell value
            if (_.isObject(parsedValue)) {
                // do not create new entry, if cell will be cleared
                cellEntry = _.isNull(parsedValue.result) ? self.getCellEntry(address) : getOrCreateCellEntry(address);
                if (cellEntry) {
                    delete cellEntry.formula;
                    _(cellEntry).extend(parsedValue);
                }
            }

            // update formatting attributes
            if (_.isObject(attributes)) {
                cellEntry = cellEntry || getOrCreateCellEntry(address);
                if (_.isObject(cellEntry.attrs)) {
                    documentStyles.extendAttributes(cellEntry.attrs, attributes);
                } else {
                    cellEntry.attrs = _.copy(attributes, true);
                }
            }
        }

        /**
         * Deletes the specified cell entry from the map.
         *
         * @param {Number[]} address
         *  The logical address of the cell to be deleted.
         */
        function deleteCellEntry(address) {
            delete map[getKey(address)];
        }

        /**
         * Moves the specified cell entry to a new position in the collection.
         *
         * @param {Number[]} address
         *  The logical address of the cell to be moved.
         *
         * @param {Number} cols
         *  The number of columns the cell entry will be moved. A negative
         *  value will move the entry to a preceding column. MUST result in a
         *  valid column index.
         *
         * @param {Number} rows
         *  The number of rows the cell entry will be moved. A negative value
         *  will move the entry to a preceding row. MUST result in a valid row
         *  index.
         */
        function moveCellEntry(address, cols, rows) {
            var cellEntry = self.getCellEntry(address);
            if (cellEntry) {
                map[getKey([address[0] + cols, address[1] + rows])] = cellEntry;
                deleteCellEntry(address);
            }
        }

        /**
         * Invokes the passed cell iterator with the correct collection entry.
         */
        function invokeIterator(colEntry, rowEntry, originalRange, iterator, context, existing) {

            var // the cell address
                address = [colEntry.index, rowEntry.index],
                // the cell entry object
                cellEntry = self.getCellEntry(address);

            // invoke the iterator function
            return (!existing || _.isObject(cellEntry)) ? iterator.call(context, address, cellEntry, colEntry, rowEntry, originalRange) : undefined;
        }

        /**
         * Deletes all cell entries that exist in the passed range.
         */
        function deleteEntriesInRange(range) {
            self.iterateCellsInRanges(range, deleteCellEntry, { existing: true });
        }

        /**
         * Initializes this collection after the active sheet has been changed.
         */
        function changeActiveSheetHandler() {

            var // the model of the active sheet
                sheetModel = view.getActiveSheetModel();

            // get collections from the current active sheet
            colCollection = sheetModel.getColCollection();
            rowCollection = sheetModel.getRowCollection();

            // clear the map and set bounding range to default values
            map = {};
            boundRange = colDimension.interval = rowDimension.interval = null;
        }

        /**
         * Updates the column or row interval of the cells contained in this
         * cell collection. Deletes all cell entries not contained in the
         * collection anymore.
         */
        function updateInterval(type, interval, options) {

            var // column or row intervals and indexes
                dimension = (type === 'columns') ? colDimension : rowDimension;

            // set the new interval
            dimension.interval = _.clone(interval);

            // update this collection, if column and row interval are valid
            if (colDimension.interval && rowDimension.interval) {

                // update the bounding range
                boundRange = SheetUtils.makeRangeFromIntervals(colDimension.interval, rowDimension.interval);

                // TODO: smart request for cell updates
                view.requestUpdate({ cells: [boundRange] });

                // TODO: trigger exact information about changed cells
                self.trigger('change:cells');
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the cell entry with the specified cell address from the
         * collection for read-only access.
         *
         * @param {Number[]} address
         *  The logical address of the cell.
         *
         * @returns {Object|Null}
         *  The cell entry; or null, if the entry does not exist. This object
         *  MUST NOT be changed.
         */
        this.getCellEntry = function (address) {
            var cellEntry = map[getKey(address)];
            return cellEntry ? cellEntry : null;
        };

        /**
         * Invokes the passed iterator function for all visible cells in this
         * collection for read-only access.
         *
         * @param {Function} iterator
         *  The iterator function called for all visible cells. Receives the
         *  following parameters:
         *  (1) {Number[2]} address
         *      The logical address of the current cell.
         *  (2) {Object|Null} cellEntry
         *      The contents and formatting attributes of the current cell, in
         *      the attributes 'display', 'result', 'formula', and 'attrs'. The
         *      contents of this object MUST NOT be changed. The value null
         *      will be passed for entries not contained in this collection,
         *      representing empty cells without contents, AND without
         *      formatting attributes.
         *  (3) {Object} colEntry
         *      The collection entry of the column collection.
         *  (4) {Object} rowEntry
         *      The collection entry of the row collection.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the cells will be visited in reversed order.
         *  @param {Boolean} [options.existing=false]
         *      If set to true, the iterator will only be called for existing
         *      cell entries in this collection. The iterator function will
         *      always receive a valid cell entry object.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateCells = function (iterator, options) {

            var // the calling context for the iterator function
                context = Utils.getOption(options, 'context'),
                // whether to iterate in reversed order
                reverse = Utils.getBooleanOption(options, 'reverse', false),
                // whether to visit existing entries only
                existing = Utils.getBooleanOption(options, 'existing', false);

            // invoke iterator for all cells in this collection
            return rowCollection.iterateVisibleEntries(rowDimension.interval, function (rowEntry) {
                return colCollection.iterateVisibleEntries(colDimension.interval, function (colEntry) {
                    return invokeIterator(colEntry, rowEntry, undefined, iterator, context, existing);
                }, { reverse: reverse });
            }, { reverse: reverse });
        };

        /**
         * Invokes the passed iterator function for specific visible cells in
         * one column or row, and moving into the specified direction.
         *
         * @param {Number[]} address
         *  The logical address of the cell that will be visited first (the
         *  option 'options.skipStartCell' can be used if iteration shall start
         *  with the nearest visible neighbor of the cell, and not with the
         *  cell itself).
         *
         * @param {String} direction
         *  How to move to the next cells while iterating. Supported values are
         *  'up', 'down', 'left', or 'right'.
         *
         * @param {Function} iterator
         *  The iterator function called for all visited cells. Receives the
         *  following parameters:
         *  (1) {Number[2]} address
         *      The logical address of the current cell.
         *  (2) {Object|Null} cellEntry
         *      The contents and formatting attributes of the current cell, in
         *      the attributes 'display', 'result', 'formula', and 'attrs'. The
         *      contents of this object MUST NOT be changed. The value null
         *      will be passed for entries not contained in this collection,
         *      representing empty cells without contents, AND without
         *      formatting attributes.
         *  (3) {Object} colEntry
         *      The collection entry of the column collection.
         *  (4) {Object} rowEntry
         *      The collection entry of the row collection.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *  @param {Boolean} [options.existing=false]
         *      If set to true, the iterator will only be called for existing
         *      cell entries in this collection. The iterator function will
         *      always receive a valid cell entry object.
         *  @param {Boolean} [options.skipStartCell=false]
         *      If set to true, iteration will start at the nearest visible
         *      neighbor of the cell specified in the 'address' parameter
         *      instead of that cell.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateNextCells = function (address, direction, iterator, options) {

            var // the calling context for the iterator function
                context = Utils.getOption(options, 'context'),
                // whether to visit existing entries only
                existing = Utils.getBooleanOption(options, 'existing', false),
                // number of cells to skip before iteration starts
                skipCount = Utils.getBooleanOption(options, 'skipStartCell', false) ? 1 : 0,
                // the fixed collection entry
                fixedEntry = null,
                // the column/row interval to be visited
                interval = null;

            function getFixedEntry(collection, interval, index) {
                return SheetUtils.intervalContainsIndex(interval, index) ? collection.getVisibleEntry(index) : null;
            }

            switch (direction) {
            case 'up':
                if ((fixedEntry = getFixedEntry(colCollection, colDimension.interval, address[0]))) {
                    if ((interval = SheetUtils.getIntersectionInterval(rowDimension.interval, { first: 0, last: address[1] - skipCount }))) {
                        return rowCollection.iterateVisibleEntries(interval, function (rowEntry) {
                            return invokeIterator(fixedEntry, rowEntry, undefined, iterator, context, existing);
                        }, { reverse: true });
                    }
                }
                break;

            case 'down':
                if ((fixedEntry = getFixedEntry(colCollection, colDimension.interval, address[0]))) {
                    if ((interval = SheetUtils.getIntersectionInterval(rowDimension.interval, { first: address[1] + skipCount, last: app.getModel().getMaxRow() }))) {
                        return rowCollection.iterateVisibleEntries(interval, function (rowEntry) {
                            return invokeIterator(fixedEntry, rowEntry, undefined, iterator, context, existing);
                        });
                    }
                }
                break;

            case 'left':
                if ((fixedEntry = getFixedEntry(rowCollection, rowDimension.interval, address[1]))) {
                    if ((interval = SheetUtils.getIntersectionInterval(colDimension.interval, { first: 0, last: address[0] - skipCount }))) {
                        return colCollection.iterateVisibleEntries(interval, function (colEntry) {
                            return invokeIterator(colEntry, fixedEntry, undefined, iterator, context, existing);
                        }, { reverse: true });
                    }
                }
                break;

            case 'right':
                if ((fixedEntry = getFixedEntry(rowCollection, rowDimension.interval, address[1]))) {
                    if ((interval = SheetUtils.getIntersectionInterval(colDimension.interval, { first: address[0] + skipCount, last: app.getModel().getMaxCol() }))) {
                        return colCollection.iterateVisibleEntries(interval, function (colEntry) {
                            return invokeIterator(colEntry, fixedEntry, undefined, iterator, context, existing);
                        });
                    }
                }
                break;

            default:
                Utils.error('CellCollection.iterateNextCells(): invalid direction "' + direction + '"');
            }
        };

        /**
         * Returns the logical address of the cell range covered by this cell
         * collection. The range address will be build by the columns and rows
         * available in the associated column and row collections.
         *
         * @returns {Object}
         *  The range address covered by the column and row collection.
         */
        this.getRange = function () {
            return _.copy(boundRange, true);
        };

        /**
         * Returns whether this cell collection covers the passed cell range
         * completely.
         *
         * @param {Object} range
         *  The logical cell range address to be checked.
         *
         * @returns {Boolean}
         *  Whether this collection covers the passed cell range completely.
         */
        this.containsRange = function (range) {
            return SheetUtils.rangeContainsRange(boundRange, range);
        };

        /**
         * Invokes the passed iterator function for all visible cells in the
         * passed cell ranges.
         *
         * @param {Object|Array} ranges
         *  The logical address of a single range, or an array of logical
         *  range addresses whose cells will be iterated.
         *
         * @param {Function} iterator
         *  The iterator function called for all visible cells in the passed
         *  ranges. The ranges will be processed independently, cells covered
         *  by several ranges will be visited multiple times (see method
         *  'CellCollection.iterateCellsInRows()' for an alternative). Receives
         *  the following parameters:
         *  (1) {Number[2]} address
         *      The logical address of the current cell.
         *  (2) {Object|Null} cellEntry
         *      The contents and formatting attributes of the current cell, in
         *      the attributes 'display', 'result', 'formula', and 'attrs'. The
         *      contents of this object MUST NOT be changed. The value null
         *      will be passed for entries not contained in this collection,
         *      representing empty cells without contents, AND without
         *      formatting attributes.
         *  (3) {Object} colEntry
         *      The collection entry of the column collection.
         *  (4) {Object} rowEntry
         *      The collection entry of the row collection.
         *  (5) {Object} originalRange
         *      The logical address of the range containing the current cell
         *      (one of the ranges contained in the 'ranges' parameter passed
         *      to this method).
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the ranges AND the cells in each range will be
         *      visited in reversed order.
         *  @param {Boolean} [options.existing=false]
         *      If set to true, the iterator will only be called for existing
         *      cell entries in this collection. The iterator function will
         *      always receive a valid cell entry object.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateCellsInRanges = function (ranges, iterator, options) {

            var // the calling context for the iterator function
                context = Utils.getOption(options, 'context'),
                // whether to iterate in reversed order
                reverse = Utils.getBooleanOption(options, 'reverse', false),
                // whether to visit existing entries only
                existing = Utils.getBooleanOption(options, 'existing', false),
                // whether only over outer cells shall be iterated
                outerCellsOnly = Utils.getBooleanOption(options, 'outerCellsOnly', false);

            // iterate the intersection of the passed ranges and the range available in this collection
            return SheetUtils.iterateIntersectionRanges(ranges, boundRange, function (intersectRange, originalRange) {
                var colInt = SheetUtils.getColInterval(intersectRange),
                    rowInt = SheetUtils.getRowInterval(intersectRange);
                return rowCollection.iterateVisibleEntries(rowInt, function (rowEntry) {
                    return colCollection.iterateVisibleEntries(colInt, function (colEntry) {
                        if (!outerCellsOnly || SheetUtils.isRangeBorderCell(originalRange, [colEntry.index, rowEntry.index])) {
                            return invokeIterator(colEntry, rowEntry, originalRange, iterator, context, existing);
                        }
                    }, { reverse: reverse });
                }, { reverse: reverse });
            }, { reverse: reverse });
        };

        /**
         * Invokes the passed iterator function exactly once for each visible
         * cell covered by the passed cell ranges.
         *
         * @param {Object|Array} ranges
         *  The logical address of a single range, or an array of logical
         *  range addresses whose cells will be iterated.
         *
         * @param {Function} iterator
         *  The iterator function called exactly once for each visible cell in
         *  the passed ranges (even if the cell is covered by multiple ranges).
         *  The cells will be visited row by row from top to bottom, and inside
         *  the rows with ascending column index. Example: If the entire
         *  columns B and D are passed to this method, the iterator function
         *  will be called for cells B1, D1, B2, D2, and so on. Receives the
         *  following parameters:
         *  (1) {Number[2]} address
         *      The logical address of the current cell.
         *  (2) {Object|Null} cellEntry
         *      The contents and formatting attributes of the current cell, in
         *      the attributes 'display', 'result', 'formula', and 'attrs'.The
         *      contents of this object MUST NOT be changed. The value null
         *      will be passed for entries not contained in this collection,
         *      representing empty cells without contents, AND without
         *      formatting attributes.
         *  (3) {Object} colEntry
         *      The collection entry of the column collection.
         *  (4) {Object} rowEntry
         *      The collection entry of the row collection.
         *  (5) {Object} rowBandRange
         *      The logical address of a range containing the current cell.
         *      These ranges are created with the algorithm that divides and
         *      merges the original ranges in order to create a list of
         *      distinct ranges. The range address has been reduced to visible
         *      columns and rows, but may contain hidden columns or rows
         *      inside.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the rows will be visited in reversed order from
         *      the bottom-most row to the top-most row, and the cells in each
         *      row will be visited from right-most column to left-most column.
         *  @param {Boolean} [options.existing=false]
         *      If set to true, the iterator will only be called for existing
         *      cell entries in this collection. The iterator function will
         *      always receive a valid cell entry object.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateCellsInRows = function (ranges, iterator, options) {

            var // the calling context for the iterator function
                context = Utils.getOption(options, 'context'),
                // whether to iterate in reversed order
                reverse = Utils.getBooleanOption(options, 'reverse', false),
                // whether to visit existing entries only
                existing = Utils.getBooleanOption(options, 'existing', false),
                // the row bands containing sub ranges with equal column intervals
                rowBands = [],
                // the start index in rowBands used to search for matching bands
                startIndex = 0;

            // restrict passed ranges to the available range of this collection
            ranges = SheetUtils.getIntersectionRanges(ranges, boundRange);

            // sort the ranges by start row index
            ranges.sort(function (r1, r2) { return r1.start[1] - r2.start[1]; });

            // build the row bands array
            _(ranges).each(function (range) {

                var // the start and end row index of the range
                    firstRow = range.start[1], lastRow = range.end[1],
                    // the current row band
                    index = 0, rowBand = null;

                // update the row bands containing the range
                for (index = startIndex; index < rowBands.length; index += 1) {
                    rowBand = rowBands[index];

                    if (rowBand.last < firstRow) {
                        // row band is above the range, ignore it in the next iterations
                        startIndex = index + 1;
                    } else if (rowBand.first < firstRow) {
                        // row band needs to be split (range starts inside the row band)
                        rowBands.splice(index, 0, { first: rowBand.first, last: firstRow - 1, ranges: _.clone(rowBand.ranges) });
                        rowBand.first = firstRow;
                        startIndex = index + 1;
                        // next iteration of the for loop will process the current row band again
                    } else {

                        if (lastRow < rowBand.last) {
                            // row band needs to be split (range ends inside the row band)
                            rowBands.splice(index + 1, 0, { first: lastRow + 1, last: rowBand.last, ranges: _.clone(rowBand.ranges) });
                            rowBand.last = lastRow;
                        }
                        // row band contains the range completely
                        rowBand.ranges.push(range);
                        // continue with next range, if end of range found
                        if (lastRow === rowBand.last) { return; }
                        // adjust start index in case a new row band needs to be appended
                        firstRow = rowBand.last + 1;
                    }
                }

                // no row band found, create and append a new row band
                rowBands.push({ first: firstRow, last: lastRow, ranges: [range] });
            });

            // reduce bands to visible rows, remove row bands without a corresponding row entry completely
            Utils.iterateArray(rowBands, function (rowBand, index) {
                var visibleInterval = rowCollection.getVisibleInterval(rowBand);
                if (visibleInterval) { _(rowBand).extend(visibleInterval); } else { rowBands.splice(index, 1); }
            }, { reverse: true });

            // call the iterator function for all resulting ranges in the row bands
            return Utils.iterateArray(rowBands, function (rowBand) {

                var // build the column intervals for the ranges in all row bands (this merges overlapping band ranges)
                    colIntervals = SheetUtils.getColIntervals(rowBand.ranges);

                // reduce to visible columns, remove column intervals without corresponding entries
                Utils.iterateArray(colIntervals, function (interval, index) {
                    var visibleInterval = colCollection.getVisibleInterval(interval);
                    if (visibleInterval) { colIntervals[index] = visibleInterval; } else { colIntervals.splice(index, 1); }
                }, { reverse: true });

                // call iterator function for all cells in the row band
                return rowCollection.iterateVisibleEntries(rowBand, function (rowEntry) {
                    // iterate row by row through all column intervals
                    return Utils.iterateArray(colIntervals, function (interval) {
                        var rowBandRange = SheetUtils.makeRangeFromIntervals(interval, rowBand);
                        return colCollection.iterateVisibleEntries(interval, function (colEntry) {
                            return invokeIterator(colEntry, rowEntry, rowBandRange, iterator, context, existing);
                        }, { reverse: reverse });
                    }, { reverse: reverse });
                }, { reverse: reverse });
            }, { reverse: reverse });
        };

        /**
         * Updates this cell collection according to the new column interval
         * displayed in the column header pane.
         */
        this.setColInterval = function (interval, options) {
            updateInterval('columns', interval, options);
        };

        /**
         * Updates this cell collection according to the new row interval
         * displayed in the row header pane.
         */
        this.setRowInterval = function (interval, options) {
            updateInterval('rows', interval, options);
        };

        /**
         * Imports the passed cell data received from a 'docs:update' event
         * notification.
         *
         * @param {Array} ranges
         *  The cell range addresses containing the passed cell data.
         *
         * @param {Array} cellData
         *  An array with cell layout descriptor objects, as received from the
         *  'docs:update' event notification.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.importCellEntries = function (ranges, cellData) {

            // clear the old cell entries
            _(ranges).each(deleteEntriesInRange);

            // map all existing cells by their position
            _(cellData).each(function (data) {
                if (_.isObject(data) && SheetUtils.isValidAddress(data.pos) && SheetUtils.rangeContainsCell(boundRange, data.pos)) {
                    createCellEntry(data.pos, data);
                }
            });

            // notify all change listeners
            this.trigger('change:cells', ranges);
            return this;
        };

        /**
         * Returns the merged formatting attributes of the passed cell entry.
         * If the cell entry is missing, resolves the default attributes of the
         * row entry. If that entry does not have own default attributes, uses
         * the default attributes of the column entry. If this entry does not
         * have any attributes, returns the merged attributes of the default
         * cell style sheet.
         *
         * @param {Object} [cellEntry]
         *  The cell entry from the cell collection. May be undefined (cell
         *  does not exist), in this case the default formatting of the row or
         *  column will be used.
         *
         * @param {Object} [colEntry]
         *  The entry of the column collection associated to the cell.
         *
         * @param {Object} [rowEntry]
         *  The entry of the row collection associated to the cell.
         *
         * @returns {Object}
         *  The merged attribute set containing all effective formatting
         *  attributes for the cell.
         */
        this.getMergedAttributes = function (cellEntry, colEntry, rowEntry) {

            // resolve default cell style attributes (TODO: must be updated after default style sheet has changed)
            if (!defaultAttributes) {
                defaultAttributes = cellStyles.getMergedAttributes({});
            }

            // if the cell entry is defined, use it exclusively (even if it does not contain any attributes)
            if (_.isObject(cellEntry)) {
                return _.isObject(cellEntry.attrs) ? cellStyles.getMergedAttributes(cellEntry.attrs) : defaultAttributes;
            }

            // if the row entry contains attributes, use them (ignore column default attributes)
            if (rowEntry && rowEntry.attributes.row.customFormat) {
                return rowEntry.attributes;
            }

            // if the column entry exists, always use its attributes (there is no 'customFormat'
            // flag), otherwise use the formatting defined by the default cell style
            return colEntry ? colEntry.attributes : defaultAttributes;
        };

        /**
         * Changes the contents of a single cell in this collection, and
         * triggers a 'change:cells' event for the cell.
         *
         * @param {Number[]} address
         *  The logical address of the cell.
         *
         * @param {String|Null} [value]
         *  If specified, the new unparsed string value for the cell, or null
         *  to clear the cell value. If omitted, only the cell formatting will
         *  be changed.
         *
         * @param {Object} [attributes]
         *  If specified, the new attributes for the cell.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.setCellContents = function (address, value, attributes) {

            // do nothing if the cell is not available or visible
            if (SheetUtils.rangeContainsCell(boundRange, address) &&
                colCollection.isEntryVisible(address[0]) && rowCollection.isEntryVisible(address[1])
            ) {

                // update the affected cell entry with the parsed value
                updateCellEntry(address, parseCellValue(value), attributes);

                // notify all change listeners
                this.trigger('change:cells', { start: address, end: address });
            }

            return this;
        };

        /**
         * Fills all cell in the specified ranges with the same value and
         * formatting attributes.
         *
         * @param {Object|Array} ranges
         *  The logical address of a single range, or an array of logical
         *  range addresses whose cells will be filled.
         *
         * @param {String|Null} [value]
         *  If specified, the new unparsed string value for the cell, or null
         *  to clear the cell value. If omitted, only the cell formatting will
         *  be changed.
         *
         * @param {Object} [attributes]
         *  If specified, the new attributes for the cell.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.fillCellContents = function (ranges, value, attributes) {

            var // the parsed value for all cell in all ranges
                parsedValue = parseCellValue(value);

            // update all affected cell entries
            this.iterateCellsInRows(ranges, function (address) {
                updateCellEntry(address, parsedValue, attributes);
            });

            // notify all change listeners
            this.trigger('change:cells', ranges);
            return this;
        };

        /**
         * Changes the cell border attributes for all passed cell ranges.
         *
         * @param {Object|Array} ranges
         *  The logical address of a single range, or an array of logical
         *  range addresses whose cells will be filled.
         *
         * @param {Object} borderAttributes
         *  An attribute map containing all border attributes to be changed.
         *
         * @param {Object} [options]
         *  A map with additional options that have been passed to the
         *  associated 'fillCellRange' operation.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.setBorderAttributes = function (ranges, borderAttributes, options) {

            var // whether to update visible borders only
                visibleBorders = Utils.getBooleanOption(options, 'visibleBorders', false),
                // the changed cell ranges (extended by surrounding cells)
                changedRanges = [];

            // process all visible ranges (intersecting the visible area of this collection)
            SheetUtils.iterateIntersectionRanges(ranges, boundRange, function (range, originalRange) {

                var // the extended range with surrounding cells if available
                    extendedRange = _.copy(range, true);

                // extend range by available columns/rows surrounding the original
                // range, if the respective border attribute has been passed, and
                // if the range has not been intersected in that direction
                if (('borderLeft' in borderAttributes) && (range.start[0] === originalRange.start[0]) && colCollection.isEntryVisible(range.start[0] - 1)) { extendedRange.start[0] -= 1; }
                if (('borderTop' in borderAttributes) && (range.start[1] === originalRange.start[1]) && rowCollection.isEntryVisible(range.start[1] - 1)) { extendedRange.start[1] -= 1; }
                if (('borderRight' in borderAttributes) && (range.end[0] === originalRange.end[0]) && colCollection.isEntryVisible(range.end[0] + 1)) { extendedRange.end[0] += 1; }
                if (('borderBottom' in borderAttributes) && (range.end[1] === originalRange.end[1]) && rowCollection.isEntryVisible(range.end[1] + 1)) { extendedRange.end[1] += 1; }
                changedRanges.push(extendedRange);

                // process all cells in the extended range
                self.iterateCellsInRanges(extendedRange, function (address, cellEntry) {

                    var // whether the cell column is inside the original range
                        colInside = SheetUtils.rangeContainsCol(originalRange, address[0]),
                        // whether the cell row is inside the original range
                        rowInside = SheetUtils.rangeContainsRow(originalRange, address[1]),
                        // maps target border attribute names to source border values
                        sourceBorders = {},
                        // the cell attributes of the current cell
                        cellAttributes = null;

                    // do nothing for edge cells in the extended range
                    if (!colInside && !rowInside) { return; }

                    // select the border attribute names to be applied at the current cell
                    if (rowInside) {
                        if (address[0] >= originalRange.start[0]) {
                            sourceBorders.borderLeft = (address[0] === originalRange.start[0]) ? 'borderLeft' : (address[0] > originalRange.end[0]) ? 'borderRight' : 'borderInsideVert';
                        }
                        if (address[0] <= originalRange.end[0]) {
                            sourceBorders.borderRight = (address[0] === originalRange.end[0]) ? 'borderRight' : (address[0] < originalRange.start[0]) ? 'borderLeft' : 'borderInsideVert';
                        }
                    }
                    if (colInside) {
                        if (address[1] >= originalRange.start[1]) {
                            sourceBorders.borderTop = (address[1] === originalRange.start[1]) ? 'borderTop' : (address[1] > originalRange.end[1]) ? 'borderBottom' : 'borderInsideHor';
                        }
                        if (address[1] <= originalRange.end[1]) {
                            sourceBorders.borderBottom = (address[1] === originalRange.end[1]) ? 'borderBottom' : (address[1] < originalRange.start[1]) ? 'borderTop' : 'borderInsideHor';
                        }
                    }

                    // remap source border names to source border values
                    _(sourceBorders).each(function (sourceBorderName, targetBorderName) {
                        if (sourceBorderName in borderAttributes) {
                            sourceBorders[targetBorderName] = borderAttributes[sourceBorderName];
                        } else {
                            delete sourceBorders[targetBorderName];
                        }
                    });

                    // create missing data, if all borders will be set
                    if (!visibleBorders) {
                        cellEntry = cellEntry || createCellEntry(address);
                        cellEntry.attrs = cellEntry.attrs || {};
                        cellEntry.attrs.cell = cellEntry.attrs.cell || {};
                    }

                    // the cell attribute map (do nothing if not existing, border is invisible)
                    cellAttributes = cellEntry && cellEntry.attrs && cellEntry.attrs.cell;
                    if (!cellAttributes) { return; }

                    // process all cell borders active for the current cell
                    _(sourceBorders).each(function (sourceBorder, targetBorderName) {
                        // update the attributes of the cell entry with the new border
                        if (!visibleBorders) {
                            cellAttributes[targetBorderName] = _.clone(sourceBorder);
                        } else if (Border.isVisibleBorder(cellAttributes[targetBorderName])) {
                            cellAttributes[targetBorderName] = _({}).extend(cellAttributes[targetBorderName], sourceBorder);
                        }
                    });
                });
            });

            // notify all change listeners
            this.trigger('change:cells', changedRanges);
            return this;
        };

        /**
         * Inserts new cells into the passed cell ranges, and moves the old
         * cells down or to the right.
         *
         * @param {Object} range
         *  The logical address of the cell range to be inserted.
         *
         * @param {String} direction
         *  Either 'columns' to move the existing cells horizontally (to the
         *  right or left according to the sheet orientation), or 'rows' to
         *  move the existing cells down.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.insertCells = function (range, direction) {

            var // whether to move cells horizontally (across columns)
                columns = (direction === 'columns'),
                // the address array index used to move the cells
                addrIndex = columns ? 0 : 1,
                // the last available column/row index in the sheet
                lastIndex = columns ? app.getModel().getMaxCol() : app.getModel().getMaxRow(),
                // number of columns the entries will be moved
                moveCols = columns ? SheetUtils.getColCount(range) : 0,
                // number of rows the entries will be moved
                moveRows = columns ? 0 : SheetUtils.getRowCount(range),
                // the cell range with all cells to be moved
                moveRange = _.copy(range, true),
                // saving info about already added cols/rows to reference cell of merged cells
                addedSpans = {};

            // move the existing cell entries
            moveRange.end[addrIndex] = lastIndex - (columns ? moveCols : moveRows);
            if (moveRange.start[addrIndex] <= moveRange.end[addrIndex]) {
                self.iterateCellsInRanges(moveRange, function (address) {
                    moveCellEntry(address, moveCols, moveRows);
                }, { reverse: true, existing: true });
            }

            // insert new cell entries according to preceding cells
            self.iterateCellsInRanges(range, function (address) {
                var addr = _.clone(address),
                    nextAddress = null,
                    newCellEntry = null,
                    clEntry = null,
                    nextClEntry = null;

                if (addr[addrIndex] > 0) {
                    addr[addrIndex]--;
                    clEntry = self.getCellEntry(addr);
                    // 'attrs.cell' need to be copied into the new cell
                    if (clEntry) {
                        if (('attrs' in clEntry) && ('cell' in clEntry.attrs)) {

                            // copying all cell attributes
                            newCellEntry = newCellEntry || {};
                            newCellEntry.attrs = { cell: _.copy(clEntry.attrs.cell, true) };

                            // deleting all cell border attributes
                            _(OUTER_BORDER_NAMES).each(function (borderName) {
                                delete newCellEntry.attrs.cell[borderName];
                            });

                            nextAddress = _.clone(address);
                            nextAddress[addrIndex]++;
                            nextClEntry = self.getCellEntry(nextAddress);

                            if (nextClEntry) {
                                // Copying only those borders only, that belong to both cell entries: clEntry and nextClEntry
                                // -> additionally these borders must be equal in type, width and color
                                if (('attrs' in nextClEntry) && ('cell' in nextClEntry.attrs)) {
                                    _(OUTER_BORDER_NAMES).each(function (borderName) {
                                        // both borders have to exist and have to be identical
                                        if ((borderName in clEntry.attrs.cell) && (borderName in nextClEntry.attrs.cell) &&
                                            (Border.isEqual(clEntry.attrs.cell[borderName], nextClEntry.attrs.cell[borderName]))) {
                                            newCellEntry.attrs.cell[borderName] = _.clone(clEntry.attrs.cell[borderName]);
                                        }
                                    });
                                }
                            }
                        }
                    }
                }

                if (newCellEntry) {
                    createCellEntry(address, newCellEntry);
                }
            });

            // notify all change listeners
            this.trigger('insert:cells', range);
            return this;
        };

        /**
         * Removes the passed cell ranges from the collection, and moves the
         * remaining cells up or to the left.
         *
         * @param {Object} range
         *  The logical address of the cell range to be deleted.
         *
         * @param {String} direction
         *  Either 'columns' to move the remaining cells horizontally (to the
         *  left or right according to the sheet orientation), or 'rows' to
         *  move the remaining cells up.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.deleteCells = function (range, direction) {

            var // whether to move cells horizontally (across columns)
                columns = direction === 'columns',
                // the address array index used to move the cells
                addrIndex = columns ? 0 : 1,
                // the last available column/row index in the sheet
                lastIndex = columns ? app.getModel().getMaxCol() : app.getModel().getMaxRow(),
                // number of columns the entries will be moved
                moveCols = columns ? -SheetUtils.getColCount(range) : 0,
                // number of rows the entries will be moved
                moveRows = columns ? 0 : -SheetUtils.getRowCount(range),
                // the cell range with all cells to be moved
                moveRange = _.copy(range, true);

            // delete the existing cell entries in the deleted range
            deleteEntriesInRange(range);

            // move the following cell entries
            moveRange.start[addrIndex] = range.end[addrIndex] + 1;
            moveRange.end[addrIndex] = lastIndex;
            if (moveRange.start[addrIndex] <= moveRange.end[addrIndex]) {
                self.iterateCellsInRanges(moveRange, function (address) {
                    moveCellEntry(address, moveCols, moveRows);
                }, { existing: true });
            }

            // notify all change listeners
            this.trigger('delete:cells', range);
            return this;
        };

        this.destroy = function () {
            this.events.destroy();
        };

        // initialization -----------------------------------------------------

        // initialize column/row collection on change of active sheet
        view.on('change:activesheet', changeActiveSheetHandler);

    } // class CellCollection

    // exports ================================================================

    return CellCollection;

});
