/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/sheetmodel',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/model/format/attributedmodel',
     'io.ox/office/drawinglayer/model/drawingcollection',
     'io.ox/office/spreadsheet/model/colrowcollection',
     'io.ox/office/spreadsheet/model/mergecollection',
     'io.ox/office/spreadsheet/model/sheetdrawingmodel'
    ], function (Utils, AttributedModel, DrawingCollection, ColRowCollection, MergeCollection, SheetDrawingModel) {

    'use strict';

    // class SheetModel =======================================================

    /**
     * Represents a single sheet in the spreadsheet document.
     *
     * @constructor
     *
     * @extends AttributedModel
     *
     * @param {SpreadsheetApplication} app
     *  The application instance with the document model containing this sheet.
     *
     * @param {Object} [attributes]
     *  An attribute set with initial formatting attributes for the sheet.
     */
    function SheetModel(app, attributes) {

        var // self reference
            self = this,

            // the column and row collections (size and formatting of all columns/rows)
            colCollection = null,
            rowCollection = null,

            // the collection of merged cells in this sheet
            mergeCollection = null,

            // the collection of drawing objects contained in this sheet
            drawingCollection = new DrawingCollection(app, {
                clientResolveModelHandler: resolveDrawingModel,
                clientRegisterModelHandler: registerDrawingModel
            }),

            // maps drawing model indexes (used as logical positions) to drawing models
            drawingModels = [],

            // all view settings of this sheet
            viewSettings = {
                activePane: 'bottomRight'
            },

            // the sheet selection (cells and drawings)
            selection = {
                ranges: [{ start: [0, 0], end: [0, 0] }],
                activeRange: 0,
                activeCell: [0, 0],
                drawings: []
            };

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, app, 'sheet', attributes);

        // private methods ----------------------------------------------------

        /**
         * Tries to return a drawing model contained in this sheet at the
         * specified logical position.
         *
         * @param {Number[]} position
         *  An arbitrary logical document position.
         *
         * @returns {DrawingModel|Null}
         *  The drawing model at the specified logical position, if available;
         *  otherwise null.
         */
        function resolveDrawingModel(position) {
            return ((position.length === 1) && drawingModels[position[0]]) || null;
        }

        /**
         * Registers a new drawing model that has just been inserted into the
         * drawing collection of this sheet.
         *
         * @param {DrawingModel} model
         *  The new drawing model.
         *
         * @param {Number[]} position
         *  The logical position of the new drawing object.
         *
         * @returns {Boolean}
         *  Whether the passed logical position is valid, and the drawing model
         *  has been registered successfully.
         */
        function registerDrawingModel(model, position) {
            if ((position.length === 1) && (0 <= position[0]) && (position[0] <= drawingModels.length)) {
                drawingModels.splice(position[0], 0, model);
                return true;
            }
            return false;
        }

        /**
         * Processes a new drawing object inserted into the drawing collection.
         */
        function insertDrawingHandler(event, model, position) {
            // extend the API of the drawing model with sheet-specific methods
            SheetDrawingModel.call(model, app);
        }

        /**
         * Processes a drawing object removed from the drawing collection.
         */
        function deleteDrawingHandler(event, model, position) {
            drawingModels.splice(position[0], 1);
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the collection of all columns in this sheet, containing
         * their size and formatting attributes.
         *
         * @returns {ColRowCollection}
         *  The collection of all columns in this sheet.
         */
        this.getColCollection = function () {
            return colCollection;
        };

        /**
         * Returns the collection of all rows in this sheet, containing their
         * size and formatting attributes.
         *
         * @returns {ColRowCollection}
         *  The collection of all rows in this sheet.
         */
        this.getRowCollection = function () {
            return rowCollection;
        };

        /**
         * Returns the collection of all merged cells, containing their ranges.
         *
         * @returns {MergeCollection}
         *  The collection of all merged cells in this sheet.
         */
        this.getMergeCollection = function () {
            return mergeCollection;
        };

        // drawing layer ------------------------------------------------------

        /**
         * Returns the drawing collection of this sheet, containing the models
         * of all drawing objects contained in this sheet.
         *
         * @returns {DrawingCollection}
         *  The collection of all drawing models in this sheet.
         */
        this.getDrawingCollection = function () {
            return drawingCollection;
        };

        /**
         * Invokes the passed iterator function for all drawing models in this
         * sheet. The drawing models will be visited in order of their logical
         * document positions (Z order).
         *
         * @param {Function} iterator
         *  The iterator function called for all drawing models. Receives the
         *  current drawing model as first parameter, and its logical position
         *  as second parameter. If the iterator returns the Utils.BREAK
         *  object, the iteration process will be stopped immediately.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the drawing models will be visited in reversed
         *      order.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateDrawingModels = function (iterator, options) {
            return Utils.iterateArray(drawingModels, function (model, index) {
                return iterator.call(this, model, [index]);
            }, options);
        };

        // view settings ------------------------------------------------------

        /**
         * Returns a specific property of the view settings of this sheet.
         *
         * @param {String} propName
         *  The name of the view property.
         *
         * @returns {Any}
         *  A clone of the property value.
         */
        this.getViewProperty = function (propName) {
            return (propName in viewSettings) ? _.copy(viewSettings[propName], true) : null;
        };

        /**
         * Changes a specific property of the view settings of this sheet, and
         * triggers a generic 'change:viewproperty' and a specific
         * 'change:viewproperty:***' event.
         *
         * @param {String} propName
         *  The name of the view property.
         *
         * @param {Any} propValue
         *  The new value of the view property. Must neither be undefined nor
         *  null. If the passed value is different to the current value, this
         *  sheet model will trigger a 'change:viewproperty' event with the
         *  property name and the new property value, followed by a
         *  'change:viewproperty:***' event with the lower-case property name
         *  in the vent type name, and the new property value as parameter.
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.setViewProperty = function (propName, propValue) {
            if (propName in viewSettings) {
                if (!_.isUndefined(propValue) && !_.isNull(propValue)) {
                    if (!_.isEqual(viewSettings[propName], propValue)) {
                        viewSettings[propName] = _.copy(propValue, true);
                        this.trigger('change:viewproperty', propName, propValue)
                            .trigger('change:viewproperty:' + propName.toLowerCase(), propValue);
                    }
                } else {
                    Utils.warn('SheetModel.setViewProperty(): invalid property value "' + propValue + '"');
                }
            } else {
                Utils.warn('SheetModel.setViewProperty(): unknown property "' + propName + '"');
            }
            return this;
        };

        // selection ----------------------------------------------------------

        /**
         * Returns an object containing all information about the selection in
         * this sheet.
         *
         * @returns {Object}
         *  The selection data, in the following properties:
         *  - {Array} ranges
         *      Array of range addresses for all selected ranges.
         *  - {Number} activeRange
         *      Element index of the active range in the 'ranges' property.
         *  - {Number[]} activeCell
         *      Logical cell address of the active cell in the active range.
         *  - {Array} drawings
         *      Array with logical positions of all selected drawing frames.
         */
        this.getSelection = function () {
            return _.copy(selection, true);
        };

        /**
         * Returns, whether a drawing object is currently selected in this
         * sheet.
         *
         * @returns {Boolean}
         *  Whether a drawing object is currently selected in this sheet.
         */
        this.hasDrawingSelection = function () {
            return selection.drawings.length > 0;
        };

        /**
         * Changes the selection properties of this sheet, and triggers
         * 'before:selection' and 'change:selection' events.
         *
         * @param {Object} newSelection
         *  The selection data, in the following properties:
         *  - {Array} ranges
         *      Array of logical range addresses.
         *  - {Number} activeRange
         *      Element index of the active range in the 'ranges' property.
         *  - {Number[]} activeCell
         *      Logical address of the active cell in the active range.
         *  - {Array} drawings
         *      Array with logical positions of all selected drawing frames.
         *  If the passed selection is different to the current selection, this
         *  sheet model will trigger a 'before:selection' event with the old
         *  selection as parameter (before actually changing the selection),
         *  and a 'change:selection' event with the new selection as parameter
         *  (after changing the selection).
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.setSelection = function (newSelection) {
            if (_.isObject(newSelection)) {
                if (!_.isEqual(selection, newSelection)) {
                    this.trigger('before:selection', selection);
                    selection = _.copy(newSelection, true);
                    this.trigger('change:selection', newSelection);
                }
            } else {
                Utils.warn('SheetModel.setSelection(): invalid value "' + newSelection + '"');
            }
            return this;
        };

        this.destroy = function () {
            this.events.destroy();
            colCollection.destroy();
            rowCollection.destroy();
            mergeCollection.destroy();
            drawingCollection.destroy();
            colCollection = rowCollection = mergeCollection = drawingCollection = drawingModels = null;
        };

        // initialization -----------------------------------------------------

        // late initialization of column/row collections (constructor uses this sheet model)
        colCollection = new ColRowCollection(app, this, true);
        rowCollection = new ColRowCollection(app, this, false);
        mergeCollection = new MergeCollection(app, this);

        // forward events from the drawing collection to own listeners
        drawingCollection.on({
            'insert:drawing': insertDrawingHandler,
            'delete:drawing': deleteDrawingHandler
        });

    } // class SheetModel

    // exports ================================================================

    // derive this class from class AttributedModel
    return AttributedModel.extend({ constructor: SheetModel });

});
