/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/model',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/model/editmodel',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/model/operations',
     'io.ox/office/spreadsheet/model/sheetmodel',
     'io.ox/office/spreadsheet/model/format/documentstyles'
    ], function (Utils, EditModel, SheetUtils, Operations, SheetModel, SpreadsheetDocumentStyles) {

    'use strict';

    // class SpreadsheetModel =================================================

    /**
     * Represents a 'workbook', the spreadsheet document model. Implements
     * execution of all supported operations.
     *
     * Triggers the events supported by the base class EditModel, and the
     * following additional events:
     * - 'insert:sheet': After a new sheet has been inserted into the document.
     *      The event handler receives the zero-based index of the new sheet in
     *      the collection of all sheets.
     * - 'delete:sheet': After a sheet has been removed from the document. The
     *      event handler receives the zero-based index of the sheet in the
     *      collection of all sheets before it has been removed.
     * - 'move:sheet': After a sheet has been moved to a new position. The
     *      event handler receives the zero-based index of the new position in
     *      the collection of all sheets, and the zero-based index of the old
     *      position.
     * - 'rename:sheet': After a sheet has been renamed. The event handler
     *      receives the zero-based index of the renamed sheet in the
     *      collection of all sheets, and the new name of the sheet.
     *
     * @constructor
     *
     * @extends EditModel
     *
     * @param {SpreadsheetApplication} app
     *  The application containing this document model.
     */
    function SpreadsheetModel(app) {

        var // self reference
            self = this,

            // all existing sheets and their names, by zero-based index
            sheetsCollection = [],

            // container for all style sheets of all attribute families
            documentStyles = new SpreadsheetDocumentStyles(app),

            // the maximum column/row index in a sheet
            maxCol = 0, maxRow = 0;

        // base constructor ---------------------------------------------------

        EditModel.call(this, app, Operations, documentStyles, { remoteUndo: true });

        // private methods ----------------------------------------------------

        /**
         * Calculates the range address of an entire sheet.
         */
        function updateMaxColRow(attributes) {
            maxCol = attributes.cols - 1;
            maxRow = attributes.rows - 1;
        }

        /**
         * Returns the sheet info structure at the specified index.
         *
         * @param {Number|String} sheet
         *  The zero-based index of the sheet, or the case-insensitive sheet
         *  name.
         *
         * @returns {Object|Null}
         *  The info structure of the sheet at the passed index or with the
         *  passed name, containing the sheet name and the reference to the
         *  sheet model instance, or null if the passed index or name is
         *  invalid.
         */
        function getSheetInfo(sheet) {

            var // the resulting sheet info
                sheetInfo = null;

            if (_.isNumber(sheet)) {
                // get sheet info structure by index
                sheetInfo = sheetsCollection[sheet];
            } else if (_.isString(sheet)) {
                // get sheet info structure by name
                sheet = sheet.toLowerCase();
                sheetInfo = _(sheetsCollection).find(function (sheetInfo) { return sheetInfo.name.toLowerCase() === sheet; });
            }

            return _.isObject(sheetInfo) ? sheetInfo : null;
        }

        /**
         * Returns whether the passed sheet name is valid (non-empty and not
         * existing in the document).
         *
         * @param {String} sheetName
         *  The desired name of the sheet.
         *
         * @param {Number} [sheet]
         *  If specified, the zero-based index of the sheet that is allowed to
         *  contain the passed name (used while renaming an existing sheet).
         *
         * @returns {Boolean}
         *  Whether the sheet could be renamed to the specified name. The name
         *  must not be empty, and must not be equal to the name of any other
         *  sheet (case-insensitive).
         */
        function isSheetNameValid(sheetName, sheet) {

            var // an existing sheet with the passed name
                sheetModelByName = self.getSheetModel(sheetName),
                // the model of the specified sheet
                sheetModelByIndex = _.isNumber(sheet) ? self.getSheetModel(sheet) : null;

            // check name (no *other* sheet must contain this name)
            return (sheetName.length > 0) && (!_.isObject(sheetModelByName) || !_.isObject(sheetModelByIndex) || (sheetModelByName === sheetModelByIndex));
        }

        // methods ------------------------------------------------------------

        /**
         * Registers a handler function that will be invoked when an operation
         * with the specified name which addresses a specific sheet will be
         * applied.
         *
         * @param {String} name
         *  The name of the operation, as contained in the 'name' attribute of
         *  the operation object.
         *
         * @param {Function} handler
         *  The function that will be invoked for the operation, if it contains
         *  a valid 'sheet' property addressing an existing sheet. Will be
         *  called in the context of this document model instance. Receives the
         *  following parameters:
         *  (1) {SheetModel} sheetModel
         *      The sheet model instance addressed by the operation.
         *  (2) {Object} operation
         *      The complete operation object.
         *  (3) {Boolean} external
         *      The external state that has been passed to the method
         *      EditModel.applyOperations().
         *  See method EditModel.registerOperationHandler() for more details.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.registerSheetOperationHandler = function (name, handler) {
            this.registerOperationHandler(name, function (operation, external) {
                var sheetModel = this.getSheetModel(Utils.getIntegerOption(operation, 'sheet', -1));
                return _.isObject(sheetModel) && handler.call(this, sheetModel, operation, external);
            });
            return this;
        };

        /**
         * Registers a handler function that will be invoked when an operation
         * with the specified name which addresses a specific column/row
         * interval in a specific sheet will be applied.
         *
         * @param {String} name
         *  The name of the operation, as contained in the 'name' attribute of
         *  the operation object.
         *
         * @param {String} type
         *  Whether the operation addresses 'columns' or 'rows'.
         *
         * @param {Function} handler
         *  The function that will be invoked for the operation, if it contains
         *  a valid 'sheet' property addressing an existing sheet, a valid
         *  'start' property addressing an existing column or row, and an
         *  optional 'end' property addressing an existing column or row that
         *  follows the start column/row. Will be called in the context of this
         *  document model instance. Receives the following parameters:
         *  (1) {SheetModel} sheetModel
         *      The sheet model instance addressed by the operation.
         *  (2) {Object} interval
         *      The column/row interval, in the properties 'first' and 'last'.
         *  (3) {Object} operation
         *      The complete operation object.
         *  (4) {Boolean} external
         *      The external state that has been passed to the method
         *      EditModel.applyOperations().
         *  See method EditModel.registerOperationHandler() for more details.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.registerIntervalOperationHandler = function (name, type, handler) {
            this.registerSheetOperationHandler(name, function (sheetModel, operation, external) {
                var start = Utils.getIntegerOption(operation, 'start', -1),
                    end = Utils.getIntegerOption(operation, 'end', start),
                    max = (type === 'columns') ? maxCol : maxRow;
                return (0 <= start) && (start <= end) && (end <= max) &&
                    handler.call(this, sheetModel, { first: start, last: end }, operation, external);
            });
            return this;
        };

        /**
         * Registers a handler function that will be invoked when an operation
         * with the specified name which addresses a specific cell range in a
         * specific sheet will be applied.
         *
         * @param {String} name
         *  The name of the operation, as contained in the 'name' attribute of
         *  the operation object.
         *
         * @param {Function} handler
         *  The function that will be invoked for the operation, if it contains
         *  a valid 'sheet' property addressing an existing sheet, a valid
         *  'start' property addressing a start cell, and an optional 'end'
         *  property addressing an end cell that is not located before the
         *  start cell. Will be called in the context of this document model
         *  instance. Receives the following parameters:
         *  (1) {SheetModel} sheetModel
         *      The sheet model instance addressed by the operation.
         *  (2) {Object} range
         *      The logical range address, in the properties 'start' and 'end'.
         *  (3) {Object} operation
         *      The complete operation object.
         *  (4) {Boolean} external
         *      The external state that has been passed to the method
         *      EditModel.applyOperations().
         *  See method EditModel.registerOperationHandler() for more details.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.registerRangeOperationHandler = function (name, handler) {
            this.registerSheetOperationHandler(name, function (sheetModel, operation, external) {
                var start = Utils.getArrayOption(operation, 'start'),
                    end = Utils.getArrayOption(operation, 'end', start);
                return SheetUtils.isValidAddress(start) && SheetUtils.isValidAddress(end) &&
                    (start[0] <= end[0]) && (end[0] <= maxCol) && (start[1] <= end[1]) && (end[1] <= maxRow) &&
                    handler.call(this, sheetModel, { start: start, end: end }, operation, external);
            });
            return this;
        };

        /**
         * Returns the cell style sheet container of this document.
         *
         * @returns {CellStyles}
         *  The cell style sheet container.
         */
        this.getCellStyles = function () {
            return this.getDocumentStyles().getStyleSheets('cell');
        };

        // sheet range --------------------------------------------------------

        /**
         * Returns the column index of the rightmost cells in a sheet.
         *
         * @returns {Number}
         *  The column index of the rightmost cells in a sheet.
         */
        this.getMaxCol = function () {
            return maxCol;
        };

        /**
         * Returns the row index of the bottom cells in a sheet.
         *
         * @returns {Number}
         *  The row index of the bottom cells in a sheet.
         */
        this.getMaxRow = function () {
            return maxRow;
        };

        /**
         * Returns the logical range address of the entire area in a sheet,
         * from the top-left cell to the bottom-right cell.
         *
         * @returns {Object}
         *  The logical range address of an entire sheet.
         */
        this.getSheetRange = function () {
            return { start: [0, 0], end: [maxCol, maxRow] };
        };

        /**
         * Returns whether the passed range address covers one or more entire
         * columns in the active sheet.
         *
         * @param {Object} range
         *  A logical range address.
         *
         * @returns {Boolean}
         *  Whether the passed range address covers one or more entire columns.
         */
        this.isColRange = function (range) {
            return (range.start[1] === 0) && (range.end[1] === maxRow);
        };

        /**
         * Returns the logical address of the cell range covering the specified
         * column interval.
         *
         * @param {Object|Number} interval
         *  The column interval, in the zero-based column index properties
         *  'first' and 'last', or a single zero-based column index.
         *
         * @returns {Object}
         *  The logical address of the column range.
         */
        this.makeColRange = function (interval) {
            return {
                start: [_.isNumber(interval) ? interval : interval.first, 0],
                end: [_.isNumber(interval) ? interval : interval.last, maxRow]
            };
        };

        /**
         * Returns the logical addresses of the cell ranges covering the
         * specified column intervals.
         *
         * @param {Object|Array} intervals
         *  A single column interval, or an array of column intervals. Each
         *  interval object contains the zero-based index properties 'first'
         *  and 'last'.
         *
         * @returns {Array}
         *  The logical addresses of the column ranges.
         */
        this.makeColRanges = function (intervals) {
            return _.chain(intervals).getArray().map(_.bind(this.makeColRange, this)).value();
        };

        /**
         * Returns whether the passed range address covers one or more entire
         * rows in the active sheet.
         *
         * @param {Object} range
         *  A logical range address.
         *
         * @returns {Boolean}
         *  Whether the passed range address covers one or more entire rows.
         */
        this.isRowRange = function (range) {
            return (range.start[0] === 0) && (range.end[0] === maxCol);
        };

        /**
         * Returns the logical address of the cell range covering the specified
         * row interval.
         *
         * @param {Object|Number} interval
         *  The row interval, in the zero-based row index properties 'first'
         *  and 'last', or a single zero-based row index.
         *
         * @returns {Object}
         *  The logical address of the row range.
         */
        this.makeRowRange = function (interval) {
            return {
                start: [0, _.isNumber(interval) ? interval : interval.first],
                end: [maxCol, _.isNumber(interval) ? interval : interval.last]
            };
        };

        /**
         * Returns the logical addresses of the cell ranges covering the
         * specified row intervals.
         *
         * @param {Object|Array} intervals
         *  A single row interval, or an array of row intervals. Each interval
         *  object contains the zero-based index properties 'first' and 'last'.
         *
         * @returns {Array}
         *  The logical addresses of the row ranges.
         */
        this.makeRowRanges = function (intervals) {
            return _.chain(intervals).getArray().map(_.bind(this.makeRowRange, this)).value();
        };

        // sheets -------------------------------------------------------------

        /**
         * Returns the number of sheets in the spreadsheet document.
         *
         * @returns {Number}
         *  The number of sheets in the document.
         */
        this.getSheetCount = function () {
            return sheetsCollection.length;
        };

        /**
         * Returns the names of all sheets in the spreadsheet document.
         *
         * @returns {String[]}
         *  The names of all sheets in the document.
         */
        this.getSheetNames = function () {
            return _(sheetsCollection).pluck('name');
        };

        /**
         * Returns whether a sheet at the specified index or with the specified
         * case-insensitive name exists in the spreadsheet document.
         *
         * @param {Number|String} sheet
         *  The zero-based index of the sheet, or the case-insensitive sheet
         *  name.
         *
         * @returns {Boolean}
         *  Whether a sheet exists at the passed index or with the passed name.
         */
        this.hasSheet = function (sheet) {
            return _.isObject(getSheetInfo(sheet));
        };

        /**
         * Returns the exact name of the sheet at the specified index or with
         * the specified case-insensitive name.
         *
         * @param {Number|String} sheet
         *  The zero-based index of the sheet, or the case-insensitive sheet
         *  name.
         *
         * @returns {String|Null}
         *  The name of the sheet at the passed index or with the passed name,
         *  or null if the passed index or name is invalid.
         */
        this.getSheetName = function (sheet) {
            var sheetInfo = getSheetInfo(sheet);
            return _.isObject(sheetInfo) ? sheetInfo.name : null;
        };

        /**
         * Returns the model instance for the sheet at the specified index or
         * with the specified name.
         *
         * @param {Number|String} sheet
         *  The zero-based index of the sheet, or the case-insensitive sheet
         *  name.
         *
         * @returns {SheetModel|Null}
         *  The model instance for the sheet at the passed index or with the
         *  specified name, or null if the passed index or name is invalid.
         */
        this.getSheetModel = function (sheet) {
            var sheetInfo = getSheetInfo(sheet);
            return _.isObject(sheetInfo) ? sheetInfo.model : null;
        };

        /**
         * Creates a new sheet in this document.
         *
         * @param {Number} sheet
         *  The zero-based insertion index of the sheet.
         *
         * @param {String} sheetName
         *  The name of the new sheet. Must not be empty. Must not be equal to
         *  the name of any existing sheet.
         *
         * @param {Object} [attrs]
         *  Attribute set containing initial formatting attributes for the new
         *  sheet.
         *
         * @returns {SheetModel|Null}
         *  The model instance of the new sheet.
         */
        this.insertSheet = function (sheet, sheetName, attrs) {
            // remove all NPCs from passed sheet name
            sheetName = Utils.cleanString(sheetName);
            // check that the sheet can be named as specified (prevent internal application error on failing operation)
            if (!isSheetNameValid(sheetName)) { return null; }
            // generate and apply the operation
            return this.applyOperations({ name: Operations.INSERT_SHEET, sheet: sheet, sheetName: sheetName, attrs: attrs }) ? this.getSheetModel(sheet) : null;
        };

        /**
         * Removes a sheet from this document.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet.
         *
         * @returns {Boolean}
         *  Whether deleting the sheet was successful.
         */
        this.deleteSheet = function (sheet) {
            // check that the sheet can be removed (prevent internal application error on failing operation)
            return (this.getSheetCount() > 1) && this.applyOperations({ name: Operations.DELETE_SHEET, sheet: sheet });
        };

        /**
         * Renames a sheet in the spreadsheet document.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet to be renamed.
         *
         * @param {String} sheetName
         *  The new name of the sheet. Must not be empty. Must not be equal to
         *  the name of any other sheet (case-insensitive).
         *
         * @returns {Boolean}
         *  Whether renaming the sheet was successful.
         */
        this.setSheetName = function (sheet, sheetName) {
            // remove all NPCs from passed sheet name
            sheetName = Utils.trimAndCleanString(sheetName);
            // check that the sheet can be renamed (prevent internal application error on failing operation)
            if (!isSheetNameValid(sheetName, sheet)) { return false; }
            // do not generate the operation, if passed name matches old name
            if (this.getSheetName(sheet) === sheetName) { return true; }
            // generate and apply the operation
            return this.applyOperations({ name: Operations.SET_SHEET_NAME, sheet: sheet, sheetName: sheetName });
        };

        /**
         * Called when document import has failed. Inserts an empty sheet, if
         * this model does not contain any sheets. This helps preventing to
         * write if-else statements for empty documents at thousand places in
         * the view code (there will always be an active sheet with selection
         * etc.).
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.prepareInvalidDocument = function () {
            if (sheetsCollection.length === 0) {
                sheetsCollection.push({ model: new SheetModel(app), name: 'Sheet1' });
            }
            return this;
        };

        // operations registration --------------------------------------------

        this.registerOperationHandler(Operations.INSERT_SHEET, function (operation) {

            var // the sheet index passed in the operation
                sheet = Utils.getIntegerOption(operation, 'sheet', -1),
                // the sheet name passed in the operation
                sheetName = Utils.getStringOption(operation, 'sheetName', ''),
                // the sheet attributes passed in the operation
                attributes = Utils.getObjectOption(operation, 'attrs'),
                // the model of the new sheet
                sheetModel = null;

            // check the sheet index and name (must not exist yet)
            if ((sheet < 0) || (sheet > this.getSheetCount()) || !isSheetNameValid(sheetName)) {
                return false;
            }

            // create and insert the new sheet
            sheetModel = new SheetModel(app, attributes);
            sheetsCollection.splice(sheet, 0, { model: sheetModel, name: sheetName });
            self.trigger('insert:sheet', sheet);
            return true;
        });

        this.registerOperationHandler(Operations.DELETE_SHEET, function (operation) {

            var // the sheet index passed in the operation
                sheet = Utils.getIntegerOption(operation, 'sheet', -1),
                // the sheet object
                sheetModel = this.getSheetModel(sheet);

            // the sheet must exist, there must remain at least one sheet in the document
            if (!_.isObject(sheetModel) || (this.getSheetCount() === 1)) {
                return false;
            }

            // finally, remove the sheet from the document model
            sheetModel.destroy();
            sheetsCollection.splice(sheet, 1);
            this.trigger('delete:sheet', sheet);
            return true;
        });

        this.registerOperationHandler(Operations.MOVE_SHEET, function (operation) {
        });

        this.registerOperationHandler(Operations.SET_SHEET_NAME, function (operation) {

            var // the sheet index passed in the operation
                sheet = Utils.getIntegerOption(operation, 'sheet', -1),
                // the sheet object
                sheetInfo = getSheetInfo(sheet),
                // the sheet name passed in the operation
                sheetName = Utils.getStringOption(operation, 'sheetName', '');

            // the sheet must exist, check name (no other sheet must contain this name)
            if (!_.isObject(sheetInfo) || !isSheetNameValid(sheetName, sheet)) {
                return false;
            }

            // no no events, if passed name matches old name
            if (sheetInfo.name !== sheetName) {
                sheetInfo.name = sheetName;
                this.trigger('rename:sheet', sheet, sheetName);
            }

            return true;
        });

        this.registerSheetOperationHandler(Operations.SET_SHEET_ATTRIBUTES, function (sheetModel, operation) {
            var attributes = Utils.getObjectOption(operation, 'attrs');
            return _.isObject(attributes) && sheetModel.setAttributes(attributes);
        });

        this.registerIntervalOperationHandler(Operations.INSERT_COLUMNS, 'columns', function (sheetModel, interval) {
            return sheetModel.getColCollection().insertEntries(interval);
        });

        this.registerIntervalOperationHandler(Operations.DELETE_COLUMNS, 'columns', function (sheetModel, interval) {
            return sheetModel.getColCollection().deleteEntries(interval);
        });

        this.registerIntervalOperationHandler(Operations.SET_COLUMN_ATTRIBUTES, 'columns', function (sheetModel, interval, operation) {
            var attributes = Utils.getObjectOption(operation, 'attrs');
            return _.isObject(attributes) && sheetModel.getColCollection().setAttributes(interval, attributes);
        });

        this.registerIntervalOperationHandler(Operations.INSERT_ROWS, 'rows', function (sheetModel, interval) {
            return sheetModel.getRowCollection().insertEntries(interval);
        });

        this.registerIntervalOperationHandler(Operations.DELETE_ROWS, 'rows', function (sheetModel, interval) {
            return sheetModel.getRowCollection().deleteEntries(interval);
        });

        this.registerIntervalOperationHandler(Operations.SET_ROW_ATTRIBUTES, 'rows', function (sheetModel, interval, operation) {
            var attributes = Utils.getObjectOption(operation, 'attrs');
            return _.isObject(attributes) && sheetModel.getRowCollection().setAttributes(interval, attributes);
        });

        this.registerRangeOperationHandler(Operations.INSERT_CELLS, function (sheetModel, range) {
            // TODO
            return false;
        });

        this.registerRangeOperationHandler(Operations.DELETE_CELLS, function (sheetModel, range) {
            // TODO
            return false;
        });

        this.registerSheetOperationHandler(Operations.MERGE_CELLS, function (sheetModel, operation) {
            var start = Utils.getArrayOption(operation, 'start', []),
                end = Utils.getArrayOption(operation, 'end', []),
                type = Utils.getStringOption(operation, 'type', 'merge');
            return sheetModel.getMergeCollection().mergeCells(start, end, type);
        });

        this.registerSheetOperationHandler(Operations.INSERT_DRAWING, function (sheetModel, operation) {
            var start = Utils.getArrayOption(operation, 'start', []),
                type = Utils.getStringOption(operation, 'type', ''),
                attributes = Utils.getObjectOption(operation, 'attrs');
            return sheetModel.getDrawingCollection().insertModel(start, type, attributes);
        });

        this.registerSheetOperationHandler(Operations.DELETE_DRAWING, function (sheetModel, operation) {
            var start = Utils.getArrayOption(operation, 'start', []);
            return sheetModel.getDrawingCollection().deleteModel(start);
        });

        this.registerSheetOperationHandler(Operations.SET_DRAWING_ATTRIBUTES, function (sheetModel, operation) {
            var start = Utils.getArrayOption(operation, 'start', []),
                attributes = Utils.getObjectOption(operation, 'attrs'),
                drawingModel = sheetModel.getDrawingCollection().findModel(start, { deep: true });
            if (_.isObject(attributes) && !_.isObject(drawingModel)) { return false; }
            drawingModel.setAttributes(attributes);
            return true;
        });

        // operations generated locally, to be ignored silently
        this.registerOperationHandler(Operations.FILL_CELL_RANGE, $.noop);
        this.registerOperationHandler(Operations.SET_CELL_CONTENTS, $.noop);

        // initialization -----------------------------------------------------

        // set the column/row count, listen to document attribute changes
        updateMaxColRow(documentStyles.getAttributes());
        documentStyles.on('change:attributes', function (event, attributes) {
            updateMaxColRow(attributes);
        });

    } // class SpreadsheetModel

    // exports ================================================================

    // derive this class from class EditModel
    return EditModel.extend({ constructor: SpreadsheetModel });

});
