/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/mergecollection',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/model/modelobject',
     'io.ox/office/spreadsheet/utils/sheetutils'
    ], function (Utils, ModelObject, SheetUtils) {

    'use strict';

    // class MergeCollection ==================================================

    /**
     * Collects information about all merged cells of a single sheet in a spreadsheet document.
     *
     * Triggers the following events:
     * - 'insert:entries'
     *      After new merged cells have been inserted into the sheet.
     *      // TODO: The event handler receives the index interval of the inserted entries.
     * - 'delete:entries'
     *      After merged cells have been deleted from the sheet.
     *      // TODO: The event handler receives the index interval of the deleted entries.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this collection instance.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     */
    function MergeCollection(app, sheetModel) {

        var // self reference
            self = this,

            // array of merged cells, described as range addresses
            // - contains ranges as objects with 'start' and 'end'
            mergedRanges = [];

        // base constructor ---------------------------------------------------

        ModelObject.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Triggers the specified event, but not while the document is still
         * being imported. TODO: Event name might be: change:mergedCells
         */
        function triggerEvent(type, changedRanges) {

            // debug: dump contents of the collection
            if (app.isImportFinished()) {
                Utils.info('MergeCollection: length=' + mergedRanges.length);
                _(mergedRanges).each(function (range, index) {
                    Utils.log('  ' + index + ': range=' + SheetUtils.getRangeName(range));
                });
            }

            self.trigger(type, changedRanges);  // sending list of changed merged cells
        }

        /**
         * Adds one new range to the list of existing merged ranges.
         * A new range might be already included in the array of merged ranges
         * or it might be bigger than existing merged ranges.
         * -> A new range cannot be part of an existing merged range because of
         * auto expanding of merged cells during selection.
         */
        function addRangeToMergeRanges(range) {
            // 1. step: Removing all existing merged ranges, that
            // are part of the new merged range.
            mergedRanges = _.reject(mergedRanges, function (oneRange) {
                return SheetUtils.rangeContainsRange(range, oneRange);
            });
            // 2. step: Adding the new merged range.
            mergedRanges.push(range);
        }

        /**
         * Dumps the content of an ranges array to the console.
         *
         * @param {String} text
         *  An additional text displayed in the console.
         *
         * @param {Array} ranges
         *  Array of range addresses.
         */
        function dumpRanges(text, ranges) {
            Utils.log(text + ' : ' + JSON.stringify(ranges));
        }

        /**
         * Handling the insertion of columns into the current sheet.
         */
        function insertColumnsHandler(event, interval) {
            insertColRow(interval, 'column');
        }

        /**
         * Handling the insertion of rows into the current sheet.
         */
        function insertRowsHandler(event, interval) {
            insertColRow(interval, 'row');
        }

        /**
         * Handling the deletion of columns from the current sheet.
         */
        function deleteColumnsHandler(event, interval) {
            deleteColRow(interval, 'column');
        }

        /**
         * Handling the deletion of rows from the current sheet.
         */
        function deleteRowsHandler(event, interval) {
            deleteColRow(interval, 'row');
        }

        /**
         * Calculating new merged cell ranges after inserting columns or rows.
         */
        function insertColRow(interval, type) {

            var // collection of all merged ranges that will be modified and need to be rendered again
                changedRanges = [],
                // the length of the interval
                size = SheetUtils.getIntervalSize(interval),
                // the index of the cell position to be evaluated
                index = (type === 'column') ? 0 : 1;

            // iterating over all merged ranges
            _(mergedRanges).each(function (range) {

                if (interval.first > range.end[index]) {
                    // do nothing, new columns/row behind/below merged cell
                } else if (interval.first <= range.start[index]) {
                    // case: shifting cell completely to the right/bottom
                    changedRanges.push(_.copy(range, true)); // old position
                    range.start[index] += size;
                    range.end[index] += size;
                    changedRanges.push(_.copy(range, true));  // new position

                } else {
                    // case: expanding cell
                    range.end[index] += size;
                    changedRanges.push(_.copy(range, true));  // new position (includes old position)
                }

            });

            // trigger change events with array of changed merged ranges
            if (! _.isEmpty(changedRanges)) {
                triggerEvent('changed:mergedCells', changedRanges);
            }

        }

        /**
         * Calculating new merged cell ranges after deleting columns or rows.
         */
        function deleteColRow(interval, type) {

            var // collection of all merged ranges that will be modified and need to be rendered again
                changedRanges = [],
                // collection of merged ranges that will be removed completely
                deleteRanges = [],
                // the length of the interval
                size = SheetUtils.getIntervalSize(interval),
                // the intersection of the merged cell range and the specified interval
                intersectionInterval = null,
                // the index of the cell position to be evaluated
                index = (type === 'column') ? 0 : 1,
                // the second index, that will not be evaluated
                altIndex =  (type === 'column') ? 1 : 0;

            // iterating over all merged ranges
            _(mergedRanges).each(function (range) {

                if (interval.first > range.end[index]) {
                    // do nothing, removed columns/rows behind/below merged cell
                } else if (interval.last < range.start[index]) {
                    // case: shifting cell completely to the top/left
                    changedRanges.push(_.copy(range, true)); // old position
                    range.start[index] -= size;
                    range.end[index] -= size;
                    changedRanges.push(_.copy(range, true));  // new position
                } else {
                    // case: shrinking cell or remove merged cell from list
                    // -> calculating the number of rows/columns that will be removed from merged cell
                    intersectionInterval = SheetUtils.getIntersectionInterval({ first: range.start[index], last: range.end[index] }, interval);
                    if (intersectionInterval) {
                        changedRanges.push(_.copy(range, true));  // old range position (includes not always new position)
                        size = SheetUtils.getIntervalSize(intersectionInterval);

                        // checking remaining merged cell -> completely removed or simple 1x1 cell remaining

                        if (((SheetUtils.getColRowCount(range, index) - size) ===  0) ||
                            (((SheetUtils.getColRowCount(range, index) - size) ===  1) && (SheetUtils.getColRowCount(range, altIndex) ===  1)))  {
                            // this is no longer a merged cell -> removing from list of merged cells
                            deleteRanges.push(_.copy(range, true));
                        } else {
                            // decreasing size of merged cell range
                            range.end[index] -= size;

                            // there might be columns/rows left/top of the merged cell, that will be removed, too
                            // -> shifting remaining merged cell completely to left/top
                            if (range.start[index] > interval.first) {
                                size = range.start[index] - interval.first;
                                range.start[index] -= size;
                                range.end[index] -= size;
                            }

                            changedRanges.push(_.copy(range, true));  // new range cell
                        }
                    }
                }
            });

            // removing merged cell ranges completely
            if (! _.isEmpty(deleteRanges)) {
                _(deleteRanges).each(function (range) {
                    removeRangeFromMergedRanges(range);
                });
            }

            // trigger change events with array of changed merged ranges
            if (! _.isEmpty(changedRanges)) {
                triggerEvent('changed:mergedCells', changedRanges);
            }

        }

        /**
         * Removing one range completely from list of merged cell ranges.
         *
         * @param {Object} range
         *  The range address of the merged cell range, that shall be expanded.
         */
        function removeRangeFromMergedRanges(range) {

            mergedRanges = _.reject(mergedRanges, function (oneRange) {
                return SheetUtils.equalRanges(oneRange, range);
            });
        }

        // methods ------------------------------------------------------------

        /**
         * Returns, whether the cell at the specified logical address is the upper
         * left corner of a merged cell range.
         *
         * @param {Number[]} address
         *  The logical address of a cell.
         *
         * @returns {Boolean}
         *  Whether the cell at the specified logical address is the upper
         * left corner of a merged cell range.
         */
        this.isReferenceCell = function (address) {
            return self.getMergedRange(address) !== null;
        };

        /**
         * Returns, whether the cell at the specified logical address is a
         * hidden cell inside a merged cell range.
         *
         * @param {Number[]} address
         *  The logical address of a cell.
         *
         * @returns {Boolean}
         *  Whether the cell at the specified logical address is a
         *  hidden cell inside a merged cell range.
         */
        this.isHiddenCell = function (address) {
            return self.getReferenceAddress(address) !== null;
        };

        /**
         * Returns, whether the cell at the specified logical address is part
         * of a merged cell. It can be the reference cell or a hidden cell.
         *
         * @param {Number[]} address
         *  The logical address of a cell.
         *
         * @returns {Boolean}
         *  Whether the cell is part of a merged cell range.
         */
        this.isMergedCell = function (address) {

            return _.find(mergedRanges, function (oneRange) {
                return SheetUtils.rangeContainsCell(oneRange, address);
            }) !== undefined;
        };

        /**
         * Returns, whether a specific range is in the list of the
         * merged cells 'mergedRanges'.
         *
         * @param {Object} range
         *  A range address.
         *
         * @returns {Boolean}
         *  Whether the specified range can be found in the list
         *  of merged cell ranges.
         */
        this.isMergedRange = function (range) {

            return _.find(mergedRanges, function (oneRange) {
                return SheetUtils.equalRanges(oneRange, range);
            }) !== undefined;
        };

        /**
         * Returns the logical range address of a merged cell range starting at
         * the specified cell address. The specified address must be the address
         * of the upper left cell in the range. If it is not the upper left cell
         * in the range (a hidden cell) or if it is not located inside a merge
         * range, 'null' is returned.
         *
         * @param {Number[]} address
         *  The logical address of a cell.
         *
         * @returns {Object|Null}
         *  The logical range address of a merged range whose top-left address
         *  is equal to the address passed to this method; or null, if no
         *  merged range exists at the address.
         */
        this.getMergedRange = function (address) {

            var range = _.find(mergedRanges, function (oneRange) {
                    return _.isEqual(oneRange.start, address);
                });

            return range ? _.copy(range, true) : null;
        };

        /**
         * Returns the logical reference address for a hidden merged cell. Given a
         * specific cell address, the reference address of the upper left corner of
         * the merged cell range is returned, if the given cell address is part of
         * the merged cell range. If the given cell address is not part of a merged
         * cell range, or if it is already the reference address, 'null' is returned.
         *
         * @param {Number[]} address
         *  The logical address of a cell.
         *
         * @returns {Number[]|Null}
         *  The logical cell address of the upper left cell of a merged cell range, if
         *  address describes a hidden cell inside a merged range. If no merge range
         *  exists at the address or if the address is already the upper left corner
         *  of a merged range, 'null' is returned.
         */
        this.getReferenceAddress = function (address) {

            var range = _.find(mergedRanges, function (oneRange) {
                    return SheetUtils.rangeContainsCell(oneRange, address);
                });

            return ((range) && (range.start) && (! _.isEqual(range.start, address))) ? _.copy(range.start, true) : null;
        };

        /**
         * Returns whether a list of ranges contain at least one merged range.
         * The address of the upper left cell of the ranges must be located
         * at least inside one of the specified ranges.
         *
         * @param {Array} ranges
         *  Array of range addresses.
         *
         * @returns {Boolean}
         *  Whether there is at least one range containing a merged range.
         */
        this.rangesContainMergedRange = function (ranges) {

            return _.find(ranges, function (oneRange) {
                return self.rangeContainsMergedRange(oneRange);
            }) !== undefined;
        };

        /**
         * Returns whether an arbitrary range contains at least one of the
         * ranges stored in the 'mergedRanges' array. This is the case, if
         * the upper left corner cell is located inside the specified range.
         * This function requires, that merged ranges are completely
         * included into the specified range (automatic expansion of
         * selection).
         *
         * @param {Object} range
         *  A range address.
         *
         * @returns {Boolean}
         *  Whether there is at least one merged range inside the specified
         *  range.
         */
        this.rangeContainsMergedRange = function (range) {

            return _.find(mergedRanges, function (oneRange) {
                return SheetUtils.rangeContainsCell(range, oneRange.start);
            }) !== undefined;
        };

        /**
         * Recalculates the list of merged ranges in the current sheet. This function
         * is called from the spreadsheet model from the operation handler for the
         * operation 'Operations.MERGE_CELLS'.
         *
         * @param {Number[]} start
         *  The upper left corner of the range, that will be merged or unmerged.
         *
         * @param {Number[]} end
         *  The bottom right corner of the range, that will be merged or unmerged.
         *
         * @param {String} type
         *  The merge type. Must be one of: 'merge', 'horizontal', 'vertical',
         *  or 'unmerge'.
         */
        this.mergeCells = function (start, end, type) {

            var // collection of all merged ranges that will be modified and need to be rendered again
                changedRanges = [],
                // one single merge range
                mergeRange = { start: start, end: end };

            // Taking care of ranges, that cover existing merged ranges only partially.
            // -> unmerging all partially covered merged cell ranges.
            // This can only happen with external operations, because of auto expansion of selection.
            mergedRanges = _.reject(mergedRanges, function (oneRange) {
                var isPartial = (! SheetUtils.equalRanges(mergeRange, oneRange)) && (SheetUtils.getIntersectionRange(mergeRange, oneRange) !== null);
                if (isPartial) {
                    changedRanges.push(oneRange);  // collecting ranges, that will be removed and need to be rendered again
                }
                return isPartial;
            });

            // Handling the different merge types
            if (type === 'merge') {
                addRangeToMergeRanges(mergeRange);
                changedRanges.push(mergeRange); // this is sufficient, because unmerged ranges inside this range will also be rendered
            } else if (type === 'horizontal') {
                Utils.iterateRange(start[1], end[1] + 1, function (row) {
                    mergeRange = { start: [start[0], row], end: [end[0], row] };
                    addRangeToMergeRanges(mergeRange);
                    changedRanges.push(mergeRange); // this is sufficient, because unmerged ranges inside this range will also be rendered
                });
            } else if (type === 'vertical') {
                Utils.iterateRange(start[0], end[0] + 1, function (col) {
                    mergeRange = { start: [col, start[1]], end: [col, end[1]] };
                    addRangeToMergeRanges(mergeRange);
                    changedRanges.push(mergeRange); // this is sufficient, because unmerged ranges inside this range will also be rendered
                });
            } else if (type === 'unmerge') {
                // unmerging all merged cells in the specified range
                // -> assuming that all merged ranges are completely included into the specified range
                // -> requires auto expanding of merged cells during selection
                mergedRanges = _.reject(mergedRanges, function (oneRange) {
                    var contains = SheetUtils.rangeContainsCell(mergeRange, oneRange.start);
                    if (contains) {
                        changedRanges.push(oneRange);  // collecting ranges, that will be removed
                    }
                    return contains;
                });
            }

            // trigger change events with array of changed merged ranges
            if (! _.isEmpty(changedRanges)) {
                triggerEvent('change:mergedCells', changedRanges);
            }

            dumpRanges('Merged ranges', mergedRanges);
            dumpRanges('Changed merged ranges', changedRanges);
        };

        // initialization -----------------------------------------------------

        // Recalculating merged cells after inserting or deleting columns or rows
        sheetModel.getColCollection().on('insert:entries', insertColumnsHandler);
        sheetModel.getColCollection().on('delete:entries', deleteColumnsHandler);
        sheetModel.getRowCollection().on('insert:entries', insertRowsHandler);
        sheetModel.getRowCollection().on('delete:entries', deleteRowsHandler);

    } // class MergeCollection

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: MergeCollection });

});
