/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/app/controller',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/model/format/border',
     'io.ox/office/editframework/model/format/mergedborder',
     'io.ox/office/editframework/app/editcontroller',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/model/format/cellstyles'
    ], function (Utils, Border, MergedBorder, EditController, SheetUtils, CellStyles) {

    'use strict';

    // global private function ------------------------------------------------

    /**
     * Returns the merged border properties from all visible border attributes.
     * Ambiguous border properties will be set to the value null.
     *
     * @param {Object} borderAttributes
     *  An attribute map containing only border attributes (name/value pairs).
     *
     * @returns {Border}
     *  A border object containing the properties 'style', 'width', and
     *  'color'. If the property is equal in all visible border attributes, it
     *  will be set as property value in this object, otherwise the property
     *  will be set to the value null. If no border is visible at all, an
     *  invisible border object will be returned.
     *  Important: MergedBorder.mergeBorders cannot be used here, because the
     *  borderAttributes contains already merged borders.
     */
    function mergeBorderProperties(borderAttributes) {

        var // the merged border attributes
            mergedBorder = {};

        _.each(borderAttributes, function (oneBorder) {
            mergedBorder = MergedBorder.mergeMergedBorders(mergedBorder, oneBorder);
        });

        return mergedBorder;
    }

    // class SpreadsheetController ============================================

    /**
     * The controller of a OX Spreadsheet application.
     *
     * @constructor
     *
     * @extends EditController
     */
    function SpreadsheetController(app) {

        var // self reference
            self = this,

            // the spreadsheet model
            model = null,

            // the spreadsheet view
            view = null,

            // all the little controller items
            items = {

                // view -------------------------------------------------------

                'view/sheet/active': {
                    get: function () { return view.getActiveSheet(); },
                    set: function (sheet) { view.setActiveSheet(sheet); }
                },

                'view/sheet/prev': {
                    enable: function () { return view.getActiveSheet() > 0; },
                    set: function (sheet) { view.setActiveSheet(view.getActiveSheet() - 1); },
                    shortcut: { keyCode: 'PAGE_UP', ctrlOrMeta: true, alt: true }
                },

                'view/sheet/next': {
                    enable: function () { return view.getActiveSheet() + 1 < model.getSheetCount(); },
                    set: function (sheet) { view.setActiveSheet(view.getActiveSheet() + 1); },
                    shortcut: { keyCode: 'PAGE_DOWN', ctrlOrMeta: true, alt: true }
                },

                'view/cell/editable': {
                    parent: 'document/editable',
                    enable: function () { return !view.hasDrawingSelection(); }
                },

                'view/attributes': {
                    parent: 'view/cell/editable',
                    get: function () { return view.getCellContents().attrs; }
                },

                'view/subtotals/sum': {
                    get: function () { return 0; }
                },

                'view/split': {
                    get: function () { return view.getSplitMode(); },
                    set: function (state) { view.setSplitMode(state); }
                },

                'view/freeze': {
                    get: function () { return view.getFreezeMode(); },
                    set: function (state) { view.setFreezeMode(state); }
                },

                // sheet operations -------------------------------------------

                'sheet/insert': {
                    parent: 'document/editable',
                    set: function () { view.insertSheet(); }
                },

                'sheet/delete': {
                    parent: 'document/editable',
                    enable: function () { return model.getSheetCount() > 1; },
                    set: function () { view.deleteSheet(); }
                },

                'sheet/name': {
                    parent: 'document/editable',
                    get: function () { return model.getSheetName(view.getActiveSheet()); },
                    set: function (sheetName) { view.renameSheet(sheetName); }
                },

                // column operations ------------------------------------------

                'column/insert': {
                    parent: 'view/cell/editable',
                    enable: function () { return view.columnsInsertable(); },
                    set: function () { view.insertColumns(); }
                },

                'column/delete': {
                    parent: 'view/cell/editable',
                    set: function () { view.deleteColumns(); }
                },

                'column/attributes': {
                    parent: 'view/attributes',
                    get: function (attributes) { return attributes.column; }
                },

                'column/visible': {
                    parent: 'column/attributes',
                    get: function (attributes) { return attributes.visible; }
                },

                'column/width': {
                    parent: 'column/attributes',
                    get: function (attributes) { return attributes.width; },
                    set: function (value) { view.setColumnWidth(value); }
                },

                // row operations ---------------------------------------------

                'row/insert': {
                    parent: 'view/cell/editable',
                    enable: function () { return view.rowsInsertable(); },
                    set: function () { view.insertRows(); }
                },

                'row/delete': {
                    parent: 'view/cell/editable',
                    set: function () { view.deleteRows(); }
                },

                'row/attributes': {
                    parent: 'view/attributes',
                    get: function (attributes) { return attributes.row; }
                },

                'row/visible': {
                    parent: 'row/attributes',
                    get: function (attributes) { return attributes.visible; }
                },

                'row/height': {
                    parent: 'row/attributes',
                    get: function (attributes) { return attributes.height; },
                    set: function (value) { view.setRowHeight(value); }
                },

                // cell attributes --------------------------------------------

                'cell/stylesheet': {
                    parent: 'view/attributes',
                    get: function (attributes) { return attributes.styleId; },
                    set: function (styleId) { view.fillCellContents(undefined, { styleId: styleId }, { clear: true }); }
                },

                'cell/attributes': {
                    parent: 'view/attributes',
                    get: function (attributes) { return attributes.cell; }
                },

                'cell/fillcolor': {
                    parent: 'cell/attributes',
                    get: function (attributes) { return attributes.fillColor; },
                    set: function (color) { view.setCellAttribute('fillColor', color); }
                },

                'cell/alignhor': {
                    parent: 'cell/attributes',
                    get: function (attributes) { return attributes.alignHor; },
                    set: function (alignment) { view.setCellAttribute('alignHor', alignment); }
                },

                'cell/alignvert': {
                    parent: 'cell/attributes',
                    get: function (attributes) { return attributes.alignVert; },
                    set: function (alignment) { view.setCellAttribute('alignVert', alignment); }
                },

                'cell/resetAttributes': {
                    parent: 'cell/attributes',
                    set: function () { view.clearAttributes(); }
                },

                // cell border attributes -------------------------------------

                // border mode, as boolean map with position keys (left, top, ...)
                'cell/border/mode': {
                    parent: 'view/cell/editable',
                    get: function () { return view.getBorderMode(); },
                    options: function () { return { showInsideHor: view.hasMultipleRowsSelected(), showInsideVert: view.hasMultipleColumnsSelected() }; },
                    set: function (borderMode) { view.setBorderMode(borderMode); }
                },

                // all merged border attributes of the selection
                'cell/border/attributes': {
                    parent: 'view/cell/editable',
                    get: function () { return view.getBorderAttributes(); }
                },

                // a single merged border for all borders in the selection
                'cell/border/value': {
                    parent: 'cell/border/attributes',
                    enable: function () { return _(view.getBorderAttributes()).any(MergedBorder.isVisibleBorder); },
                    get: function () { return mergeBorderProperties(view.getBorderAttributes()); }
                },

                'cell/border/style': {
                    parent: 'cell/border/value',
                    get: function (border) { return border.style; },
                    set: function (value) { view.setBorderAttributes({ style: value }); }
                },

                // width of the border lines, in points (rounded to 1/2 points)
                'cell/border/width': {
                    parent: 'cell/border/value',
                    get: function (border) { return _.isNumber(border.width) ? Utils.convertHmmToLength(border.width, 'pt', 0.5) : null; },
                    set: function (value) { view.setBorderAttributes({ width: Utils.convertLengthToHmm(value, 'pt') }); }
                },

                'cell/border/color': {
                    parent: 'cell/border/value',
                    get: function (border) { return border.color; },
                    set: function (value) { view.setBorderAttributes({ color: value }); }
                },

                // merging cells
                'cell/merge': {
                    parent: 'view/cell/editable',
                    get: function () { return view.selectionHasMergedCells(); },
                    set: function (value) {
                        if (value === 'toggleMerge') {  // simulating toggle behavior for merging cells
                            value = view.selectionHasMergedCells() ? 'unmerge' : 'merge';
                        }
                        view.mergeCells(value);
                    }
                },

                // character attributes ---------------------------------------

                'character/attributes': {
                    parent: 'view/attributes',
                    get: function (attributes) { return attributes.character; }
                },

                'character/bold': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.bold; },
                    set: function (state) { view.setCharacterAttribute('bold', state); }
                },

                'character/italic': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.italic; },
                    set: function (state) { view.setCharacterAttribute('italic', state); }
                },

                'character/underline': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.underline; },
                    set: function (state) { view.setCharacterAttribute('underline', state); }
                },

                'character/strike': {
                    parent: 'character/attributes',
                    get: function (attributes) { return _.isString(attributes.strike) ? (attributes.strike !== 'none') : null; },
                    set: function (state) { view.setCharacterAttribute('strike', state ? 'single' : 'none'); }
                },

                'character/color': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.color; },
                    set: function (color) { view.setCharacterAttribute('color', color); }
                },

                'character/fontname': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.fontName; },
                    set: function (fontName) { view.setCharacterAttribute('fontName', fontName); }
                },

                'character/fontsize': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.fontSize; },
                    set: function (fontSize) { view.setCharacterAttribute('fontSize', fontSize); }
                }

            };

        // base constructor ---------------------------------------------------

        EditController.call(this, app);

        // initialization -----------------------------------------------------

        // register item definitions
        this.registerDefinitions(items);

        // initialization after construction
        app.on('docs:init', function () {

            // model and view are not available at construction time
            model = app.getModel();
            view = app.getView();

            // update GUI after changed selection, and after receiving view updates
            view.on('change:selection change:layoutdata', function () { self.update(); });
        });

    } // class SpreadsheetController

    // exports ================================================================

    // derive this class from class EditController
    return EditController.extend({ constructor: SpreadsheetController });

});
