/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/drawingframe',
    ['io.ox/office/tk/utils',
     'gettext!io.ox/office/drawing'
    ], function (Utils, gt) {

    'use strict';

    var // the CSS class name of the drawing content node
        CONTENT_CLASS = 'content',

        // the CSS class name of the selection root node
        SELECTION_CLASS = 'selection',

        // maps drawing object types to GUI names and icons
        OBJECT_TYPE_SETTINGS = {
            chart:    { type: gt('Chart'),   icon: 'icon-bar-chart' },
            image:    { type: gt('Image'),   icon: 'icon-picture' },
            shape:    { type: gt('Shape'),   icon: 'icon-picture' },
            diagram:  { type: gt('Diagram'), icon: 'icon-sitemap' }
        },

        DEFAULT_TYPE_SETTINGS = { type: gt('Drawing'), icon: 'icon-picture' };

    // private global functions ===============================================

    /**
     * Creates an image node in the passed content node with the specified
     * source URL.
     */
    function createImageNodeFromUrl(contentNode, srcUrl) {
        contentNode.empty().append($('<img>', { src: srcUrl }));
    }

    /**
     * Creates an image node or an SVG image in the passed content node,
     * according to the passed image data.
     */
    function createImageNodeFromData(contentNode, imageData) {
        if (/^data:/.test(imageData)) {
            createImageNodeFromUrl(contentNode, imageData);
        } else if (/^<svg/.test(imageData)) {
            contentNode[0].innerHTML = imageData;
        } else {
            Utils.warn('DrawingFrame.createImageNodeFromData() - unsupported source data for the image object');
        }
    }

    /**
     * Calculates the offset and size of the bitmap in an image frame for one
     * dimension (either horizontally or vertically), according to the passed
     * cropping settings.
     *
     * @param {Number} frameSize
     *  Visible size (with or height) of the drawing frame containing the image
     *  bitmap, in 1/100 of millimeters.
     *
     * @param {Number} leadingCrop
     *  The leading cropping value (left/top), in percent.
     *
     * @param {Number} trailingCrop
     *  The trailing cropping value (right/bottom), in percent.
     *
     * @returns {Object}
     *  An object containing 'offset' and 'size' CSS attributes specifying how
     *  to format the bitmap node (in pixels with 'px' unit name).
     */
    function calculateBitmapSettings(frameSize, leadingCrop, trailingCrop) {

        var // sum of leading and trailing cropping (must not exceed a specific amount)
            totalCrop = leadingCrop + trailingCrop,
            // resulting settings for the bitmap
            size = 0, offset = 0;

        // do not crop away more than 90% of the bitmap
        if (totalCrop > 90) {
            leadingCrop *= (90 / totalCrop);
            trailingCrop *= (90 / totalCrop);
        }

        // bitmap size and offset, according to object size and cropping
        size = frameSize * 100 / (100 - leadingCrop - trailingCrop);
        offset = (size * leadingCrop) / 100;

        // convert to CSS pixels
        return {
            offset: Utils.convertHmmToCssLength(-offset, 'px', 1),
            size: Utils.convertHmmToCssLength(size, 'px', 1)
        };
    }

    // static class DrawingFrame ==============================================

    /**
     * Contains common static helper functions to display and handle drawing
     * frames in any document.
     */
    var DrawingFrame = {};

    // constants --------------------------------------------------------------

    /**
     * The CSS class used to mark drawing frame nodes.
     *
     * @constant
     */
    DrawingFrame.NODE_CLASS = 'drawing';

    /**
     * A jQuery selector that matches nodes representing a drawing frame.
     *
     * @constant
     */
    DrawingFrame.NODE_SELECTOR = 'div.' + DrawingFrame.NODE_CLASS;

    // methods ----------------------------------------------------------------

    /**
     * Returns a new drawing frame node.
     *
     * @param {String} type
     *  The type of the drawing.
     *
     * @returns {jQuery}
     *  A new drawing frame with empty content node, as jQuery object.
     */
    DrawingFrame.createDrawingFrame = function (type) {
        var contentNode = $('<div>').addClass(CONTENT_CLASS);
        return $('<div>', { contenteditable: false }).addClass(DrawingFrame.NODE_CLASS).data('type', type).append(contentNode);
    };

    /**
     * Returns whether the passed node is the root node of a drawing frame.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is the root node of a drawing frame.
     */
    DrawingFrame.isDrawingFrame = function (node) {
        return $(node).is(DrawingFrame.NODE_SELECTOR);
    };

    /**
     * Returns the type of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {String}
     *  The type of the specified drawing frame, as specified while the frame
     *  has been created.
     */
    DrawingFrame.getDrawingType = function (drawingFrame) {
        return $(drawingFrame).first().data('type');
    };

    /**
     * Returns the root content node of the specified drawing frame containing
     * all type specific contents of the drawing.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The root content node of the specified drawing frame.
     */
    DrawingFrame.getContentNode = function (drawingFrame) {
        return $(drawingFrame).first().children('.' + CONTENT_CLASS);
    };

    /**
     * Inserts replacement layout nodes for unsupported drawing types. Inserts
     * the passed name and description of the drawing as text.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {String} [name]
     *  The name of the drawing frame.
     *
     * @param {String} [description]
     *  The description text for the drawing frame.
     */
    DrawingFrame.insertReplacementNodes = function (drawingFrame, name, description) {

        var // the type of the drawing frame
            type = DrawingFrame.getDrawingType(drawingFrame),
            typeSettings = OBJECT_TYPE_SETTINGS[type] || DEFAULT_TYPE_SETTINGS,
            // the content node
            contentNode = DrawingFrame.getContentNode(drawingFrame),
            // the inner width and height available in the content node
            innerWidth = Math.max(0, drawingFrame.outerWidth() - 2),
            innerHeight = Math.max(0, drawingFrame.outerHeight() - 2),
            // the vertical padding in the content node
            verticalPadding = Utils.minMax(Math.min(innerWidth, innerHeight) / 24, 1, 6),
            // the font size of the picture icon
            pictureIconSize = Utils.minMax(Math.min(innerWidth - 16, innerHeight - 28), 8, 72),
            // the base font size of the text
            fontSize = Utils.minMax(Math.min(innerWidth, innerHeight) / 4, 9, 13);

        // set border width at the content node, insert the picture icon
        contentNode
            .empty()
            .addClass('placeholder')
            .css({
                padding: Math.floor(verticalPadding) + 'px ' + Math.floor(verticalPadding * 2) + 'px',
                fontSize: fontSize + 'px'
            })
            .append(
                $('<div>')
                    .addClass('abs background-icon')
                    .css('line-height', innerHeight + 'px')
                    .append(Utils.createIcon(typeSettings.icon).css('font-size', pictureIconSize + 'px')),
                $('<p>').text(name || typeSettings.type)
            );

        // insert description if there is a reasonable amount of space available
        if ((innerWidth >= 20) && (innerHeight >= 20)) {
            contentNode.append($('<p>').text(description));
        }
    };

    // formatting -------------------------------------------------------------

    /**
     * Updates the CSS formatting of the passed drawing frame, according to the
     * passed generic formatting attributes.
     *
     * @param {EditApplication} app
     *  The application instance containing the drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame to be updated. If this object is a
     *  jQuery collection, uses the first DOM node it contains.
     *
     * @param {Object} mergedAttributes
     *  The drawing attribute set (a map of attribute maps as name/value pairs,
     *  keyed by the 'drawing' attribute family), containing the effective
     *  attribute values merged from style sheets and explicit attributes.
     */
    DrawingFrame.updateFormatting = function (app, drawingFrame, mergedAttributes) {

        var // the drawing attributes of the passed attribute set
            drawingAttributes = mergedAttributes.drawing,
            // value of the 'transform' CSS attribute, e.g. for flipped drawings
            transformations = '',
            // type of the drawing object: 'image', ...
            type = DrawingFrame.getDrawingType(drawingFrame),
            // the content node inside the drawing frame
            contentNode = DrawingFrame.getContentNode(drawingFrame),
            // the replacement data for unsupported drawings
            replacementData = Utils.getStringOption(drawingAttributes, 'replacementData', '');

        // updates all formatting for image objects
        function updateImageFormatting() {

            var // current width of the drawing frame, in 1/100 mm
                drawingWidth = Utils.convertLengthToHmm(drawingFrame.width(), 'px'),
                // current height of the drawing frame, in 1/100 mm
                drawingHeight = Utils.convertLengthToHmm(drawingFrame.height(), 'px'),
                // horizontal offset/size of cropped bitmaps, as CSS attributes
                horizontalSettings = null, verticalSettings = null,
                // IE requires to swap cropping settings if image is flipped
                swapHorCrop = _.browser.IE && drawingAttributes.fliph,
                swapVertCrop = _.browser.IE && drawingAttributes.flipv,
                // effective cropping values
                cropLeft = swapHorCrop ? drawingAttributes.cropRight : drawingAttributes.cropLeft,
                cropRight = swapHorCrop ? drawingAttributes.cropLeft : drawingAttributes.cropRight,
                cropTop = swapVertCrop ? drawingAttributes.cropBottom : drawingAttributes.cropTop,
                cropBottom = swapVertCrop ? drawingAttributes.cropTop : drawingAttributes.cropBottom;

            // create the image or SVG node (considered to be constant at runtime)
            if (contentNode.children().length === 0) {
                if (drawingAttributes.imageData.length > 0) {
                    createImageNodeFromData(contentNode, drawingAttributes.imageData);
                } else if (drawingAttributes.imageUrl.length > 0) {
                    // convert relative URL to an application-specific absolute URL
                    createImageNodeFromUrl(contentNode, /:\/\//.test(drawingAttributes.imageUrl) ? drawingAttributes.imageUrl : app.getFilterModuleUrl({ action: 'getfile', get_filename: drawingAttributes.imageUrl }));
                } else {
                    Utils.warn('DrawingFrame.updateFormatting() - missing source data for the image object');
                }
            }

            // set bitmap cropping
            if ((drawingWidth > 0) && (drawingHeight > 0)) {
                horizontalSettings = calculateBitmapSettings(drawingWidth, cropLeft, cropRight);
                verticalSettings = calculateBitmapSettings(drawingHeight, cropTop, cropBottom);
                contentNode.find('>img').css({
                    left: horizontalSettings.offset,
                    top: verticalSettings.offset,
                    width: horizontalSettings.size,
                    height: verticalSettings.size
                });
            }
        }

        // updates all formatting for unsupported objects
        function updateUnsupportedFormatting() {
            if (replacementData.length === 0) {
                // insert default replacement nodes, if no replacement data is available
                DrawingFrame.insertReplacementNodes(drawingFrame, drawingAttributes.name, drawingAttributes.description);
            } else if (!drawingFrame.data('has-replacement')) {
                // replacement data MUST NOT change at runtime
                createImageNodeFromData(contentNode, replacementData);
                drawingFrame.data('has-replacement', true);
            }
        }

        // set text attribute to support tooltip for the drawing
        Utils.setControlTooltip(drawingFrame, drawingAttributes.name);

        // vertically or horizontally flipped drawings
        if (drawingAttributes.flipH && drawingAttributes.flipV) {
            transformations = 'scale(-1)';
        } else if (drawingAttributes.flipH) {
            transformations = 'scaleX(-1)';
        } else if (drawingAttributes.flipV) {
            transformations = 'scaleY(-1)';
        }

        // always set the CSS attribute, to be able to remove existing transformations
        Utils.setCssAttributeWithPrefixes(contentNode, 'transform', transformations);

        // update attributes specific to the drawing type
        switch (type) {
        case 'image':
        case 'ole':
            updateImageFormatting();
            break;
        default:
            updateUnsupportedFormatting();
        }
    };

    // selection --------------------------------------------------------------

    /**
     * Returns the selection root node of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The selection root node of the specified drawing frame. Will be an
     *  empty collection, if the drawing frame is not selected currently.
     */
    DrawingFrame.getSelectionNode = function (drawingFrame) {
        return $(drawingFrame).first().children('.' + SELECTION_CLASS);
    };

    /**
     * Returns whether the specified drawing frame is currently selected.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether the specified drawing frame is selected.
     */
    DrawingFrame.isSelected = function (drawingFrame) {
        return $(drawingFrame).hasClass(Utils.SELECTED_CLASS);
    };

    /**
     * Creates or updates additional nodes displayed while a drawing frame is
     * selected.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {Object} [options]
     *  A map with options to control the appearance of the selection. The
     *  following options are supported:
     *  @param {Boolean} [options.movable=false]
     *      If set to true, mouse pointer will be changed to a movable pointer
     *      while the mouse hovers the drawing.
     *  @param {Boolean} [options.resizable=false]
     *      If set to true, mouse pointer will be changed to an appropriate
     *      resize pointer while the mouse hovers a resize handle.
     *
     * @returns {jQuery}
     *  The selection root node contained in the drawing frame, as jQuery
     *  object.
     */
    DrawingFrame.drawSelection = function (drawingFrame, options) {

        var // whether the drawing is movable
            movable = Utils.getBooleanOption(options, 'movable', false),
            // whether the drawing is resizable
            resizable = Utils.getBooleanOption(options, 'resizable', false),
            // the selection root node
            selectionNode = DrawingFrame.getSelectionNode(drawingFrame);

        // create new selection nodes if missing
        if (selectionNode.length === 0) {

            // create a new selection node
            selectionNode = $('<div>').addClass(SELECTION_CLASS);
            $(drawingFrame).first().addClass(Utils.SELECTED_CLASS).append(selectionNode);

            // create the border and tracker node
            selectionNode.append($('<div>').addClass('border'), $('<div>').addClass('tracker'));

            // create resizer handles
            _(['t', 'b', 'l', 'r', 'tl', 'tr', 'bl', 'br']).each(function (pos) {
                selectionNode.append($('<div>').addClass('handle').attr('data-pos', pos));
            });
        }

        // update state classes
        return selectionNode.toggleClass('movable', movable).toggleClass('resizable', resizable);
    };

    /**
     * Removes the selection node from the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {Object} [options]
     *  A map with options to control the appearance of the selection. The
     *  following options are supported:
     *  @param {Boolean} [options.movable=false]
     *      If set to true, mouse pointer will be changed to a movable pointer
     *      while the mouse hovers the drawing.
     *  @param {Boolean} [options.resizable=false]
     *      If set to true, mouse pointer will be changed to an appropriate
     *      resize pointer while the mouse hovers a resize handle.
     *
     * @returns {jQuery}
     *  The selection root node contained in the drawing frame, as jQuery
     *  object.
     */
    DrawingFrame.clearSelection = function (drawingFrame) {
        $(drawingFrame).first().removeClass(Utils.SELECTED_CLASS);
        DrawingFrame.getSelectionNode(drawingFrame).remove();
    };

    // exports ================================================================

    return DrawingFrame;

});
