/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/model/drawingcollection',
    ['io.ox/core/event',
     'io.ox/office/tk/utils',
     'io.ox/office/drawinglayer/model/drawingmodel',
     'io.ox/office/drawinglayer/model/drawingmodelfactory'
    ], function (Events, Utils, DrawingModel, DrawingModelFactory) {

    'use strict';

    // class DrawingCollection ================================================

    /**
     * A collection containing models of drawing objects.
     *
     * @constructor
     *
     * @extends Events
     *
     * @param {EditApplication} app
     *  The application instance containing this drawing collection.
     *
     * @param {Object} options
     *  A map with additional options. The following options are supported:
     *  @param {Function} options.clientResolveModelHandler
     *      A function that receives an arbitrary logical document position,
     *      and must return the model of a drawing object at that position, if
     *      available; otherwise null.
     *  @param {Function} options.clientRegisterModelHandler
     *      A function that receives a new drawing model that has just been
     *      inserted into this collection, and the logical document position of
     *      the new drawing model. The function has to perform client-side
     *      registration of the drawing model according to the document
     *      position, and returns whether registration of the drawing model at
     *      the passed logical position was successful.
     */
    function DrawingCollection(app, options) {

        var // self reference
            self = this,

            // factory for drawing model instances
            modelFactory = new DrawingModelFactory(app),

            // the client resolves logical document positions to models
            clientResolveModelHandler = Utils.getFunctionOption(options, 'clientResolveModelHandler', $.noop),

            // the client has to register new drawing models
            clientRegisterModelHandler = Utils.getFunctionOption(options, 'clientRegisterModelHandler', $.noop),

            // all drawing models, mapped by internal unique identifier
            drawingModels = {},

            // cached number of drawing models in this collection (faster than counting map properties)
            modelCount = 0;

        // base constructor ---------------------------------------------------

        Events.extend(this);

        // private methods ----------------------------------------------------

        /**
         * Returns the drawing model at the specified logical document position
         * contained in this drawing collection.
         */
        function getModel(position) {

            var // let client resolve the drawing model
                model = clientResolveModelHandler.call(self, position);

            // check that the model is part of this collection
            if (model && !(model.getUid() in drawingModels)) {
                Utils.warn('DrawingCollection.getModel() - unknown drawing model');
                return null;
            }

            return model;
        }

        /**
         * Tries to find the drawing collection embedded in a drawing object
         * inside this collection.
         *
         * @param {Number[]} position
         *  The logical document position of the target drawing model.
         *
         * @returns {Object|Null}
         *  A result object containing the following properties:
         *  - {DrawingCollection} collection
         *      The drawing collection of a drawing object inside this
         *      collection.
         *  - {Number[]} position
         *      The remaining logical document position relative to the drawing
         *      collection.
         * If no drawing object with an own embedded collection has been found,
         * the value null will be returned.
         */
        function findEmbeddedCollection(position) {

            var // an existing drawing model
                model = null,
                // the collection embedded in the drawing model
                collection = null;

            for (var length = 1; length < position.length; length += 1) {
                if ((model = getModel(position.slice(0, length)))) {
                    collection = model.getCollection();
                    return collection ? { collection: collection, position: position.slice(length) } : null;
                }
            }

            return null;
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the factory that provides a method to create new drawing
         * model instances intended to be inserted into this drawing
         * collection.
         *
         * @returns {DrawingModelFactory}
         *  The drawing model factory of this drawing collection.
         */
        this.getModelFactory = function () {
            return modelFactory;
        };

        /**
         * Returns the number of drawings inserted directly into this drawing
         * collection (excluding the drawing objects embedded in other drawing
         * objects).
         *
         * @returns {Number}
         *  The number of drawing objects contained in this collection.
         */
        this.getModelCount = function () {
            return modelCount;
        };

        /**
         * Tries to find the drawing model at the passed logical document
         * position.
         *
         * @param {Number[]} position
         *  The logical document position of the drawing model. May specify a
         *  position inside another drawing object.
         *
         * @param {Object} [options]
         *  A map with additional options. The following options are supported:
         *  @param {Boolean} [options.deep=false]
         *      If set to true, searches for drawing objects embedded in other
         *      drawing objects.
         *  @param {String} [options.type]
         *      If specified, the drawing model must be of the specified type.
         *      If a drawing model with another type exists at the specified
         *      document position, this method returns null.
         *
         * @returns {DrawingModel|Null}
         *  The drawing model at the specified logical document position; or
         *  null, if no drawing model has been found.
         */
        this.findModel = function (position, options) {

            var // whether to search deeply in embedded collections
                deep = Utils.getBooleanOption(options, 'deep', false),
                // whether to restrict to a specific type
                type = Utils.getStringOption(options, 'type'),
                // embedded collection addressed by the passed position
                embedInfo = null,
                // the drawing model
                model = null;

            // existing drawing object with embedded collection: search drawing model there
            if (deep && (embedInfo = findEmbeddedCollection(position))) {
                return embedInfo.collection.findModel(embedInfo.position, options);
            }

            // find the drawing model at the specified position directly
            model = getModel(position);

            // check the type of the drawing model if specified
            return (model && (!_.isString(type) || (type === model.getType()))) ? model : null;
        };

        /**
         * Invokes the passed iterator function for all drawing models in this
         * collection, in no specific order.
         *
         * @param {Function} iterator
         *  The iterator function called for all drawing models. Receives the
         *  current drawing model as first parameter. If the iterator returns
         *  the Utils.BREAK object, the iteration process will be stopped
         *  immediately.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateModels = function (iterator, options) {

            var // the calling context for the iterator function
                context = Utils.getOption(options, 'context');

            return _(drawingModels).any(function (model) {
                return iterator.call(context, model) === Utils.BREAK;
            }) ? Utils.BREAK : undefined;
        };

        /**
         * Creates and stores the model of a new drawing object based on the
         * passed settings.
         *
         * @param {Number[]} position
         *  The logical document position of the new drawing model. May specify
         *  a position inside another drawing object (if the array without the
         *  last element points to an existing drawing object). If that drawing
         *  model supports embedding drawing objects, the new drawing model
         *  will be inserted into the drawing collection of that drawing object
         *  instead into this collection.
         *
         * @param {String} type
         *  The type of the drawing object.
         *
         * @param {Object} [attributes]
         *  An attribute set with initial formatting attributes for the drawing
         *  object.
         *
         * @returns {Boolean}
         *  Whether the new drawing model has been created successfully.
         */
        this.insertModel = function (position, type, attributes) {

            var // embedded collection addressed by the passed position
                embedInfo = null,
                // the new drawing model
                model = null;

            // existing drawing object with embedded collection found: create new drawing model there
            if ((embedInfo = findEmbeddedCollection(position))) {
                return embedInfo.collection.insertModel(embedInfo.position, type, attributes);
            }

            // create and store the new drawing model
            model = modelFactory.createModel(type, attributes);
            if (!model) { return false; }
            drawingModels[model.getUid()] = model;
            modelCount += 1;

            // register the new drawing model at the client application
            if (!clientRegisterModelHandler.call(this, model, position)) { return false; }

            // notify all listeners
            this.trigger('insert:drawing', model, position);

            // listen to attribute changes of the drawing model
            model.on('change:attributes', function (event, newAttributes, oldAttributes) {
                self.trigger('change:drawing', model, newAttributes, oldAttributes);
            });

            return true;
        };

        /**
         * Removes the model of a drawing object located at the passed logical
         * document position.
         *
         * @param {Number[]} position
         *  The logical document position of the drawing model. May specify a
         *  position inside another drawing object (if the array without the
         *  last element points to an existing drawing object).
         *
         * @returns {Boolean}
         *  Whether the drawing model has been removed successfully.
         */
        this.deleteModel = function (position) {

            var // embedded collection addressed by the passed position
                embedInfo = null,
                // the drawing model
                model = null;

            // existing drawing object with embedded collection found: create new drawing model there
            if ((embedInfo = findEmbeddedCollection(position))) {
                return embedInfo.collection.deleteModel(embedInfo.position);
            }

            // find the drawing model at the specified logical position
            if (!(model = this.findModel(position))) { return false; }

            // notify all listeners
            this.trigger('delete:drawing', model, position);

            // remove and destroy the drawing model
            delete drawingModels[model.getUid()];
            model.destroy();
            modelCount -= 1;

            return true;
        };

        this.destroy = function () {
            _(drawingModels).invoke('destroy');
            this.events.destroy();
        };

        // initialization -----------------------------------------------------

        // register common drawing types
        modelFactory
            .registerModelType('shape', DrawingModel)
            .registerModelType('group', DrawingModel)
            .registerModelType('image', DrawingModel)
            .registerModelType('diagram', DrawingModel)
            .registerModelType('chart', DrawingModel)
            .registerModelType('ole', DrawingModel)
            .registerModelType('undefined', DrawingModel);

    } // class DrawingCollection

    // exports ================================================================

    return _.makeExtendable(DrawingCollection);

});
