/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Oliver Specht <oliver.specht@open-xchange.com>
 */

define('io.ox/office/text/format/lists',
    ['io.ox/office/tk/utils',
     'io.ox/office/framework/model/format/container',
     'io.ox/office/text/operations',
     'gettext!io.ox/office/text'
    ], function (Utils, Container, Operations, gt) {

    'use strict';

    var // the default bullet list definition with different bullets per list level
        DEFAULT_BULLET_LIST_DEFINITION = {
            listLevel0: { textAlign: 'left', indentLeft: 1270,     numberFormat: 'bullet', listStartValue: 1, fontName: 'Symbol', levelText: '\uf0b7', indentFirstLine: -635 },
            listLevel1: { textAlign: 'left', indentLeft: 2 * 1270, numberFormat: 'bullet', listStartValue: 1, fontName: 'Times New Roman', levelText: '\u25CB', indentFirstLine: -635 },
            listLevel2: { textAlign: 'left', indentLeft: 3 * 1270, numberFormat: 'bullet', listStartValue: 1, fontName: 'Times New Roman', levelText: '\u25A0', indentFirstLine: -635 },
            listLevel3: { textAlign: 'left', indentLeft: 4 * 1270, numberFormat: 'bullet', listStartValue: 1, fontName: 'Symbol', levelText: '\uf0b7', indentFirstLine: -635 },
            listLevel4: { textAlign: 'left', indentLeft: 5 * 1270, numberFormat: 'bullet', listStartValue: 1, fontName: 'Times New Roman', levelText: '\u25CB', indentFirstLine: -635 },
            listLevel5: { textAlign: 'left', indentLeft: 6 * 1270, numberFormat: 'bullet', listStartValue: 1, fontName: 'Times New Roman', levelText: '\u25A0', indentFirstLine: -635 },
            listLevel6: { textAlign: 'left', indentLeft: 7 * 1270, numberFormat: 'bullet', listStartValue: 1, fontName: 'Symbol', levelText: '\uf0b7', indentFirstLine: -635 },
            listLevel7: { textAlign: 'left', indentLeft: 8 * 1270, numberFormat: 'bullet', listStartValue: 1, fontName: 'Times New Roman', levelText: '\u25CB', indentFirstLine: -635 },
            listLevel8: { textAlign: 'left', indentLeft: 9 * 1270, numberFormat: 'bullet', listStartValue: 1, fontName: 'Times New Roman', levelText: '\u25A0', indentFirstLine: -635 }
        },

        // the default numbered list definition with different number formats per list level (as used in MSO2010)
        DEFAULT_NUMBERING_LIST_DEFINITION = {
            listLevel0: { numberFormat: 'decimal',     listStartValue: 1, indentLeft: 1270,     indentFirstLine: -635, textAlign: 'left',  levelText: '%1.'},
            listLevel1: { numberFormat: 'lowerLetter', listStartValue: 1, indentLeft: 2 * 1270, indentFirstLine: -635, textAlign: 'left',  levelText: '%2.'},
            listLevel2: { numberFormat: 'upperLetter', listStartValue: 1, indentLeft: 3 * 1270, indentFirstLine: -635, textAlign: 'right', levelText: '%3.'},
            listLevel3: { numberFormat: 'lowerRoman',  listStartValue: 1, indentLeft: 4 * 1270, indentFirstLine: -635, textAlign: 'left',  levelText: '%4.'},
            listLevel4: { numberFormat: 'upperRoman',  listStartValue: 1, indentLeft: 5 * 1270, indentFirstLine: -635, textAlign: 'left',  levelText: '%5.'},
            listLevel5: { numberFormat: 'decimal',     listStartValue: 1, indentLeft: 6 * 1270, indentFirstLine: -635, textAlign: 'right', levelText: '%6.'},
            listLevel6: { numberFormat: 'lowerLetter', listStartValue: 1, indentLeft: 7 * 1270, indentFirstLine: -635, textAlign: 'left',  levelText: '%7.'},
            listLevel7: { numberFormat: 'upperLetter', listStartValue: 1, indentLeft: 8 * 1270, indentFirstLine: -635, textAlign: 'left',  levelText: '%8.'},
            listLevel8: { numberFormat: 'lowerRoman',  listStartValue: 1, indentLeft: 9 * 1270, indentFirstLine: -635, textAlign: 'right', levelText: '%9.'}
        },

        // all predefined list styles for the GUI controls, mapped by list identifier
        PREDEFINED_LISTSTYLES = (function () {

            /**
             * Creates and returns a style definition for a bullet list with
             * the same bullet character in all list levels.
             *
             * @param {Function} levelCallback
             *  A function that generates a style definition for a single list
             *  level. Receives the list level in the first parameter.
             *
             * @param {String} listKey
             *  A unique key for the list style.
             *
             * @param {String} listLabel
             *  The button label for the list style used in GUI from controls.
             *
             * @param {String} tooltip
             *  A tool tip for the list style used in GUI from controls.
             *
             * @returns {Object}
             *  An object with an attribute 'definition' and the string
             *  attributes 'listKey', 'listLabel', and 'tooltip'. The list
             *  definition object containing the attributes 'listLevel0' to
             *  'listLevel8' referring to list style definitions with an
             *  indentation according to the respective list level.
             */
            function createListStyle(levelCallback, listKey, listLabel, tooltip) {
                var definition = {};
                _(9).times(function (level) {
                    definition['listLevel' + level] = levelCallback(level);
                });
                return { definition: definition, listKey: listKey, listLabel: listLabel, tooltip: tooltip };
            }

            /**
             * Creates and returns a style definition for a bullet list with
             * the same bullet character in all list levels.
             *
             * @param {String} bulletKey
             *  A unique key for the bullet list style.
             *
             * @param {String} bulletCharacter
             *  The bullet character for all list levels.
             *
             * @param {String} tooltip
             *  A tool tip for the list style used in GUI from controls.
             *
             * @returns {Object}
             *  An object with an attribute 'definition' and the string
             *  attributes 'listLabel' and 'tooltip'. The list definition
             *  object containing the attributes 'listLevel0' to 'listLevel8'
             *  referring to list style definitions with an indentation
             *  according to the respective list level.
             */
            function createBulletListStyle(bulletKey, bulletCharacter, tooltip) {
                return createListStyle(function (level) {
                    return {
                        numberFormat: 'bullet',
                        listStartValue: 1,
                        indentLeft: (level + 1) * 1270,
                        indentFirstLine: -635,
                        textAlign: 'left',
                        levelText: bulletCharacter,
                        fontName: 'Times New Roman'
                    };
                }, bulletKey, bulletCharacter, tooltip);
            }

            /**
             * Creates and returns a style definition for a numbered list with
             * the same numbering style in all list levels.
             *
             * @param {String} numberFormat
             *  One of the predefined number formats 'decimal', 'lowerLetter',
             *  'upperLetter', 'lowerRoman', or 'upperRoman'.
             *
             * @param {String} textBefore
             *  Fixed text inserted before each numeric list label.
             *
             * @param {String} textAfter
             *  Fixed text inserted after each numeric list label.
             *
             * @param {String} tooltip
             *  A tool tip for the list style used in GUI from controls.
             *
             * @returns {Object}
             *  An object with an attribute 'definition' and the string
             *  attributes 'listLabel' and 'tooltip'. The list definition
             *  object containing the attributes 'listLevel0' to 'listLevel8'
             *  referring to list style definitions with an indentation
             *  according to the respective list level.
             */
            function createNumberedListStyle(numberFormat, textBefore, textAfter, tooltip) {
                return createListStyle(function (level) {
                    return {
                        numberFormat: numberFormat,
                        listStartValue: 1,
                        indentLeft: (level + 1) * 1270,
                        indentFirstLine: -635,
                        textAlign: 'left',
                        levelText: textBefore + '%' + (level + 1) + textAfter
                    };
                }, textBefore + numberFormat + textAfter, textBefore + formatNumber(1, numberFormat) + textAfter, tooltip);
            }

            // generate and return the array with all predefined list styles
            return {
                L20000: createBulletListStyle('filled-circle',       '\u25CF', /*#. symbol in bullet lists (filled) */ gt('Filled circle')),
                L20001: createBulletListStyle('small-filled-circle', '\u2022', /*#. symbol in bullet lists (filled) */ gt('Small filled circle')),
                L20002: createBulletListStyle('circle',              '\u25CB', /*#. symbol in bullet lists (not filled) */ gt('Circle')),
                L20003: createBulletListStyle('small-circle',        '\u25E6', /*#. symbol in bullet lists (not filled) */ gt('Small circle')),
                L20004: createBulletListStyle('filled-square',       '\u25A0', /*#. symbol in bullet lists (filled) */ gt('Filled square')),
                L20005: createBulletListStyle('small-filled-square', '\u25AA', /*#. symbol in bullet lists (filled) */ gt('Small filled square')),
                L20006: createBulletListStyle('square',              '\u25A1', /*#. symbol in bullet lists (not filled) */ gt('Square')),
                L20007: createBulletListStyle('small-square',        '\u25AB', /*#. symbol in bullet lists (not filled) */ gt('Small square')),
                L20008: createBulletListStyle('filled-diamond',      '\u2666', /*#. symbol in bullet lists (filled) */ gt('Filled diamond')),
                L20009: createBulletListStyle('diamond',             '\u25CA', /*#. symbol in bullet lists (not filled) */ gt('Diamond')),
                L20010: createBulletListStyle('filled-triangle',     '\u25BA', /*#. symbol in bullet lists (filled) */ gt('Filled triangle')),
                L20011: createBulletListStyle('chevron',             '>',      /*#. symbol in bullet lists */ gt('Chevron')),
                L20012: createBulletListStyle('arrow',               '\u2192', /*#. symbol in bullet lists */ gt('Arrow')),
                L20013: createBulletListStyle('dash',                '\u2013', /*#. symbol in bullet lists */ gt('Dash')),
                L20014: createBulletListStyle('none',                ' ',      /*#. bullet lists without symbols */ gt('No symbol')),

                L30000: createNumberedListStyle('decimal',     '',  '',  /*#. label style in numbered lists */ gt('Decimal numbers')),
                L30001: createNumberedListStyle('decimal',     '',  '.', /*#. label style in numbered lists */ gt('Decimal numbers')),
                L30002: createNumberedListStyle('decimal',     '',  ')', /*#. label style in numbered lists */ gt('Decimal numbers')),
                L30003: createNumberedListStyle('decimal',     '(', ')', /*#. label style in numbered lists */ gt('Decimal numbers')),
                L30010: createNumberedListStyle('lowerLetter', '',  '',  /*#. label style in numbered lists */ gt('Small Latin letters')),
                L30011: createNumberedListStyle('lowerLetter', '',  '.', /*#. label style in numbered lists */ gt('Small Latin letters')),
                L30012: createNumberedListStyle('lowerLetter', '',  ')', /*#. label style in numbered lists */ gt('Small Latin letters')),
                L30013: createNumberedListStyle('lowerLetter', '(', ')', /*#. label style in numbered lists */ gt('Small Latin letters')),
                L30020: createNumberedListStyle('upperLetter', '',  '',  /*#. label style in numbered lists */ gt('Capital Latin letters')),
                L30021: createNumberedListStyle('upperLetter', '',  '.', /*#. label style in numbered lists */ gt('Capital Latin letters')),
                L30022: createNumberedListStyle('upperLetter', '',  ')', /*#. label style in numbered lists */ gt('Capital Latin letters')),
                L30023: createNumberedListStyle('upperLetter', '(', ')', /*#. label style in numbered lists */ gt('Capital Latin letters')),
                L30030: createNumberedListStyle('lowerRoman',  '',  '',  /*#. label style in numbered lists */ gt('Small Roman numbers')),
                L30031: createNumberedListStyle('lowerRoman',  '',  '.', /*#. label style in numbered lists */ gt('Small Roman numbers')),
                L30032: createNumberedListStyle('lowerRoman',  '',  ')', /*#. label style in numbered lists */ gt('Small Roman numbers')),
                L30033: createNumberedListStyle('lowerRoman',  '(', ')', /*#. label style in numbered lists */ gt('Small Roman numbers')),
                L30040: createNumberedListStyle('upperRoman',  '',  '',  /*#. label style in numbered lists */ gt('Capital Roman numbers')),
                L30041: createNumberedListStyle('upperRoman',  '',  '.', /*#. label style in numbered lists */ gt('Capital Roman numbers')),
                L30042: createNumberedListStyle('upperRoman',  '',  ')', /*#. label style in numbered lists */ gt('Capital Roman numbers')),
                L30043: createNumberedListStyle('upperRoman',  '(', ')', /*#. label style in numbered lists */ gt('Capital Roman numbers'))
            };

        }()); // end of PREDEFINED_LISTSTYLES local scope

    // static private function ================================================

    function isLevelEqual(defaultLevel, compareLevel) {
        var ret = defaultLevel !== undefined && compareLevel !== undefined &&
        defaultLevel.numberFormat === compareLevel.numberFormat &&
        defaultLevel.indentLeft === compareLevel.indentLeft &&
        defaultLevel.indentFirstLine === compareLevel.indentFirstLine &&
        defaultLevel.textAlign === compareLevel.textAlign &&
        defaultLevel.levelText === compareLevel.levelText &&
        defaultLevel.fontName === compareLevel.fontName &&
        defaultLevel.listStartValue === compareLevel.listStartValue;
        return ret;
    }

    function isDefinitionEqual(defaultDefinition, compareDefinition) {
        return isLevelEqual(defaultDefinition.listLevel0, compareDefinition.listLevel0) &&
            isLevelEqual(defaultDefinition.listLevel1, compareDefinition.listLevel1) &&
            isLevelEqual(defaultDefinition.listLevel2, compareDefinition.listLevel2) &&
            isLevelEqual(defaultDefinition.listLevel3, compareDefinition.listLevel3) &&
            isLevelEqual(defaultDefinition.listLevel4, compareDefinition.listLevel4) &&
            isLevelEqual(defaultDefinition.listLevel5, compareDefinition.listLevel5) &&
            isLevelEqual(defaultDefinition.listLevel6, compareDefinition.listLevel6) &&
            isLevelEqual(defaultDefinition.listLevel7, compareDefinition.listLevel7) &&
            isLevelEqual(defaultDefinition.listLevel8, compareDefinition.listLevel8);
    }

    function isArrayDefinitionEqual(arrayDefinition, compareDefinition) {
        return isLevelEqual(arrayDefinition.listLevels[0], compareDefinition.listLevel0) &&
            isLevelEqual(arrayDefinition.listLevels[1], compareDefinition.listLevel1) &&
            isLevelEqual(arrayDefinition.listLevels[2], compareDefinition.listLevel2) &&
            isLevelEqual(arrayDefinition.listLevels[3], compareDefinition.listLevel3) &&
            isLevelEqual(arrayDefinition.listLevels[4], compareDefinition.listLevel4) &&
            isLevelEqual(arrayDefinition.listLevels[5], compareDefinition.listLevel5) &&
            isLevelEqual(arrayDefinition.listLevels[6], compareDefinition.listLevel6) &&
            isLevelEqual(arrayDefinition.listLevels[7], compareDefinition.listLevel7) &&
            isLevelEqual(arrayDefinition.listLevels[8], compareDefinition.listLevel8);
    }
    /**
     * Converts the passed value to upper-case Latin letters.
     *
     * @param {Number} value
     *  One-based number to be converted.
     *
     * @returns {String}
     *  The upper-case representation of the passed value in Latin letters.
     *  Counts from 'A', 'B', 'C', ..., 'Z', 'AA', 'BB', 'CC', and so on.
     */
    function convertToLetters(value) {

        var letter = String.fromCharCode(65 + ((value - 1) % 26)),
            count = Math.floor((value - 1) / 26) + 1;

        return Utils.repeatString(letter, count);
    }

    /**
     * Converts the passed value to upper-case Roman numbers.
     *
     * @param {Number} value
     *  One-based number to be converted.
     *
     * @returns {String}
     *  The upper-case Roman number representation of the passed value.
     */
    function convertToRoman(value) {

        var result = '',
            romanDigitsArr = ['M', 'D', 'C', 'L', 'X', 'V', 'I'],
            romanValArr = [1000, 500, 100,  50,  10,   5,   1];

        if (value > 0) {
            var index = 0;
            for (;index < 7; index++) {
                while (value >= romanValArr[index]) {
                    result += romanDigitsArr[index];
                    value -= romanValArr[index];
                }
                var position = 7;
                for (; position > index; position--) {
                    var tempVal = romanValArr[index] - romanValArr[position];
                    if ((romanValArr[position] < tempVal) && (tempVal <= value)) {
                        result += romanDigitsArr[position] + romanDigitsArr[index];
                        value -= tempVal;
                    }
                }
            }
        }
        return result;
    }

    function parseRoman(text) {
        var romanSmallArr = ['m', 'd', 'c', 'l', 'x', 'v', 'i'],
        romanValArr = [1000, 500, 100,  50,  10,   5,   1],
        ret = {},
        lowerText = text.toLowerCase(),
        startValue = 0;
        ret.caps = lowerText !== text;
        var index = 0, lastValue = 1000;
        for (; index < text.length; ++index) {
            var char = lowerText.charAt(index);
            if (char === '.')
                break;
            var position = $.inArray(char, romanSmallArr);
            if (position < 0)
                return {};
            var value = romanValArr[position];
            if (lastValue < value) {
                startValue = startValue - lastValue + (value - lastValue);
            } else {
                startValue += value;
            }
            lastValue = value;
        }
        if (startValue > 0) {
            ret.startnumber = startValue;
            ret.numberFormat = lowerText !== text ? 'upperRoman' : 'lowerRoman';
        }
        return ret;
    }

    function formatNumber(value, format) {
        switch (format.toLowerCase()) {
        case "decimal":
            return value.toString();
        case "lowerletter":
            return convertToLetters(value).toLowerCase();
        case "upperletter":
            return convertToLetters(value);
        case "lowerroman":
            return convertToRoman(value).toLowerCase();
        case "upperroman":
            return convertToRoman(value);
        }
        Utils.error('Lists.formatNumber() - unknown number format: ' + format);
        return '';
    }

    // class Lists ============================================================

    /**
     * Contains the definitions of lists.
     *
     * @constructor
     *
     * @extends Container
     *
     * @param {EditApplication} app
     *  The root application instance.
     */
    function Lists(app, documentStyles) {

        var // list definitions
            lists = [],

            // listIds used in insertList operation
            insertedListIds = {},

            // defaults
            defaultNumberingNumId,

            defaultBulletNumId,

            characterStyles = documentStyles.getStyleSheets('character');

        // base constructor ---------------------------------------------------

        Container.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Creates the label text of the current numbered paragraph.
         * For picture numbered bullet list it returns the URI of the picture
         *
         * @param levelIndexes array of indexes of all numbering levels
         * @param ilvl current indentation level
         * @param listDefinition properties of the list used
         * @returns {Object}
         *  text element contains label text
         *  imgsrc element contains URI of the picture used
         *
         *  In case imgsrc is set then text will be empty
         */
        function formatNumberType(levelIndexes, ilvl, listDefinition, listParagraphIndex) {
            var ret = {},
                seqNo = 0,
                end = listDefinition.listUnifiedNumbering === true ? ilvl: 0;
            var topLevelformat = listDefinition.listLevels[ilvl];
            var levelText = topLevelformat.levelText ? topLevelformat.levelText : '';
            ret.labelFollowedBy = listDefinition.listLevels[ilvl].labelFollowedBy;
            ret.tabpos = listDefinition.listLevels[ilvl].tabpos;

            for (;ilvl >= end; --ilvl) {
                var levelFormat = listDefinition.listLevels[ilvl];
                if (listDefinition.listUnifiedNumbering === true) {
                    seqNo += listParagraphIndex;
                }
                else {
                    seqNo = levelIndexes === undefined ? 0 :
                        levelIndexes[ilvl] + (levelFormat && levelFormat.listStartValue !== undefined ? levelFormat.listStartValue - 1 : 0);
                }
                var levelToken = '%' + (ilvl + 1);
                var indexpos = levelText.indexOf(levelToken);
                if (indexpos < 0 && levelFormat.numberFormat !== 'bullet')
                    continue;
                var replacetext = '';
                switch (levelFormat.numberFormat) {
                case "bullet":
                    if (levelFormat.levelPicBulletUri) {
                        ret.imgsrc = levelFormat.levelPicBulletUri;
                        ret.imgwidth = levelFormat.width;
                        replacetext = '';
                    }
                    else {
                        var charCode = levelFormat.levelText ? levelFormat.levelText.charCodeAt(0) : -1;
                        if (charCode > 0 && (charCode < 0xE000 || charCode > 0xF8FF)) {
                            replacetext = levelFormat.levelText;
                        }
                        else
                            replacetext = charCode === 0xf0b7 ? "\u2022" : "\u25cf";
                    }
                    break;
                case "none":
                    replacetext = '';
                    break;
                default:
                    replacetext = formatNumber(seqNo, levelFormat.numberFormat);
                }
                ret.color = levelFormat.color;
                if (!ret.color && levelFormat.styleId)
                    ret.color = characterStyles.getStyleSheetAttributes(levelFormat.styleId).character.color;

                if (levelFormat.levelPicBulletUri) {
                    levelText = '';
                    break;
                }
                else if (levelFormat.numberFormat === 'bullet') {
                    levelText = replacetext;
                    break;
                }
                else
                    levelText = levelText.replace(levelToken, replacetext);
            }
            ret.text = levelText;
            return ret;
        }

        // methods ------------------------------------------------------------

        /**
         * Adds a new list to this container. An existing list definition
         * with the specified identifier will be replaced.
         *
         * @param {String} listIdentifier
         *  The name of of the new list definition.
         *
         * @param {Object} listDefinition
         *  The attributes of the list definition.
         *
         * @returns {Lists}
         *  A reference to this instance.
         */
        this.addList = function (operation) {

            var newPosition = lists.length;
            lists[newPosition] = {};
            var list = lists[newPosition];
            list.listIdentifier = operation.listStyleId;
            list.listUnifiedNumbering = operation.listUnifiedNumbering;
            //list.listStyleId = listStyleId;
            list.listLevels = [];
            if (operation.listDefinition) {
                list.listLevels[0] = operation.listDefinition.listLevel0;
                list.listLevels[1] = operation.listDefinition.listLevel1;
                list.listLevels[2] = operation.listDefinition.listLevel2;
                list.listLevels[3] = operation.listDefinition.listLevel3;
                list.listLevels[4] = operation.listDefinition.listLevel4;
                list.listLevels[5] = operation.listDefinition.listLevel5;
                list.listLevels[6] = operation.listDefinition.listLevel6;
                list.listLevels[7] = operation.listDefinition.listLevel7;
                list.listLevels[8] = operation.listDefinition.listLevel8;
                if (operation.listDefinition.defaultlist) {
                    if (operation.listDefinition.defaultlist === 'bullet')
                        defaultBulletNumId = operation.listStyleId;
                    else
                        defaultNumberingNumId = operation.listStyleId;
                } else {
                    if (defaultBulletNumId === undefined) {
                        if (isDefinitionEqual(DEFAULT_BULLET_LIST_DEFINITION, operation.listDefinition) === true)
                            defaultBulletNumId = operation.listStyleId;
                    }
                    if (defaultNumberingNumId === undefined) {
                        if (isDefinitionEqual(DEFAULT_NUMBERING_LIST_DEFINITION, operation.listDefinition) === true)
                            defaultNumberingNumId = operation.listStyleId;
                    }
                }
            }
            // notify listeners
            this.triggerChangeEvent('add', operation.listStyleId);

            return this;
        };

        /**
         * Remove an existing list
         *
         * @param {String} listIdentifier
         *  The name of of the list to be removed.
         *
         * @returns {Lists}
         *  A reference to this instance.
         */
        this.deleteList = function (listIdentifier) {
            var index = 0;
            for (; index < lists.length; ++index) {
                if (lists[index].listIdentifier === listIdentifier) {
                    lists.splice(index, 1);
                    if (defaultNumberingNumId === listIdentifier)
                        defaultNumberingNumId = undefined;
                    if (defaultBulletNumId === listIdentifier)
                        defaultBulletNumId = undefined;
                    break;
                }
            }
            return this;
        };

        /**
         * Gives access to a single list definition.
         *
         * @param name the name of the list to return.
         * @returns {Lists}
         *  A reference to this instance.
         */
        this.getList = function (name) {
            var index = 0,
                ret;
            for (; index < lists.length; ++index) {
                if (lists[index].listIdentifier === name) {
                    ret = lists[index];
                    break;
                }
            }
            return ret;
        };

        /**
         * Gives access to all list definitions.
         */
        this.getLists = function () {
            return lists;
        };

        /**
         * @param {String} type
         *  either bullet or numbering
         * @returns {integer}
         *  the Id of a default bullet or numbered numbering. If this default numbering definition is not available then it will be created
         */
        this.getDefaultNumId = function (type) {
            return type === 'bullet' ? defaultBulletNumId : defaultNumberingNumId;
        };

        /**
         * creates a list id not used, yet
         * @returns {String}
         *  new list id
         */
        this.getFreeListId = function () {
            var freeId = (lists.length + 1),
                sFreeId = 'L' + freeId;

            function hasListId(list, listId) {
                var ret = false;
                _.each(list, function (element) {
                    if (element.listIdentifier === listId)
                        ret = true;
                });
                return ret;
            }
            
            while (hasListId(lists, sFreeId)) {
                ++freeId;
                sFreeId = 'L' + freeId;
            }
            return sFreeId;
        };

        /**
         * @param {String} type
         *  either bullet or numbering
         * @param {Object} options
         *  can contain symbol - the bullet symbol
         *              listStartValue - start index of an ordered list
         * @returns {Object}
         *  the operation that creates the requested list
         *
         */
        this.getDefaultListOperation = function (type, options) {
            var sFreeId = this.getFreeListId();

            var newOperation = { name: Operations.INSERT_LIST, listStyleId: sFreeId };
            if (type === 'bullet') {
                newOperation.listDefinition = _.copy(DEFAULT_BULLET_LIST_DEFINITION, true);
                if (options && options.symbol && options.symbol !== '*') {
                    newOperation.listDefinition.listLevel0.levelText = options.symbol;
                } else {
                    newOperation.listDefinition.defaultlist = type;
                }
            } else {
                newOperation.listDefinition = _.copy(DEFAULT_NUMBERING_LIST_DEFINITION, true);
                var defaultlist = true;
                if (options) {
                    if (options.listStartValue) {
                        newOperation.listDefinition.listLevel0.listStartValue = options.listStartValue;
                        defaultlist = false;
                    }
                    if (options.numberFormat) {
                        newOperation.listDefinition.listLevel0.numberFormat = options.numberFormat;
                        defaultlist = false;
                    }
                }
                if (defaultlist) {
                    newOperation.listDefinition.defaultlist = type;
                }
            }
            return newOperation;
        };

        /**
         * @param {String} listId
         *  selected list style id
         * @param {Object} options
         *  can contain symbol - the bullet symbol
         *              listStartValue - start index of an ordered list
         * @returns {Object}
         *  the operation that creates the requested list
         *
         */
        this.getSelectedListStyleOperation = function (listId, options) {

            var newOperation = { name: Operations.INSERT_LIST, listStyleId: listId };

            if ((listId in PREDEFINED_LISTSTYLES) && !(listId in insertedListIds)) {
                newOperation.listDefinition = _.copy(PREDEFINED_LISTSTYLES[listId].definition, true);
                insertedListIds[listId] = true;  // avoiding inserting list more than once
            } else {
                newOperation = null;
            }

            return newOperation;
        };

        /**
         * Merge a level of a list definition into an existing listStyle and create an insertList operation
         * (if no such list exists) and return that operation and it's listId
         *
         * @param {String} listId
         *  selected list style id
         * @param {Object} options
         *  can contain symbol - the bullet symbol
         *              listStartValue - start index of an ordered list
         * @returns {Object}
         *  containing the operation that creates the requested list (optional)
         *  and the id of the merged list
         *
         */
        this.mergeListStyle = function (existingListStyle, modifyingStyle, level) {
            var operationAndStyle = {listOperation: null, listStyleId: null},
                sourceLevel = null,
                sourceListDefinition;
                //newOperation = { name: Operations.INSERT_LIST, listStyleId: listId };
            if ((modifyingStyle in PREDEFINED_LISTSTYLES)) {
                operationAndStyle.listOperation = { name: Operations.INSERT_LIST, listStyleId: this.getFreeListId() };
                operationAndStyle.listOperation.listDefinition = _.copy(this.getList(existingListStyle), true);
                sourceListDefinition = PREDEFINED_LISTSTYLES[modifyingStyle].definition;
                _(9).times(function (l) {
                    sourceLevel = l === level ? sourceListDefinition['listLevel' + l] : operationAndStyle.listOperation.listDefinition.listLevels[l];
                    operationAndStyle.listOperation.listDefinition['listLevel' + l] = _.copy(sourceLevel, true);

                });

                //search for the new list definition in the existing list styles to prevent creation of identical styles
                operationAndStyle.listStyleId = operationAndStyle.listOperation.listStyleId;
                _.each(lists, function (element, key, list) {
                    if (operationAndStyle.listOperation && isArrayDefinitionEqual(element, operationAndStyle.listOperation.listDefinition)) {
                        operationAndStyle.listStyleId = list[key].listIdentifier;
                        operationAndStyle.listOperation = null;
                    }
                });



            } else {
                operationAndStyle.listStyleId = existingListStyle;
            }
            return operationAndStyle;
        };

        /**
         * Returns whether the specified list is a bullet list.
         *
         * @param {String} listStyleId
         *  The identifier of a list.
         *
         * @param {Number} listLevel
         *  the indent level of the list in the paragraph.
         *
         * @returns {Boolean}
         *  Whether the specified list identifier points to a bullet list.
         */
        this.isBulletsList = function (listStyleId, listLevel) {

            var currentList = this.getList(listStyleId),
                listLevelDef = currentList ? currentList.listLevels[listLevel] : null;

            return listLevelDef && ((listLevelDef.numberFormat === 'bullet') || (listLevelDef.numberFormat === 'none'));
        };

        /**
         * Returns whether the specified list is a numbered list.
         *
         * @param {String} listStyleId
         *  The identifier of a list.
         *
         * @param {Number} listLevel
         *  the indent level of the list in the paragraph.
         *
         * @returns {Boolean}
         *  Whether the specified list identifier points to a numbered list.
         */
        this.isNumberingList = function (listStyleId, listLevel) {

            var currentList = this.getList(listStyleId),
                listLevelDef = currentList ? currentList.listLevels[listLevel] : null;

            return listLevelDef && (listLevelDef.numberFormat !== 'bullet') && (listLevelDef.numberFormat !== 'none');
        };

        /**
         * Generates the numbering Label for the given paragraph
         *
         * @param listId identifier of the applied numbering definition
         * @param ilvl indent level, zero based
         * @param levelIndexes array of sequential position of the current paragraph
         *      contains an array with ilvl + 1 elements that determines the sequential position of the current paragraph within the numbering
         *
         * @returns {Object} containing:
         *          indent
         *          labelwidth
         *          text
         *          tbd.
         */
        this.formatNumber = function (listStyleId, ilvl, levelIndexes, listParagraphIndex) {
            var ret = {};
            var currentList = this.getList(listStyleId);
            if (currentList === undefined) {
                return "?";
            }
            var levelFormat = currentList.listLevels[ilvl];
            if (levelFormat === undefined) {
                return "??";
            }
            var format = formatNumberType(levelIndexes, ilvl, currentList, listParagraphIndex);
            _.extend(ret, format);
            ret.indent = levelFormat.indentLeft;
            ret.firstLine = levelFormat.indentFirstLine ? levelFormat.indentLeft + levelFormat.indentFirstLine : levelFormat.indentLeft;
            return ret;
        };

        /**
         * @param text possible numbering label text
         *
         * @returns {integer} listId
         *
         */
        this.detectListSymbol = function (text) {
            var ret = {};
            if (text.length === 1 && (text === '-' || text === '*')) {
                // bullet
                ret.numberFormat = 'bullet';
                ret.symbol = text;
            } else if (text.substring(text.length - 1) === '.') {
                var sub = text.substring(0, text.length - 1);
                var startnumber = parseInt(sub, 10);
                if (startnumber > 0) {
                    ret.numberFormat = 'decimal';
                    ret.listStartValue = startnumber;
                } else {
                    var roman = parseRoman(text);
                    if (roman.startnumber > 0) {
                        ret.numberFormat = roman.numberFormat;
                        ret.listStartValue = roman.startnumber;
                    }
                }
            }
            return ret;
        };

        this.findIlvl = function (listStyleId, paraStyle) {
            var list = this.getList(listStyleId);
            if (list === undefined) {
                return -1;
            }
            var ilvl = 0;
            for (; ilvl < 9; ++ilvl) {
                var levelFormat = list.listLevels[ilvl];
                if (levelFormat.paraStyle === paraStyle)
                    return ilvl;
            }
            return -1;
        };

        this.findPrevNextStyle = function (listStyleId, paraStyle, prev) {
            var list = this.getList(listStyleId);
            if (list === undefined) {
                return '';
            }
            var ilvl = 0;
            for (; ilvl < 9; ++ilvl) {
                var levelFormat = list.listLevels[ilvl];
                if (levelFormat.paraStyle === paraStyle) {
                    var ret = '';
                    if (prev) {
                        if (ilvl > 0 && list.listLevels[ilvl - 1].paraStyle)
                            ret = list.listLevels[ilvl - 1].paraStyle;
                    } else if (ilvl < 8 && list.listLevels[ilvl + 1].paraStyle) {
                        ret = list.listLevels[ilvl + 1].paraStyle;
                    }
                    return ret;
                }
            }
            return -1;
        };

    } // class Lists

    // constants --------------------------------------------------------------

    /**
     * A constant that will be used in the GUI to represent the default style
     * of bullet lists or numbered lists.
     */
    Lists.DEFAULT_VALUE = '__DEFAULT__';

    // static methods ---------------------------------------------------------

    /**
     * Returns all predefined bullet list styles.
     *
     * @returns {Object}
     *  A map with list style identifiers as keys, and objects as values
     *  containing the attributes 'definition' with the list style definition,
     *  'listLabel' containing the bullet text, and 'tooltip' containing a GUI
     *  tool tip string for the list style.
     */
    Lists.getPredefinedBulletListStyles = function () {

        var // the resulting bullet list styles, mapped by list identifier
            bulletListStyles = {};

        _(PREDEFINED_LISTSTYLES).each(function (listStyle, listStyleId) {
            if (listStyle.definition.listLevel0.numberFormat === 'bullet') {
                bulletListStyles[listStyleId] = listStyle;
            }
        });

        return bulletListStyles;
    };

    /**
     * Returns all predefined numbered list styles.
     *
     * @returns {Object}
     *  A map with list style identifiers as keys, and objects as values
     *  containing the attributes 'definition' with the list style definition,
     *  'listlabel' containing a string with the number 1 formatted according
     *  to the list style, and 'tooltip' containing a GUI tool tip string for
     *  the list style.
     */
    Lists.getPredefinedNumberedListStyles = function () {

        var // the resulting numbered list styles, mapped by list identifier
            numberedListStyles = {};

        _(PREDEFINED_LISTSTYLES).each(function (listStyle, listStyleId) {
            if (listStyle.definition.listLevel0.numberFormat !== 'bullet') {
                numberedListStyles[listStyleId] = listStyle;
            }
        });

        return numberedListStyles;
    };

    // exports ================================================================

    // derive this class from class Container
    return Container.extend({ constructor: Lists });

});
