/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/statusbar',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/control/label',
     'io.ox/office/tk/control/button',
     'io.ox/office/framework/view/pane',
     'io.ox/office/framework/view/component',
     'io.ox/office/spreadsheet/view/controls',
     'gettext!io.ox/office/spreadsheet'
    ], function (Utils, Label, Button, Pane, Component, SpreadsheetControls, gt) {

    'use strict';

    // private global functions ===============================================

    /**
     * Returns the subtotal label for the specified type and value.
     *
     * @param {String} type
     *  The type identifier of the subtotal.
     *
     * @param {Number} value
     *  The actual value of the subtotal.
     *
     * @returns {String}
     *  The status bar label of the subtotal.
     */
    function getSubtotalLabel(type, value) {
        switch (type) {
        case 'sum':
            //#. %1$d is the sum of all values in a selected cell range in a spreadsheet
            //#, c-format
            return gt('Sum: %1$d', _.noI18n(value));
        case 'min':
            //#. %1$d is the minimum of all values in a selected cell range in a spreadsheet
            //#, c-format
            return gt('Minimum: %1$d', _.noI18n(value));
        case 'max':
            //#. %1$d is the maximum of all values in a selected cell range in a spreadsheet
            //#, c-format
            return gt('Maximum: %1$d', _.noI18n(value));
        }
        return '';
    }

    // class StatusBar ========================================================

    /**
     * @constructor
     *
     * @extends Pane
     */
    function StatusBar(app) {

        var // self reference
            self = this,

            // the view component for all sheet controls
            sheetComponent = null,

            // the sheet selector, as radio group, and its scroll buttons
            sheetGroup = null,
            scrollPrevButton = null,
            scrollNextButton = null,

            // the sheet selectors, as drop-down menu (with and without sheet name as label)
            largeSheetList = null,
            smallSheetList = null,

            // the view component for displaying the subtotals
            subtotalComponent = null;

        // base constructor ---------------------------------------------------

        Pane.call(this, app, 'statusbar', { position: 'bottom', classes: 'status-bar inline' });

        // private methods ----------------------------------------------------

        /**
         * Refreshes the layout of the status bar after its size has been
         * changed.
         */
        function refreshLayout() {

            var // the root node of the status bar
                rootNode = self.getNode(),
                // the root node of the sheet component
                sheetCompNode = sheetComponent.getNode(),
                // the available inner width in the status bar
                availableWidth = rootNode.width(),
                // the remaining width for the sheet selector
                remainingWidth = 0;

            // hide all additional sheet selector controls
            largeSheetList.hide();
            smallSheetList.hide();
            scrollPrevButton.hide();
            sheetGroup.show();
            scrollNextButton.hide();

            // reset sheet selector to automatic width
            sheetGroup.reset(false);

            // check whether the sheet tabs need to be shortened and made scrollable
            if (availableWidth < sheetCompNode.outerWidth()) {
                if (availableWidth < 300) {
                    // show large sheet drop-down list for small devices
                    sheetGroup.hide();
                    largeSheetList.show();
                    // calculate and set remaining space for the sheet drop-down button
                    remainingWidth = largeSheetList.getNode().outerWidth() + (availableWidth - sheetCompNode.outerWidth());
                    largeSheetList.setWidth(remainingWidth);
                } else {
                    // show additional sheet selector controls
                    smallSheetList.show();
                    scrollPrevButton.show();
                    sheetGroup.reset(true);
                    scrollNextButton.show();
                    // calculate and set remaining space for the sheet tabs
                    remainingWidth = sheetGroup.getNode().outerWidth() + (availableWidth - sheetCompNode.outerWidth());
                    sheetGroup.setWidth(remainingWidth);
                }
            }
        }

        /**
         * Handles tracking events on the scroll buttons. Scrolls the sheet
         * tabs in the sheet selector repeatedly until tracking is stopped.
         * @param event
         */
        var autoRepetitionHandler = (function () {

            var scrollMethod = null,
                animationTimer = null;

            function scroll() {
                if (!animationTimer) {
                    animationTimer = scrollMethod.call(sheetGroup);
                    animationTimer.always(function () { animationTimer = null; });
                }
            }

            return function (event) {
                switch (event.type) {
                case 'tracking:start':
                    scrollMethod = $(this).is('.scroll-button.next') ? sheetGroup.scrollToNext : sheetGroup.scrollToPrev;
                    scroll();
                    break;
                case 'tracking:repeat':
                    scroll();
                    break;
                }
            };

        }()); // end of autoRepetitionHandler() local scope

        // initialization -----------------------------------------------------

        this.addViewComponent(sheetComponent = new Component(app, 'sheets')
            .addGroup('view/sheet/active', largeSheetList = new SpreadsheetControls.ActiveSheetList(app, { showNames: true }))
            .addGroup('view/sheet/active', smallSheetList = new SpreadsheetControls.ActiveSheetList(app))
            .addPrivateGroup(scrollPrevButton = new Button({ classes: 'scroll-button prev', icon: 'icon-angle-left' }))
            .addGroup('view/sheet/active', sheetGroup = new SpreadsheetControls.ActiveSheetGroup(app))
            .addPrivateGroup(scrollNextButton = new Button({ classes: 'scroll-button next', icon: 'icon-angle-right' }))
            .addGroup('sheet/insert', new Button({ icon: 'icon-plus', tooltip: 'Insert sheet' }))
        );

        // refresh layout after status bar changes its size, or after the sheet list has changed
        app.getView().on('refresh:layout', refreshLayout);
        sheetGroup.on('refresh:layout', refreshLayout);

        // handle tracking events on scroll buttons
        scrollPrevButton.getNode().add(scrollNextButton.getNode())
            .enableTracking({ autoRepeat: true })
            .on('tracking:start tracking:repeat', autoRepetitionHandler);

    } // class StatusBar

    // exports ================================================================

    // derive this class from class Pane
    return Pane.extend({ constructor: StatusBar });

});
