/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/headerpane',
    ['io.ox/core/event',
     'io.ox/office/tk/utils',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/utils/colrowintervals'
    ], function (Events, Utils, SheetUtils, ColRowIntervals) {

    'use strict';

    var // the names of all tracking events but 'tracking:start'
        TRACKING_EVENT_NAMES = 'tracking:move tracking:scroll tracking:end tracking:cancel';

    // class HeaderPane =======================================================

    /**
     * Represents a single row or column header pane in the spreadsheet view.
     * If the view has been split or frozen at a specific cell position, the
     * view will consist of up to four header panes (left and right column
     * header panes, and top and bottom row header panes).
     *
     * @constructor
     *
     * @extends Events
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this header pane.
     *
     * @param {String} paneSide
     *  The position of this header pane (one of 'left', 'right', 'top',
     *  'bottom', or 'corner').
     */
    function HeaderPane(app, paneSide) {

        var // self reference
            self = this,

            // the spreadsheet view
            view = app.getView(),

            // type of this header pane, according to passed pane side
            columns = (paneSide === 'left') || (paneSide === 'right'),

            // the container node of this header pane
            rootNode = $('<div>').addClass('header-pane unselectable ' + (columns ? 'columns' : 'rows')),

            // the content node with the header cell and resizer nodes
            contentNode = $('<div>').addClass('header-content noI18n').appendTo(rootNode),

            // the column/row collection for this header pane
            collection = view.getColRowCollection(paneSide),

            // the absolute scroll position in the associated grid panes
            scrollPos = 0,

            // distance between cell A1 and the top-left visible cell of this pane
            hiddenSize = 0,

            // the type of the current tracking mode ('select', 'resize', null)
            trackingType = null;

        // base constructor ---------------------------------------------------

        Events.extend(this);

        // private methods ----------------------------------------------------

        /**
         * Updates the position and size of the content node (containing the
         * cell header nodes) according to the current scroll position and pane
         * side layout data.
         */
        function updateContentPosition() {

            var // position of the content node
                offset = collection.getOffset() - self.getFirstVisibleOffset(),
                // size of the content node
                size = collection.getSize();

            contentNode.css(columns ? { left: offset, width: size } : { top: offset, height: size });
        }

        /**
         * Renders all visible column header cells according to the current
         * pane side layout data received from a view layout update
         * notification.
         */
        function renderColumnCells(selection) {

            var // start position of first column
                leftOffset = collection.getOffset(),
                // the height of the entire header pane (used as line height)
                lineHeight = rootNode.height(),
                // the HTML mark-up for all visible column header cells
                markup = '';

            // generate HTML mark-up text for all visible header cells
            collection.iterateEntries(function (colEntry) {

                var cellStyleAttr = ' style="left:' + (colEntry.offset - leftOffset) + 'px;width:' + colEntry.size + 'px;line-height:' + lineHeight + 'px;"',
                    cellText = SheetUtils.getColumnName(colEntry.index),
                    resizeStyleAttr = ' style="left:' + (colEntry.offset + colEntry.size - leftOffset - 2) + 'px;"',
                    dataAttr = ' data-index="' + colEntry.index + '"';

                markup += '<div class="cell"' + cellStyleAttr + dataAttr + '>' + cellText + '</div><div class="resizer"' + resizeStyleAttr + dataAttr + '></div>';
            });

            // insert entire HTML mark-up into the cell container node
            contentNode[0].innerHTML = markup;
        }

        /**
         * Renders all visible row header cells according to the current pane
         * side layout data received from a view layout update notification.
         */
        function renderRowCells(selection) {

            var // start position of first row
                topOffset = collection.getOffset(),
                // the HTML mark-up for all visible row header cells
                markup = '';

            // generate HTML mark-up text for all visible header cells
            collection.iterateEntries(function (rowEntry) {

                var cellStyleAttr = ' style="top:' + (rowEntry.offset - topOffset) + 'px;height:' + rowEntry.size + 'px;line-height:' + rowEntry.size + 'px;"',
                    cellText = SheetUtils.getRowName(rowEntry.index),
                    resizeStyleAttr = ' style="top:' + (rowEntry.offset + rowEntry.size - topOffset - 2) + 'px;"',
                    dataAttr = ' data-index="' + rowEntry.index + '"';

                markup += '<div class="cell"' + cellStyleAttr + dataAttr + '>' + cellText + '</div><div class="resizer"' + resizeStyleAttr + dataAttr + '></div>';
            });

            // insert entire HTML mark-up into the cell container node
            contentNode[0].innerHTML = markup;
        }

        /**
         * Redraws the selection highlighting in the header cells of this
         * header pane.
         */
        function renderSelection(selection) {

            var // the merged ranges of selected columns/rows
                intervals = new ColRowIntervals(selection.ranges, columns ? 'columns' : 'rows').getArray(),
                // current index into the intervals array
                intervalIndex = 0;

            // find the first interval that ends at or before the passed index
            function findInterval(index) {
                while ((intervalIndex < intervals.length) && (intervals[intervalIndex].last < index)) { intervalIndex += 1; }
                return intervals[intervalIndex];
            }

            // find the first interval, and iterate all cell nodes
            findInterval(collection.getFirstIndex());
            Utils.iterateSelectedDescendantNodes(contentNode, '.cell', function (cellNode) {
                var index = Utils.getElementAttributeAsInteger(cellNode, 'data-index'),
                    interval = findInterval(index);
                $(cellNode).toggleClass('selected', _.isObject(interval) && (interval.first <= index));
            }, undefined, { children: true });
        }

        /**
         * Redraws the header cells of this header pane, after the selection in
         * the sheet has been changed.
         */
        function selectionHandler(event, selection) {
            if (trackingType === 'resize') { $.cancelTracking(); }
            renderSelection(selection);
        }

        /**
         * Updates this header pane according to the row/column sizes in the
         * updated column/row collection received from the view.
         */
        function updateGridHandler(event, changedFlags) {

            // do nothing if the layout data has not been changed
            if (!changedFlags[paneSide]) { return; }

            // generate visible header cells of this header pane
            Utils.takeTime('HeaderPane.updateGridHandler(): paneSide=' + paneSide, function () {
                // update size and position of the content node
                updateContentPosition();
                // generate HTML mark-up text for all visible header cells
                (columns ? renderColumnCells : renderRowCells)();
                // redraw the selection highlighting
                renderSelection(view.getCellSelection());
            });
        }

        /**
         * Returns the column/row entry matching the 'data-index' attribute of
         * the target DOM node in the passed 'tracking:start' event.
         *
         * @param {jQuery.Event} event
         *  The 'tracking:start' event.
         *
         * @returns {Object|Null}
         *  The matching entry from the column/row collection.
         */
        function getTrackingStartEntry(event) {
            var index = Utils.getElementAttributeAsInteger(event.target, 'data-index', -1);
            return collection.getEntryByIndex(index);
        }

        /**
         * Returns the current absolute sheet position in the header pane for
         * the passed tracking event.
         *
         * @param {jQuery.Event} event
         *  The tracking event.
         *
         * @returns {Number}
         *  The absolute sheet position, in pixels.
         */
        function getTrackingOffset(event) {
            return (columns ? (event.pageX - contentNode.offset().left) : (event.pageY - contentNode.offset().top)) + collection.getOffset();
        }

        /**
         * Handles all tracking events while cell selection is active.
         */
        var selectionTrackingHandler = (function () {

            var // the column/row index where tracking has started
                startIndex = null,
                // current index (prevent updates while tracking over the same column/row)
                currentIndex = -1;

            function makeCell(index1, index2) {
                return columns ? [index1, index2] : [index2, index1];
            }

            // makes an adjusted range between start index and current index
            function makeRange() {
                var maxIndex = view.getSheetLayoutData()[columns ? 'rows' : 'cols'] - 1;
                return SheetUtils.adjustRange({ start: makeCell(startIndex, 0), end: makeCell(currentIndex, maxIndex) });
            }

            // initializes the selection according to the passed tracking event
            function initializeTracking(event) {

                var // the initial column/row entry
                    entry = getTrackingStartEntry(event),
                    // whether to append a new range to the selection
                    append = !event.shiftKey && (event.ctrlKey || event.metaKey),
                    // whether to extend the active range
                    extend = event.shiftKey && !event.ctrlKey && !event.metaKey;

                if (!entry) {
                    $.cancelTracking();
                    return;
                }

                currentIndex = entry.index;
                if (extend) {
                    // pressed SHIFT key: modify the active range (from active cell to tracking start position)
                    startIndex = view.getActiveCell()[columns ? 0 : 1];
                    view.changeActiveRange(makeRange());
                } else {
                    // no SHIFT key: create/append a new selection range
                    startIndex = currentIndex;
                    // set active cell to current first visible cell in the column/row
                    view.selectRange(makeRange(), {
                        append: append,
                        active: makeCell(startIndex, Math.max(0, view.getVisibleHeaderPane(!columns).getFirstVisibleIndex()))
                    });
                }
                view.scrollPaneSideToEntry(paneSide, currentIndex);
            }

            // updates the selection according to the passed tracking event
            function updateSelection(event) {

                var // the column/row entry currently hovered
                    entry = collection.getNearestEntryByOffset(getTrackingOffset(event));

                if (entry && (currentIndex >= 0) && (entry.index !== currentIndex)) {
                    currentIndex = entry.index;
                    view.changeActiveRange(makeRange());
                }
            }

            // updates the scroll position while tracking
            function updateScroll(event) {
                view.scrollPaneSide(paneSide, scrollPos + (columns ? event.scrollX : event.scrollY));
            }

            // finalizes the selection tracking
            function finalizeTracking(event) {
                if (currentIndex >= 0) {
                    view.scrollPaneSideToEntry(paneSide, currentIndex);
                    currentIndex = -1;
                }
                trackingType = null;
                $(event.delegateTarget).off(TRACKING_EVENT_NAMES);
                view.grabFocus();
            }

            // return the actual selectionTrackingHandler() method
            return function (event) {

                switch (event.type) {
                case 'tracking:start':
                    initializeTracking(event);
                    break;
                case 'tracking:move':
                    updateSelection(event);
                    break;
                case 'tracking:scroll':
                    updateScroll(event);
                    updateSelection(event);
                    break;
                case 'tracking:end':
                case 'tracking:cancel':
                    // cancel selection tracking: keep current selection
                    updateSelection(event);
                    finalizeTracking(event);
                    break;
                }
            };

        }()); // end of selectionTrackingHandler() local scope

        /**
         * Handles all tracking events while resizing a column/row is active.
         */
        var resizeTrackingHandler = (function () {

            var // entry of the column/row currently resized
                entry = null,
                // difference between exact start position and trailing position of column/row
                correction = 0,
                // the current size of the column/row while tracking
                currentSize = 0,
                // minimum/maximum scroll position allowed while resizing
                minScrollPos = 0, maxScrollPos = 0,
                // maximum size of columns/rows
                MAX_SIZE = Utils.convertHmmToLength(columns ? SheetUtils.MAX_COLUMN_WIDTH : SheetUtils.MAX_ROW_HEIGHT, 'px', 1);

            // initializes column/row resizing according to the passed tracking event
            function initializeTracking(event) {
                if ((entry = getTrackingStartEntry(event))) {
                    // calculate difference between exact tracking start position and end of column/row
                    // (to prevent unwanted resize actions without moving the tracking point)
                    correction = getTrackingOffset(event) - (entry.offset + entry.size);
                    currentSize = entry.size;
                    minScrollPos = Math.max(0, entry.offset - 20 - hiddenSize);
                    maxScrollPos = Math.max(minScrollPos, entry.offset + MAX_SIZE + 20 - (columns ? rootNode.width() : rootNode.height()) - hiddenSize);
                    self.trigger('resize:start', entry.offset, currentSize);
                }
            }

            // updates column/row resizing according to the passed tracking event
            function updateSize(event) {
                if (entry) {
                    currentSize = Utils.minMax(getTrackingOffset(event) - correction - entry.offset, 0, MAX_SIZE);
                    self.trigger('resize:update', entry.offset, currentSize);
                }
            }

            // updates scroll position according to the passed tracking event
            function updateScroll(event) {
                var newScrollPos = scrollPos + (columns ? event.scrollX : event.scrollY);
                if (entry) {
                    if ((newScrollPos < scrollPos) && (scrollPos > minScrollPos)) {
                        view.scrollPaneSide(paneSide, Math.max(newScrollPos, minScrollPos));
                    } else if ((newScrollPos > scrollPos) && (scrollPos < maxScrollPos)) {
                        view.scrollPaneSide(paneSide, Math.min(newScrollPos, maxScrollPos));
                    }
                }
            }

            // finalizes the resize tracking
            function finalizeTracking(event, applySize) {
                if (applySize && entry && (currentSize !== entry.size)) {
                    (columns ? view.setColumnWidth : view.setRowHeight)(entry.index, null, Utils.convertLengthToHmm(currentSize, 'px'));
                }
                self.trigger('resize:end');
                trackingType = null;
                $(event.delegateTarget).off(TRACKING_EVENT_NAMES);
                view.grabFocus();
            }

            // return the actual resizeTrackingHandler() method
            return function (event) {

                switch (event.type) {
                case 'tracking:start':
                    initializeTracking(event);
                    break;
                case 'tracking:move':
                    updateSize(event);
                    break;
                case 'tracking:scroll':
                    updateScroll(event);
                    updateSize(event);
                    break;
                case 'tracking:end':
                    updateSize(event);
                    finalizeTracking(event, true);
                    break;
                case 'tracking:cancel':
                    finalizeTracking(event, false);
                    break;
                }
            };

        }()); // end of resizeTrackingHandler() local scope

        /**
         * Handles 'tracking:start' events end decides whether to select cells,
         * or to resize columns or rows.
         */
        function trackingStartHandler(event) {
            if ($(event.target).is('.cell')) {
                trackingType = 'select';
                $(event.delegateTarget).on(TRACKING_EVENT_NAMES, selectionTrackingHandler);
                selectionTrackingHandler.call(this, event);
            } else if ($(event.target).is('.resizer')) {
                trackingType = 'resize';
                $(event.delegateTarget).on(TRACKING_EVENT_NAMES, resizeTrackingHandler);
                resizeTrackingHandler.call(this, event);
            }
        }

        /**
         * Handles a changed document edit mode. In read-only mode, disables
         * the possibility to modify the size of rows or columns.
         */
        function editModeHandler(event, editMode) {
            rootNode.toggleClass('readonly', !editMode);
            if (!editMode && (trackingType === 'resize')) {
                $.cancelTracking();
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the root DOM node of this header pane.
         *
         * @returns {jQuery}
         *  The root node of this header pane, as jQuery object.
         */
        this.getNode = function () {
            return rootNode;
        };

        /**
         * Initializes the settings and layout of this header pane.
         *
         * @param {Object} settings
         *  The view settings of the pane side represented by this header pane.
         *
         * @returns {HeaderPane}
         *  A reference to this instance.
         */
        this.initialize = function (settings) {

            // visibility
            rootNode.toggle(settings.visible);

            // position and size
            rootNode.css(columns ?
                { left: settings.offset, width: settings.size, height: view.getCornerPane().getNode().height() } :
                { top: settings.offset, height: settings.size, width: view.getCornerPane().getNode().width() }
            );

            // initialize auto-scrolling
            rootNode.enableTracking({
                selector: '.cell, .resizer',
                autoScroll: settings.scrollable ? (columns ? 'horizontal' : 'vertical') : false,
                borderNode: rootNode,
                borderMargin: -30,
                borderSize: 60,
                minSpeed: 2,
                maxSpeed: 250
            });

            // other layout options
            hiddenSize = settings.hiddenSize;

            // update the header contents
            updateContentPosition();

            return this;
        };

        /**
         * Returns the offset of the first pixel in the entire sheet visible in
         * this header pane.
         *
         * @returns {Number}
         *  The position of the first visible pixel, in pixels.
         */
        this.getFirstVisibleOffset = function () {
            return hiddenSize + scrollPos;
        };

        /**
         * Returns the zero-based index of the first visible column/row in this
         * header pane. The column/row is considered visible, it at least the
         * half of its total size is really visible.
         *
         * @returns {Number}
         *  The zero-based index of the first visible column/row. If no layout
         *  data is currently available, returns -1.
         */
        this.getFirstVisibleIndex = function () {

            var // the offset of the first visible pixel
                offset = this.getFirstVisibleOffset(),
                // the collection entry covering the first pixel
                entry = collection.getEntryByOffset(offset),
                // the column/row index
                index = -1;

            // layout data may be outdated
            if (entry) {

                // store current index, looking up following collection entry may fail
                index = entry.index;

                // check if the entry is visible less than half, go to next entry in that case
                if (entry.offset + entry.size - offset < entry.size / 2) {
                    if ((entry = collection.findNextEntryByIndex(index + 1))) {
                        index = entry.index;
                    }
                }
            }

            return index;
        };

        /**
         * Changes the scroll position of this header pane.
         *
         * @param {Number} pos
         *  The new scroll position.
         *
         * @returns {HeaderPane}
         *  A reference to this instance.
         */
        this.scroll = function (pos) {
            scrollPos = Math.max(0, pos);
            updateContentPosition();
            return this;
        };

        // initialization -----------------------------------------------------

        // listen to update and selection events from the view
        view.on({
            'change:selection': selectionHandler,
            'update:grid': updateGridHandler
        });

        // tracking for column/row selection with mouse/touch
        rootNode.on('tracking:start', trackingStartHandler);

        // enable/disable operations that modify the document according to edit mode
        app.getModel().on('change:editmode', editModeHandler);

    } // class HeaderPane

    // exports ================================================================

    return HeaderPane;

});
