/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/sheetutils', ['io.ox/office/tk/utils'], function (Utils) {

    'use strict';

    var // maximum number of columns that will ever be allowed in a sheet (A to ZZZZ)
        MAX_COL_COUNT = 475254,

        // maximum number of rows that will ever be allowed in a sheet (1 to 99,999,999)
        MAX_ROW_COUNT = 99999999,

        // regular expression pattern matching a valid column name (A to ZZZZ)
        COL_PATTERN = '[A-Z]{1,4}',

        // regular expression pattern matching a row name (1 to 99,999,999)
        ROW_PATTERN = '0*[1-9][0-9]{0,7}',
        // regular expression object matching a column name only
        RE_COL_NAME = new RegExp('^' + COL_PATTERN + '$'),

        // regular expression object matching a row name only
        RE_ROW_NAME = new RegExp('^' + ROW_PATTERN + '$'),

        // regular expression object matching a cell address, row and column part in parentheses
        RE_ADDRESS = new RegExp('^(' + COL_PATTERN + ')(' + ROW_PATTERN + ')$');

    // static class SheetUtils ================================================

    var SheetUtils = {};

    // constants --------------------------------------------------------------

    // TODO: get maximum size of columns/rows from somewhere
    /**
     * Maximum width of columns, in 1/100 millimeters.
     *
     * @constant
     */
    SheetUtils.MAX_COLUMN_WIDTH = 20000;

    // TODO: get maximum size of columns/rows from somewhere
    /**
     * Maximum height of rows, in 1/100 millimeters.
     *
     * @constant
     */
    SheetUtils.MAX_ROW_HEIGHT = 20000;

    // methods ----------------------------------------------------------------

    /**
     * Returns whether the passed value is a valid logical cell address.
     *
     * @param {Any} address
     *  Any value that will be checked whether it is a valid logical cell
     *  address.
     *
     * @returns {Boolean}
     *  Whether the passed value is an array with two numerical non-negative
     *  elements. The first element (column index) must be less than the global
     *  maximum number of supported columns (currently 475,254 - columns A to
     *  ZZZZ); the second element (row index) must be less than the global
     *  maximum number of supported rows (currently 99,999,999). This static
     *  method does not take the real grid size of a spreadsheet document into
     *  account.
     */
    SheetUtils.isValidAddress = function (address) {
        return _.isArray(address) && (address.length === 2) &&
            _.isFinite(address[0]) && (0 <= address[0]) && (address[0] < MAX_COL_COUNT) &&
            _.isFinite(address[1]) && (0 <= address[1]) && (address[1] < MAX_ROW_COUNT);
    };

    /**
     * Returns whether the passed value is a valid logical cell range address.
     *
     * @param {Any} range
     *  Any value that will be checked whether it is a valid logical cell range
     *  address.
     *
     * @returns {Boolean}
     *  Whether the passed value is an object with the two properties 'start'
     *  and 'end', both containing valid cell addresses (see method
     *  SheetUtils.isValidAddress()). Does not check whether the start and end
     *  positions are ordered.
     */
    SheetUtils.isValidRange = function (range) {
        return _.isObject(range) && SheetUtils.isValidAddress(range.start) && SheetUtils.isValidAddress(range.end);
    };

    /**
     * Checks the column and row indexes contained in the passed range, and
     * swaps them (in-place) if they are not ordered.
     *
     * @param {Object} range
     *  The logical address of a range, that will be adjusted if required.
     *
     * @returns {Object}
     *  A reference to the passed range.
     */
    SheetUtils.adjustRange = function (range) {
        var tmp = null;
        // sort columns and rows independently
        if (range.start[0] > range.end[0]) { tmp = range.start[0]; range.start[0] = range.end[0]; range.end[0] = tmp; }
        if (range.start[1] > range.end[1]) { tmp = range.start[1]; range.start[1] = range.end[1]; range.end[1] = tmp; }
        return range;
    };

    /**
     * Returns a new range with column and row indexes in the correct order.
     *
     * @param {Object} range
     *  The logical address of a range.
     *
     * @returns {Object}
     *  A new adjusted cell range address.
     */
    SheetUtils.getAdjustedRange = function (range) {
        return {
            start: [Math.min(range.start[0], range.end[0]), Math.min(range.start[1], range.end[1])],
            end: [Math.max(range.start[0], range.end[0]), Math.max(range.start[1], range.end[1])]
        };
    };

    /**
     * Returns the number of columns covered by the passed cell range.
     *
     * @param {Object} range
     *  The logical address of the range.
     *
     * @returns {Number}
     *  The number of columns in the passed range.
     */
    SheetUtils.getColCount = function (range) {
        return range.end[0] - range.start[0] + 1;
    };

    /**
     * Returns the number of rows covered by the passed cell range.
     *
     * @param {Object} range
     *  The logical address of the range.
     *
     * @returns {Number}
     *  The number of rows in the passed range.
     */
    SheetUtils.getRowCount = function (range) {
        return range.end[1] - range.start[1] + 1;
    };

    /**
     * Returns the number of cells covered by the passed cell range.
     *
     * @param {Object} range
     *  The logical address of the range.
     *
     * @returns {Number}
     *  The number of cells in the passed range.
     */
    SheetUtils.getCellCount = function (range) {
        return SheetUtils.getColCount(range) * SheetUtils.getRowCount(range);
    };

    /**
     * Returns whether the passed range contains the specified column index.
     *
     * @param {Object} range
     *  The logical address of the range.
     *
     * @param {Number} col
     *  The zero-based column index to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed range contains the column index.
     */
    SheetUtils.rangeContainsCol = function (range, col) {
        return (range.start[0] <= col) && (col <= range.end[0]);
    };

    /**
     * Returns whether the passed range contains the specified row index.
     *
     * @param {Object} range
     *  The logical address of the range.
     *
     * @param {Number} row
     *  The zero-based row index to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed range contains the row index.
     */
    SheetUtils.rangeContainsRow = function (range, row) {
        return (range.start[1] <= row) && (row <= range.end[1]);
    };

    /**
     * Returns whether the passed range contains the specified cell address.
     *
     * @param {Object} range
     *  The logical address of the range.
     *
     * @param {Number[]} address
     *  The logical cell address to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed range contains the cell address.
     */
    SheetUtils.rangeContainsCell = function (range, address) {
        return SheetUtils.rangeContainsCol(range, address[0]) && SheetUtils.rangeContainsRow(range, address[1]);
    };

    /**
     * Returns whether the passed ranges overlap with at least one cell.
     *
     * @param {Object} range1
     *  The logical address of the first range.
     *
     * @param {Object} range2
     *  The logical address of the second range.
     *
     * @returns {Boolean}
     *  Whether the passed ranges are overlapping.
     */
    SheetUtils.rangesOverlap = function (range1, range2) {
        return (range1.start[0] <= range2.end[0]) && (range2.start[0] <= range1.end[0]) && (range1.start[1] <= range2.end[1]) && (range2.start[1] <= range1.end[1]);
    };

    /**
     * Returns the intersecting range of the passed ranges.
     *
     * @param {Object} range1
     *  The logical address of the first range.
     *
     * @param {Object} range2
     *  The logical address of the second range.
     *
     * @returns {Object|Null}
     *  The address of the cell range covered by both passed ranges, if
     *  existing; otherwise null.
     */
    SheetUtils.getIntersectionRange = function (range1, range2) {
        return SheetUtils.rangesOverlap(range1, range2) ? {
            start: [Math.max(range1.start[0], range2.start[0]), Math.max(range1.start[1], range2.start[1])],
            end: [Math.min(range1.end[0], range2.end[0]), Math.min(range1.end[1], range2.end[1])]
        } : null;
    };

    /**
     * Returns the string representation of the passed column index.
     *
     * @param {Number} col
     *  The zero-based column index.
     *
     * @returns {String}
     *  The upper-case string representation of the column index.
     */
    SheetUtils.getColumnName = function (col) {
        var name = '';
        while (col >= 0) {
            name = String.fromCharCode(65 + col % 26) + name;
            col = Math.floor(col / 26) - 1;
        }
        return name;
    };

    /**
     * Returns the string representation of the passed row index.
     *
     * @param {Number} row
     *  The zero-based row index.
     *
     * @returns {String}
     *  The one-based string representation of the row index.
     */
    SheetUtils.getRowName = function (row) {
        return String(row + 1);
    };

    /**
     * Returns the string representation of the passed logical cell address.
     *
     * @param {Number[]} address
     *  The logical cell address as two-element array, containing the
     *  zero-based column and row indexes of the cell.
     *
     * @returns {String}
     *  The string representation of the cell position, in A1 notation.
     */
    SheetUtils.getCellName = function (address) {
        return SheetUtils.getColumnName(address[0]) + SheetUtils.getRowName(address[1]);
    };

    /**
     * Returns the string representation of the passed logical range address.
     *
     * @param {Object} range
     *  The logical cell range address as object with 'start' and 'end'
     *  properties, both containing logical cell addresses.
     *
     * @param {Object} [options]
     *  A map of options controlling the result of this method. The following
     *  options are supported:
     *  @param {Boolean} [options.shorten=false]
     *      If set to true, a cell range referring to a single cell will be
     *      converted to a cell address instead of a range address consisting
     *      of two equal cell addresses.
     *
     * @returns {String}
     *  The string representation of the cell range, in A1:A1 notation.
     */
    SheetUtils.getRangeName = function (range, options) {
        var name = SheetUtils.getCellName(range.start);
        if (!Utils.getBooleanOption(options, 'shorten', false) || !_.isEqual(range.start, range.end)) {
            name += ':' + SheetUtils.getCellName(range.end);
        }
        return name;
    };

    /**
     * Returns the column index of the passed column string representation.
     *
     * @param {String} name
     *  The string representation of a column (case-insensitive).
     *
     * @returns {Number}
     *  The zero-based column index, if the passed column name is valid,
     *  otherwise -1.
     */
    SheetUtils.parseColumnName = function (name) {
        var col = -1;
        name = name.toUpperCase();
        if (RE_COL_NAME.test(name)) {
            for (var ichar = 0; ichar < name.length; ichar += 1) {
                col = (col + 1) * 26 + name.charCodeAt(ichar) - 65;
            }
        }
        return (col < MAX_COL_COUNT) ? col : -1;
    };

    /**
     * Returns the row index of the passed row string representation.
     *
     * @param {String} name
     *  The string representation of a row (one-based).
     *
     * @returns {Number}
     *  The zero-based row index, if the passed row name is valid, otherwise
     *  the value -1.
     */
    SheetUtils.parseRowName = function (name) {
        var row = RE_ROW_NAME.test(name) ? (parseInt(name, 10) - 1) : -1;
        return (row < MAX_ROW_COUNT) ? row : -1;
    };

    SheetUtils.parseCellName = function (name) {
    };

    SheetUtils.parseRangeName = function (name, options) {
    };


    // exports ================================================================

    return SheetUtils;

});
