/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/colrowintervals', ['io.ox/office/tk/utils'], function (Utils) {

    'use strict';

    // class ColRowIntervals ==================================================

    /**
     * Represents a sorted array of distinct column or row intervals, created
     * from the passed unordered array of cell range addresses.
     *
     * @param {Object[]} ranges
     *  The ranges list, as array of logical range positions (objects with
     *  'start' and 'end' properties containing logical cell positions).
     *
     * @param {String} type
     *  Either 'columns' or 'rows'. Depending on this type, the appropriate
     *  property names will be used while reading the layout data received from
     *  the 'docs:update' event notification.
     */
    function ColRowIntervals(ranges, type) {

        var // the resulting intervals
            intervals = [];

        // private methods ----------------------------------------------------

        function initialize() {

            var // index in the logical cell address (columns or rows)
                index = (type === 'columns') ? 0 : 1,
                // column or row intervals from passed cell ranges
                origIntervals = _(ranges).map(function (range) {
                    return { first: range.start[index], last: range.end[index] };
                });

            // sort intervals by start index
            origIntervals.sort(function (i1, i2) { return i1.first - i2.first; });

            // merge overlapping intervals
            _(origIntervals).each(function (interval) {

                var // last interval in the resulting list
                    lastInterval = _.last(intervals);

                if (!lastInterval || (lastInterval.last + 1 < interval.first)) {
                    // append a new interval
                    intervals.push({ first: interval.first, last: interval.last });
                } else {
                    // extend end of last interval
                    lastInterval.last = Math.max(lastInterval.last, interval.last);
                }
            });

            // add 'count' property to all intervals
            _(intervals).each(function (interval) {
                interval.count = interval.last - interval.first + 1;
            });
        }

        // methods ------------------------------------------------------------

        /**
         * Calls the passed iterator function for all intervals.
         *
         * @param {Function} iterator
         *  The iterator function called for all intervals in this list.
         *  Receives the interval as first parameter, and its array index as
         *  second parameter. Each interval object contains the properties
         *  'first' (first index included in the interval), 'last' (last index
         *  included), and 'count' (number of indexes included). If the
         *  iterator returns the Utils.BREAK object, the iteration process will
         *  be stopped immediately.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the intervals will be visited in reversed
         *      order.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateIntervals = function (iterator, options) {

            var context = Utils.getOption(options, 'context');

            if (Utils.getBooleanOption(options, 'reverse', false)) {
                for (var index = intervals.length - 1; index >= 0; index -= 1) {
                    if (iterator.call(context, intervals[index], index) === Utils.BREAK) {
                        return Utils.BREAK;
                    }
                }
            } else {
                for (var index = 0; index < intervals.length; index += 1) {
                    if (iterator.call(context, intervals[index], index) === Utils.BREAK) {
                        return Utils.BREAK;
                    }
                }
            }
        };

        /**
         * Returns the raw array of all intervals contained in this list.
         *
         * @returns {Object[]}
         *  All intervals in this list, as plain JavaScript array.
         */
        this.getArray = function () {
            return intervals;
        };

        /**
         * Returns whether the specified index is contained in this interval
         * list.
         *
         * @param {Number} index
         *  The index to be checked whether it is contained in the intervals.
         *
         * @returns {Boolean}
         *  Whether the specified index is contained in the index intervals.
         */
        this.intervalsContainIndex = function (index) {
            return _(intervals).any(function (interval) {
                return (interval.first <= index) && (index <= interval.last);
            });
        };

        /**
         * Returns whether the passed intervals are contained completely in
         * this interval list.
         *
         * @param {ColRowIntervals} intervals
         *  The interval list to be checked.
         *
         * @returns {Boolean}
         *  Whether this interval list contains the passed interval list
         *  completely.
         */
        this.containsIntervals = function (intervals) {
        };

        // initialization -----------------------------------------------------

        initialize();

    } // class ColRowIntervals

    // exports ================================================================

    return ColRowIntervals;

});
