/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/colrowcollection', ['io.ox/office/tk/utils'], function (Utils) {

    'use strict';

    var // property names for the pane side data received from 'docs:update' events
        PROPERTY_NAMES = {
            columns: { array: 'cols', index: 'col', offset: 'left', size: 'width' },
            rows:    { array: 'rows', index: 'row', offset: 'top',  size: 'height' }
        };

    // class ColRowCollection =================================================

    /**
     * Collects layout information about a range of visible columns or rows.
     *
     * Each collection entry is an object with the following properties:
     * - {Number} index
     *      The zero-based index of the column/row.
     * - {Number} offset
     *      The absolute position of the left column border/top row border in
     *      the sheet, in pixels.
     * - {Number} size
     *      The column width/row height, in pixels.
     * - {Object} [attrs]
     *      The column/row attribute set.
     *
     * @constructor
     *
     * @param {String} type
     *  Either 'columns' or 'rows'. Depending on this type, the appropriate
     *  property names will be used while reading the layout data received from
     *  the 'docs:update' event notification.
     */
    function ColRowCollection(type) {

        var // the property names for column or row properties
            PROP_NAMES = PROPERTY_NAMES[type],

            // all column/row data entries, as array, sorted by column/row index
            array = null,

            // all column/row data entries, as map, mapped by column/row index
            map = null,

            // first available column/row index (may point to hidden column/row)
            firstIndex = 0,

            // last available column/row index (may point to hidden column/row)
            lastIndex = 0,

            // absolute position of the first entry
            startOffset = 0,

            // absolute size of all entries
            totalSize = 0;

        // methods ------------------------------------------------------------

        /**
         * Clears this collection.
         *
         * @returns {ColRowCollection}
         *  A reference to this instance.
         */
        this.clear = function () {
            array = [];
            map = {};
            startOffset = totalSize = 0;
            return this;
        };

        /**
         * Initializes this collection with the data received from a
         * 'docs:update' event notification.
         *
         * @param {Object} paneSideData
         *  The original layout data of a pane side, as received from the
         *  'docs:update' event notification. Expects the properties 'cols' and
         *  'left' (if this is a columns collection), or 'rows' and 'top' (if
         *  this is a rows collection).
         *
         * @returns {ColRowCollection}
         *  A reference to this instance.
         */
        this.initialize = function (paneSideData) {

            var // the offset of the first column/row in this collection, in pixels
                offset = Utils.convertHmmToLength(paneSideData[PROP_NAMES.offset], 'px', 1),
                // the data array containing column/row entries
                dataArray = _.clone(Utils.getArrayOption(paneSideData, PROP_NAMES.array, []));

            // initialize class members
            this.clear();
            startOffset = offset;

            // sort the array by column/row indexes
            dataArray.sort(function (entry1, entry2) { return entry1[PROP_NAMES.index] - entry2[PROP_NAMES.index]; });

            // create the entries in the array and map of this collection
            _(dataArray).each(function (data) {

                var // the index of the first column/row
                    index = data[PROP_NAMES.index],
                    // the end index of the column/row range (half open)
                    endIndex = index + Utils.getIntegerOption(data, 'count', 1, 1),
                    // the size of the column/row, in pixels
                    size = Math.max(3, Utils.convertHmmToLength(data[PROP_NAMES.size], 'px', 1));

                // generate all collection entries
                for (; index < endIndex; index += 1, offset += size) {
                    array.push(map[index] = { index: index, offset: offset, size: size, attrs: data.attrs });
                }
            });

            // initialize first and last available column/row indexes
            firstIndex = this.isEmpty() ? 0 : Utils.getIntegerOption(paneSideData, 'first', _.first(array).index);
            lastIndex = this.isEmpty() ? 0 : Utils.getIntegerOption(paneSideData, 'last', _.last(array).index);

            // get the total size of all entries
            totalSize = offset - startOffset;

            return this;
        };

        /**
         * Copies the contents of the passed collection into this collection.
         *
         * @param {ColRowCollection} collection
         *  The source collection whose contents will be cloned into this
         *  collection.
         *
         * @returns {ColRowCollection}
         *  A reference to this instance.
         */
        this.cloneFrom = function (collection) {

            // initialize class members
            this.clear();

            // copy all collection entries
            collection.iterateEntries(function (entry) {
                array.push(map[entry.index] = entry);
            });

            // copy other class members
            firstIndex = collection.getFirstIndex();
            lastIndex = collection.getLastIndex();
            startOffset = collection.getOffset();
            totalSize = collection.getSize();

            return this;
        };

        /**
         * Calls the passed iterator function for all collection entries.
         *
         * @param {Function} iterator
         *  The iterator function called for all entries in this collection.
         *  Receives the collection entry as first parameter. If the iterator
         *  returns the Utils.BREAK object, the iteration process will be
         *  stopped immediately.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Number} [options.first]
         *      The index of the collection entry where iteration starts. If
         *      omitted, iteration starts at the first entry in the collection.
         *  @param {Number} [options.last]
         *      The index of the collection entry where iteration ends
         *      (including that entry). If omitted, iteration ends at the last
         *      entry in the collection.
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateEntries = function (iterator, options) {

            var first = Utils.getIntegerOption(options, 'first', firstIndex),
                last = Utils.getIntegerOption(options, 'last', lastIndex),
                context = Utils.getOption(options, 'context');

            return _(array).any(function (entry) {
                return (first <= entry.index) && (entry.index <= last) && (iterator.call(context, entry) === Utils.BREAK);
            }) ? Utils.BREAK : undefined;
        };

        /**
         * Returns whether this collection is empty.
         *
         * @returns {Boolean}
         *  Whether this collection is empty.
         */
        this.isEmpty = function () {
            return array.length === 0;
        };

        /**
         * Returns the index of the first column/row covered by this
         * collection. If this column/row is hidden, the collection will not
         * contain a corresponding entry. To get the first visible column/row,
         * the method ColRowCollection.getFirstEntry() can be used.
         *
         * @returns {Number}
         *  The index of the first column/row covered by this collection.
         */
        this.getFirstIndex = function () {
            return firstIndex;
        };

        /**
         * Returns the index of the last column/row covered by this collection.
         * If this column/row is hidden, the collection will not contain a
         * corresponding entry. To get the last visible column/row, the method
         * ColRowCollection.getLastEntry() can be used.
         *
         * @returns {Number}
         *  The index of the last column/row covered by this collection.
         */
        this.getLastIndex = function () {
            return lastIndex;
        };

        /**
         * Returns whether this collection covers the column7row with the
         * specified index. There must not necessarily exist an entry for the
         * passed index, if the column/row is hidden. To get the entry of a
         * visible column/row, the method ColRowCollection.getEntryByIndex()
         * can be used.
         *
         * @param {Number} index
         *  The zero-based index of the column/row.
         *
         * @returns {Boolean}
         *  Returns whether this collection covers the specified index.
         */
        this.containsIndex = function (index) {
            return !this.isEmpty() && (firstIndex <= index) && (index <= lastIndex);
        };

        /**
         * Returns the leading entry of the collection describing the first
         * visible column/row in the range.
         *
         * @returns {Object|Null}
         *  The collection entry of the first column/row; or null, if the
         *  collection is empty.
         */
        this.getFirstEntry = function () {
            return _(array).first() || null;
        };

        /**
         * Returns the trailing entry of the collection describing the last
         * visible column/row in the range.
         *
         * @returns {Object|Null}
         *  The collection entry of the last column/row; or null, if the
         *  collection is empty.
         */
        this.getLastEntry = function () {
            return _(array).last() || null;
        };

        /**
         * Returns whether this collection contains an entry for the specified
         * column/row. In this case, the column/row will especially be visible.
         *
         * @param {Number} index
         *  The zero-based index of the column/row.
         *
         * @returns {Boolean}
         *  Returns whether this collection contains the specified entry.
         */
        this.containsEntry = function (index) {
            return index in map;
        };

        /**
         * Returns the entry with the specified column/row index from the
         * collection.
         *
         * @param {Number} index
         *  The zero-based index of the column/row.
         *
         * @returns {Object|Null}
         *  The collection entry of the column/row with the specified index; or
         *  null, if the entry does not exist.
         */
        this.getEntryByIndex = function (index) {
            return map[index] || null;
        };

        /**
         * Returns an available entry for the specified column/row index. If no
         * entry contains the exact index, returns the first entry that follows
         * the specified column/row.
         *
         * @param {Number} index
         *  The zero-based index of the column/row.
         *
         * @returns {Object|Null}
         *  The collection entry of the column/row with the specified index, or
         *  following the specified index; or null, if no entry has been found.
         */
        this.findNextEntryByIndex = function (index) {
            return this.containsIndex(index) ? (map[index] || _(array).find(function (entry) { return entry.index > index; }) || null) : null;
        };

        /**
         * Returns an available entry for the specified column/row index. If no
         * entry contains the exact index, returns the first entry that
         * precedes the specified column/row.
         *
         * @param {Number} index
         *  The zero-based index of the column/row.
         *
         * @returns {Object|Null}
         *  The collection entry of the column/row with the specified index, or
         *  preceding the specified index; or null, if no entry has been found.
         */
        this.findPrevEntryByIndex = function (index) {
            return this.containsIndex(index) ? (map[index] || Utils.findLast(array, function (entry) { return entry.index < index; }) || null) : null;
        };

        /**
         * Returns the entry from the collection that covers the specified
         * pixel position in the sheet.
         *
         * @param {Number} offset
         *  The absolute position in the sheet, that needs to be resolved to
         *  the column/row entry, in pixels.
         *
         * @returns {Object|Null}
         *  The collection entry of the column/row covering the specified sheet
         *  position; or null, if the entry does not exist.
         */
        this.getEntryByOffset = function (offset) {
            return _(array).find(function (entry) { return (entry.offset <= offset) && (offset < entry.offset + entry.size); }) || null;
        };

        /**
         * Returns the entry from the collection that covers the specified
         * pixel position in the sheet. If the passed position is outside the
         * range covered by the collection, returns the first or last entry,
         * whichever is nearer to the position.
         *
         * @param {Number} offset
         *  The absolute position in the sheet, that needs to be resolved to
         *  the column/row entry, in pixels.
         *
         * @returns {Object|Null}
         *  The collection entry of the column/row covering the specified sheet
         *  position; or null, if collection is empty.
         */
        this.getNearestEntryByOffset = function (offset) {
            return (array.length === 0) ? null :
                (offset <= startOffset) ? this.getFirstEntry() :
                (offset >= startOffset + totalSize) ? this.getLastEntry() :
                this.getEntryByOffset(offset);
        };

        /**
         * Returns the position and size of the collection entry with the
         * specified column/row index.
         *
         * @param {Number} index
         *  The zero-based index of the column/row.
         *
         * @returns {Object|Null}
         *  An object with the properties 'offset' and 'size' containing the
         *  position and size of the collection entry, in pixels; or null, if
         *  the passed index is not covered by this collection.
         */
        this.getEntryPosition = function (index) {

            var entry = null, offset = 0, size = 0;

            if (this.containsIndex(index)) {
                entry = this.findNextEntryByIndex(index);
                offset = entry ? entry.offset : (startOffset + totalSize);
                size = (entry && (entry.index === index)) ? entry.size : 0;
                return { offset: offset, size: size };
            }

            return null;
        };

        /**
         * Returns the position and size of the specified range of collection
         * entries.
         *
         * @param {Number} first
         *  The zero-based index of the first column/row in the range.
         *
         * @param {Number} last
         *  The zero-based index of the last column/row in the range.
         *
         * @returns {Object|Null}
         *  An object with the properties 'offset' and 'size' containing the
         *  position and size of the range of collection entries, in pixels; or
         *  null, if the passed range is not covered by this collection.
         */
        this.getRangePosition = function (first, last) {

            var firstEntry = null, lastEntry = null, offset = null, size = null;

            if ((first <= lastIndex) && (last >= firstIndex)) {
                firstEntry = this.findNextEntryByIndex(first) || this.getFirstEntry();
                lastEntry = this.findPrevEntryByIndex(last) || this.getLastEntry();
                offset = firstEntry ? firstEntry.offset : (startOffset + totalSize);
                size = (lastEntry ? (lastEntry.offset + lastEntry.size) : startOffset) - offset;
                return { offset: offset, size: size };
            }

            return null;
        };

        /**
         * Returns the absolute position of the first entry contained in the
         * collection.
         *
         * @returns {Number}
         *  The absolute position of the first collection entry, in pixels.
         */
        this.getOffset = function () {
            return startOffset;
        };

        /**
         * Returns the total size of all entries contained in the collection.
         *
         * @returns {Number}
         *  The total size of all collection entries, in pixels.
         */
        this.getSize = function () {
            return totalSize;
        };

        // initialization -----------------------------------------------------

        // initialize class members
        this.clear();

    } // class ColRowCollection

    // exports ================================================================

    return ColRowCollection;

});
