/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/framework/view/drawingframe',
    ['io.ox/office/tk/utils',
     'gettext!io.ox/office/framework'
    ], function (Utils, gt) {

    'use strict';

    var // the CSS class name of the drawing content node
        CONTENT_CLASS = 'content',

        // the CSS class name of the selection root node
        SELECTION_CLASS = 'selection',

        // maps drawing object types to GUI names and icons
        OBJECT_TYPE_SETTINGS = {
            chart:    { type: gt('Chart'),   icon: 'icon-bar-chart' },
            image:    { type: gt('Image'),   icon: 'icon-picture' },
            shape:    { type: gt('Shape'),   icon: 'icon-picture' },
            diagram:  { type: gt('Diagram'), icon: 'icon-sitemap' }
        },

        DEFAULT_TYPE_SETTINGS = { type: gt('Drawing'), icon: 'icon-picture' };

    // static class DrawingFrame ==============================================

    /**
     * Contains common static helper functions to display and handle drawing
     * frames in any document.
     */
    var DrawingFrame = {};

    // constants --------------------------------------------------------------

    /**
     * The CSS class used to mark drawing frame nodes.
     *
     * @constant
     */
    DrawingFrame.NODE_CLASS = 'drawing';

    /**
     * A jQuery selector that matches nodes representing a drawing frame.
     *
     * @constant
     */
    DrawingFrame.NODE_SELECTOR = 'div.' + DrawingFrame.NODE_CLASS;

    // methods ----------------------------------------------------------------

    /**
     * Returns a new drawing frame node.
     *
     * @param {String} type
     *  The type of the drawing.
     *
     * @returns {jQuery}
     *  A new drawing frame with empty content node, as jQuery object.
     */
    DrawingFrame.createDrawingFrame = function (type) {
        var contentNode = $('<div>').addClass(CONTENT_CLASS);
        return $('<div>', { contenteditable: false }).addClass(DrawingFrame.NODE_CLASS).data('type', type).append(contentNode);
    };

    /**
     * Returns whether the passed node is the root node of a drawing frame.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is the root node of a drawing frame.
     */
    DrawingFrame.isDrawingFrame = function (node) {
        return $(node).is(DrawingFrame.NODE_SELECTOR);
    };

    /**
     * Returns the type of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {String}
     *  The type of the specified drawing frame, as specified while the frame
     *  has been created.
     */
    DrawingFrame.getDrawingType = function (drawingFrame) {
        return $(drawingFrame).first().data('type');
    };

    /**
     * Returns the root content node of the specified drawing frame containing
     * all type specific contents of the drawing.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The root content node of the specified drawing frame.
     */
    DrawingFrame.getContentNode = function (drawingFrame) {
        return $(drawingFrame).first().children('.' + CONTENT_CLASS);
    };

    /**
     * Inserts replacement layout nodes for unsupported drawing types. Inserts
     * the passed name and description of the drawing as text.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {String} [name]
     *  The name of the drawing frame.
     *
     * @param {String} [description]
     *  The description text for the drawing frame.
     */
    DrawingFrame.insertReplacementNodes = function (drawingFrame, name, description) {

        var // the type of the drawing frame
            type = DrawingFrame.getDrawingType(drawingFrame),
            typeSettings = OBJECT_TYPE_SETTINGS[type] || DEFAULT_TYPE_SETTINGS,
            // the content node
            contentNode = DrawingFrame.getContentNode(drawingFrame),
            // the inner width and height available in the content node
            innerWidth = Math.max(0, drawingFrame.outerWidth() - 2),
            innerHeight = Math.max(0, drawingFrame.outerHeight() - 2),
            // the vertical padding in the content node
            verticalPadding = Utils.minMax(Math.min(innerWidth, innerHeight) / 24, 1, 6),
            // the font size of the picture icon
            pictureIconSize = Utils.minMax(Math.min(innerWidth - 16, innerHeight - 28), 8, 72),
            // the base font size of the text
            fontSize = Utils.minMax(Math.min(innerWidth, innerHeight) / 4, 9, 13);

        // set border width at the content node, insert the picture icon
        contentNode
            .empty()
            .addClass('placeholder')
            .css({
                padding: Math.floor(verticalPadding) + 'px ' + Math.floor(verticalPadding * 2) + 'px',
                fontSize: fontSize + 'px'
            })
            .append(
                $('<div>')
                    .addClass('abs background-icon')
                    .css('line-height', innerHeight + 'px')
                    .append(Utils.createIcon(typeSettings.icon).css('font-size', pictureIconSize + 'px')),
                $('<p>').text(name || typeSettings.type)
            );

        // insert description if there is a reasonable amount of space available
        if ((innerWidth >= 20) && (innerHeight >= 20)) {
            contentNode.append($('<p>').text(description));
        }
    };

    // selection --------------------------------------------------------------

    /**
     * Returns the selection root node of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The selection root node of the specified drawing frame. Will be an
     *  empty collection, if the drawing frame is not selected currently.
     */
    DrawingFrame.getSelectionNode = function (drawingFrame) {
        return $(drawingFrame).first().children('.' + SELECTION_CLASS);
    };

    /**
     * Returns whether the specified drawing frame is currently selected.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether the specified drawing frame is selected.
     */
    DrawingFrame.isSelected = function (drawingFrame) {
        return DrawingFrame.getSelectionNode(drawingFrame).length > 0;
    };

    /**
     * Creates or updates additional nodes displayed while a drawing frame is
     * selected.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {Object} [options]
     *  A map with options to control the appearance of the selection. The
     *  following options are supported:
     *  @param {Boolean} [options.movable=false]
     *      If set to true, mouse pointer will be changed to a movable pointer
     *      while the mouse hovers the drawing.
     *  @param {Boolean} [options.resizable=false]
     *      If set to true, mouse pointer will be changed to an appropriate
     *      resize pointer while the mouse hovers a resize handle.
     *
     * @returns {jQuery}
     *  The selection root node contained in the drawing frame, as jQuery
     *  object.
     */
    DrawingFrame.drawSelection = function (drawingFrame, options) {

        var // whether the drawing is movable
            movable = Utils.getBooleanOption(options, 'movable', false),
            // whether the drawing is resizable
            resizable = Utils.getBooleanOption(options, 'resizable', false),
            // the selection root node
            selectionNode = DrawingFrame.getSelectionNode(drawingFrame);

        // create new selection nodes if missing
        if (selectionNode.length === 0) {

            // create a new selection node
            selectionNode = $('<div>').addClass(SELECTION_CLASS);
            $(drawingFrame).first().append(selectionNode);

            // create the border and tracker node
            selectionNode.append($('<div>').addClass('border'), $('<div>').addClass('tracker'));

            // create resizer handles
            _(['tl', 't', 'tr', 'r', 'br', 'b', 'bl', 'l']).each(function (pos) {
                selectionNode.append($('<div>').addClass('handle').attr('data-pos', pos));
            });
        }

        // update state classes
        return selectionNode.toggleClass('movable', movable).toggleClass('resizable', resizable);
    };

    /**
     * Removes the selection node from the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {Object} [options]
     *  A map with options to control the appearance of the selection. The
     *  following options are supported:
     *  @param {Boolean} [options.movable=false]
     *      If set to true, mouse pointer will be changed to a movable pointer
     *      while the mouse hovers the drawing.
     *  @param {Boolean} [options.resizable=false]
     *      If set to true, mouse pointer will be changed to an appropriate
     *      resize pointer while the mouse hovers a resize handle.
     *
     * @returns {jQuery}
     *  The selection root node contained in the drawing frame, as jQuery
     *  object.
     */
    DrawingFrame.clearSelection = function (drawingFrame) {
        DrawingFrame.getSelectionNode(drawingFrame).remove();
    };

    // exports ================================================================

    return DrawingFrame;

});
