/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Malte Timmermann <malte.timmermann@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/framework/model/undomanager', ['io.ox/office/tk/utils'], function (Utils) {

    'use strict';

    // class UndoAction =======================================================

    /**
     * An instance of UndoAction is used by the undo manager to store undo and
     * redo operations and to apply them in the document model.
     *
     * @constructor
     *
     * @param {Object|Object[]} [undoOperations]
     *  One ore more operations that form a logical undo action. When the undo
     *  action is executed, the operations will be applied in the exact order
     *  as passed in this parameter; they will NOT be applied in reversed order
     *  as it would be the case if the operations have been added in multiple
     *  undo actions. The parameter may be a single operation object, an array
     *  of operation objects, or omitted.
     *
     * @param {Object|Object[]} [redoOperations]
     *  One ore more operations that form a logical redo action. When the redo
     *  action is executed, the operations will be applied in the exact order
     *  as passed in this parameter. The parameter may be a single operation
     *  object, an array of operation objects, or omitted.
     */
    function UndoAction(undoOperations, redoOperations) {
        this.undoOperations = [];
        this.redoOperations = [];
        this.appendOperations(undoOperations, redoOperations);
    }

    /**
     * Appends the passed undo and redo operations to the own operations
     * arrays. Undo operations will be inserted at the beginning of the undo
     * operations array, and redo operations will be appended to the redo
     * operations array.
     */
    UndoAction.prototype.appendOperations = function (undoOperations, redoOperations) {

        // add the undo operations at the beginning of the array
        if (_.isArray(undoOperations)) {
            this.undoOperations = _.copy(undoOperations, true).concat(this.undoOperations);
        } else if (_.isObject(undoOperations)) {
            this.undoOperations.unshift(_.copy(undoOperations, true));
        }

        // add the redo operations at the end of the array
        if (_.isArray(redoOperations)) {
            this.redoOperations = this.redoOperations.concat(_.copy(redoOperations, true));
        } else if (_.isObject(redoOperations)) {
            this.redoOperations.push(_.copy(redoOperations, true));
        }
    };

    // class UndoManager ======================================================

    /**
     * The UndoManager mix-in class adds undo/redo functionality to the
     * document model. Also the grouping and merging of undo/redo operations is
     * a task of the undo manager.
     *
     * Triggers the following events:
     * - 'undo:before': Before undo operations are about to be applied by the
     *      method UndoManager.undo().
     * - 'undo:after': After undo operations have been applied by the method
     *      UndoManager.undo().
     * - 'redo:before': Before redo operations are about to be applied by the
     *      method UndoManager.redo().
     * - 'redo:after': After redo operations have been applied by the method
     *      UndoManager.redo().
     * - 'undogroup:open': After a new undo group was opened by the method
     *      UndoManager.enterUndoGroup().
     * - 'undogroup:close': After an undo group was close by the method
     *      UndoManager.enterUndoGroup().
     *
     * @constructor
     *
     * @internal
     *  This is a mix-in class supposed to extend an existing instance of the
     *  class EditModel. Expects the symbol 'this' to be bound to an instance
     *  of EditModel.
     *
     * @param {EditApplication} app
     *  The application containing this undo manager instance.
     *
     * @param {Object} [options]
     *  A map with options controlling the behavior of the undo manager. The
     *  following options are supported:
     *  @param {Function} [options.mergeUndoActionHandler]
     *      A function that will be called while adding a new undo action onto
     *      the undo stack, and that can try to merge the operations of the new
     *      undo action into the operations of the last undo action already
     *      stored in the undo stack. Receives the following parameters:
     *      (1) {UndoAction} prevAction
     *          The last undo action stored in the undo stack.
     *      (2) {UndoAction} nextAction
     *          The new action about to be pushed onto the undo stack.
     *      Must return true, if the new undo action has been merged into the
     *      existing undo action, in that case the new undo action will be
     *      discarded. All other return values result in pushing the new undo
     *      action onto the undo stack.
     */
    function UndoManager(app, options) {

        var // self reference
            self = this,

            // all undo actions
            actions = [],

            // callback function that implements merging two actions to one action
            mergeUndoActionHandler = Utils.getFunctionOption(options, 'mergeUndoActionHandler', $.noop),

            // index of the next redo action in the 'actions' array
            currentAction = 0,

            // number of nested action groups
            groupLevel = 0,

            // current undo action in grouped mode
            groupAction = null,

            // whether undo or redo operations are currently processed
            processing = false;

        // private methods ----------------------------------------------------

        /**
         * Opens a new undo group. If no undo group is open yet, triggers an
         * 'undogroup:open' event.
         */
        function openUndoGroup() {

            // notify listeners when top-level undo group has been opened
            if (groupLevel === 0) {
                self.trigger('undogroup:open');
            }

            groupLevel += 1;
        }

        /**
         * Closes the most recent undo group. If no undo group is open anymore
         * afterwards, triggers an 'undogroup:close' event.
         */
        function closeUndoGroup() {

            // push existing group action to action stack on last group level
            groupLevel -= 1;
            if ((groupLevel === 0) && groupAction) {
                pushAction(groupAction);
                groupAction = null;
            }

            // notify listeners when top-level undo group has been closed
            if (groupLevel === 0) {
                self.trigger('undogroup:close');
            }
        }

        /**
         * Pushes the passed new undo action onto the undo stack. Shortens the
         * undo stack if the current undo action is not the topmost action on
         * the stack.
         *
         * @param {UndoAction} newAction
         *  The new undo action to be pushed onto the undo stack.
         */
        function pushAction(newAction) {

            var // last action on action stack
                lastAction = null;

            // truncate main undo stack and push the new action
            if (currentAction < actions.length) {

                // remove undone actions, push new action without merging
                actions.splice(currentAction);

            } else {

                // try to merge with last action, if stack has not been truncated
                lastAction = _.last(actions);

                // try to merge an 'insertText' operation for a single character with the last action on stack
                if (_.isObject(lastAction) && (mergeUndoActionHandler.call(self, lastAction, newAction) === true)) {
                    return;
                }
            }

            actions.push(newAction);
            currentAction = actions.length;
        }

        // methods ------------------------------------------------------------

        /**
         * Clears all undo actions from the stack of the undo manager.
         *
         * @returns {UndoManager}
         *  A reference to this instance.
         */
        this.clearUndoActions = function () {
            actions = [];
            currentAction = 0;
            return this;
        };

        /**
         * Returns whether the undo manager is enabled (importing the document
         * has been finished, and the undo manager is currently not processing
         * any undo or redo operations).
         *
         * @returns {Boolean}
         *  Whether the undo manager is enabled.
         */
        this.isUndoEnabled = function () {
            return !processing && app.isImportFinished();
        };

        /**
         * Opens an undo action group and executes the specified callback
         * function. All undo operations added by the callback function will be
         * collected in the action group and act like a single undo action.
         * Nested calls are supported.
         *
         * @param {Function} callback
         *  The callback function that will be executed while the undo action
         *  group is open. If the function returns a Deferred object or a
         *  Promise, the undo group remains opened asynchronously until this
         *  object will be resolved or rejected by the callback.
         *
         * @param {Object} [context]
         *  The context the callback function will be bound to.
         *
         * @returns {Any}
         *  The return value of the callback function. In asynchronous mode,
         *  this will be the Deferred object or Promise returned by the
         *  callback function.
         */
        this.enterUndoGroup = function (callback, context) {

            var // the return value of the callback function
                result = null;

            openUndoGroup();
            result = callback.call(context);
            $.when(result).always(closeUndoGroup);

            return result;
        };

        /**
         * Creates a new undo action that will apply the passed undo and redo
         * operations in the document model.
         *
         * @constructor
         *
         * @param {Object|Object[]} [undoOperations]
         *  One ore more operations that form a logical undo action. When the
         *  undo action is executed, the operations will be applied in the
         *  exact order as passed in this parameter; they will NOT be applied
         *  in reversed order (as would be the case if the operations had been
         *  added in multiple undo actions). The parameter may be a single
         *  operation object, an array of operation objects, or omitted.
         *
         * @param {Object|Object[]} [redoOperations]
         *  One ore more operations that form a logical redo action. When the
         *  redo action is executed, the operations will be applied in the
         *  exact order as passed in this parameter. The parameter may be a
         *  single operation object, an array of operation objects, or omitted.
         *
         * @returns {UndoManager}
         *  A reference to this instance.
         */
        this.addUndo = function (undoOperations, redoOperations) {

            // check that undo manager is valid and either undo or redo operations have been passed
            if (this.isUndoEnabled() && (_.isObject(undoOperations) || _.isObject(redoOperations))) {

                if (groupLevel > 0) {
                    // active group action: insert operations into its operation arrays
                    if (groupAction) {
                        groupAction.appendOperations(undoOperations, redoOperations);
                    } else {
                        groupAction = new UndoAction(undoOperations, redoOperations);
                    }
                } else {
                    // create and insert a new action
                    pushAction(new UndoAction(undoOperations, redoOperations));
                }
            }

            return this;
        };

        /**
         * Returns the number of undo actions available on the undo stack.
         */
        this.undoAvailable = function () {
            return currentAction;
        };

        /**
         * Applies the undo operations of the specified number of undo actions.
         *
         * @param {Number} [count=1]
         *  The number of undo actions whose undo operations will be applied.
         *
         * @returns {UndoManager}
         *  A reference to this instance.
         */
        this.undo = function (count) {

            this.trigger('undo:before');

            // apply the undo operations
            processing = true;
            try {
                count = _.isNumber(count) ? count : 1;
                while ((currentAction > 0) && (count > 0)) {
                    currentAction -= 1;
                    this.applyOperations(actions[currentAction].undoOperations);
                    count -= 1;
                }
            } finally {
                processing = false;
            }

            this.trigger('undo:after');
            return this;
        };

        /**
         * Returns the number of redo actions available on the redo stack.
         */
        this.redoAvailable = function () {
            return actions.length - currentAction;
        };

        /**
         * Applies the redo operations of the specified number of undo actions.
         *
         * @param {Number} [count=1]
         *  The number of undo actions whose redo operations will be applied.
         *
         * @returns {UndoManager}
         *  A reference to this instance.
         */
        this.redo = function (count) {

            this.trigger('redo:before');

            // apply the redo operations
            processing = true;
            try {
                count = _.isNumber(count) ? count : 1;
                while ((currentAction < actions.length) && (count > 0)) {
                    this.applyOperations(actions[currentAction].redoOperations);
                    currentAction += 1;
                    count -= 1;
                }
            } finally {
                processing = false;
            }

            this.trigger('redo:after');
            return this;
        };

    } // class UndoManager

   // exports =================================================================

    return UndoManager;

});
