/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/framework/model/format/fonts',
    ['io.ox/office/tk/utils',
     'io.ox/office/framework/model/format/container'
    ], function (Utils, Container) {

    'use strict';

    var // all known fonts, mapped by font name
        PREDEFINED_FONTS = {

            'Helvetica':            { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Arial'] } },
            'Arial':                { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Helvetica'] } },
            'Verdana':              { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Arial'] } },
            'Tahoma':               { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Arial'] } },
            'Calibri':              { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Arial'] } },
            'Albany':               { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Arial'] }, hidden: true },
            'Impact':               { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Arial'] } },
            'Helvetica Neue':       { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Helvetica'] }, hidden: true },
            'Open Sans':            { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Helvetica Neue'] }, hidden: true },
            'Open Sans Light':      { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Open Sans'] }, hidden: true },
            'Open Sans Semibold':   { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Open Sans'] }, hidden: true },
            'Open Sans Extrabold':  { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Open Sans Semibold'] }, hidden: true },

            'Times':                { cssDefault: 'serif', description: { pitch: 'variable', altNames: ['Times New Roman'] }, hidden: true },
            'Times New Roman':      { cssDefault: 'serif', description: { pitch: 'variable', altNames: ['Times'] } },
            'Georgia':              { cssDefault: 'serif', description: { pitch: 'variable', altNames: ['Times New Roman'] } },
            'Palatino':             { cssDefault: 'serif', description: { pitch: 'variable', altNames: ['Times New Roman'] } },
            'Cambria':              { cssDefault: 'serif', description: { pitch: 'variable', altNames: ['Times New Roman'] } },
            'Thorndale':            { cssDefault: 'serif', description: { pitch: 'variable', altNames: ['Times New Roman'] }, hidden: true },
            'Book Antiqua':         { cssDefault: 'serif', description: { pitch: 'variable', altNames: ['Palatino'] } },

            'Courier':              { cssDefault: 'monospace', description: { pitch: 'fixed', altNames: ['Courier New'] }, hidden: true },
            'Courier New':          { cssDefault: 'monospace', description: { pitch: 'fixed', altNames: ['Courier'] } },
            'Andale Mono':          { cssDefault: 'monospace', description: { pitch: 'fixed', altNames: ['Courier New'] } },
            'Consolas':             { cssDefault: 'monospace', description: { pitch: 'fixed', altNames: ['Courier New'] } },
            'Cumberland':           { cssDefault: 'monospace', description: { pitch: 'fixed', altNames: ['Courier New'] }, hidden: true }
        };

    // class Fonts ============================================================

    /**
     * Contains the definitions of all known fonts in a document.
     *
     * @constructor
     *
     * @extends Container
     *
     * @param {EditApplication} app
     *  The root application instance.
     */
    function Fonts(app) {

        var // all registered fonts, mapped by capitalized font name
            fonts = {};

        // base constructor ---------------------------------------------------

        Container.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Removes the calculated CSS font families from all registered fonts.
         */
        function deleteAllCssFontFamilies() {
            _(fonts).each(function (font) { delete font.cssFontFamily; });
        }

        /**
         * Calculates and returns the CSS font family attribute containing the
         * specified font name and all alternative font names.
         */
        function calculateCssFontFamily(fontName) {

            var // the resulting collection of CSS font names
                cssFontNames = [fontName],
                // the last found CSS default font family
                lastCssDefault = null;

            // collects all alternative font names of the fonts in the passed array
            function collectAltNames(fontNames) {

                var // alternative font names of the passed fonts
                    altFontNames = [];

                // collect all alternative font names
                _(fontNames).each(function (fontName) {
                    if ((fontName in fonts) && _.isArray(fonts[fontName].description.altNames)) {
                        altFontNames = altFontNames.concat(fonts[fontName].description.altNames);
                        if (_.isString(fonts[fontName].cssDefault)) {
                            lastCssDefault = fonts[fontName].cssDefault;
                        }
                    }
                });

                // unify the new alternative font names and remove font names
                // already collected in cssFontNames (prevent cyclic references)
                altFontNames = _.chain(altFontNames).unique().difference(cssFontNames).value();

                // insert the new alternative font names into result list, and
                // call recursively until no new font names have been found
                if (altFontNames.length > 0) {
                    cssFontNames = cssFontNames.concat(altFontNames);
                    collectAltNames(altFontNames);
                }
            }

            // collect alternative font names by starting with passed font name
            collectAltNames([fontName]);

            // finally, add the CSS default family
            if (_.isString(lastCssDefault)) {
                cssFontNames.push(lastCssDefault);
            }

            // enclose all fonts with white-space into quote characters
            cssFontNames = _(cssFontNames).map(function (fontName) {
                return (/ /).test(fontName) ? ('"' + fontName + '"') : fontName;
            });

            // return the CSS font-family attribute value
            return cssFontNames.join(',');
        }

        // methods ------------------------------------------------------------

        /**
         * Adds a new font description to this container. An existing font with
         * the specified name will be replaced.
         *
         * @param {String} fontName
         *  The name of of the new font.
         *
         * @param {Object} description
         *  The font description of the font.
         *
         * @returns {Fonts}
         *  A reference to this instance.
         */
        this.addFont = function (fontName, description) {

            // fonts are keyed by capitalized names
            fontName = Utils.capitalizeWords(fontName);

            // store passed font attributes
            description = _.copy(description, true);
            fonts[fontName] = { description: description, hidden: false };

            // validate alternative names
            if (_.isArray(description.altNames)) {
                description.altNames = _.chain(description.altNames).without('').map(Utils.capitalizeWords).value();
            } else {
                description.altNames = [];
            }

            // add predefined alternative names
            if ((fontName in PREDEFINED_FONTS) && _.isArray(PREDEFINED_FONTS[fontName].description.altNames)) {
                description.altNames = description.altNames.concat(PREDEFINED_FONTS[fontName].description.altNames);
            }

            // unify alternative names, remove own font name
            description.altNames = _.chain(description.altNames).unique().without(fontName).value();

            // add default CSS font family if known
            if (fontName in PREDEFINED_FONTS) {
                fonts[fontName].cssDefault = PREDEFINED_FONTS[fontName].cssDefault;
            }

            // remove calculated CSS font families of all fonts, notify listeners
            deleteAllCssFontFamilies();
            this.triggerChangeEvent('add', fontName);

            return this;
        };

        /**
         * Returns the names of all predefined and registered fonts. Predefined
         * fonts with the hidden flag that have not been registered will not be
         * included.
         *
         * @returns {String[]}
         *  The names of all predefined and registered fonts, as array.
         *
         */
        this.getFontNames = function () {
            var fontNames = [];
            _(fonts).each(function (fontEntry, fontName) {
                if (!Utils.getBooleanOption(fontEntry, 'hidden', false)) {
                    fontNames.push(fontName);
                }
            });
            return fontNames;
        };

        /**
         * Returns the value of the CSS 'font-family' attribute containing the
         * specified font name and all alternative font names.
         *
         * @param {String} fontName
         *  The name of of the font (case-insensitive).
         *
         * @returns {String}
         *  The value of the CSS 'font-family' attribute containing the
         *  specified font name and all alternative font names.
         */
        this.getCssFontFamily = function (fontName) {

            // fonts are keyed by capitalized names
            fontName = Utils.capitalizeWords(fontName);

            // default for unregistered fonts: just the passed font name
            if (!(fontName in fonts)) {
                return fontName;
            }

            // calculate and cache the CSS font family
            if (!_.isString(fonts[fontName].cssFontFamily)) {
                fonts[fontName].cssFontFamily = calculateCssFontFamily(fontName);
            }

            return fonts[fontName].cssFontFamily;
        };

        // initialization -----------------------------------------------------

        // start with all predefined fonts
        fonts = _.copy(PREDEFINED_FONTS, true);

    } // class Fonts

    // static methods ---------------------------------------------------------

    /**
     * Returns the first font name specified in the passed font family string
     * received from a CSS font-family attribute. The CSS font family may
     * consist of a comma separated list of font names, where the font names
     * may be enclosed in quote characters.
     *
     * @param {String} cssFontFamily
     *  The value of a CSS font-family attribute.
     *
     * @returns {String}
     *  The first font name from the passed string, without quote characters.
     */
    Fonts.getFirstFontName = function (cssFontFamily) {

        var // extract first font name from comma separated list
            fontName = $.trim(cssFontFamily.split(',')[0]);

        // Remove leading and trailing quote characters (be tolerant and allow
        // mixed quotes such as "Font' or even a missing quote character on one
        // side of the string). After that, trim again and return.
        return Utils.capitalizeWords($.trim(fontName.match(/^["']?(.*?)["']?$/)[1]));
    };

    // exports ================================================================

    // derive this class from class Container
    return Container.extend({ constructor: Fonts });

});
