/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/framework/model/format/container',
    ['io.ox/core/event',
     'io.ox/office/tk/utils'
    ], function (Events, Utils) {

    'use strict';

    // class Container ========================================================

    /**
     * Generic base class for style sheet containers and other custom
     * formatting containers stored in an instance of the DocumentStyles class.
     * Provides helper functions and event functionality useful in any derived
     * class.
     *
     * @constructor
     *
     * @param {EditApplication} app
     *  The root application instance.
     */
    function Container(app) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        // add event hub (mix-in class)
        Events.extend(this);

        // methods ------------------------------------------------------------

        /**
         * Immediately triggers the specified event for this container, and
         * prepares to trigger a deferred 'change' event that will be triggered
         * once after the current script has been executed. This method must be
         * called always after contents of this container have been changed,
         * added, or removed.
         *
         * Note: This method DOES NOT trigger any events while the document is
         * imported.
         *
         * @returns {Container}
         *  A reference to this instance.
         */
        this.triggerChangeEvent = function () {
            // do not trigger any events while importing the document
            return this;
        };

        this.destroy = function () {
            this.events.destroy();
        };

        // initialization -----------------------------------------------------

        // create real implementation of Container.triggerChangeEvent() after import
        app.on('docs:import:success', function () {

            // direct callback: called every time when Container.triggerChangeEvent() has been called
            function triggerDirectEvent(type, data) {
                return self.trigger(type, data);
            }

            // deferred callback: called once, after current script ends
            function triggerDeferredEvent() {
                self.trigger('change');
            }

            // create and return the debounced Container.triggerChangeEvent() method
            self.triggerChangeEvent = app.createDebouncedMethod(triggerDirectEvent, triggerDeferredEvent);
        });

    } // class Container

    // exports ================================================================

    return _.makeExtendable(Container);

});
