/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/framework/model/editmodel',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/config',
     'io.ox/office/framework/model/basemodel',
     'io.ox/office/framework/model/undomanager',
     'io.ox/office/framework/model/operationsgenerator'
    ], function (Utils, Config, BaseModel, UndoManager, OperationsGenerator) {

    'use strict';

    // class EditModel ========================================================

    /**
     * The base class for editable document models. Adds generic support for
     * operations and undo action management.
     *
     * Triggers the events supported by the mix-in class UndoManager, and the
     * following additional events:
     * - 'operations:before': Before an array of operations will be applied via
     *      the method EditModel.applyOperations(). The event handler receives
     *      the operations array, and the external state passed to the method.
     *      If a single operation has been passed to that method, it will be
     *      converted to an array with one element before calling the event
     *      handler.
     * - 'operations:after': After an array of operations have been applied via
     *      the method EditModel.applyOperations(), regardless if successfully
     *      or not. The event handler receives the external state passed to the
     *      method EditModel.applyOperations().
     * - 'operations:success': Directly after the event 'operations:after', if
     *      the operations have been applied successfully. The event handler
     *      receives the operations array, and the external state passed to the
     *      method EditModel.applyOperations(). If a single operation has been
     *      passed to that method, it will be converted to an array with one
     *      element before calling the event handler.
     * - 'operations:error': Directly after the event 'operations:after', if
     *      applying the operations has failed. The event handler receives the
     *      external state passed to the method EditModel.applyOperations().
     * - 'change:editmode': When the edit mode of the document model has been
     *      changed. Event handlers receive the new state of the edit mode.
     *
     * @constructor
     *
     * @extends BaseModel
     * @extends UndoManager
     *
     * @param {EditApplication} app
     *  The application containing this document model instance.
     *
     * @param {Function} OperationsGeneratorClass
     *  The constructor function of the operations generator. Must be a sub
     *  class of class OperationsGenerator.
     *
     * @param {DocumentStyles} documentStyles
     *  Global collection with the style sheet containers and custom formatting
     *  containers of a document.
     *
     * @param {Object} [options]
     *  Additional options that will be passed to the UndoManager mix-in class.
     */
    function EditModel(app, OperationsGeneratorClass, documentStyles, options) {

        var // self reference for local functions
            self = this,

            // maps all operation names to operation handler functions
            operationHandlers = {},

            // whether operations are currently applied (prevent recursive calls)
            processingOperations = false,

            // edit mode, received dynamically from server (one editing user at a time)
            editMode = null,

            // the total number of operations applied successfully
            totalOperations = 0,

            // a counter for the operations, that adds all single operations, taking care of merged
            // operations. So this number must be equal on every client, independent from merge
            // processes on client or server side.
            operationStateNumber = -1;

        // base constructors --------------------------------------------------

        BaseModel.call(this, app);
        // add undo manager (mix-in class)
        UndoManager.call(this, app, options);

        // private methods ----------------------------------------------------

        /**
         * Called before an operations array will be applied at the document
         * model. Triggers an 'operations:before' event.
         *
         * @param {Array} operations
         *  The operations to be applied. Will be passed to the listeners of
         *  the 'operations:before' event.
         *
         * @param {Boolean} external
         *  Whether the operations have been received from the server. Will be
         *  passed to the listeners of the 'operations:before' event.
         *
         * @returns {Boolean}
         *  False if this method has been called recursively, i.e. the model
         *  already processes operations. This is considered a fatal error.
         */
        function beginProcessOperations(operations, external) {

            if (processingOperations) {
                Utils.error('EditModel.beginProcessOperations(): recursive call - currently processing operations.');
                return false;
            }

            self.trigger('operations:before', operations, external);
            processingOperations = true;
            return true;
        }

        /**
         * Called after an operations array has been applied at the document
         * model. Triggers an 'operations:after' event, and one of the
         * 'operations:success' or 'operations:error' events.
         *
         * @param {Boolean} success
         *  Whether applying the operations has succeeded.
         *
         * @param {Array} operations
         *  The operations that have been applied. Will be passed to the
         *  listeners of the operations events.
         *
         * @param {Boolean} external
         *  Whether the operations have been received from the server. Will be
         *  passed to the listeners of the operations events.
         */
        function endProcessOperations(success, operations, external) {
            processingOperations = false;
            self.trigger('operations:after', operations, external);
            if (success) {
                self.trigger('operations:success', operations, external);
            } else {
                self.setEditMode(false);
                self.trigger('operations:error', operations, external);
            }
        }

        /**
         * Handles a number for the operation state to find problems
         * in the communication between different clients earlier. This
         * is currently only used in debug mode.
         *
         * @param {Object} operation
         *  A single operation to be applied at the document model.
         *
         * @param {Boolean} external
         *  Whether the operation has been received from the server.
         */
        function handleOperationStateNumber(operation, external) {

            // checking the correct operation state number of an external operation
            // The operation state number must be identical on every client.
            // The operation state number is sent from server, when the document is loaded
            if (external) {

                // During loading the document, the initial operation state number
                // is sent by the server. If we don't have a valid value use it.
                if ((operation.osn >= 0) && (operationStateNumber === -1)) {
                    operationStateNumber = operation.osn;
                }

                // check for the correct operation state number of an external operation
                if (_.isNumber(operation.osn) && (operationStateNumber !== operation.osn)) {
                    // If this client is already synchronized -> there is a conflict in the operation state number
                    Utils.error('EditModel.applyOperation(): Wrong document state. Operation expected state ' + operation.osn + '. But local state is: ' + operationStateNumber);
                    return false;
                }
            } else {
                // Preparing internal operations, so that remote clients can control the correct operation state number.
                // A. setting current operation state number as required base for this operation
                operation.osn = operationStateNumber;
                // B. setting length of this operation -> this might be increased by operation merge processes
                operation.opl = 1;
            }

            if (operationStateNumber >= 0) {
                // increasing the operation state number of the document
                operationStateNumber = operationStateNumber + (operation.opl || 1);
            }

            return true;
        }

        /**
         * Executes the handler function for the passed operation.
         *
         * @param {Object} operation
         *  A single operation to be applied at the document model.
         *
         * @param {Boolean} external
         *  Whether the operation has been received from the server.
         *
         * @returns {Boolean}
         *  Whether the operation has been applied successfully.
         */
        function applyOperation(operation, external) {

            var // the function that executes the operation
                operationHandler = null,
                // success value of operation
                success = false;

            // check operation
            if (!_.isObject(operation) || !_.isString(operation.name)) {
                Utils.error('EditModel.applyOperation(): expecting operation object.');
                return false;
            }

            // handle an operation state number. This number needs to be synchronized
            // between all clients and has then to be identical for all clients.
            if (!handleOperationStateNumber(operation, external)) {
                return false;
            }

            // get and check operation handler
            operationHandler = operationHandlers[operation.name];
            if (!_.isFunction(operationHandler)) {
                Utils.error('EditModel.applyOperation(): invalid operation name "' + operation.name + '".');
                return false;
            }

            // execute the operation handler
            try {
                success = operationHandler.call(self, operation, external) !== false;
            } catch (ex) {
                Utils.exception(ex);
            }

            if (success) {
                totalOperations += 1;
            } else {
                Utils.error('EditModel.applyOperation(): operation ' + (totalOperations + 1) + ' failed: ' + JSON.stringify(operation));
            }
            return success;
        }

        /**
         * Executes the handler functions for all passed operations. If the
         * undo manager is active, all applied operations will be embedded into
         * a single undo group action.
         *
         * @param {Array} operations
         *  The operations to be applied at the document model.
         *
         * @returns {Boolean}
         *  Whether applying all operations was successful.
         */
        function applyOperationsSync(operations, external) {

            var // the boolean result for synchronous mode
                success = false;

            // if the operation's origin is not an external client and
            // we don't have edit rights then we skip the operations silently.
            if (!external && !editMode) {
                return true;
            }

            // enter operations mode (prevent recursive calls)
            if (beginProcessOperations(operations, external)) {
                // apply all operations at once, group into an undo action
                self.enterUndoGroup(function () {
                    success = _(operations).all(function (operation) {
                        return applyOperation(operation, external);
                    });
                });
            }

            // post-processing (trigger end events, leave operations mode),
            // also if beginProcessOperations() has failed (notify the error)
            endProcessOperations(success, operations, external);
            return success;
        }

        /**
         * Executes the handler functions for all passed operations in an
         * asynchronous background loop. If the undo manager is active, all
         * applied operations will be embedded into a single undo group action.
         *
         * @param {Array} operations
         *  The operations to be applied at the document model.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved after all
         *  operations have been applied successfully, or rejected immediately
         *  after applying an operation has failed. The Promise regularly sends
         *  progress updates while applying the operations.
         */
        function applyOperationsAsync(operations, external, delay) {

            var // the result Promise for asynchronous mode
                promise = null;

            // enter operations mode (prevent recursive calls)
            if (beginProcessOperations(operations, external)) {
                // apply all operations in a background task, group into an undo action
                promise = self.enterUndoGroup(function () {
                    // return the Deferred to enterUndoGroup() to defer the open undo group
                    return app.processArrayDelayed(function (chunk) {

                        // if the operation's origin is not an external client and
                        // we don't have edit rights then we skip the operations silently.
                        if (!external && !editMode) {
                            return Utils.BREAK;
                        }

                        return _(chunk).all(function (operation) {
                            return applyOperation(operation, external);
                        }) ? undefined : $.Deferred().reject();
                    }, operations, { chunkLength: 50, delay: delay });
                }, undefined, { async: true });
            } else {
                // currently applying operations (called recursively)
                promise = $.Deferred().reject();
            }

            // post-processing (trigger end events, leave operations mode)
            return promise.done(function () {
                endProcessOperations(true, operations, external);
            }).fail(function () {
                endProcessOperations(false, operations, external);
            });
        }

        // methods ------------------------------------------------------------

        /**
         * Registers a function that will be executed when an operation with
         * the specified name will be applied.
         *
         * @param {String} name
         *  The name of the operation, as contained in the 'name' attribute of
         *  the operation object.
         *
         * @param {Function} handler
         *  The function that will be executed for the operation. All operation
         *  handler functions will be called in the context of this document
         *  model instance. Receives the operation object as first parameter,
         *  and the external state that has been passed to the method
         *  EditModel.applyOperations() as second parameter. If the operation
         *  handler returns the boolean value false, further processing of
         *  operations will be stopped immediately, and an 'operations:error'
         *  event will be triggered. Every other return value (also undefined)
         *  indicates that the operation has been applied successfully.
         *
         * @returns {EditModel}
         *  A reference to this instance.
         */
        this.registerOperationHandler = function (name, handler) {
            operationHandlers[name] = handler;
            return this;
        };

        /**
         * Creates and returns an initialized operations generator instance (an
         * instance of the class specified in the constructor parameter
         * OperationsGeneratorClass).
         *
         * @returns {OperationsGenerator}
         *  A new operations generator instance.
         */
        this.createOperationsGenerator = function () {
            return new OperationsGeneratorClass(documentStyles);
        };

        /**
         * Executes the handler functions for all passed operations. If the
         * undo manager is active, all applied operations will be embedded into
         * a single undo group action.
         *
         * @param {Object|Array} operations
         *  A single operation, or an array with operations to be applied at
         *  the document model.
         *
         * @returns {Boolean}
         *  Whether applying all operations was successful.
         */
        this.applyOperations = function (operations) {
            // convert parameter to an array
            operations = _(operations).getArray();
            // do not trigger any events if array is empty
            return (operations.length === 0) || applyOperationsSync(operations, false);
        };

        /**
         * Executes the handler functions for all operations contained in the
         * passed actions. The operations of each action will be enclosed into
         * distinct undo groups automatically.
         *
         * @param {Object|Array} actions
         *  A single action, or an array with actions containing operations to
         *  be applied at the document model.
         *
         * @param {Object} [options]
         *  A map with additional options. The following options are supported:
         *  @param {Boolean} [options.external=false]
         *      If set to true, the actions have been received from an external
         *      sender. Otherwise, the actions have been generated by the
         *      document model itself.
         *  @param {Boolean} [options.async=false]
         *      If set to true, the actions will be applied asynchronously in a
         *      browser timeout loop. The Promise returned by this method will
         *      be resolved or rejected after applying all actions, and will be
         *      notified about the progress of the operation.
         *  @param {Number} [options.delay=0]
         *      In asynchronous mode, the delay time between actions and
         *      between chunks of multiple operations in a single action.
         *
         * @returns {Boolean|jQuery.Promise}
         *  In synchronous mode (option 'options.async' is false), returns the
         *  boolean value true, if applying all actions was successful, or
         *  false, if the operations of any action has failed. In asynchronous
         *  mode (option 'options.async' is true), returns the Promise of a
         *  Deferred object that will be resolved after all actions have been
         *  applied successfully, or rejected immediately after applying an
         *  action has failed. The caller has to wait for the Promise before
         *  new actions can be applied. The Promise regularly sends progress
         *  updates while applying the actions.
         */
        this.applyActions = function (actions, options) {

            var // the result Deferred object for asynchronous mode
                def = null,
                // the timer processing the array asynchronously
                timer = null,
                // whether to execute asynchronously
                async = Utils.getBooleanOption(options, 'async', false),
                // delay time for asynchronous mode
                delay = Utils.getIntegerOption(options, 'delay', 0, 0),
                // whether the actions have been received from the server
                external = Utils.getBooleanOption(options, 'external', false),
                // total number of operations
                operationCount = 0,
                // number of operations already applied
                operationIndex = 0;

            // convert parameter to an array, filter for actions containing
            // at least one operation, get total number of operations
            actions = _.chain(actions).getArray().filter(function (action) {
                var count = _.isArray(action.operations) ? action.operations.length : 0;
                operationCount += count;
                return count > 0;
            }).value();

            // no operations: early exit
            if (operationCount === 0) {
                return async ? $.when() : true;
            }

            // apply actions, get resulting Deferred object
            if (async) {

                // create an own Deferred object, for custom progress calculation
                def = $.Deferred();

                // apply the actions in a browser timeout loop
                timer = app.processArrayDelayed(function (action) {

                    var // the operations contained in the action (action is passed as one-element array)
                        operations = action[0].operations;

                    // apply the operations and calculate the progress
                    return applyOperationsAsync(operations, external, delay)
                        .progress(function (progress) {
                            def.notify((operationIndex + operations.length * progress) / operationCount);
                        })
                        .done(function () {
                            operationIndex += operations.length;
                        });
                }, actions, { chunkLength: 1, delay: delay });

                // forward the result of the timer to the own Deferred object
                timer.done(function () { def.notify(1).resolve(); }).fail(function () { def.reject(); });
                return def.promise();
            }

            // apply all actions at once, but bail out on error
            return _(actions).all(function (action) {
                // applyOperationsSync() creates undo groups by itself
                return applyOperationsSync(action.operations, external);
            });
        };

        /**
         * Returns whether operations are currently processed.
         *
         * @returns {Boolean}
         *  Whether the model processes operations.
         */
        this.isProcessingOperations = function () {
            return processingOperations;
        };

        /**
         * Returns whether the document can be modified by the current user.
         *
         * @returns {Boolean}
         *  Whether the document can be modified.
         */
        this.getEditMode = function () {
            return editMode;
        };

        /**
         * Enables or disables the edit mode of this document model.
         *
         * @param {Boolean} state
         *  The new edit mode state. If set to true, the user can modify the
         *  document, otherwise the user can view the document contents, but
         *  cannot edit them.
         *
         * @returns {EditModel}
         *  A reference to this instance.
         */
        this.setEditMode = function (state) {
            if (editMode !== state) {
                editMode = state;
                // notify state change to listeners
                this.trigger('change:editmode', editMode);
            }
            return this;
        };

        /**
         * Returns the global collection with the style sheet containers and
         * custom formatting containers of a document.
         *
         * @returns {DocumentStyles}
         *  The global document styles collection of the document.
         */
        this.getDocumentStyles = function () {
            return documentStyles;
        };

        /**
         * Returns the style sheet container for the passed style family.
         *
         * @param {String} family
         *  The name of the attribute family.
         *
         * @returns {StyleSheets}
         *  The style sheets container registered for the passed style family.
         */
        this.getStyleSheets = function (styleFamily) {
            return documentStyles.getStyleSheets(styleFamily);
        };

        /**
         * Returns the fonts container.
         */
        this.getFonts = function () {
            return documentStyles.getContainer('fonts');
        };

        /**
         * Returns the themes container.
         */
        this.getThemes = function () {
            return documentStyles.getContainer('themes');
        };

        /**
         * Returns the total number of operations applied successfully.
         */
        this.getOperationsCount = function () {
            return totalOperations;
        };

        /**
         * Returns the operation state number of the document.
         */
        this.getOperationStateNumber = function () {
            return operationStateNumber;
        };

        /**
         * Destroys this document model instance.
         */
        this.destroy = (function () {
            var destroyMethod = self.destroy;
            return function () {
                documentStyles.destroy();
                documentStyles = null;
                destroyMethod.call(self);
            };
        }());

        // operation handlers -------------------------------------------------

        // setting default attributes is not undoable
        this.registerOperationHandler(OperationsGenerator.SET_DOCUMENT_ATTRIBUTES, function (operation) {

            var // passed attributes from operation
                attributes = _.isObject(operation.attrs) ? operation.attrs : {};

            // global document attributes
            documentStyles.setAttributes(attributes.document);

            // default attribute values of other style families
            _(attributes).each(function (defaultValues, family) {
                var styleSheets = documentStyles.getStyleSheets(family);
                if (styleSheets) {
                    styleSheets.setAttributeDefaultValues(defaultValues);
                }
            });
        });

        this.registerOperationHandler(OperationsGenerator.INSERT_THEME, function (operation) {
            this.getThemes().addTheme(operation.themeName, operation.attrs);
        });

        this.registerOperationHandler(OperationsGenerator.INSERT_FONT_DESCRIPTION, function (operation) {
            this.getFonts().addFont(operation.fontName, operation.attrs);
        });

        this.registerOperationHandler(OperationsGenerator.INSERT_STYLESHEET, function (operation) {

            var // target style sheet container
                styleSheets = this.getStyleSheets(operation.type),
                // passed attributes from operation
                attributes = _.isObject(operation.attrs) ? operation.attrs : {};

            if (!_.isObject(styleSheets)) {
                Utils.warn('EditModel.insertStyleSheet(): invalid style family: "' + operation.type + '"');
                return false;
            }

            // TODO: undo for insertion, needs implementation of DELETE_STYLESHEET
            // if (this.isUndoEnabled()) {
            //     this.addUndo({ name: Operations.DELETE_STYLESHEET, type: operation.type, styleId: operation.styleId }, operation);
            // }

            styleSheets.addStyleSheet(operation.styleId, operation.styleName, operation.parent, attributes,
                { hidden: operation.hidden, priority: operation.uiPriority, defStyle: operation['default'] });
        });

        this.registerOperationHandler(OperationsGenerator.DELETE_STYLESHEET, function (operation) {
            // TODO: implement
        });

        this.registerOperationHandler(OperationsGenerator.NOOP, $.noop);

    } // class EditModel

    // exports ================================================================

    // derive this class from class BaseModel
    return BaseModel.extend({ constructor: EditModel });

});
