/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/text/format/tablestyles',
    ['io.ox/office/tk/utils',
     'io.ox/office/framework/model/format/color',
     'io.ox/office/framework/model/format/border',
     'io.ox/office/framework/model/format/lineheight',
     'io.ox/office/framework/model/format/stylesheets',
     'io.ox/office/text/dom',
     'io.ox/office/text/table'
    ], function (Utils, Color, Border, LineHeight, StyleSheets, DOM, Table) {

    'use strict';

    var // definitions for table attributes
        DEFINITIONS = {

            /**
             * Width of the table, as number in 1/100 of millimeters.
             */
            width: {
                def: 'auto',
                format: function (element, width) {
                    if (width === 'auto') {
                        element.css('width', '100%');
                    } else {
                        element.css('width', Utils.convertHmmToCssLength(width, 'px', 1));
                    }
                }
            },

            /**
             * Fill color of the table.
             */
            fillColor: {
                def: Color.AUTO,
                format: function (element, color) {
                    element.css('background-color', this.getCssColor(color, 'fill'));
                },
                preview: function (element, color) {
                    if (!Color.isTransparentColor(color, 'fill')) {
                        element.css('background-color', this.getCssColor(color, 'fill'));
                    }
                }
            },

            /**
             * Grid width of columns in relative units. It is an array of numbers
             */
            tableGrid: {
                def: [],
                scope: 'element'
            },

            /**
             * Array containing information, if conditional attributes will be
             * used. As default value, all styles will be used, so that this
             * array can be empty.
             */
            exclude: {
                def: [],
                scope: 'element'
            },

            /**
             * Left border of the table (will be set in the table cells).
             */
            borderLeft: { def: Border.NONE },

            /**
             * Top border of the table (will be set in the table cells).
             */
            borderTop: { def: Border.NONE },

            /**
             * Right border of the table (will be set in the table cells).
             */
            borderRight: { def: Border.NONE },

            /**
             * Bottom border of the table (will be set in the table cells).
             */
            borderBottom: { def: Border.NONE },

            /**
             * Inner horizontal borders inside the table (will be set in the
             * table cells).
             */
            borderInsideHor: { def: Border.NONE },

            /**
             * Inner vertical borders inside the table (will be set in the
             * table cells).
             */
            borderInsideVert: { def: Border.NONE },

            /**
             * Top padding of the whole table
             */
            paddingTop: { def: 0 },

            /**
             * Bottom padding of the whole table
             */
            paddingBottom: { def: 0 },

            /**
             * Left padding of the whole table
             */
            paddingLeft: { def: 0 },

            /**
             * Right padding of the whole table
             */
            paddingRight: { def: 0 }

        },

        // the names of all border properties in cells, mapped by border mode properties
        CELL_BORDER_ATTRIBUTES = { left: 'borderLeft', right: 'borderRight', top: 'borderTop', bottom: 'borderBottom', insideh: 'borderInsideHor', insidev: 'borderInsideVert' },

        // the conditional table attributes (only 'wholeTable' has no geometric dependencies)
        GEOMETRIC_TABLE_ATTRIBUTES = ['firstRow', 'lastRow', 'firstCol', 'lastCol', 'band1Vert', 'band2Vert', 'band1Hor', 'band2Hor', 'northEastCell', 'northWestCell', 'southEastCell', 'southWestCell'];

    // class TableStyles ======================================================

    /**
     * Contains the style sheets for table formatting attributes. The CSS
     * formatting will be written to table elements and their rows and cells.
     *
     * @constructor
     *
     * @extends StyleSheets
     *
     * @param {TextApplication} app
     *  The root application instance.
     *
     * @param {DocumentStyles} documentStyles
     *  Collection with the style containers of all style families.
     */
    function TableStyles(app, documentStyles) {

        // base constructor ---------------------------------------------------

        StyleSheets.call(this, app, documentStyles, 'table', DEFINITIONS, {
            styleAttributesResolver: resolveTableStyleAttributes,
            elementAttributesResolver: resolveTableElementAttributes,
            formatHandler: updateTableFormatting,
            previewHandler: updateTablePreviewFormatting
        });

        // private methods ----------------------------------------------------

        /**
         * Returns the attributes of the specified attribute family contained
         * in table style sheets. Resolves the conditional attributes that
         * match the position of the passed source element.
         *
         * @param {Object} styleAttributes
         *  The complete 'attributes' object of a table style sheet.
         *
         * @param {jQuery} tableNode
         *  The DOM table node referring to a table style sheet, as jQuery
         *  object.
         *
         * @param {jQuery} [cellNode]
         *  The DOM cell node that has initially requested the formatting
         *  attributes of a table style sheet, as jQuery object.
         *
         * @returns {Object}
         *  The formatting attributes extracted from the passed style sheet
         *  attributes object, as map of attribute value maps (name/value
         *  pairs), keyed by attribute family.
         */
        function resolveTableStyleAttributes(styleAttributes, tableNode, cellNode) {

            var // table size
                lastTableRow = Table.getRowCount(tableNode) - 1,
                lastTableCol = Table.getColumnCount(tableNode) - 1,

                // the grid row range covered by the cell
                rowRange = (cellNode.length > 0) ? Table.getGridRowRangeOfCell(cellNode) : null,
                isFirstRow = rowRange && (rowRange.start === 0),
                isLastRow = rowRange && (rowRange.end === lastTableRow),

                // the grid grid column range covered by the cell
                colRange = (cellNode.length > 0) ? Table.getGridColumnRangeOfCell(cellNode) : null,
                isFirstCol = colRange && (colRange.start === 0),
                isLastCol = colRange && (colRange.end === lastTableCol),

                // the excluded conditional keys
                excludedConditionalKeys = StyleSheets.getExplicitAttributes(tableNode, 'table').exclude,
                firstRowIncluded = isConditionalKeyIncluded('firstRow'),
                lastRowIncluded = isConditionalKeyIncluded('lastRow'),
                firstColIncluded = isConditionalKeyIncluded('firstCol'),
                lastColIncluded = isConditionalKeyIncluded('lastCol'),

                // conditional key of the inner horizontal/vertical bands, according to cell position
                bandKey = null, bandSize = 1, bandIndex = 0,

                // the extracted style attributes according to the position of the table cell
                attributes = {};

            // whether the passed conditional key is included according to the 'exclude' attribute
            function isConditionalKeyIncluded(conditionalKey) {
                return !_.isArray(excludedConditionalKeys) || !_(excludedConditionalKeys).contains(conditionalKey);
            }

            // copies a border attribute (table or cell) to an outer cell border attribute
            function updateOuterCellBorder(sourceAttributes, outerBorderName, innerBorderName, isOuterBorder) {

                var // table and cell attributes (either may be missing)
                    tableAttributes = _.isObject(sourceAttributes.table) ? sourceAttributes.table : {},
                    cellAttributes = _.isObject(sourceAttributes.cell) ? sourceAttributes.cell : {},
                    // the source border attribute value
                    border = null;

                if (isOuterBorder) {
                    // copy outer table border to cell border if border is missing in cell attributes
                    border = _.isObject(cellAttributes[outerBorderName]) ? null : tableAttributes[outerBorderName];
                } else {
                    // copy inner table or cell border (cell border wins) to outer cell border
                    border = _.isObject(cellAttributes[innerBorderName]) ? cellAttributes[innerBorderName] : tableAttributes[innerBorderName];
                }

                // insert existing border to the specified outer cell border
                if (_.isObject(border)) {
                    (attributes.cell || (attributes.cell = {}))[outerBorderName] = border;
                }
            }

            // copies a padding attribute from a table to a cell, if required
            function updateCellPadding(sourceAttributes, paddingName) {

                var // table and cell attributes (either may be missing)
                    tableAttributes = _.isObject(sourceAttributes.table) ? sourceAttributes.table : {},
                    cellAttributes = _.isObject(sourceAttributes.cell) ? sourceAttributes.cell : {},
                    // the padding attribute value
                    padding = (paddingName in cellAttributes) ? cellAttributes[paddingName] : tableAttributes[paddingName];

                // insert existing border to the specified outer cell border
                if (_.isNumber(padding)) {
                    (attributes.cell || (attributes.cell = {}))[paddingName] = padding;
                }
            }

            // merges the specified conditional attributes into the 'attributes' object
            function mergeConditionalAttributes(conditionalKey, isHorizontalBand, isVerticalBand) {

                var // the attributes at the passed conditional key
                    conditionalAttributes = styleAttributes[conditionalKey];

                if (_.isObject(conditionalAttributes)) {

                    // copy all attributes from the style sheet to the result object
                    documentStyles.extendAttributes(attributes, conditionalAttributes);

                    // copy inner borders to outer cell borders, if cell is located inside the current table area
                    updateOuterCellBorder(conditionalAttributes, 'borderTop', 'borderInsideHor', !isVerticalBand || isFirstRow);
                    updateOuterCellBorder(conditionalAttributes, 'borderBottom', 'borderInsideHor', !isVerticalBand || isLastRow);
                    updateOuterCellBorder(conditionalAttributes, 'borderLeft', 'borderInsideVert', !isHorizontalBand || isFirstCol);
                    updateOuterCellBorder(conditionalAttributes, 'borderRight', 'borderInsideVert', !isHorizontalBand || isLastCol);

                    updateCellPadding(conditionalAttributes, 'paddingTop');
                    updateCellPadding(conditionalAttributes, 'paddingBottom');
                    updateCellPadding(conditionalAttributes, 'paddingLeft');
                    updateCellPadding(conditionalAttributes, 'paddingRight');
                }
            }

            // wholeTable: always
            mergeConditionalAttributes('wholeTable', true, true);

            // inner horizontal bands
            if (rowRange && !(firstRowIncluded && isFirstRow) && !(lastRowIncluded && isLastRow) && isConditionalKeyIncluded('bandsHor')) {
                // size of horizontal bands, TODO: replace '1' with attribute value
                bandSize = Math.floor(Math.max(1, 1));
                // ignore first row in calculation of band index
                bandIndex = Math.floor((rowRange.start - (firstRowIncluded ? 1 : 0)) / bandSize);
                // resolve odd or even band attributes
                bandKey = ((bandIndex % 2) === 0) ? 'band1Hor' : 'band2Hor';
                mergeConditionalAttributes(bandKey, true, false);
            }

            // inner vertical bands
            if (colRange && !(firstColIncluded && isFirstCol) && !(lastColIncluded && isLastCol) && isConditionalKeyIncluded('bandsVert')) {
                // size of vertical bands, TODO: replace '1' with attribute value
                bandSize = Math.floor(Math.max(1, 1));
                // ignore first column in calculation of band index
                bandIndex = Math.floor((colRange.start - (firstColIncluded ? 1 : 0)) / bandSize);
                // resolve odd or even band attributes
                bandKey = ((bandIndex % 2) === 0) ? 'band1Vert' : 'band2Vert';
                mergeConditionalAttributes(bandKey, false, true);
            }

            // cell inside first/last row/column
            if (firstColIncluded && isFirstCol) {
                mergeConditionalAttributes('firstCol', false, true);
            }
            if (lastColIncluded && isLastCol) {
                mergeConditionalAttributes('lastCol', false, true);
            }
            if (firstRowIncluded && isFirstRow) {
                mergeConditionalAttributes('firstRow', true, false);
            }
            if (lastRowIncluded && isLastRow) {
                mergeConditionalAttributes('lastRow', true, false);
            }

            // single corner cells (only if inside active first/last row AND column areas)
            if (firstRowIncluded && firstColIncluded && isFirstRow && isFirstCol) {
                mergeConditionalAttributes('northWestCell', false, false);
            }
            if (firstRowIncluded && lastColIncluded && isFirstRow && isLastCol) {
                mergeConditionalAttributes('northEastCell', false, false);
            }
            if (lastRowIncluded && firstColIncluded && isLastRow && isFirstCol) {
                mergeConditionalAttributes('southWestCell', false, false);
            }
            if (lastRowIncluded && lastColIncluded && isLastRow && isLastCol) {
                mergeConditionalAttributes('southEastCell', false, false);
            }

            return attributes;
        }

        /**
         * Returns the attributes of the specified attribute family contained
         * in table style sheets. Resolves the conditional attributes that
         * match the position of the passed source element.
         *
         * @param {Object} elementAttributes
         *  The explicit attributes of the table element, as map of attribute
         *  value maps (name/value pairs), keyed by attribute family.
         *
         * @param {jQuery} tableNode
         *  The DOM table node, as jQuery object.
         *
         * @param {jQuery} [cellNode]
         *  The DOM cell node that has initially requested the formatting
         *  attributes, as jQuery object.
         *
         * @returns {Object}
         *  The resolved explicit formatting attributes, as map of attribute
         *  value maps (name/value pairs), keyed by attribute family.
         */
        function resolveTableElementAttributes(elementAttributes, tableNode, cellNode) {
            return resolveTableStyleAttributes({ wholeTable: elementAttributes }, tableNode, cellNode);
        }

        /**
         * Will be called for every table element whose attributes have been
         * changed. Repositions and reformats the table according to the passed
         * attributes.
         *
         * @param {jQuery} table
         *  The <table> element whose table attributes have been changed, as
         *  jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute value maps (name/value pairs), keyed by
         *  attribute family, containing the effective attribute values merged
         *  from style sheets and explicit attributes.
         *
         * @param {Boolean} async
         *  If set to true, the table formatting will be updated asynchronously
         *  in a browser timeout loop.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  the table is completely formatted.
         */
        function updateTableFormatting(table, mergedAttributes, async) {

            var // the table styles/formatter
                tableStyles = documentStyles.getStyleSheets('table'),
                // the table row styles/formatter
                tableRowStyles = documentStyles.getStyleSheets('row'),
                // the table cell styles/formatter
                tableCellStyles = documentStyles.getStyleSheets('cell'),
                // the paragraph styles/formatter
                paragraphStyles = documentStyles.getStyleSheets('paragraph'),
                // the column widths
                tableGrid = mergedAttributes.table.tableGrid,
                // the explicit table attributes
                tableAttributes = StyleSheets.getExplicitAttributes(table).table,
                // all row nodes in the table
                rowNodes = DOM.getTableRows(table),
                // the number of rows updated at a time
                chunkLength = 0,
                // whether the table has a 'simple' style
                isSimpleTableStyle = false,
                // whether this update was triggered by an GUI event
                isGUIOperation = false,
                // whether cells were added to the table or removed from the table
                isGUIInsertOperation = false,
                // the result Deferred object
                def = null;

            // checking if the attributes have table geometric dependencies
            // -> then a complete reformatting of the table is required.
            // Otherwise only the new cells (insert operations) or no cells
            // (delete operations) need to be formatted.
            function checkSimpleTableStyle() {
                var isSimple = true,
                    styleSheetAttributeMap = tableStyles.getStyleSheetAttributeMap(mergedAttributes.styleId);
                _.each(GEOMETRIC_TABLE_ATTRIBUTES, function (attribute) {
                    if ((attribute in styleSheetAttributeMap) && (! _.contains(tableAttributes.exclude, attribute))) {
                        isSimple = false;
                    }
                });
                return isSimple;
            }

            // checking if the attributes have table geometric dependencies
            if (mergedAttributes.styleId) {
                isSimpleTableStyle = checkSimpleTableStyle();
            }

            if (table.data('gui')) {
                isGUIOperation = true;
                if (table.data('gui') === 'insert') {
                    isGUIInsertOperation = true;
                }
                table.removeData('gui');
            }

            // updates the formatting of the passed table rows
            function updateRowsFormatting(currentRowNodes) {
                currentRowNodes.each(function () {

                    // update the table row itself
                    tableRowStyles.updateElementFormatting(this);

                    // update table cells (do NOT pass merged attributes passed
                    // to the updateTableFormatting() method as base attributes,
                    // attributes must be recalculated for each single cell!)
                    $(this).children('td').each(function () {
                        var cellFormatting = true;
                        // updating only new cells, if this update was triggered from a GUI event
                        // and if the table has a simple table style -> otherwise all cells need to be updated
                        if ((isGUIOperation && isGUIInsertOperation && isSimpleTableStyle && (! $(this).data('newCell'))) ||
                            (isGUIOperation && !isGUIInsertOperation && isSimpleTableStyle)) {
                            cellFormatting = false;
                        }

                        $(this).removeData('newCell');

                        if (cellFormatting) {
                            tableCellStyles.updateElementFormatting(this);
                        } else {
                            // checking if a cell contains tab stops
                            if ($(this).find(DOM.TAB_NODE_SELECTOR).length > 0) {
                                // updating the tabs is always required, because the width of a cell might have changed -> updating all paragraphs in the table cell
                                Utils.iterateSelectedDescendantNodes(DOM.getCellContentNode(this), DOM.PARAGRAPH_NODE_SELECTOR, function (paragraph) {
                                    paragraphStyles.updateTabStops(paragraph);
                                }, undefined, { children: true });
                            }
                        }
                    });
                });
            }

            // update column widths
            Table.updateColGroup(table, tableGrid);

            if (async === true) {
                // process the table asynchronously (about 40 cells at a time)
                chunkLength = Math.max(1, Math.round(40 / Math.max(1, tableGrid.length)));
                def = app.processArrayDelayed(updateRowsFormatting, rowNodes, { chunkLength: chunkLength });
            } else {
                // update all rows at once (but asynchron, so that for example a new column is visible immediately)
                def = app.executeDelayed(function () {
                    updateRowsFormatting(rowNodes);
                });
            }

            return def.promise();
        }

        /**
         * Will be called for tables used as preview elements in the GUI.
         *
         * @param {jQuery} table
         *  The preview table node, as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         */
        function updateTablePreviewFormatting(table, mergedAttributes) {

            var // the table cell styles/formatter
                tableCellStyles = documentStyles.getStyleSheets('cell');

            // iterate over all cells in the table to set the cell attributes
            DOM.getTableCells(table).each(function () {
                // do NOT pass merged attributes passed to this method as base
                // attributes; attributes must be recalculated for each single cell
                tableCellStyles.updateElementFormatting(this, { preview: true });
            });
        }

    } // class TableStyles

    // static methods ---------------------------------------------------------

    TableStyles.getBorderStyleFromAttributes = function (attributes) {

        var allWidths = [],
            width = 'none';

        if ((attributes) && (attributes.borderLeft) && (Border.isVisibleBorder(attributes.borderLeft))) { allWidths.push(attributes.borderLeft.width); }
        if ((attributes) && (attributes.borderRight) && (Border.isVisibleBorder(attributes.borderRight))) { allWidths.push(attributes.borderRight.width); }
        if ((attributes) && (attributes.borderTop) && (Border.isVisibleBorder(attributes.borderTop))) { allWidths.push(attributes.borderTop.width); }
        if ((attributes) && (attributes.borderBottom) && (Border.isVisibleBorder(attributes.borderBottom))) { allWidths.push(attributes.borderBottom.width); }
        if ((attributes) && (attributes.borderInsideHor) && (Border.isVisibleBorder(attributes.borderInsideHor))) { allWidths.push(attributes.borderInsideHor.width); }
        if ((attributes) && (attributes.borderInsideVert) && (Border.isVisibleBorder(attributes.borderInsideVert))) { allWidths.push(attributes.borderInsideVert.width); }

        allWidths = _.uniq(allWidths);

        if (allWidths.length === 1) {
            width = allWidths[0];
            width = Utils.convertHmmToLength(width, 'pt', 0.1);  // converting from 1/100 mm to pt
        }

        return width;
    };

    TableStyles.getAttributesFromBorderStyle = function (borderWidth, attributes) {

        var borderLeft = _.clone(attributes.borderLeft),
            borderRight = _.clone(attributes.borderRight),
            borderTop = _.clone(attributes.borderTop),
            borderBottom = _.clone(attributes.borderBottom),
            borderInsideHor = _.clone(attributes.borderInsideHor),
            borderInsideVert = _.clone(attributes.borderInsideVert);

        // converting from pt to 1/100 mm
        borderWidth = Utils.convertLengthToHmm(borderWidth, 'pt');

        borderLeft.width = borderWidth;
        borderRight.width = borderWidth;
        borderTop.width = borderWidth;
        borderBottom.width = borderWidth;
        borderInsideHor.width = borderWidth;
        borderInsideVert.width = borderWidth;

        return {
            borderLeft:       borderLeft,
            borderRight:      borderRight,
            borderTop:        borderTop,
            borderBottom:     borderBottom,
            borderInsideHor : borderInsideHor,
            borderInsideVert: borderInsideVert
        };

    };

    // Light Shading table style attributes

    TableStyles.getLightShadingTableStyleAttributes = function (colorValue) {

        var SINGLE_BORDER_LINE = { style: 'single', color: { type: 'scheme', value: colorValue }, width: 35 },
            BAND_COLOR = { type: 'scheme', value: colorValue, transformations: [{ type: 'tint', value: 24706}] },
            ROW_CELL_ATTRIBUTES = { borderLeft: Border.NONE, borderTop: SINGLE_BORDER_LINE, borderRight: Border.NONE, borderBottom: SINGLE_BORDER_LINE, borderInsideHor: Border.NONE, borderInsideVert: Border.NONE },
            BAND_CELL_ATTRIBUTES = { borderLeft: Border.NONE, borderRight: Border.NONE, borderInsideHor: Border.NONE, borderInsideVert: Border.NONE, fillColor: BAND_COLOR };

        return {
            wholeTable: {
                table: { borderTop: SINGLE_BORDER_LINE, borderBottom: SINGLE_BORDER_LINE },
                paragraph: { lineHeight: LineHeight.SINGLE, marginBottom: 0 },
                character: { color: { type: 'scheme', value: colorValue, transformations: [{ type: 'shade', value: 74902 }] } }
            },
            firstRow: { cell: ROW_CELL_ATTRIBUTES, paragraph: { lineHeight: LineHeight.SINGLE, marginTop: 0, marginBottom: 0 }, character: { bold: true } },
            lastRow: { cell: ROW_CELL_ATTRIBUTES, paragraph: { lineHeight: LineHeight.SINGLE, marginTop: 0, marginBottom: 0 }, character: { bold: true } },
            firstCol: { character: { bold: true } },
            lastCol: { character: { bold: true } },
            band1Vert: { cell: BAND_CELL_ATTRIBUTES },
            band1Hor: { cell: BAND_CELL_ATTRIBUTES }
        };
    };

    // Medium Shading 1 table style attributes

    TableStyles.getMediumShading1TableStyleAttributes = function (colorValue) {

        var SCHEME_COLOR = { type: 'scheme', value: colorValue },
            BAND_COLOR = { type: 'scheme', value: colorValue, transformations: [{ type: 'tint', value: 24706 }] },
            SINGLE_BORDER_LINE = { style: 'single', color: SCHEME_COLOR, width: 35 },
            DOUBLE_BORDER_LINE = { style: 'double', color: SCHEME_COLOR, width: 26 };

        return {
            wholeTable: {
                table: { borderLeft: SINGLE_BORDER_LINE, borderTop: SINGLE_BORDER_LINE, borderRight: SINGLE_BORDER_LINE, borderBottom: SINGLE_BORDER_LINE, borderInsideHor: SINGLE_BORDER_LINE },
                paragraph: { lineHeight: LineHeight.SINGLE, marginBottom: 0 }
            },
            firstRow: {
                cell: { borderLeft: SINGLE_BORDER_LINE, borderTop: SINGLE_BORDER_LINE, borderRight: SINGLE_BORDER_LINE, borderBottom: SINGLE_BORDER_LINE, borderInsideHor: Border.NONE, borderInsideVert: Border.NONE, fillColor: SCHEME_COLOR },
                paragraph: { lineHeight: LineHeight.SINGLE, marginTop: 0, marginBottom: 0 },
                character: { bold: true, color: { type: 'scheme', value: 'background1' } }
            },
            lastRow: {
                cell: { borderLeft: SINGLE_BORDER_LINE, borderTop: DOUBLE_BORDER_LINE, borderRight: SINGLE_BORDER_LINE, borderBottom: SINGLE_BORDER_LINE, borderInsideHor: Border.NONE, borderInsideVert: Border.NONE },
                paragraph: { lineHeight: LineHeight.SINGLE, marginTop: 0, marginBottom: 0 },
                character: { bold: true }
            },
            firstCol: { character: { bold: true } },
            lastCol: { character: { bold: true } },
            band1Vert: { cell: { fillColor: BAND_COLOR } },
            band1Hor: { cell: { borderInsideHor: Border.NONE, borderInsideVert: Border.NONE, fillColor: BAND_COLOR } },
            band2Hor: { cell: { borderInsideHor: Border.NONE, borderInsideVert: Border.NONE } }
        };
    };

    // Medium Shading 2 table style attributes

    TableStyles.getMediumShading2TableStyleAttributes = function (colorValue) {

        var SCHEME_COLOR = { type: 'scheme', value: colorValue },
            BACK_COLOR = { type: 'scheme', value: 'background1' },
            BAND_COLOR = { type: 'scheme', value: 'background1', transformations: [{ type: 'shade', value: 84706 }] },
            SINGLE_BORDER_LINE = { style: 'single', color: Color.AUTO, width: 79 },
            DOUBLE_BORDER_LINE = { style: 'double', color: Color.AUTO, width: 26 },
            NORTH_CELL_ATTRIBUTES = { borderLeft: Border.NONE, borderTop: SINGLE_BORDER_LINE, borderRight: Border.NONE, borderBottom: SINGLE_BORDER_LINE, borderInsideHor: Border.NONE, borderInsideVert: Border.NONE };

        return {
            wholeTable: {
                table: { borderTop: SINGLE_BORDER_LINE, borderBottom: SINGLE_BORDER_LINE },
                paragraph: { lineHeight: LineHeight.SINGLE, marginBottom: 0 }
            },
            firstRow: {
                cell: { borderLeft: Border.NONE, borderTop: SINGLE_BORDER_LINE, borderRight: Border.NONE, borderBottom: SINGLE_BORDER_LINE, borderInsideHor: Border.NONE, borderInsideVert: Border.NONE, fillColor: SCHEME_COLOR },
                paragraph: { lineHeight: LineHeight.SINGLE, marginTop: 0, marginBottom: 0 },
                character: { bold: true, color: BACK_COLOR }
            },
            lastRow: {
                cell: { borderLeft: Border.NONE, borderTop: DOUBLE_BORDER_LINE, borderRight: Border.NONE, borderBottom: SINGLE_BORDER_LINE, borderInsideHor: Border.NONE, borderInsideVert: Border.NONE, fillColor: BACK_COLOR },
                paragraph: { lineHeight: LineHeight.SINGLE, marginTop: 0, marginBottom: 0 },
                character: { color: Color.AUTO }
            },
            firstCol: {
                cell: { borderLeft: Border.NONE, borderTop: Border.NONE, borderRight: Border.NONE, borderBottom: SINGLE_BORDER_LINE, borderInsideHor: Border.NONE, borderInsideVert: Border.NONE, fillColor: SCHEME_COLOR },
                character: { bold: true, color: BACK_COLOR }
            },
            lastCol: {
                cell: { borderLeft: Border.NONE, borderRight: Border.NONE, borderInsideHor: Border.NONE, borderInsideVert: Border.NONE, fillColor: SCHEME_COLOR },
                character: { bold: true, color: BACK_COLOR }
            },
            band1Vert: { cell: { borderLeft: Border.NONE, borderRight: Border.NONE, borderInsideHor: Border.NONE, borderInsideVert: Border.NONE, fillColor: BAND_COLOR } },
            band1Hor: { cell: { fillColor: BAND_COLOR } },
            northEastCell: { cell: NORTH_CELL_ATTRIBUTES },
            northWestCell: { cell: NORTH_CELL_ATTRIBUTES, character: { color: BACK_COLOR } }
        };
    };

    // Medium Grid 1 table style attributes

    TableStyles.getMediumGrid1TableStyleAttributes = function (colorValue) {

        var BAND_COLOR = { type: 'scheme', value: colorValue, transformations: [{ type: 'tint', value: 49804 }] },
            SINGLE_BORDER_LINE = { style: 'single', color: { type: 'scheme', value: colorValue }, width: 35 },
            THICK_BORDER_LINE = { style: 'single', color: { type: 'scheme', value: colorValue }, width: 79 };

        return {
            wholeTable: {
                table: { borderLeft: SINGLE_BORDER_LINE, borderTop: SINGLE_BORDER_LINE, borderRight: SINGLE_BORDER_LINE, borderBottom: SINGLE_BORDER_LINE, borderInsideHor: SINGLE_BORDER_LINE, borderInsideVert: SINGLE_BORDER_LINE },
                cell: { fillColor: { type: 'scheme', value: colorValue, transformations: [{ type: 'tint', value: 24706 }] } },
                paragraph: { lineHeight: LineHeight.SINGLE, marginBottom: 0 }
            },
            firstRow: { character: { bold: true } },
            lastRow: { cell: { borderTop: THICK_BORDER_LINE }, character: { bold: true } },
            firstCol: { character: { bold: true }},
            lastCol: { character: { bold: true }},
            band1Vert: { cell: { fillColor: BAND_COLOR } },
            band1Hor: { cell: { fillColor: BAND_COLOR } }
        };
    };

    // exports ================================================================

    // derive this class from class StyleSheets
    return StyleSheets.extend({ constructor: TableStyles });

});
