/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/text/drawingResize',
    ['io.ox/office/tk/utils',
     'io.ox/office/framework/view/drawingframe',
     'io.ox/office/text/dom',
     'io.ox/office/text/operations',
     'io.ox/office/text/position',
     'io.ox/office/text/table'
    ], function (Utils, DrawingFrame, DOM, Operations, Position, Table) {

    'use strict';

    var // the names of all tracking events but 'tracking:start'
        TRACKING_EVENT_NAMES = 'tracking:move tracking:scroll tracking:end tracking:cancel';

    // static class DrawingResize =============================================

    var DrawingResize = {};

    // static methods ---------------------------------------------------------

    /**
     * Draws a selection box for the specified drawing node and registers
     * mouse handlers for moving and resizing.
     *
     * @param {TextApplication} app
     *  The application instance containing the drawing.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node to be selected, as DOM node or jQuery object.
     */
    DrawingResize.drawDrawingSelection = function (app, drawingNode) {

        var // the text editor
            editor = app.getModel(),
            // the root node of the editor DOM
            rootNode = editor.getNode(),
            // the node containing the scroll bar
            scrollNode = app.getView().getContentRootNode(),
            // object containing information about movable and resizable
            options = { movable: true, resizable: true },  // -> parameter?
            // the container element used to visualize the selection
            selectionBox = null,
            // saving the selected resize node
            resizeNode = null,
            // the container element used to visualize the movement and resizing
            moveBox = null;

        // common deinitialization after move or resize tracking
        function leaveTracking() {
            moveBox.removeClass('tracking').css({ left: '', top: '', right: '', bottom: '', width: '', height: '' });
        }

        /**
         * Handling all tracking events while moving a drawing.
         */
        var drawingMoveHandler = (function () {

            var // size of the drawing
                drawingWidth = 0, drawingHeight = 0,
                // the current position change (in px)
                shiftX = 0, shiftY = 0,
                // the current scroll position
                scrollX = 0, scrollY = 0,
                // the start scroll position
                startScrollX = 0, startScrollY = 0,
                // left/top distance from drawing to event point (in px)
                leftDrawingDistance = 0, topDrawingDistance = 0,
                // right/bottom distance from drawing to event point (in px)
                // rightDrawingDistance = 0, bottomDrawingDistance = 0,
                // the container node of a page element that contains all top-level content nodes (paragraphs and tables)
                pageContent = DOM.getPageContentNode(rootNode),
                // the width of the page content node
                pageContentWidth = Math.round(pageContent.width()),
                // the height of the page content node
                pageContentHeight = Math.round(pageContent.height()),
                // whether the mouse was really moved (Fix for 28633)
                mouseMoved = false;

            // initialing the moving of the drawing according to the passed tracking event
            function startMoveDrawing(event) {

                // storing old height and width of drawing
                drawingWidth = drawingNode.width();
                drawingHeight = drawingNode.height();

                leftDrawingDistance = Utils.round(event.pageX - drawingNode.offset().left, 1);
                topDrawingDistance = Utils.round(event.pageY - drawingNode.offset().top, 1);
                // rightDrawingDistance = drawingWidth - leftDrawingDistance;
                // bottomDrawingDistance = drawingHeight - topDrawingDistance;

                startScrollX = scrollNode.scrollLeft();
                startScrollY = scrollNode.scrollTop();
            }

            // updating the drawing position according to the passed tracking event
            function updateMove(event) {

                // some local variables to control, that the drawing (its upper left corner) is not moved outside the document (28220).
                var minLeftPosition = Math.round(pageContent.offset().left),
                    maxRightPosition = minLeftPosition + pageContentWidth,
                    minTopPosition = Math.round(pageContent.offset().top),
                    maxBottomPosition = minTopPosition + pageContentHeight;

                // make move box visible when tracking position has moved
                moveBox.addClass('tracking');

                // reading scrollPosition again. Maybe it was not updated or not updated completely.
                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;

                shiftX = event.pageX - event.startX + scrollX;
                shiftY = event.pageY - event.startY + scrollY;

                // only move the moveBox, if the mouse was really moved (Fix for 28633)
                mouseMoved = ((shiftX !== 0) || (shiftY !== 0));

                if (mouseMoved) {
                    // the upper left corner of the drawing always has to be inside the page content node -> reduce or increase shiftX and shiftY, if necessary
                    if (event.pageX - leftDrawingDistance < minLeftPosition) { shiftX = shiftX + minLeftPosition - event.pageX + leftDrawingDistance; }
                    if (event.pageY - topDrawingDistance < minTopPosition) { shiftY = shiftY + minTopPosition - event.PageY + topDrawingDistance; }
                    if (event.pageX - leftDrawingDistance > maxRightPosition) { shiftX = shiftX + maxRightPosition - event.pageX + leftDrawingDistance; }
                    if (event.pageY - topDrawingDistance > maxBottomPosition) { shiftY = shiftY + maxBottomPosition - event.pageY + topDrawingDistance; }

                    if ((_.isNumber(shiftX)) && (_.isNumber(shiftY)) && (shiftX !== 0) || (shiftY !== 0)) {
                        moveBox.css({ left: shiftX, top: shiftY, width: drawingWidth, height: drawingHeight });
                    }
                }
            }

            // updates scroll position according to the passed tracking event
            function updateMoveScroll(event) {

                // update scrollPosition with suggestion from event
                if (event.scrollX) {
                    scrollNode.scrollLeft(scrollNode.scrollLeft() + event.scrollX);
                }

                if (event.scrollY) {
                    scrollNode.scrollTop(scrollNode.scrollTop() + event.scrollY);
                }

                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;
            }

            // handling the drawing position, when moving is stopped according to the passed tracking event
            function stopMoveDrawing(event) {

                // mouse up handler
                var generator = editor.createOperationsGenerator(),
                    // the horizontal move of the drawing in 1/100 mm
                    moveX = 0,
                    // the vertical move of the drawing in 1/100 mm
                    moveY = 0,
                    // the logical position where the drawing is located before it needs to be moved
                    updatePosition = null,
                    // the attributes of the drawing node
                    drawingNodeAttrs = null,
                    // the anchorHorOffset property of the drawing
                    anchorHorOffset = 0,
                    // the anchorVertOffset property of the drawing
                    anchorVertOffset = 0,
                    // the anchorHorBase property of the drawing
                    anchorHorBase = 0,
                    // the anchorVertBase property of the drawing
                    anchorVertBase = 0,
                    // the anchorHorAlign property of the drawing
                    anchorHorAlign = 0,
                    // the anchorVertAlign property of the drawing
                    anchorVertAlign = 0,
                    // the saved anchorHorOffset property of the drawing before move
                    oldAnchorHorOffset = 0,
                    // the saved anchorVertOffset property of the drawing before move
                    oldAnchorVertOffset = 0,
                    // paragraph node required for checking implicit paragraphs
                    localParagraph = null,
                    // logical paragraph position required for checking implicit paragraphs
                    localPosition = null,
                    // the logical destination for moved images
                    destPosition = null,
                    // current drawing width, in 1/100 mm
                    drawingWidth = 0,
                    // the paragraph element containing the drawing node
                    paragraph = null,
                    // total width of the paragraph, in 1/100 mm
                    paraWidth = 0,
                    // is it necessary to move the image?
                    moveImage = false,
                    // whether the drawing was moved or not
                    imageMoved = false,
                    // whether the image was moved downwards caused by the resize operation
                    moveDownwards = false,
                    // position of the mouse up event shifted into the document borders
                    trimmedPosition = null,
                    // the height of the offset node, in 1/100 mm
                    offsetNodeHeight = 0,
                    // the current position sent by the event (in px)
                    currentX = 0, currentY = 0,
                    // the position of the top left corner of the drawing
                    drawingLeftX = 0, drawingTopY = 0,
                    // the number of floated upper drawings in a paragraph
                    upperDrawings = 0;

                function adaptPositionIntoDocument(posX, posY) {

                    var // the left offset of the page content node -> the drawing must be right of this position
                        minLeftPosition = Math.round(pageContent.offset().left),
                        // the left offset of the page content node plus its width -> the drawing must be left of this position
                        maxRightPosition = minLeftPosition + pageContentWidth,
                        // the top position of the page content, not the page itself
                        minTopPosition = Math.round(pageContent.offset().top),
                        // adding the height of the page content (can be only one paragraph) to the top position of the page content
                        maxBottomPosition = minTopPosition + pageContentHeight;

                    // taking care of leftDrawingDistance and topDrawingDistance, so that the upper left corner is inside the pageContent.
                    if (posX < minLeftPosition + leftDrawingDistance) { posX = minLeftPosition + leftDrawingDistance; }
                    if (posX > maxRightPosition + leftDrawingDistance) { posX = maxRightPosition + leftDrawingDistance; }
                    if (posY < minTopPosition + topDrawingDistance) { posY = minTopPosition + topDrawingDistance; }
                    if (posY > maxBottomPosition + topDrawingDistance) { posY = maxBottomPosition + topDrawingDistance; }

                    return { posX: posX, posY: posY };
                }

                function isPositionInsideNode(node, posX, posY) {

                    if (! (node instanceof $)) { node = $(node); }

                    return ((Math.round(node.offset().left) <= posX) && (posX <= Math.round(node.offset().left + node.outerWidth())) &&
                            (Math.round(node.offset().top) <= posY) && (posY <= Math.round(node.offset().top + node.outerHeight())));
                }

                function iterateSelectorNodes(topNode, currentNode, posX, posY, selector, skipSelector, options) {

                    var selectorNode = null,
                        reverse = Utils.getBooleanOption(options, 'reverse', false);

                    while (currentNode) {

                        if (isPositionInsideNode(currentNode, posX, posY)) {
                            selectorNode = currentNode;
                            break;
                        }

                        if (reverse) {
                            currentNode = Utils.findPreviousNode(topNode, currentNode, selector, skipSelector);
                        } else {
                            currentNode = Utils.findNextNode(topNode, currentNode, selector, skipSelector);
                        }
                    }

                    return selectorNode;
                }

                function getParagraphAtPosition(topNode, startNode, shiftX, shiftY, posX, posY) {

                    var searchPrevious = true,
                        searchFollowing = true,
                        paragraph = null,
                        tableCell = null;

                    if ((shiftX > 0) && (shiftY > 0)) { searchPrevious = false; }
                    if ((shiftX < 0) && (shiftY < 0)) { searchFollowing = false; }

                    if (paragraph) { searchFollowing = false; }

                    if (searchFollowing) {
                        paragraph = iterateSelectorNodes(topNode, Utils.getDomNode(startNode), posX, posY, DOM.PARAGRAPH_NODE_SELECTOR, DrawingFrame.NODE_SELECTOR, {'reverse': false});
                    }

                    if (paragraph) { searchPrevious = false; }
                    else { searchPrevious = true; }

                    if (searchPrevious) {
                        paragraph = iterateSelectorNodes(topNode, Utils.getDomNode(startNode), posX, posY, DOM.PARAGRAPH_NODE_SELECTOR, DrawingFrame.NODE_SELECTOR, {'reverse': true});
                    }

                    // maybe the paragraph is in a table cell with a cell neighbor that is much higher -> use last paragraph in this cell
                    if (! paragraph) {
                        tableCell = iterateSelectorNodes(topNode, Utils.getDomNode(startNode), posX, posY, DOM.TABLE_CELLNODE_SELECTOR, DrawingFrame.NODE_SELECTOR, {'reverse': false});

                        if (! tableCell) {
                            tableCell = iterateSelectorNodes(topNode, Utils.getDomNode(startNode), posX, posY, DOM.TABLE_CELLNODE_SELECTOR, DrawingFrame.NODE_SELECTOR, {'reverse': true});
                        }

                        if ((tableCell) && (DOM.isTableCellNode(tableCell))) {
                            paragraph = DOM.getCellContentNode(tableCell)[0].lastChild;  // the last paragraph of the cell content
                        }
                    }

                    if (paragraph) { paragraph = $(paragraph); }

                    return paragraph;
                }

                // begin of stopMoveDrawing

                if (! mouseMoved) {
                    return;
                }

                // setting position of tracking:end event
                currentX = event.pageX;
                currentY = event.pageY;

                // scrolling to the correct position -> absolutely necessary!
                scrollNode.scrollLeft(scrollX + startScrollX).scrollTop(scrollY + startScrollY);

                // shifting currentX and currentY to position inside the document
                trimmedPosition = adaptPositionIntoDocument(currentX, currentY);
                currentX = trimmedPosition.posX;
                currentY = trimmedPosition.posY;

                // calculating the real shift of the drawing
                shiftX = currentX - event.startX + scrollX;
                shiftY = currentY - event.startY + scrollY;

                // top left corner of the drawing
                drawingLeftX = currentX - leftDrawingDistance;
                drawingTopY = currentY - topDrawingDistance;

                updatePosition = Position.getOxoPosition(rootNode, drawingNode, 0);

                if ((_.isNumber(shiftX)) && (_.isNumber(shiftY)) && ((shiftX !== 0) || (shiftY !== 0))) {

                    paragraph = drawingNode.parent();
                    // converting to 1/100 mm
                    moveX = Utils.convertLengthToHmm(shiftX, 'px');
                    moveY = Utils.convertLengthToHmm(shiftY, 'px');
                    drawingWidth = Utils.convertLengthToHmm(drawingNode.width(), 'px');
                    paraWidth = Utils.convertLengthToHmm(paragraph.width(), 'px');
                    // evaluating attributes
                    drawingNodeAttrs = editor.getStyleSheets('drawing').getElementAttributes(drawingNode).drawing;
                    oldAnchorHorOffset = drawingNodeAttrs.anchorHorOffset;
                    oldAnchorVertOffset = drawingNodeAttrs.anchorVertOffset ? drawingNodeAttrs.anchorVertOffset : 0;
                    anchorHorBase = drawingNodeAttrs.anchorHorBase;
                    anchorVertBase = drawingNodeAttrs.anchorVertBase;
                    anchorHorAlign = drawingNodeAttrs.anchorHorAlign;
                    anchorVertAlign = drawingNodeAttrs.anchorVertAlign;

                    if ((oldAnchorHorOffset === undefined) || (oldAnchorHorOffset === 0)) {
                        // anchorHorOffset has to be calculated corresponding to the left paragraph border
                        if (anchorHorAlign === 'right') {
                            oldAnchorHorOffset = paraWidth - drawingWidth;
                        } else if (anchorHorAlign === 'center') {
                            oldAnchorHorOffset = (paraWidth - drawingWidth) / 2;
                        } else {
                            oldAnchorHorOffset = 0;
                        }
                    }

                    anchorHorOffset = oldAnchorHorOffset;
                    anchorVertOffset = oldAnchorVertOffset;

                    // checking position of mouse up event
                    // -> is the top-left corner of the drawing still in the same paragraph?
                    if (isPositionInsideNode(paragraph, drawingLeftX, drawingTopY)) {   // -> new position is in the same paragraph (before the drawing is shifted!)

                        if (moveX !== 0) {
                            anchorHorOffset = oldAnchorHorOffset + moveX;
                            anchorHorAlign = 'offset';
                            anchorHorBase = 'column';
                            if (anchorHorOffset < 0) { anchorHorOffset = 0; }
                            else if (anchorHorOffset > (paraWidth - drawingWidth)) { anchorHorOffset = paraWidth - drawingWidth; }
                        }

                        if (moveY !== 0) {
                            anchorVertAlign = 'offset';
                            anchorVertBase = 'paragraph';

                            anchorVertOffset = Utils.convertLengthToHmm((drawingTopY - paragraph.offset().top), 'px');
                            // anchorVertOffset always has to be >= 0, not leaving the paragraph ('< 0' should never happen here)
                            if (anchorVertOffset < 0) { anchorVertOffset = 0; }

                            // moving the drawing is required, if an already existing offset node is not sufficient
                            if ((drawingNode.prev().length) && (DOM.isOffsetNode(drawingNode.prev()))) {
                                offsetNodeHeight = Utils.convertLengthToHmm(drawingNode.prev().height(), 'px');
                            }

                            // Moving the drawing to the beginning of the paragraph
                            // There might be no offset node before (offsetNodeHeight === 0) or there can be an offset node
                            // that reaches the beginning of the paragraph or there can also be an offset node, that starts
                            // in the middle of the paragraph. The latter can be achieved by swichting from inline to floated
                            // and then moving down the drawing inside the paragraph. If such an offset node is not high enough,
                            // the drawing must be moved to the beginning of the paragraph behind other floated drawings.
                            if ((moveY < 0) && ((- moveY) > offsetNodeHeight)) {
                                destPosition = _.clone(updatePosition);
                                destPosition[destPosition.length - 1] = Position.getLeadingFloatingDrawingCount(paragraph, drawingTopY - paragraph.offset().top);
                                moveImage = true;
                            }

                            // If the drawing is moved downwards in a paragraph, it is possible, that following drawings will
                            // be shifted downwards, too. It is better, that the position of such a drawing is moved, too, so that
                            // following drawings are not (or less) modified.
                            if (moveY > 0) {
                                // calculating number of drawings, that have less distance to the top edge of the paragraph
                                upperDrawings = Position.getLeadingFloatingDrawingCount(paragraph, drawingTopY - paragraph.offset().top);
                                upperDrawings -= 1;  // reducing by 1, because of the moved drawing itself
                                if (updatePosition[updatePosition.length - 1] < upperDrawings) {
                                    destPosition = _.clone(updatePosition);
                                    destPosition[destPosition.length - 1] = upperDrawings + 1;  // moving from [0,0] to [0,2], so that it ends at [0,1]
                                    moveImage = true;
                                    moveDownwards = true;
                                }
                            }
                        }

                    } else {   // -> new position is in another paragraph (before the drawing is moved!)

                        // paragraph has to be determined from the coordinates of the top left corner of the drawing
                        // -> moving operation for the drawing is always required
                        paragraph = getParagraphAtPosition(rootNode, paragraph, shiftX, shiftY, drawingLeftX, drawingTopY);

                        if (paragraph) {

                            paraWidth = Utils.convertLengthToHmm(paragraph.width(), 'px');

                            anchorVertAlign = 'offset';
                            anchorVertBase = 'paragraph';
                            anchorVertOffset = Utils.convertLengthToHmm((drawingTopY - paragraph.offset().top), 'px');

                            anchorHorAlign = 'offset';
                            anchorHorBase = 'column';
                            anchorHorOffset = Utils.convertLengthToHmm((drawingLeftX - paragraph.offset().left), 'px');

                            destPosition = Position.getOxoPosition(rootNode, paragraph, 0);
                            // Moving behind already existing floated drawings
                            destPosition.push(Position.getLeadingFloatingDrawingCount(paragraph, drawingTopY - paragraph.offset().top));

                            moveImage = true;

                        } else {
                            // No paragraph found: Do not call set Attributes and not moveImage
                            moveImage = false;
                            anchorHorOffset = oldAnchorHorOffset;
                            anchorVertOffset = oldAnchorVertOffset;
                        }
                    }

                    if ((moveImage) && (! _.isEqual(updatePosition, destPosition))) {

                        // check, if destPosition is located in a non-implicit paragraph -> otherwise create paragraph
                        // similar to editor.handleImplicitParagraph
                        localPosition = _.clone(destPosition);

                        if (localPosition.pop() === 0) {  // is this an empty paragraph?
                            localParagraph = Position.getParagraphElement(rootNode, localPosition);
                            if ((DOM.isImplicitParagraphNode(localParagraph)) && (Position.getParagraphNodeLength(localParagraph) === 0)) {
                                // removing implicit paragraph node
                                $(localParagraph).remove();
                                // creating new paragraph explicitly
                                generator.generateOperation(Operations.PARA_INSERT, {start: localPosition});
                            }
                        }

                        generator.generateOperation(Operations.MOVE, {
                            start: updatePosition,
                            end: updatePosition,
                            to: destPosition
                        });

                        updatePosition = _.clone(destPosition); // for setting attributes required
                        if (moveDownwards) {
                            updatePosition[updatePosition.length - 1] -= 1;  // moving from [0,0] to [0,2], so that it ends at [0,1]
                        }

                        imageMoved = true;
                    }

                    if ((anchorHorOffset !== oldAnchorHorOffset) || (anchorVertOffset !== oldAnchorVertOffset)) {

                        generator.generateOperation(Operations.SET_ATTRIBUTES, {
                            attrs: { drawing: { anchorHorOffset: anchorHorOffset, anchorVertOffset: anchorVertOffset, anchorHorAlign: anchorHorAlign, anchorVertAlign: anchorVertAlign, anchorHorBase: anchorHorBase, anchorVertBase: anchorVertBase } },
                            start: updatePosition
                        });

                    }
                }

                // apply the operations (undo group is created automatically)
                editor.applyOperations(generator.getOperations());

                // set new text selection after moving the drawing, this will repaint the selection
                if (imageMoved) {
                    editor.getSelection().setTextSelection(updatePosition, Position.increaseLastIndex(updatePosition));
                }
            }

            // finalizes the move tracking
            function finalizeMoveDrawing(event) {
                leaveTracking();
                moveBox.off(TRACKING_EVENT_NAMES);

                // Resetting variables for new mouseup events without mousemove
                shiftX = shiftY = scrollX = scrollY = 0;
            }

            // return the actual drawingMoveHandler() method
            return function (event) {

                switch (event.type) {
                case 'tracking:start':
                    startMoveDrawing(event);
                    break;
                case 'tracking:move':
                    updateMove(event);
                    break;
                case 'tracking:scroll':
                    updateMoveScroll(event);
                    break;
                case 'tracking:end':
                    stopMoveDrawing(event);
                    finalizeMoveDrawing(event);
                    break;
                case 'tracking:cancel':
                    finalizeMoveDrawing(event);
                    break;
                }
            };

        }()); // end of drawingMoveHandler() local scope

        /**
         * Handles all tracking events while resizing a drawing.
         */
        var drawingResizeHandler = (function () {

            var // original size of the drawing
                oldWidth = 0, oldHeight = 0,
                // the size of the resized drawing (in px)
                finalWidth = 0, finalHeight = 0,
                // the current scroll position
                scrollX = 0, scrollY = 0,
                // the initial scroll position
                startScrollX = 0, startScrollY = 0,
                // whether resizing is available in horizontal/vertical direction
                useX = false, useY = false, topResize = false,
                // correction factor for resizing to the left/top
                scaleX = 0, scaleY = 0,
                // the maximum width of the drawing on the page or in a table cell
                maxDrawingWidth = null,
                // the maximum height of the drawing (sometimes same scaling as width)
                maxDrawingHeight = null,
                // vertical shift of the drawing in px
                moveY = 0;

            // initializes resizing the drawing according to the passed tracking event
            function startResizeDrawing(event) {

                var pos = $(event.target).attr('data-pos'),
                    // the paragraph element containing the drawing node
                    paragraph = null;

                // storing old height and width of drawing
                oldWidth = finalWidth = drawingNode.width();
                oldHeight = finalHeight = drawingNode.height();

                // collecting information about the handle node
                useX = /[lr]/.test(pos);
                useY = /[tb]/.test(pos);
                topResize = /[t]/.test(pos);

                if (/l/.test(pos)) {
                    scaleX = -1;
                    moveBox.css({ left: 'auto', right: 0 });
                } else {
                    scaleX = 1;
                    moveBox.css({ left: 0, right: 'auto' });
                }

                if (/t/.test(pos)) {
                    scaleY = -1;
                    moveBox.css({ top: 'auto', bottom: 0 });
                } else {
                    scaleY = 1;
                    moveBox.css({ top: 0, bottom: 'auto' });
                }

                startScrollX = scrollNode.scrollLeft();
                startScrollY = scrollNode.scrollTop();

                // reduce the width of the drawing, if the width is bigger than the width of
                // the page plus the page's right margin or the width of the paragraph in a table cell
                paragraph = drawingNode.parent();
                maxDrawingWidth = paragraph.width();
                // maxDrawingWidth = paragraph.width() - Math.max(Utils.convertCssLength(drawingNode.css('margin-left'), 'px', 1), 0) - Math.max(Utils.convertCssLength(drawingNode.css('margin-right'), 'px', 1), 0);
                if (DOM.isPageContentNode(paragraph.parent())) {
                    maxDrawingWidth += Utils.convertCssLength(rootNode.css('padding-right'), 'px', 1);
                }
                // in the case of non-deforming resize, there is also a maximum value for the height
                if (useX && useY) {
                    maxDrawingHeight = oldHeight * maxDrawingWidth / oldWidth;
                }
            }

            // updates scroll position according to the passed tracking event
            function updateResizeScroll(event) {

                // update scrollPosition with suggestion from event
                scrollNode
                    .scrollLeft(scrollNode.scrollLeft() + event.scrollX)
                    .scrollTop(scrollNode.scrollTop() + event.scrollY);

                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;
            }

            // updates resizing the drawing according to the passed tracking event
            function updateResize(event) {

                var deltaX = useX ? ((event.pageX - event.startX + scrollX) * scaleX) : 0,
                    deltaY = useY ? ((event.pageY - event.startY + scrollY) * scaleY) : 0,
                    // the scaling factor for the width
                    scaleWidth = 1,
                    // the scaling factor for the height
                    scaleHeight = 1;

                if (topResize) {
                    moveY = deltaY;
                }

                // update drawing size
                finalWidth = Math.max(0, oldWidth + deltaX);
                finalHeight = Math.max(0, oldHeight + deltaY);

                // The maximum value for the width is 'maxDrawingWidth' -> avoid endless scrolling
                // There is no limit for the height (only in non-deforming resizing).
                if (finalWidth >= maxDrawingWidth) {
                    finalWidth = maxDrawingWidth;
                }

                if (useX && useY && (finalHeight >= maxDrawingHeight)) {
                    finalHeight = maxDrawingHeight;
                }

                // use the same scaling factor for vertical and horizontal resizing, if both are enabled
                // -> the larger number wins
                if (useX && useY) {
                    scaleWidth = finalWidth / oldWidth;
                    scaleHeight = finalHeight / oldHeight;

                    if (scaleWidth > scaleHeight) {
                        finalHeight = scaleWidth * oldHeight;
                    } else {
                        finalWidth = scaleHeight * oldWidth;
                    }
                }

                // make move box visible when tracking position has moved
                moveBox.addClass('tracking').css({ width: finalWidth, height: finalHeight });
            }

            // resizes the drawing according to the passed tracking event
            function stopResizeDrawing(event) {

                var generator = editor.createOperationsGenerator(),
                    // drawing attributes for the operation
                    attributes = {},
                    // the final scaling of the drawing width and height
                    finalWidthScaling = 1,
                    // the vertical move of the drawing in 1/100 mm
                    moveYHmm = 0,
                    // the paragraph containing the resized drawing
                    paragraph = null,
                    // the logical position to which a drawing needs to be moved to
                    destPosition = null,
                    // the logical position where the drawing is located before it needs to be moved
                    updatePosition = Position.getOxoPosition(rootNode, drawingNode, 0),
                    // whether the image needs to be moved because of the resize operation
                    moveImage = false,
                    // whether the image was moved caused by the resize operation
                    imageMoved = false,
                    // whether the image was moved downwards caused by the resize operation
                    moveDownwards = false,
                    // the attributes of the drawing node
                    drawingNodeAttrs = null,
                    // the anchorVertBase property of the drawing
                    anchorVertBase = null,
                    // the anchorVertAlign property of the drawing
                    anchorVertAlign = null,
                    // the anchorVertOffset property of the drawing
                    anchorVertOffset = null,
                    // the saved anchorVertOffset property of the drawing before resize
                    oldAnchorVertOffset = 0,
                    // the new vertical offset inside the paragraph in px
                    paragraphDeltaY = 0,
                    // the height of the offset node, in 1/100 mm
                    offsetNodeHeight = 0,
                    // the number of floated upper drawings in a paragraph
                    upperDrawings = 0;

                // Calculating new width and height

                // calculate width
                if (useX && (finalWidth > 0) && (finalWidth !== oldWidth)) {
                    attributes.width = Utils.convertLengthToHmm(finalWidth, 'px');

                    if (finalWidth > maxDrawingWidth) {
                        finalWidthScaling = maxDrawingWidth / finalWidth;
                        attributes.width = Utils.convertLengthToHmm(maxDrawingWidth, 'px');
                    }
                }

                // calculate height
                if (useY && (finalHeight > 0) && (finalHeight !== oldHeight)) {
                    if (finalWidthScaling !== 1) { finalHeight *= finalWidthScaling; }
                    attributes.height = Utils.convertLengthToHmm(finalHeight, 'px');
                }

                // Check, if the drawing needs to be moved upwards
                // -> recalculating offsets and sometimes position
                if ((topResize) && (moveY !== 0) && (options.movable)) {

                    // moveY > 0 -> Drawing was increased -> Moving drawing upwards
                    // moveY < 0 -> Drawing was decreased -> Moving drawing downwards
                    moveYHmm = - Utils.convertLengthToHmm(moveY, 'px');  // negative value
                    // moveYHmm > 0 -> moving drawing downwards -> value can be added to existing vertical offset
                    // moveYHmm < 0 -> moving drawing upwards -> value can be added to existing vertical offset

                    paragraph = drawingNode.parent();
                    paragraphDeltaY = drawingNode.offset().top - moveY - paragraph.offset().top; // the new vertical offset in the paragraph
                    // -> simplification: Not moving a drawing into another paragraph with resize
                    if (paragraphDeltaY < 0) { paragraphDeltaY = 0; }

                    // evaluating attributes
                    drawingNodeAttrs = editor.getStyleSheets('drawing').getElementAttributes(drawingNode).drawing;
                    oldAnchorVertOffset = drawingNodeAttrs.anchorVertOffset ? drawingNodeAttrs.anchorVertOffset : 0;
                    anchorVertBase = drawingNodeAttrs.anchorVertBase;
                    anchorVertAlign = drawingNodeAttrs.anchorVertAlign;

                    anchorVertAlign = 'offset';
                    anchorVertBase = 'paragraph';

                    anchorVertOffset = Utils.convertLengthToHmm(paragraphDeltaY, 'px');
                    // anchorVertOffset always has to be >= 0, not leaving the paragraph ('< 0' should never happen here)
                    if (anchorVertOffset < 0) { anchorVertOffset = 0; }

                    // moving the drawing is required, if an already existing offset node is not sufficient
                    if ((drawingNode.prev().length) && (DOM.isOffsetNode(drawingNode.prev()))) {
                        offsetNodeHeight = Utils.convertLengthToHmm(drawingNode.prev().height(), 'px');
                    }

                    // Resizing the drawing to the beginning of the paragraph
                    if ((moveYHmm < 0) && ((- moveYHmm) > offsetNodeHeight)) {
                        destPosition = _.clone(updatePosition);
                        destPosition[destPosition.length - 1] = Position.getLeadingFloatingDrawingCount(paragraph, paragraphDeltaY);
                        moveImage = true;
                    }

                    // If the drawing is resized downwards (decreasing the drawing) , it is possible, that following drawings will
                    // be shifted downwards, too. It is better, that the position of such a drawing is moved, too, so that
                    // following drawings are not (or less) modified.
                    if (moveYHmm > 0) {
                        // calculating number of drawings, that have less distance to the top edge of the paragraph
                        upperDrawings = Position.getLeadingFloatingDrawingCount(paragraph, paragraphDeltaY);
                        upperDrawings -= 1;  // reducing by 1, because of the moved drawing itself
                        if (updatePosition[updatePosition.length - 1] < upperDrawings) {
                            destPosition = _.clone(updatePosition);
                            destPosition[destPosition.length - 1] = upperDrawings + 1;  // moving from [0,0] to [0,2], so that it ends at [0,1]
                            moveImage = true;
                            moveDownwards = true;
                        }
                    }

                    if (anchorVertOffset !== oldAnchorVertOffset) {
                        // Additional attributes for the setAttribute operation
                        attributes.anchorVertOffset = anchorVertOffset;
                        attributes.anchorVertAlign = anchorVertAlign;
                        attributes.anchorVertBase = anchorVertBase;

                        // Generating the move operation
                        if ((moveImage) && (! _.isEqual(updatePosition, destPosition))) {
                            // moving image not in another paragraph using resize
                            // -> checking for implicit paragraphs not required (like in stopMoveDrawing).

                            generator.generateOperation(Operations.MOVE, {
                                start: updatePosition,
                                end: updatePosition,
                                to: destPosition
                            });

                            updatePosition = _.clone(destPosition); // for setting attributes required
                            if (moveDownwards) {
                                updatePosition[updatePosition.length - 1] -= 1;  // moving from [0,0] to [0,2], so that it ends at [0,1]
                            }

                            imageMoved = true;
                        }
                    }
                }

                // Generating the setAttribute operation
                if (!_.isEmpty(attributes)) {
                    generator.generateOperation(Operations.SET_ATTRIBUTES, {
                        start: updatePosition,
                        attrs: { drawing: attributes }
                    });
                }

                // Applying the operations (undo group is created automatically)
                editor.applyOperations(generator.getOperations());

                // Setting new text selection after moving the drawing, this will repaint the selection
                if (imageMoved) {
                    editor.getSelection().setTextSelection(updatePosition, Position.increaseLastIndex(updatePosition));
                }

            }

            // finalizes the resize tracking
            function finalizeResizeDrawing(event) {
                leaveTracking();
                resizeNode.off(TRACKING_EVENT_NAMES);
                app.getView().scrollToChildNode(drawingNode);
            }

            // return the actual drawingResizeHandler() method
            return function (event) {

                switch (event.type) {
                case 'tracking:start':
                    startResizeDrawing(event);
                    break;
                case 'tracking:move':
                    updateResize(event);
                    break;
                case 'tracking:scroll':
                    updateResizeScroll(event);
                    break;
                case 'tracking:end':
                    stopResizeDrawing(event);
                    finalizeResizeDrawing(event);
                    break;
                case 'tracking:cancel':
                    finalizeResizeDrawing(event);
                    break;
                }
            };

        }()); // end of drawingResizeHandler() local scope

        /**
         * Handler for 'tracking:start' events for 'div.handle' and 'div.move'
         * nodes. This handler has to decide, which node got the event. For this
         * node all other tracking events have to be registered. Additionally the
         * handler is called with the 'tracking:start' event, so that the drawing
         * initialization happens.
         *
         * @param {jQuery.Event} event
         *  The 'tracking:start' event, that starts the moving or the resizing
         *  of a drawing.
         */
        function trackingStartHandler(event) {
            if ($(event.target).is('.handle')) {
                resizeNode = $(event.target);
                resizeNode.on(TRACKING_EVENT_NAMES, drawingResizeHandler);
                drawingResizeHandler.call(this, event);
            } else if (moveBox.is(event.target)) {
                moveBox.off(TRACKING_EVENT_NAMES); // <- should not be necessary, but it is (more and more setAttributes OPs)
                moveBox.on(TRACKING_EVENT_NAMES, drawingMoveHandler);
                drawingMoveHandler.call(this, event);
            }
        }

        // starting code of static method drawDrawingSelection()
        drawingNode = $(drawingNode);
        if (drawingNode.length !== 1) {
            Utils.error('DrawingResize.drawDrawingSelection(): single drawing node expected');
            drawingNode = drawingNode.first();
        }

        // inline drawings are not movable
        if (DOM.isInlineDrawingNode(drawingNode)) {
            options.movable = false;
        } else if (DOM.isFloatingDrawingNode(drawingNode)) {
            options.movable = true;
        }
        if (drawingNode.hasClass('horizontal-line')) {
            options.movable = false;
        }

        // the container element used to visualize the selection
        selectionBox = DrawingFrame.drawSelection(drawingNode, options);

        // initialize move tracking
        moveBox = selectionBox.children('.tracker');
        if (options.movable) {
            moveBox
                .enableTracking({
                    autoScroll: true,
                    borderNode: scrollNode,
                    borderMargin: -30,
                    borderSize: 60,
                    minSpeed: 10,
                    maxSpeed: 250
                })
                .on('tracking:start', trackingStartHandler);
        } else {
            moveBox
                .disableTracking()
                .off('tracking:start', trackingStartHandler);
        }

        // initialize resize tracking
        if (options.resizable) {
            selectionBox.children('.handle')
                .enableTracking({
                    autoScroll: true,
                    borderNode: scrollNode,
                    borderMargin: -30,
                    borderSize: 60,
                    minSpeed: 10,
                    maxSpeed: 250
                })
                .on('tracking:start', trackingStartHandler);
        } else {
            selectionBox.children('.handle')
                .disableTracking()
                .off('tracking:start', trackingStartHandler);
        }
    };

    // exports ================================================================

    return DrawingResize;

});
