/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/framework/view/editview',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/config',
     'io.ox/office/tk/control/button',
     'io.ox/office/tk/control/radiolist',
     'io.ox/office/framework/view/baseview',
     'io.ox/office/framework/view/basecontrols',
     'io.ox/office/framework/view/pane',
     'io.ox/office/framework/view/editsidepane',
     'io.ox/office/framework/view/toolbox',
     'io.ox/office/framework/view/editcontrols',
     'gettext!io.ox/office/framework',
     'less!io.ox/office/framework/view/editstyle.less'
    ], function (Utils, KeyCodes, Config, Button, RadioList, BaseView, BaseControls, Pane, EditSidePane, ToolBox, EditControls, gt) {

    'use strict';

    // class EditView =========================================================

    /**
     * @constructor
     *
     * @extends BaseView
     *
     * @param {EditApplication} app
     *  The application containing this view instance.
     *
     * @param {Object} [options]
     *  Additional options to control the appearance of the view. Supports all
     *  options that are supported by the base class BaseView.
     */
    function EditView(app, options) {

        var // self reference
            self = this,

            // the main side pane
            sidePane = null,

            // the debug pane
            debugPane = null,

            // tool pane floating over the top of the application pane
            overlayPane = null,

            // tool box in the upper overlay pane
            overlayToolBox = null;

        // base constructor ---------------------------------------------------

        BaseView.call(this, app, Utils.extendOptions(options, { initHandler: initHandler, deferredInitHandler: deferredInitHandler }));

        // private methods ----------------------------------------------------

        /**
         * Creates and initializes the debug pane.
         */
        function initDebugMode() {

            var // the operations log
                operationsNode = $('<table>'),
                // number of operations already logged
                operationsCount = 0,
                // operations not yet logged
                pendingOperations = [],
                // background loop logging all pending operations
                logOperationsTimer = null,
                // number of custom log messages already logged
                logCount = 0,
                // the debug info table
                infoNode = $('<table>').css('table-layout', 'fixed').append($('<colgroup>').append($('<col>', { width: '40px' }), $('<col>'))),
                // output node for the application state
                appStateNode = null,
                // output node for the operation state number
                osnNode = null;

            // scrolls the operations output down to its end
            function scrollDownOperationsNode() {
                var scrollNode = operationsNode.parent();
                scrollNode.scrollTop(scrollNode[0].scrollHeight);
            }

            // direct callback for logOperations(): called every time when logOperations() is called
            function registerOperations(event, operations, external) {
                // append the new operations to the pending operations queue
                pendingOperations = pendingOperations.concat(_(operations).map(function (operation) {
                    return { operation: operation, external: external };
                }));
            }

            // deferred callback for logOperations(): called once after current script execution ends
            function printSomeOperations() {

                // check if the background loop is already running
                if (logOperationsTimer) { return; }

                // create a new background loop that processes all pending operations
                logOperationsTimer = app.repeatDelayed(function () {

                    var // the operations to insert into the table
                        operations = pendingOperations.splice(0, 20),
                        // the number of operations applied successfully by the document model
                        successCount = app.getModel().getOperationsCount();

                    // abort the background loop, if all operations have been logged
                    if (operations.length === 0) { return Utils.BREAK; }

                    // log the next operations chunk
                    _(operations).each(function (entry) {

                        var operation = _.clone(entry.operation),
                            name = operation.name,
                            osn = ('osn' in operation) ? operation.osn : '',
                            opl = ('opl' in operation) ? operation.opl : '';

                        delete operation.name;
                        delete operation.osn;
                        delete operation.opl;
                        operation = JSON.stringify(operation).replace(/^\{(.*)\}$/, '$1').replace(/"(\w+)":/g, '$1:').replace(/ /g, '\xb7');
                        operationsCount += 1;
                        operationsNode.append($('<tr>')
                            .toggleClass('error', successCount < operationsCount)
                            .append(
                                $('<td>').text(_.noI18n(operationsCount)),
                                $('<td>').text(_.noI18n(entry.external ? 'E' : 'I')).attr('title', _.noI18n((entry.external ? 'Ex' : 'In') + 'ternal operation')),
                                $('<td>').text(_.noI18n(osn)).css('text-align', 'right'),
                                $('<td>').text(_.noI18n(opl)).css('text-align', 'right'),
                                $('<td>').text(_.noI18n(name)),
                                $('<td>').text(_.noI18n(operation))
                            ));
                    });
                    scrollDownOperationsNode();

                }, { delay: 50 });

                // forget reference to the timer, when all operations have been logged
                logOperationsTimer.always(function () { logOperationsTimer = null; });
            }

            // extend public interface
            self.addDebugInfoHeader = function (label) {
                infoNode.append($('<tr>').append($('<th>', { colspan: 2 }).text(_.noI18n(label))));
            };

            self.addDebugInfoNode = function (label) {
                var node = $('<td>');
                infoNode.append($('<tr>').append($('<td>').text(_.noI18n(label)), node));
                return node;
            };

            self.createDebugToolBox = function () {
                return this.createToolBox('debug', { label: _.noI18n('Debug'), visible: 'debug/enabled' })
                    .addGroup('debug/toggle', new Button({ icon: 'icon-bug', tooltip: _.noI18n('Toggle debug panel'), toggle: true }))
                    .addGroup('debug/highlight', new Button({ icon: 'icon-eye-open', tooltip: _.noI18n('Highlight DOM elements'), toggle: true }))
                    .addRightTab()
                    .addGroup('debug/action', new RadioList({ icon: 'icon-wrench',      label: _.noI18n('Actions') })
                        .createMenuSection('conn')
                        .createOptionButton('offline', { sectionId: 'conn', icon: 'icon-unlink', label: _.noI18n('Switch to offline mode') })
                        .createOptionButton('online', { sectionId: 'conn', icon: 'icon-link', label: _.noI18n('Switch to online mode') })
                        .createMenuSection('rt', { separator: true })
                        .createOptionButton('reset', { sectionId: 'rt', icon: 'icon-exclamation', label: _.noI18n('Simulate connection reset') })
                        .createOptionButton('timeout', { sectionId: 'rt', icon: 'icon-time', label: _.noI18n('Simulate connection timeout') })
                        .createOptionButton('hangup', { sectionId: 'rt', icon: 'icon-refresh', label: _.noI18n('Simulate connection hangup') })
                        .createMenuSection('app', { separator: true })
                        .createOptionButton('invalid-op', { sectionId: 'app', icon: 'icon-minus-sign', label: _.noI18n('Apply invalid operation') })
                        .createOptionButton('quit-unsaved', { sectionId: 'app', icon: 'icon-warning-sign', label: _.noI18n('Show unsaved changes dialog and quit') })
                        .createOptionButton('quit-error', { sectionId: 'app', icon: 'icon-remove-sign', label: _.noI18n('Simulate quit with error') })
                    )
                    .newLine();
            };

            self.log = function (title, message) {
                logCount += 1;
                operationsNode.append($('<tr>').addClass('log').append(
                    $('<td>').text(_.noI18n(logCount)),
                    $('<td>').text(_.noI18n('L')),
                    $('<td>'),
                    $('<td>'),
                    $('<td>').text(_.noI18n(title)),
                    $('<td>').text(_.noI18n(message))));
                scrollDownOperationsNode();
            };

            self.isDebugPaneVisible = function () {
                return debugPane.isVisible();
            };

            self.toggleDebugPane = function (state) {
                debugPane.toggle(state);
                if (state) { scrollDownOperationsNode(); }
                return this;
            };

            self.isDebugHighlight = function () {
                return app.getWindowNode().hasClass('debug-highlight');
            };

            self.toggleDebugHighlight = function (state) {
                app.getWindowNode().toggleClass('debug-highlight', state);
                return this;
            };

            // create the debug pane
            debugPane = new Pane(app, 'debug', { position: 'bottom', classes: 'debug user-select-text' });

            // add the operations table and the info table to the debug pane
            self.addPane(debugPane.hide());
            debugPane.hide().getNode().append(
                $('<table>').append(
                    $('<colgroup>').append($('<col>', { width: '70%' })),
                    $('<tr>').append(
                        $('<td>').append($('<div>', { tabindex: 0 }).append(operationsNode)),
                        $('<td>').append($('<div>', { tabindex: 0 }).append(infoNode))
                    )
                )
            );

            // log the application state
            self.addDebugInfoHeader('Application');
            appStateNode = self.addDebugInfoNode('state');
            app.on('docs:state', function (state) { appStateNode.text(_.noI18n(state)); });

            // log the operation state number of the document
            self.addDebugInfoHeader('Document');
            osnNode = self.addDebugInfoNode('osn');
            app.getModel().on('operations:after', function () {
                osnNode.text(_.noI18n(app.getModel().getOperationStateNumber()));
            });

            // log all applied operations
            operationsNode.append($('<tr>').append(
                $('<th>', { colspan: 2 }),
                $('<th>').text(_.noI18n('OSN')).attr('title', _.noI18n('Operation State Number')).css('font-size', '70%'),
                $('<th>').text(_.noI18n('OPL')).attr('title', _.noI18n('Operation Length')).css('font-size', '70%'),
                $('<th>').text(_.noI18n('Name')),
                $('<th>').text(_.noI18n('Parameters'))));

            // simply collect all operations while importing the document
            app.getModel().on('operations:after', registerOperations);

            // log operations after importing the document
            app.on('docs:import:after', function () {
                // create the debounced logOperations() method
                var logOperations = app.createDebouncedMethod(registerOperations, printSomeOperations);
                // reconnect the 'operations:after' event to the debounced method
                app.getModel().off('operations:after', registerOperations).on('operations:after', logOperations);
                // trigger deferred logging by calling 'logOperations' once without new operations
                logOperations({}, []);
            });

            // keyboard handling for debug pane
            operationsNode.add(infoNode).parent().on('keydown', function (event) {
                if (KeyCodes.matchKeyCode(event, 'A', { ctrlOrMeta: true })) {
                    var docRange = window.document.createRange();
                    docRange.setStart(event.delegateTarget, 0);
                    docRange.setEnd(event.delegateTarget, 1);
                    window.getSelection().removeAllRanges();
                    window.getSelection().addRange(docRange);
                    return false;
                }
                if (event.keyCode === KeyCodes.ESCAPE) {
                    self.grabFocus();
                    return false;
                }
            });
        }

        /**
         * Updates the visibility of control groups in the overlay pane
         * according to the current document edit mode.
         */
        function updateOverlayVisibility() {

            var // whether application is in internal error mode
                internalError = app.getState() === 'error',
                // current document edit mode
                editMode = app.getModel().getEditMode(),
                // all visible group nodes
                visibleGroupNodes = $(),
                // all hidden group nodes
                hiddenGroupNodes = $();

            // select hidden and visible groups according to application state
            overlayToolBox.iterateGroups(function (group) {

                var visibility = ' ' + Utils.getStringOption(group.getOptions(), 'visibility', 'default') + ' ',
                    visible = (internalError ? /\s(always|error)\s/ : app.isLocked() ? /\s(always)\s/ : editMode ? /\s(always|default|editmode)\s/ : /\s(always|default|readonly)\s/).test(visibility);

                if (visible) {
                    visibleGroupNodes = visibleGroupNodes.add(group.getNode());
                } else {
                    hiddenGroupNodes = hiddenGroupNodes.add(group.getNode());
                }
            });

            // hide and show all groups according to the passed edit mode
            visibleGroupNodes.closest('.group-container').removeClass('hidden');
            hiddenGroupNodes.closest('.group-container').addClass('hidden');
        }

        /**
         * Builds generic contents of this edit view instance: the side pane,
         * the overlay panes, and the debug pane. After that, calls the passed
         * callback handler to build the application-specific view elements.
         */
        function initHandler() {

            var // the application status label
                statusLabel = new BaseControls.StatusLabel(app);

            // Disable browser table controls after edit mode has changed. Execute
            // delayed to be sure to run after other DOM changes (e.g. changing the
            // content-editable mode will enable the edit handlers again).
            app.getModel().on('change:editmode', function () { self.disableBrowserEditControls(); });
            app.on('docs:import:before', function () { self.disableBrowserEditControls(); });

            // create the side pane (initializes itself internally)
            self.addPane(sidePane = new EditSidePane(app));

            // create the overlay pane
            self.addPane(overlayPane = new Pane(app, 'overlaytop', { position: 'top', classes: 'inline right', overlay: true, transparent: true, hoverEffect: true }));
            overlayPane
                .addViewComponent(overlayToolBox = new ToolBox(app, 'overlaymain')
                    .addGroup('app/view/sidepane', new Button(BaseControls.SHOW_SIDEPANE_OPTIONS))
                    .addGap()
                );

            // create the second overlay pane containing the status label
            self.addPane(new Pane(app, 'statuslabel', { position: 'top', classes: 'inline right', overlay: true, transparent: true })
                .addViewComponent(new ToolBox(app, 'statuslabel', { focusable: false }).addPrivateGroup(statusLabel))
            );

            // initialize the status label
            app.on('docs:state', function (state) {
                switch (state) {
                case 'sending':
                    statusLabel.setValue(gt('Saving changes'), { type: 'warning', delay: 1000 });
                    break;
                case 'ready':
                    statusLabel.setValue(gt('All changes saved'), { type: 'success', fadeOut: true });
                    break;
                case 'error':
                    statusLabel.setValue(null);
                    self.lockPaneLayout(function () {
                        self.toggleSidePane(false);
                        self.toggleDebugPane(true);
                        app.getWindow().setChromeless(true);
                        updateOverlayVisibility();
                    });
                    break;
                default:
                    statusLabel.setValue(null);
                }
            });

            // show/hide controls in overlay pane according to edit mode
            app.getModel().on('change:editmode', updateOverlayVisibility);

            // initially, show the side pane if there is enough room
            self.toggleSidePane(window.innerWidth >= 1000);

            // additions for debug mode
            if (Config.isDebug()) { initDebugMode(); }

            // call initialization handler passed by sub class
            Utils.getFunctionOption(options, 'initHandler', $.noop).call(self);
        }

        /**
         * Initialization after importing the document. Creates all tool boxes
         * in the side pane and overlay pane. Needed to be executed after
         * import, to be able to hide specific GUI elements depending on the
         * file type.
         */
        function deferredInitHandler() {

            // call initialization handler passed by sub class
            Utils.getFunctionOption(options, 'deferredInitHandler', $.noop).call(self);

            // add remaining controls to the overlay tool box for edit mode
            overlayToolBox
                .addGap()
                .addGroup('document/acquireedit', new EditControls.AcquireEditButton(app))
                .addGap()
                .addGroup('document/reload', new Button(EditControls.RELOAD_OPTIONS))
                .addGap()
                .addGroup('document/undo', new Button(EditControls.UNDO_OPTIONS))
                .addGroup('document/redo', new Button(EditControls.REDO_OPTIONS))
                .addGap()
                .addGroup('app/quit', new Button(Utils.extendOptions(BaseControls.QUIT_OPTIONS, { visibility: 'always' })));

            // initialize visibility of the overlay groups
            updateOverlayVisibility();
        }

        // methods ------------------------------------------------------------

        /**
         * Disables the internal resize handles shown for of tables, drawing
         * objects, etc. provided by the browser edit mode (content-editable
         * elements).
         *
         * @returns {EditView}
         *  A reference to this instance.
         */
        this.disableBrowserEditControls = function () {
            // execute delayed to prevent conflicts with ongoing event handling
            app.executeDelayed(function () {
                try {
                    // disable FireFox table manipulation handlers
                    window.document.execCommand('enableObjectResizing', false, false);
                    window.document.execCommand('enableInlineTableEditing', false, false);
                } catch (ex) {
                }
            });
            return this;
        };

        /**
         * Creates a tool box in the side pane, and registers it for automatic
         * visibility handling.
         *
         * @param {String} id
         *  The identifier for the new tool box. Must be unique across all view
         *  components in the application.
         *
         * @param {Object} [options]
         *  A map of options to control the properties of the new tool box.
         *  Supports all options supported by the method
         *  SidePane.createToolBox().
         *
         * @returns {ToolBox}
         *  The new tool box instance.
         */
        this.createToolBox = function (id, options) {
            return sidePane.createToolBox(id, options);
        };

        /**
         * Returns the customizable tool box in the overlay pane.
         *
         * @returns {ToolBox}
         *  The tool box in the overlay pane.
         */
        this.getOverlayToolBox = function () {
            return overlayToolBox;
        };

        /**
         * Returns whether the main side pane is currently visible.
         *
         * @returns {Boolean}
         *  Whether the main side pane is currently visible.
         */
        this.isSidePaneVisible = function () {
            return sidePane.isVisible();
        };

        /**
         * Changes the visibility of the main side pane and the overlay tool
         * box. If the side pane is visible, the overlay tool box is hidden,
         * and vice versa.
         *
         * @param {Boolean} state
         *  Whether to show or hide the main side pane.
         *
         * @returns {EditView}
         *  A reference to this instance.
         */
        this.toggleSidePane = function (state) {
            this.lockPaneLayout(function () {
                sidePane.toggle(state);
                overlayPane.toggle(!state);
                app.getWindow().setChromeless(!state);
            });
            return this;
        };

        /**
         * Returns whether the debug pane is currently visible.
         *
         * @returns {Boolean}
         *  Whether the debug pane is currently visible.
         */
        this.isDebugPaneVisible = function () {
            // will be replaced with actual implementation in initDebugMode()
            return false;
        };

        /**
         * Changes the visibility of the debug pane.
         *
         * @param {Boolean} state
         *  Whether to show or hide the debug pane.
         *
         * @returns {EditView}
         *  A reference to this instance.
         */
        this.toggleDebugPane = function (state) {
            // will be replaced with actual implementation in initDebugMode()
            return this;
        };

        /**
         * Returns whether the debug highlighting mode is currently active.
         *
         * @returns {Boolean}
         *  Whether the debug highlighting mode is currently active.
         */
        this.isDebugHighlight = function () {
            // will be replaced with actual implementation in initDebugMode()
            return false;
        };

        /**
         * Toggles the debug highlighting of specific DOM elements.
         *
         * @param {Boolean} state
         *  Whether to highlight specific DOM elements.
         *
         * @returns {EditView}
         *  A reference to this instance.
         */
        this.toggleDebugHighlight = function (state) {
            // will be replaced with actual implementation in initDebugMode()
            return this;
        };

        /**
         * Creates a new editable container node for clipboard functionality,
         * and inserts it into the hidden container node of the application.
         *
         * @returns {jQuery}
         *  The new empty clipboard container node.
         */
        this.createClipboardNode = function () {
            // clipboard node must be content-editable to allow system copy-to-clipboard
            // clipboard node must contain 'user-select-text' to be focusable in Chrome
            var clipboardNode = $('<div>', { contenteditable: true, 'data-focus-role': 'clipboard' }).addClass('clipboard user-select-text');
            this.insertHiddenNode(clipboardNode);
            return clipboardNode;
        };

        this.log = $.noop;

    } // class EditView

    // exports ================================================================

    // derive this class from class BaseView
    return BaseView.extend({ constructor: EditView });

});
