/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2011
 * Mail: info@open-xchange.com
 *
 * @author Kai Ahrens <kai.ahrens@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/framework/app/rtconnection',
    ['io.ox/core/event',
     'io.ox/realtime/groups',
     'io.ox/office/tk/config',
     'io.ox/office/tk/utils',
     'gettext!io.ox/office/framework'
    ], function (Events, RTGroups, DocConfig, Utils, gt) {

    'use strict';

    var // names of internal events for the real-time connection state
        INTERNAL_EVENTS = 'online offline reset',

        // names of runtime requests expecting an answer
        REQUEST_EVENTS = 'addresource getdocument',

        // names of runtime push messages
        PUSH_EVENTS = 'update hangup preparelosingeditrights declinedacquireeditrights acceptedacquireeditrights';

    // class RTConnection =====================================================

    /**
     * Represents the connection to the real-time framework, used to send
     * message to and receive message from the server.
     *
     * Triggers the following events:
     * - 'update': Update notifications containing actions created by remote
     *  clients, and state information for the application (read-only, name and
     *  identifier of the current editor, etc.).
     *
     * @extends Events
     *
     * @param {EditApplication} app
     *  The application containing this operations queue.
     *
     * @param {Object} options
     *  A map of options to be called for certain states of the realtime
     *  connection.
     *  @param {Number} [options.realTimeDelay=700]
     *      The duration in milliseconds the realtime framework will collect
     *      messages before sending them to the server.
     */
    function RTConnection(app, options) {

        var // self reference
            self = this,

            // file descriptor of the edited document
            file = app.getFileDescriptor(),

            // create the core real-time group instance for the document
            rtGroup = RTGroups.getGroup('synthetic.office://operations/' + file.folder_id + '.' + file.id),

            // private event hub for internal event handling
            eventHub = new Events(),

            // duration for the realtime framework collecting messages before sending
            realTimeDelay = Utils.getIntegerOption(options, 'realTimeDelay', 700, 0),

            traceActions = {};

        // base constructor ---------------------------------------------------

        Events.extend(this);

        // private methods ----------------------------------------------------

        /**
         * Register events for specific RT actions.
         * The event handler 'eventName' is triggered, when
         * an RT event with the action name 'eventName' is received.
         * The triggered eventName will be equal to the 'actionName'.
         * The eventHandler will be called with the received payload data
         */
        function registerRTEvents(eventNames) {
            _(eventNames.split(/\s+/)).each(function (eventName) {
                rtGroup.on('receive', function (event, stanza) {
                    if (RTConnection.debug) {
                        Utils.log('RTConnection.rtGroup.on receive notification');
                    }
                    _(stanza.getAll('office', eventName)).each(function (payload) {
                        if (RTConnection.debug) {
                            Utils.log('RTConnection.eventHub.trigger: office:' + eventName);
                        }
                        eventHub.trigger(eventName, payload.data);
                    });
                });
            });
        }

        /**
         * Determines whether a certain action should be traced on its way
         * through the realtime backend.
         *
         * @returns {Boolean}
         *  Whether to trace the stanzas for the specified action.
         */
        function shouldTrace(action) {
            return traceActions[action.toLowerCase()];
        }

        /**
         * Send a message to the connected RT object,
         * waiting for the acknowledge to this call.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  acknowledge for this request will have arrived, or
         *  rejected after a time out otherwise.
         */
        function send(action, element, data) {

            var payloads = [{ element: 'action', data: action }],
                defer;

            if (_.isString(element) && !_.isUndefined(data)) {
                payloads.push({
                    namespace: 'office',
                    element: element,
                    data: data,
                    verbatim: true
                });
            }

            defer = rtGroup.send({
                element: 'message',
                payloads: payloads,
                trace: shouldTrace(action),
                bufferinterval: realTimeDelay
            });

            // Time out detection now uses the deferred to trigger an optional timeout handler
            defer.fail(function () {
                if (RTConnection.debug) {
                    Utils.error('Send message with RT failed, payloads: ' + payloads.toString());
                }
                self.trigger('timeout');
            });

            return defer.promise();
        }

        /**
         * Send a message request to the connected RT object,
         * returning a promise object that will be resolved,
         * if the result data to this request has arrived.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  result for this request has arrived, or rejected otherwise.
         */
        function sendQuery(action, element, data) {
            var payloads = [{ element: 'action', data: action }],
                defer;

            if (_.isString(element) && !_.isUndefined(data)) {
                payloads.push({
                    namespace: 'office',
                    element: element,
                    data: data,
                    verbatim: true
                });
            }

            defer = rtGroup.query({
                element: 'message',
                payloads: payloads,
                trace: shouldTrace(action),
                bufferinterval: realTimeDelay
            });

            // Time out detection now uses the deferred to trigger an optional timeout handler
            defer.fail(function () {
                if (RTConnection.debug) {
                    Utils.error('Send query message with RT failed, payloads: ' + payloads.toString());
                }
                self.trigger('timeout');
            });

            return defer.promise();
        }

        /**
         * Send a message request to the connected RT object,
         * returning a deferred object that will be resolved,
         * if the result data to this request will have arrived.
         * @deprecated use sendQuery instead
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  result for this request has arrived, or rejected otherwise.
         */
        function sendRequest(action, element, data) {

            var // the result Deferred object
                def = $.Deferred();

            data = Utils.extendOptions(data, { uniqueId: _.uniqueId() });

            // resolves the Deferred object with the 'action' result
            function requestHandler(event, resultData) {
                // stop listening to 'requestHandler' events if we received exactly
                // one event with the message id we sent with the initial request
                if (_.isObject(resultData)) {
                    if (RTConnection.debug) {
                        Utils.log("RTConnection.sendRequest: Answer for sendRequest arrived: " + action + " , expected uniqueId: " + data.uniqueId + ', received uniqueId:' + resultData.uniqueId);
                    }
                    if (_.isString(resultData.uniqueId) && (resultData.uniqueId === data.uniqueId)) {
                        stopListening();
                        if (RTConnection.debug) {
                            Utils.log("RTConnection.sendRequest: resolve deferred with resultData");
                        }
                        def.resolve(resultData);
                    }
                } else {
                    if (RTConnection.debug) {
                        Utils.error("RTConnection.sendRequest: Unexpected answer - deferred rejected");
                    }
                    def.reject();
                }
            }

            // stops listening to action events
            function stopListening() {
                eventHub.off(action, requestHandler);
            }

            // resolves the Deferred object with the 'action' result
            eventHub.on(action, requestHandler);

            // connect and listen to the 'action' result
            if (RTConnection.debug) {
                Utils.log("RTConnection.sendRequest: Sending request: " + action + ", uniqueId: " + data.uniqueId);
            }
            send(action, element, data).fail(function () {
                if (def.state() === 'pending') {
                    def.reject({cause: 'timeout'});
                }
            });
            if (RTConnection.debug) {
                Utils.log("RTConnection.sendRequest: Waiting for answer to request " + action + " ,uniqueId: " + data.uniqueId);
            }

            // return a Promise with an abort() extension that disconnects the
            // event listener, leaving the Deferred in pending state
            return _.extend(def.promise(), { abort: stopListening });
        }

        /**
         * Handles the response from the RT low-level code and extract the
         * data part from, providing it as result via the deferred/promise.
         *
         * @param {jQuery.Deferred|jQuery.Promise} orgDeferred
         *  The original promise or deferred provided by the RT low-level
         *  code.
         *
         * @param {Number} [timeout]
         *  If specified, the delay time in milliseconds, after the Promise
         *  returned by this method will be rejected, if the passed original
         *  Deferred object is still pending.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  result for the original request has arrived, or rejected otherwise.
         */
        function handleResponse(orgDeferred, timeout) {

            var // the Deferred object to be returned
                def = $.Deferred();

            orgDeferred.then(function (response) {
                var payloads = Utils.getArrayOption(response, 'payloads', []);
                if (payloads.length > 0) {
                    // Extract the data and provide it
                    def.resolve(payloads[0].data);
                } else {
                    // Response from server is not in the expected format, therefore
                    // we reject the deferred here.
                    def.reject(response);
                }
            }, function (response) {
                var payloads = Utils.getArrayOption(response, 'payloads', []);
                if (payloads.length > 0) {
                    // Extract data even for the rejected case
                    def.reject(payloads[0].data);
                } else {
                    // If we don't find a payload just provide the original response
                    def.reject(response);
                }
            });

            // prevent endless waiting
            if (_.isNumber(timeout) && (timeout > 0)) {
                app.executeDelayed(function () { def.reject({ cause: 'timeout' }); }, { delay: timeout });
            }

            return def.promise();
        }

        /**
         * Handles the response from the RT low-level code and adds timeout
         *
         * @param {jQuery.Deferred|jQuery.Promise} orgDeferred
         *  The original promise or deferred provided by the RT low-level
         *  code.
         *
         * @param {Number} [timeout]
         *  If specified, the delay time in milliseconds, after the Promise
         *  returned by this method will be rejected, if the passed original
         *  Deferred object is still pending.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  result for the original request has arrived, or rejected otherwise.
         */
        function handleGenericResponse(orgDeferred, timeout) {
            var // the Deferred object to be returned
                def = $.Deferred();

            orgDeferred.then(function (data) { def.resolve(data); }, function (data) { def.reject(data); });

            // prevent endless waiting
            if (_.isNumber(timeout) && (timeout > 0)) {
                app.executeDelayed(function () { def.reject({ cause: 'timeout' }); }, { delay: timeout });
            }

            return def.promise();
        }

        // methods ------------------------------------------------------------

        /**
         * Connects the real-time group and sends the initial request for the
         * document import operations.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved with the
         *  getActions results.
         */
        this.connect = function () {
            if (RTConnection.debug) {
                Utils.log("RTConnection.connect called");
            }
            return handleResponse(rtGroup.join({expectWelcomeMessage: true}));
        };

        /**
         * Send the given operations to the connected RT object
         * for further distribution to other connected clients
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  ACK of the internal send arrives.
         */
        this.sendActions = function (actions) {
            if (RTConnection.debug) {
                Utils.log("RTConnection.applyactions called");
            }
            return send('applyactions', 'actions', actions);
        };

        /**
         * Sends a real-time request to the server and waits for a response.
         *
         * @param {String} action
         *  The identifier of the action sent to the server.
         *
         * @param {Number} [timeout]
         *  If specified, the delay time in milliseconds, after the Promise
         *  returned by this method will be rejected, if the query is still
         *  pending.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved with the
         *  answer of the server request, or rejected on error or timeout.
         */
        this.sendQuery = function (action, timeout) {
            if (RTConnection.debug) {
                Utils.log('RTConnection.sendQuery(' + action + ')');
            }
            return handleGenericResponse(sendQuery(action), timeout);
        };

        /**
         * Add the given resource id to the connected RT object
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved with the
         *  result data of the request.
         */
        this.addResourceId = function (resourceId) {
            if (RTConnection.debug) {
                Utils.log("RTConnection.addResourceId called");
            }
            return sendRequest('addresource', 'resource', { resourceid: resourceId });
        };

        /**
         * Get the resource id of the connected RT object for the specified document
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved with the
         *  result data of the request.
         */
        this.getDocumentId = function (data) {
            if (RTConnection.debug) {
                Utils.log("RTConnection.getDocumentId called");
            }
            return sendRequest('getdocument', 'document', Utils.extendOptions(data, { returntype: "resourceid" }));
        };

        /**
         * Send our wish to acquire the edit rights to the connected RT object
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  ACK of the internal send arrives.
         */
        this.acquireEditRights = function () {
            if (RTConnection.debug) {
                Utils.log("RTConnection.acquireedit called");
            }
            return send('acquireedit', 'state', { osn: app.getModel().getOperationStateNumber() });
        };

        /**
         * Acknowledge to the server that our preparation to lose the
         * edit rights are completed.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  ACK of the internal send arrives.
         */
        this.canLoseEditRights = function () {
            if (RTConnection.debug) {
                Utils.log("RTConnection.canloseeditrights called");
            }
            return send('canloseeditrights', 'state', { osn: app.getModel().getOperationStateNumber() });
        };

        /**
         * Close the document at the connected RT object
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved with the
         *  closing document status of the request.
         */
        this.closeDocument = function () {
            if (RTConnection.debug) {
                Utils.log("RTConnection.closeDocument called");
            }
            return handleResponse(rtGroup.leave({ expectSignOffMessage: true }), 20000);
        };

        /**
         * Disconnect from the RT object, no further calls
         * should be made at this object from now on
         */
        this.destroy = function () {
            try {
                rtGroup.destroy();
                if (RTConnection.debug) {
                    Utils.log('RTConnection.destroy(): RT group destroyed');
                }
            } catch (ex) {
                Utils.exception(ex);
            }
            eventHub.destroy();
            this.events.destroy();
        };

        // initialization -----------------------------------------------------

        if (RTConnection.debug) {
            Utils.log("RTConnection initialization");
        }

        // read traceable actions from page URL
        if (_.url.hash("office:trace")) {
            _(_.url.hash("office:trace").split(/\s*,\s*/)).each(function (action) {
                traceActions[action.toLowerCase()] = true;
            });
        }

        // register the internal event hub for request events and custom runtime events
        registerRTEvents(REQUEST_EVENTS + ' ' + PUSH_EVENTS);

        // forward runtime events to own listeners
        eventHub.on(INTERNAL_EVENTS + ' ' + PUSH_EVENTS, function (event, data) {
            self.trigger(event.type, data);
        });

    } // class RTConnection

    // global initialization ==================================================

    // enable realtime debugging
    require(['io.ox/realtime/rt']).done(function (Realtime, Config) {
        // whether to show console output of real-time core
        Realtime.debug = DocConfig.isDebug();
        // whether to show console output of this RTConnection class
        RTConnection.debug = DocConfig.isDebug();
    });

    // exports ================================================================

    return RTConnection;
});
