/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 * © 2012 Open-Xchange Inc., Tarrytown, NY, USA. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/framework/app/fileactions',
    ['io.ox/core/extensions',
     'io.ox/core/extPatterns/links',
     'io.ox/office/tk/utils',
     'io.ox/office/framework/app/extensionregistry',
     'gettext!io.ox/office/files'
    ], function (ext, links, Utils, ExtensionRegistry, gt) {

    'use strict';

    // static private functions ===============================================

    function disableActionForInvalidFiles(actionId) {
        new links.Action(actionId, {
            id: 'disable_action',
            index: 'first',
            requires: function (data) {
                if (ExtensionRegistry.isError(data.baton.data.filename)) {
                    data.stopPropagation();
                    return false;
                }
            }
        });
    }

    // static class FileActions ===============================================

    var FileActions = {};

    // methods ----------------------------------------------------------------

    /**
     * Creates all required actions and links in the OX Files application for a
     * specific OX Documents application.
     *
     * @param {String} appBaseName
     *  The base name of the application (name of the application source code
     *  directory).
     *
     * @param {Object} options
     *  A map with options controlling the behavior of the created extensions
     *  in the Files application. The following options are supported:
     *  @param {String} options.newDocumentLabel
     *      The string shown for the 'Create new document' action link inserted
     *      into the Files tool bar.
     *  @param {Number} options.newDocumentIndex
     *      The index of the 'Create new document' action link that will be
     *      inserted into the Files tool bar. The 'Create new document' action
     *      links of the various OX Document applications will be ordered
     *      increasing by this index.
     */
    FileActions.create = function (appBaseName, options) {

        var // root for extension points in Files application
            ACTION_POINT = 'io.ox/files/actions/office/' + appBaseName,
            // the module name of the edit application
            MODULE_NAME = 'io.ox/office/' + appBaseName;

        // private methods ----------------------------------------------------

        /**
         * Launches a new OX Documents application with the passed options.
         */
        function launchApplication(options) {
            ox.launch(MODULE_NAME + '/main', options);
        }

        // initialization -----------------------------------------------------

        // Creates a new empty document in the current folder of the Files application.
        if (ExtensionRegistry.supportsEditMode(appBaseName)) {
            new links.Action(ACTION_POINT + '/new', {
                action: function (baton) {
                    launchApplication({ action: 'new', folderId: baton.app.folder.get() });
                }
            });
        }

        // Loads an existing document from the current file in the Files application.
        // If the document requires conversion to a supported file format, creates the
        // converted file and loads it.
        new links.Action(ACTION_POINT + '/edit', {
            requires: function (e) {
                return e.collection.has('one', 'read', 'modify') && ExtensionRegistry.isEditable(e.context.filename, MODULE_NAME);
            },
            filter: function (obj) {
                return ExtensionRegistry.isEditable(obj.filename, MODULE_NAME);
            },
            action: function (baton) {
                if (ExtensionRegistry.isNative(baton.data.filename)) {
                    launchApplication({ action: 'load', file: baton.data });
                } else {
                    launchApplication({ action: 'convert', folderId: baton.data.folder_id, templateFile: baton.data, preserveFileName: true });
                }
            }
        });

        // Creates a new document as copy of the current file in the Files application.
        if (ExtensionRegistry.supportsEditMode(appBaseName)) {
            new links.Action(ACTION_POINT + '/editasnew', {
                requires: function (e) {
                    return e.collection.has('one', 'read') && !ExtensionRegistry.isError(e.context.filename) && ExtensionRegistry.isEditable(e.context.filename, MODULE_NAME);
                },
                filter: function (obj) {
                    return ExtensionRegistry.isEditable(obj.filename, MODULE_NAME);
                },
                action: function (baton) {
                    var convert = ExtensionRegistry.isConvertible(baton.data.filename);
                    launchApplication({ action: convert ? 'convert' : 'new', folderId: baton.data.folder_id, templateFile: baton.data });
                }
            });
        }

        // An action link in the main tool bar of the Files application used
        // to create a new empty document.
        new links.ActionLink('io.ox/files/links/toolbar/default', {
            index: 200 + Utils.getIntegerOption(options, 'newDocumentIndex', 0),
            id: appBaseName + '_new',
            label: Utils.getStringOption(options, 'newDocumentLabel', ''),
            ref: ACTION_POINT + '/new'
        });

        // An action link in the details view of a file used to load the file.
        ext.point('io.ox/files/links/inline').extend(new links.Link({
            id: appBaseName + '_edit',
            index: 150,
            prio: 'hi',
            label: gt('Edit'),
            ref: ACTION_POINT + '/edit'
        }));

        // An action link in the details view of a file used to create a new
        // document based on that file.
        ext.point('io.ox/files/links/inline').extend(new links.Link({
            id: appBaseName + '_editasnew',
            prio: 'lo',
            index: 'first',
            section: 'edit',
            label: gt('Edit as new'),
            ref: ACTION_POINT + '/editasnew'
        }));

    };

    // static initialization ==================================================

    disableActionForInvalidFiles('io.ox/files/actions/add-to-portal');
    disableActionForInvalidFiles('io.ox/files/actions/publish');

    // exports ================================================================

    return FileActions;

});
