/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/framework/app/editcontroller',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/config',
     'io.ox/office/framework/app/basecontroller'
    ], function (Utils, Config, BaseController) {

    'use strict';

    // class EditController ===================================================

    /**
     * The base class for controller classes of all OX Documents allowing to
     * edit a document.
     *
     * @constructor
     *
     * @extends BaseController
     *
     * @param {EditApplication} app
     *  The OX Documents application that has created this controller instance.
     *
     * @param {Object} [options]
     *  A map with options controlling the behavior of this controller.
     *  Supports all options supported by the base class BaseController.
     */
    function EditController(app, options) {

        var // self reference
            self = this,

            // the model instance of the passed application
            model = null,

            // the view instance of the passed application
            view = null,

            // all the little controller items
            items = {

                // application and view ---------------------------------------

                // toggle the main side pane
                'app/view/sidepane': {
                    enable: function () { return app.getState() !== 'error'; },
                    get: function () { return app.getView().isSidePaneVisible(); },
                    set: function (state) { app.getView().toggleSidePane(state); },
                    shortcut: { keyCode: 'F3', ctrlOrMeta: true, value: function (state) { return !state; } }
                },

                // document ---------------------------------------------------

                'document/acquireedit': {
                    enable: function () { return !app.isLocked() && !model.getEditMode(); },
                    set: function () { app.acquireEditRights(); }
                },

                'document/reload': {
                    enable: function () { return app.getState() === 'error'; },
                    set: function () { app.reloadDocument(); }
                },

                // to be used as parent item for all items that require edit mode
                'document/editable': {
                    enable: function () { return model.getEditMode(); }
                },

                'document/rename': {
                    parent: 'document/editable',
                    get: function () { return app.getShortFileName(); },
                    set: function (fileName) { return app.rename(fileName); }
                },

                'document/undo': {
                    parent: 'document/editable',
                    enable: function () { return model.undoAvailable() > 0; },
                    set: function () { model.undo(1); },
                    shortcut: [
                        { keyCode: 'Z', ctrlOrMeta: true },
                        { keyCode: 'BACKSPACE', alt: true }
                    ]
                },

                'document/redo': {
                    parent: 'document/editable',
                    enable: function () { return model.redoAvailable() > 0; },
                    set: function () { model.redo(1); },
                    shortcut: [
                        { keyCode: 'Y', ctrlOrMeta: true },
                        { keyCode: 'BACKSPACE', shift: true, alt: true }
                    ]
                },

                // debug mode -------------------------------------------------

                'debug/enabled': {
                    enable: function () { return Config.isDebug(); }
                },

                'debug/toggle': {
                    parent: 'debug/enabled',
                    get: function () { return view.isDebugPaneVisible(); },
                    set: function (state) { view.toggleDebugPane(state); }
                },

                'debug/highlight': {
                    parent: 'debug/enabled',
                    get: function () { return view.isDebugHighlight(); },
                    set: function (state) { view.toggleDebugHighlight(state); }
                },

                'debug/action': {
                    parent: 'debug/enabled',
                    set: function (action) { app.triggerDebugAction(action); }
                }
            };

        // base constructor ---------------------------------------------------

        BaseController.call(this, app, options);

        // initialization -----------------------------------------------------

        // register item definitions
        this.registerDefinitions(items);

        // initialization after construction (model not available at construction time)
        app.on('docs:init', function () {
            model = app.getModel();
            view = app.getView();
        });

        // update GUI after operations, or changed state of edit mode
        app.on('docs:import:after', function () {
            model.on('operations:after change:editmode', function () { self.update(); });
        });

    } // class EditController

    // exports ================================================================

    // derive this class from class BaseController
    return BaseController.extend({ constructor: EditController });

});
