/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mobile.api.facade.osgi;

import com.openexchange.config.ConfigurationService;
import com.openexchange.config.Interests;
import com.openexchange.config.Reloadable;
import com.openexchange.config.Reloadables;
import com.openexchange.mobile.api.facade.auth.AuthenticationService;
import com.openexchange.mobile.api.facade.auth.impl.DefaultAuthenticationService;
import com.openexchange.mobile.api.facade.configuration.ConfigurationHelper;
import com.openexchange.mobile.api.facade.configuration.GlobalConfiguration;
import com.openexchange.mobile.api.facade.connectors.ConnectorFactoryFactory;
import com.openexchange.mobile.api.facade.connectors.impl.DefaultConnectorFactoryFactory;
import com.openexchange.mobile.api.facade.endpoints.AccountsEndpoint;
import com.openexchange.mobile.api.facade.endpoints.AccountsEndpointV2;
import com.openexchange.mobile.api.facade.endpoints.AuthenticationEndpoint;
import com.openexchange.mobile.api.facade.endpoints.ConfigEndpoint;
import com.openexchange.mobile.api.facade.endpoints.ContactsEndpoint;
import com.openexchange.mobile.api.facade.endpoints.FoldersEndpoint;
import com.openexchange.mobile.api.facade.endpoints.MailsEndpoint;
import com.openexchange.mobile.api.facade.endpoints.ManifestEndpoint;
import com.openexchange.mobile.api.facade.endpoints.PushEndpoint;
import com.openexchange.mobile.api.facade.endpoints.UsersEndpoint;
import com.openexchange.mobile.api.facade.endpoints.VersionEndpoint;
import com.openexchange.mobile.api.facade.exceptions.UnhandledExceptionMapper;
import com.openexchange.mobile.api.facade.services.AccountsService;
import com.openexchange.mobile.api.facade.services.ConfigService;
import com.openexchange.mobile.api.facade.services.ContactsService;
import com.openexchange.mobile.api.facade.services.FoldersService;
import com.openexchange.mobile.api.facade.services.MailsService;
import com.openexchange.mobile.api.facade.services.ManifestService;
import com.openexchange.mobile.api.facade.services.PushService;
import com.openexchange.mobile.api.facade.services.UsersService;
import com.openexchange.mobile.api.facade.services.VersionService;
import com.openexchange.mobile.api.facade.utils.VersionCheckerUtil;
import com.openexchange.mobile.api.facade.weakforce.WeakforceService;
import com.openexchange.osgi.HousekeepingActivator;

import lombok.extern.slf4j.Slf4j;

/**
 * {@link MobileApiFacadeActivator}
 *
 * @author <a href="mailto:michael.koch@open-xchange.com">Michael Koch</a>
 * @author <a href="mailto:daniel.boehrs@open-xchange.com">Daniel Boehrs</a>
 * @since v1.0.0
 */
@Slf4j
public class MobileApiFacadeActivator extends HousekeepingActivator implements Reloadable {

    @Override
    protected Class<?>[] getNeededServices() {
        return new Class<?>[] { ConfigurationService.class };
    }

    @Override
    protected void startBundle() throws Exception {
        log.info("starting bundle: \"com.openexchange.mobile.api.facade\"");
        try {
            ConfigurationService configurationService = getService(ConfigurationService.class);
            ConfigurationHelper.initialize(configurationService);
            ConnectorFactoryFactory connectorFactoryFactory = new DefaultConnectorFactoryFactory();
            if (GlobalConfiguration.versionCheckEnabled) {
                VersionCheckerUtil.check(connectorFactoryFactory);
            }
            registerServices(connectorFactoryFactory);
        } catch (Exception e) {
            log.error("failed to initialize bundle: \"com.openexchange.mobile.api.facade\": " + e.getMessage());
        }
    }

    private void registerServices(ConnectorFactoryFactory connectorFactoryFactory) {
        registerService(UnhandledExceptionMapper.class, new UnhandledExceptionMapper());

        ConfigService configService = new ConfigService(connectorFactoryFactory);
        registerService(ConfigEndpoint.class, new ConfigEndpoint(configService));

        WeakforceService weakforceService = new WeakforceService();

        MultipleAuthenticationServicesHandler authenticationHandler = new MultipleAuthenticationServicesHandler(weakforceService);
        track(AuthenticationService.class, authenticationHandler.getServiceListener());

        registerService(AuthenticationService.class, new DefaultAuthenticationService(connectorFactoryFactory));

        registerService(AuthenticationEndpoint.class, new AuthenticationEndpoint(authenticationHandler, configService));

        ContactsService contactsService = new ContactsService(connectorFactoryFactory);
        registerService(ContactsEndpoint.class, new ContactsEndpoint(contactsService));

        FoldersService foldersService = new FoldersService(connectorFactoryFactory);
        registerService(FoldersEndpoint.class, new FoldersEndpoint(foldersService));

        AccountsService accountsService = new AccountsService(connectorFactoryFactory, foldersService);
        foldersService.setAccountsService(accountsService);
        registerService(AccountsEndpoint.class, new AccountsEndpoint(accountsService));
        registerService(AccountsEndpointV2.class, new AccountsEndpointV2(accountsService));

        MailsService mailsService = new MailsService(connectorFactoryFactory);
        registerService(MailsEndpoint.class, new MailsEndpoint(mailsService));

        PushService pushService = new PushService(connectorFactoryFactory);
        registerService(PushEndpoint.class, new PushEndpoint(pushService));

        UsersService usersService = new UsersService(connectorFactoryFactory);
        registerService(UsersEndpoint.class, new UsersEndpoint(usersService));

        VersionService versionService = new VersionService(connectorFactoryFactory);
        registerService(VersionEndpoint.class, new VersionEndpoint(versionService));

        ManifestService manifestService = new ManifestService(connectorFactoryFactory, versionService);
        registerService(ManifestEndpoint.class, new ManifestEndpoint(manifestService));

        registerService(Reloadable.class, this);

        openTrackers();
    }

    @Override
    public void reloadConfiguration(ConfigurationService configurationService) {
        ConfigurationHelper.initialize(configurationService);
    }

    @Override
    public Interests getInterests() {
        return Reloadables.interestsForFiles(
                ConfigurationHelper.CONFIG_FACADE_PROPERTIES,
                ConfigurationHelper.CONFIG_MOBILE_API_FACADE_CONFIG_YAML
        );
    }

}
