/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mobile.api.facade.osgi;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import com.openexchange.java.Strings;
import com.openexchange.mobile.api.facade.auth.AuthenticationService;
import com.openexchange.mobile.api.facade.auth.impl.DefaultAuthenticationService;
import com.openexchange.mobile.api.facade.configuration.HostConfiguration;
import com.openexchange.mobile.api.facade.exceptions.ApiFacadeException;
import com.openexchange.mobile.api.facade.weakforce.WeakforceService;
import com.openexchange.mobile.api.facade.weakforce.WeakforceWrapperAuthenticationService;
import com.openexchange.osgi.SimpleRegistryListener;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import org.osgi.framework.ServiceReference;

@RequiredArgsConstructor
@Slf4j
public class MultipleAuthenticationServicesHandler {

    private final class AuthenticationServiceRegistryListener implements SimpleRegistryListener<AuthenticationService> {
        @Override
        public void added(ServiceReference<AuthenticationService> serviceReference, AuthenticationService service) {
            String name = service.getClass().getName();
            authenticationServices.put(name, service);
            log.debug("Registered authentication service: " + service.getClass().getName());
        }

        @Override
        public void removed(ServiceReference<AuthenticationService> serviceReference, AuthenticationService service) {
            String name = service.getClass().getName();
            authenticationServices.remove(name);
            log.debug("Unregistered authentication service: " + service.getClass().getName());
        }
    }

    private final WeakforceService weakforceService;

    /*private*/ final Map<String, AuthenticationService> authenticationServices = new ConcurrentHashMap<>();

    @Getter
    private final SimpleRegistryListener<AuthenticationService> serviceListener = new AuthenticationServiceRegistryListener();

    public AuthenticationService getAuthenticationService(HostConfiguration configuration) {
        log.debug("Retrieving authentication service");
        String name = configuration.getAuthenticationClassName();
        if (Strings.isEmpty(name)) {
            name = DefaultAuthenticationService.class.getName();
        }
        AuthenticationService service = authenticationServices.get(name);
        if (service == null) {
            throw ApiFacadeException.authenticationNotFound();
        }
        log.debug("Using authentication service " + service.getClass().getName());
        if (configuration.isWeakforceEnabled()) {
            service = new WeakforceWrapperAuthenticationService(configuration, weakforceService, service);
        }
        return service;
    }

}
