/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mobile.api.facade.utils;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import com.openexchange.mobile.api.facade.auth.LoginCredentials;
import com.openexchange.mobile.api.facade.configuration.HostConfiguration;
import com.openexchange.mobile.api.facade.exceptions.ApiFacadeException;

import lombok.experimental.UtilityClass;

@UtilityClass
public class WeakforceUtil {

    public String generatePasswordHash(HostConfiguration configuration, LoginCredentials credentials) {
        StringBuilder s = new StringBuilder(32)
                .append(configuration.getWeakforceHashNonce())
                .append(credentials.getUsername())
                .append('\0')
                .append(credentials.getPassword());
        byte[] digest = getDigest(s.toString(), configuration.getWeakforceHashMechanism());
        return toHexString(((digest[0] & 0xFF) << 4) | ((digest[1] & 0xFF) >> 4), 4);
    }

    /**
     * All possible chars for representing a number as a String
     */
    private final char[] digits = {
            '0', '1', '2', '3', '4', '5',
            '6', '7', '8', '9', 'a', 'b',
            'c', 'd', 'e', 'f', 'g', 'h',
            'i', 'j', 'k', 'l', 'm', 'n',
            'o', 'p', 'q', 'r', 's', 't',
            'u', 'v', 'w', 'x', 'y', 'z'
    };

    /**
     * Converts the two-byte integer to a HEX string.
     */
    private String toHexString(int i, int padding) {
        char[] buf = new char[16];
        int charPos = 16;
        int mask = (1 << 4) - 1;

        do {
            buf[--charPos] = digits[i & mask];
            i >>>= 4;
        } while (i != 0);

        while ((16 - charPos) < padding) {
            buf[--charPos] = '0';
        }

        return new String(buf, charPos, 16 - charPos);
    }

    /**
     * Gets the digest of specified string.
     *
     * @param string The string to hash
     * @param algorithm The name of the algorithm
     * @return The digest
     */
    private byte[] getDigest(String string, String algorithm) {
        try {
            MessageDigest md = MessageDigest.getInstance(algorithm);
            md.update(string.getBytes("UTF-8"));
            return md.digest();
        } catch (NoSuchAlgorithmException e) {
            throw ApiFacadeException.weakforceInvalidHashAlgorithm(algorithm);
        } catch (UnsupportedEncodingException e) {
            throw ApiFacadeException.internalServerError(e);
        }
    }

}
