/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mobile.api.facade.connectors;

import java.io.ByteArrayOutputStream;
import java.io.IOException;

import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.openexchange.java.Strings;
import com.openexchange.mobile.api.facade.auth.SessionData;
import com.openexchange.mobile.api.facade.configuration.HostConfiguration;
import com.openexchange.mobile.api.facade.exceptions.ApiFacadeException;
import com.openexchange.mobile.api.facade.utils.JacksonUtil;
import com.openexchange.mobile.api.facade.utils.RequestUtil;

import lombok.extern.slf4j.Slf4j;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.StatusLine;
import org.apache.http.client.CookieStore;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.conn.ConnectTimeoutException;
import org.apache.http.cookie.Cookie;
import org.apache.http.entity.BufferedHttpEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.cookie.BasicClientCookie;
import org.apache.http.util.EntityUtils;

@Slf4j
public abstract class AbstractConnector<T> implements Connector<T, T> {

    private final RequestConfiguration configuration;

    protected final SessionData sessionData;

    private final String module;

    public AbstractConnector(RequestConfiguration configuration, String module) {
        this(configuration, null, module);
    }

    public AbstractConnector(RequestConfiguration configuration, SessionData sessionData, String module) {
        this.configuration = configuration;
        this.sessionData = sessionData;
        this.module = module;
    }

    protected abstract HttpUriRequest createRequest();

    protected void addSessionData(RequestBuilder builder) {
        HostConfiguration configuration = sessionData.getConfiguration();
        if (configuration.isForwardHostHeader()) {
            String hostname = sessionData.getHost();
            String xHostname = sessionData.getXHost();
            if (!Strings.isEmpty(hostname)) {
                builder.addHeader(RequestUtil.HEADER_HOST, hostname);
            }
            if (!Strings.isEmpty(xHostname)) {
                builder.addHeader(RequestUtil.HEADER_X_HOST, xHostname);
            }
        }
        String forwardedFor = sessionData.getForwardedFor();
        if (!Strings.isEmpty(forwardedFor)) {
            builder.addHeader(RequestUtil.HEADER_X_FORWARDED_FOR, forwardedFor);
        }
        String userAgent = sessionData.getUserAgent();
        if (!Strings.isEmpty(userAgent)) {
            builder.addHeader(RequestUtil.HEADER_USER_AGENT, userAgent);
        }
        builder.addParameter("session", sessionData.getSession());
        String trackingId = sessionData.getProperty(SessionData.PROPERTY_TRACKING_ID);
        if (!Strings.isEmpty(trackingId)) {
            builder.addParameter("trackingId", trackingId);
        }
    }

    @Override
    public String createUrl() {
        return configuration.getBaseUrl() + module;
    }

    @Override
    public T execute() {
        CloseableHttpClient httpClient = configuration.getHttpClient();
        HttpUriRequest request = createRequest();
        HttpClientContext context = HttpClientContext.create();
        if (sessionData != null) {
            CookieStore cookieStore = configuration.getCookieStore();
            for (Cookie cookie : sessionData.getCookies()) {
                BasicClientCookie basicCookie = (BasicClientCookie) cookie;
                basicCookie.setDomain(request.getURI().getHost());
                basicCookie.setSecure(false);
                cookieStore.addCookie(basicCookie);
            }
            context.setCookieStore(cookieStore);
        }
        try (CloseableHttpResponse response = httpClient.execute(request, context)) {
            return handleResponse(response);
        } catch (Exception e) {
            log.error("Failed to execute request", e);
            if (e instanceof ConnectTimeoutException) {
                throw ApiFacadeException.gatewayTimeoutError(e);
            }
            throw ApiFacadeException.internalServerError(e);
        }
    }

    @Override
    public T handleResponse(HttpResponse response) throws UnsupportedOperationException {
        try {
            StatusLine statusLine = response.getStatusLine();
            int statusCode = statusLine.getStatusCode();
            HttpEntity entity = response.getEntity();
            ObjectMapper mapper = JacksonUtil.getMapper();
            T responseBody;
            JavaType responseBodyType = getResponseBodyType(mapper);
            if (log.isDebugEnabled()) {
                entity = new BufferedHttpEntity(response.getEntity());
                ByteArrayOutputStream baos = new ByteArrayOutputStream();
                entity.writeTo(baos);
                log.debug(new String(baos.toByteArray(), "UTF-8"));
            }
            try {
                responseBody = mapper.readValue(entity.getContent(), responseBodyType);
            } catch (JsonMappingException e) {
                JavaType errorResponseBodyType = mapper.getTypeFactory().constructType(AbstractResponseMto.class);
                AbstractResponseMto errorResponseBody = mapper.readValue(entity.getContent(), errorResponseBodyType);
                throw ApiFacadeException.fromResponseBody(errorResponseBody);
            }
            EntityUtils.consume(entity);
            if (isError(responseBody)) {
                throw handleErrors(responseBody);
            } else if (statusCode < 200 || statusCode > 299) {
                throw ApiFacadeException.internalServerError("Request could not be completed successfully. Returned status code: " + statusCode);
            }
            return responseBody;
        } catch (IOException e) {
            throw ApiFacadeException.internalServerError(e);
        }
    }

    protected abstract JavaType getResponseBodyType(ObjectMapper mapper);

    protected abstract boolean isError(T responseBody);

    protected abstract ApiFacadeException handleErrors(T responseBody);

}
