/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mobile.api.facade.endpoints;

import java.util.List;

import javax.annotation.security.PermitAll;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import com.openexchange.mobile.api.facade.auth.SessionData;
import com.openexchange.mobile.api.facade.endpoints.requests.AccountRequestBody;
import com.openexchange.mobile.api.facade.endpoints.responses.AccountResponseBodyV2;
import com.openexchange.mobile.api.facade.endpoints.responses.AccountsResponseBodyV2;
import com.openexchange.mobile.api.facade.exceptions.ApiFacadeException;
import com.openexchange.mobile.api.facade.models.MailAccountData;
import com.openexchange.mobile.api.facade.services.AccountsService;
import com.openexchange.mobile.api.facade.utils.JacksonUtil;
import com.openexchange.mobile.api.facade.utils.RequestUtil;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import org.apache.http.HttpStatus;

@Path("/api-facade/v2/accounts")
@RequiredArgsConstructor
@PermitAll
@Slf4j
public class AccountsEndpointV2 {

    private final AccountsService accountsService;

    @GET
    @Produces(MediaType.APPLICATION_JSON)
    public Response getAccounts(@Context HttpServletRequest request) {
        log.info("Retrieve all accounts");
        SessionData sessionData = RequestUtil.getSessionData(request);
        String language = RequestUtil.getLanguage(request);
        List<MailAccountData> accounts = accountsService.getAccountsV2(sessionData, language);
        AccountsResponseBodyV2 responseBody = new AccountsResponseBodyV2(accounts);
        return RequestUtil.createCachableResponse(request, responseBody);
    }

    @GET
    @Path("/{id}")
    @Produces(MediaType.APPLICATION_JSON)
    public Response getAccount(@Context HttpServletRequest request, @PathParam("id") String accountId) {
        log.info("Retrieve specific account");
        SessionData sessionData = RequestUtil.getSessionData(request);
        String language = RequestUtil.getLanguage(request);
        MailAccountData account = accountsService.getAccountV2(sessionData, accountId, language);
        AccountResponseBodyV2 responseBody = new AccountResponseBodyV2(account);
        return RequestUtil.createCachableResponse(request, responseBody);
    }

    @POST
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public Response createAccount(@Context HttpServletRequest request, String rawRequestBody) {
        log.info("Create account");
        SessionData sessionData = RequestUtil.getSessionData(request);
        AccountRequestBody requestBody = JacksonUtil.parse(rawRequestBody, AccountRequestBody.class);
        boolean forceInsecureConnection = requestBody.getForceInsecureConnection() != null && requestBody.getForceInsecureConnection();
        MailAccountData account = accountsService.createAccount(sessionData, new MailAccountData(requestBody), forceInsecureConnection);
        AccountResponseBodyV2 responseBody = new AccountResponseBodyV2(account);
        return RequestUtil.jsonResponse(responseBody);
    }

    @PUT
    @Path("/{id}")
    @Consumes(MediaType.APPLICATION_JSON)
    @Produces(MediaType.APPLICATION_JSON)
    public Response updateAccount(@Context HttpServletRequest request, @PathParam("id") String accountId, String rawRequestBody) {
        log.info("Update account");
        SessionData sessionData = RequestUtil.getSessionData(request);
        AccountRequestBody requestBody = JacksonUtil.parse(rawRequestBody, AccountRequestBody.class);
        if (!accountId.equals(requestBody.getId())) {
            throw ApiFacadeException.badRequest(ApiFacadeException.Code.INVALID_ACCOUNT_DATA, "Account Ids don't match");
        }
        boolean forceInsecureConnection = requestBody.getForceInsecureConnection() != null && requestBody.getForceInsecureConnection();
        MailAccountData account = accountsService.updateAccount(sessionData, new MailAccountData(requestBody), forceInsecureConnection);
        AccountResponseBodyV2 responseBody = new AccountResponseBodyV2(account);
        return RequestUtil.jsonResponse(responseBody);
    }

    @DELETE
    @Path("/{id}")
    @Produces(MediaType.APPLICATION_JSON)
    public Response deleteAccount(@Context HttpServletRequest request, @PathParam("id") String accountId) {
        log.info("Delete account");
        SessionData sessionData = RequestUtil.getSessionData(request);
        boolean isDeleted = accountsService.deleteAccount(sessionData, accountId);
        return Response.status(isDeleted ? HttpStatus.SC_OK : HttpStatus.SC_NOT_FOUND).build();
    }

}
