/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mobile.api.facade.utils;

import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.net.ssl.SSLContext;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.ResponseBuilder;

import com.openexchange.java.Strings;
import com.openexchange.log.LogProperties;
import com.openexchange.mobile.api.facade.auth.SessionData;
import com.openexchange.mobile.api.facade.configuration.ConfigurationHelper;
import com.openexchange.mobile.api.facade.configuration.GlobalConfiguration;
import com.openexchange.mobile.api.facade.configuration.HostConfiguration;
import com.openexchange.mobile.api.facade.connectors.RequestConfiguration;
import com.openexchange.mobile.api.facade.exceptions.ApiFacadeException;
import com.vdurmont.semver4j.Semver;
import com.vdurmont.semver4j.Semver.SemverType;

import lombok.SneakyThrows;
import lombok.experimental.UtilityClass;

import org.apache.http.client.config.RequestConfig;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.TrustStrategy;
import org.apache.http.cookie.Cookie;
import org.apache.http.impl.client.BasicCookieStore;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.ssl.SSLContextBuilder;
import org.apache.http.util.TextUtils;

@UtilityClass
public class RequestUtil {

    public static final String HEADER_AUTHORIZATION = "Authorization";

    public static final String HEADER_CONTENT_TYPE = "Content-Type";

    public static final String CONTENT_TYPE_JSON = "application/json";

    public static final String HEADER_USER_AGENT = "User-Agent";

    public static final String HEADER_X_USER_AGENT = "X-User-Agent";

    public static final String HEADER_SESSION = "X-Session";

    public static final String HEADER_TRACKING_ID = "X-Tracking-ID";

    public static final String HEADER_CONTEXT_ID = "X-Context-ID";

    public static final String HEADER_USER_ID = "X-User-ID";

    public static final String HEADER_ETAG = "ETag";

    public static final String HEADER_IF_NONE_MATCH = "If-None-Match";

    public static final String HEADER_ACCEPT_LANGUAGE = "Accept-Language";

    public static final String HEADER_X_FORWARDED_FOR = "X-Forwarded-For";

    public static final String HEADER_HOST = "Host";

    public static final String HEADER_X_HOST = "X-Host";

    private static final int USER_AGENT_MAXLEN = 100;

    private static final Pattern USER_AGENT_PATTERN = Pattern.compile("^(.*)\\.(.*)\\.Mail/(\\S*) \\(.*\\)$");

    @SneakyThrows
    public RequestConfiguration createConfiguration() {
        BasicCookieStore cookieStore = new BasicCookieStore();
        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectTimeout(10000)
                .build();
        /* Java8: For Java 8:
        SSLContext sslContext = new SSLContextBuilder()
                .loadTrustMaterial(null, (TrustStrategy) (chain, authType) -> true)
                .build();
        */
        SSLContext sslContext = new SSLContextBuilder()
                .loadTrustMaterial(null, new TrustStrategy() {
                    @Override
                    public boolean isTrusted(X509Certificate[] arg0, String arg1) throws CertificateException {
                        return true;
                    }
                })
                .build();
        CloseableHttpClient httpClient = HttpClients.custom()
                .setDefaultCookieStore(cookieStore)
                .setDefaultRequestConfig(requestConfig)
                .setSSLHostnameVerifier(NoopHostnameVerifier.INSTANCE)
                .setSSLContext(sslContext)
                .build();
        return new RequestConfiguration(GlobalConfiguration.middlewareBaseUrl, httpClient, cookieStore);
    }

    public SessionData getSessionData(HttpServletRequest request) {
        String session = request.getHeader(HEADER_SESSION);
        if (Strings.isEmpty(session)) {
            throw ApiFacadeException.missingHeader(HEADER_SESSION);
        }
        List<Cookie> cookies = getCookies(request, GlobalConfiguration.middlewareBaseUrl.getHost());
        if (cookies.size() == 0) {
            throw ApiFacadeException.badRequest(ApiFacadeException.Code.MISSING_COOKIES, "Cookies missing");
        }
        String hostname = getHostHeader(request);
        String userAgent = getUserAgent(request);
        String trackingId = getTrackingId(request);
        String contextId = getContextId(request);
        String userId = getUserId(request);
        HostConfiguration hostConfiguration = ConfigurationHelper.getConfiguration(request);
        checkUserAgent(userAgent, hostConfiguration);
        return enhanceSessionData(request, hostConfiguration, hostname, userAgent, trackingId, contextId, userId, new SessionData(session, cookies));
    }

    public static SessionData enhanceSessionData(HttpServletRequest request, HostConfiguration configuration, String hostname, String userAgent, String trackingId, String contextId, String userId, SessionData sessionData) {
        if (trackingId != null) {
            sessionData.setProperty(SessionData.PROPERTY_TRACKING_ID, trackingId);
        }
        if (contextId != null) {
            sessionData.setProperty(SessionData.PROPERTY_CONTEXT_ID, contextId);
        }
        if (userId != null) {
            sessionData.setProperty(SessionData.PROPERTY_USER_ID, userId);
        }
        return sessionData
                .withHost(hostname)
                .withForwardedFor(getForwardedForWithRemoteIp(request))
                .withUserAgent(userAgent)
                .withConfiguration(configuration);
    }

    public String getUserAgent(HttpServletRequest request) {
        String userAgent = request.getHeader(HEADER_X_USER_AGENT);
        if (userAgent == null) {
            userAgent = request.getHeader(HEADER_USER_AGENT);
        }
        return userAgent;
    }

    public void checkUserAgent(String userAgent, HostConfiguration hostConfiguration) {
        if (Strings.isEmpty(userAgent)) {
            throw ApiFacadeException.missingHeader(HEADER_USER_AGENT);
        }
        UserAgentInfo userAgentInfo = parseUserAgent(userAgent);
        if (!hostConfiguration.isAllowedBrand(userAgentInfo.getBrand())) {
            throw ApiFacadeException.forbidden();
        }
        if (!minimumClientVersionSatisfied(userAgentInfo, hostConfiguration.getMinimumClientVersions())) {
            throw ApiFacadeException.clientUpdateMandatory();
        }
    }

    public boolean minimumClientVersionSatisfied(UserAgentInfo userAgentInfo, Map<String, String> minimumClientVersions) {
        String clientVersionStr = userAgentInfo.getVersion();
        String minimumClientVersionStr = getMinimumClientVersion(minimumClientVersions, userAgentInfo);
        if (Strings.isEmpty(minimumClientVersionStr)) {
            return true;
        }
        if (Strings.isEmpty(clientVersionStr)) {
            return false;
        }
        Semver clientVersion = new Semver(clientVersionStr, SemverType.LOOSE);
        Semver minimumClientVersion = new Semver(minimumClientVersionStr, SemverType.LOOSE);
        return !clientVersion.isLowerThan(minimumClientVersion);
    }

    private String getMinimumClientVersion(Map<String, String> minimumClientVersions, UserAgentInfo userAgentInfo) {
        String platform = userAgentInfo.getPlatform();
        String minimumClientVersion = minimumClientVersions.get(platform + "." + userAgentInfo.getBrand());
        if (minimumClientVersion == null) {
            minimumClientVersion = minimumClientVersions.get(platform);
        }
        return minimumClientVersion;
    }

    @SneakyThrows
    private List<Cookie> getCookies(HttpServletRequest request, String targetDomain) {
        javax.servlet.http.Cookie[] cookies = request.getCookies();
        return CookieUtil.convertForConnector(cookies, targetDomain);
    }

    public <T> Response jsonResponse(T responseBody) {
        return jsonResponse(200, responseBody);
    }

    public <T> Response jsonResponse(int statusCode, T responseBody) {
        return Response.status(statusCode).entity(JacksonUtil.toString(responseBody)).build();
    }

    public Response emptyJsonResponse() {
        return Response.ok().entity("{}").build();
    }

    public <T> Response createCachableResponse(HttpServletRequest request, T responseBody) {
        String ifNoneMatchValue = getIfNoneMatch(request);
        String responseBodyStr = JacksonUtil.toString(responseBody);
        String hashValue = String.format("\"%d\"", responseBodyStr.hashCode());
        boolean isNotModified = hashValue.equals(ifNoneMatchValue);
        ResponseBuilder responseBuilder = isNotModified ? Response.notModified() : Response.ok(responseBodyStr);
        responseBuilder.header(HEADER_ETAG, hashValue);
        return responseBuilder.build();
    }

    public String getRemoteIp(HttpServletRequest request) {
        return request.getRemoteAddr();
    }

    public String getForwardedForWithRemoteIp(HttpServletRequest request) {
        String remoteIp = getRemoteIp(request);
        String forwardedFor = request.getHeader(HEADER_X_FORWARDED_FOR);
        if (forwardedFor == null) {
            forwardedFor = remoteIp;
        } else {
            forwardedFor += ", " + remoteIp;
        }
        return forwardedFor;
    }

    public UserAgentInfo parseUserAgent(String userAgent) {
        if (TextUtils.isEmpty(userAgent) || userAgent.length() > USER_AGENT_MAXLEN) {
            throw ApiFacadeException.invalidUserAgent();
        }
        Matcher matcher = USER_AGENT_PATTERN.matcher(userAgent);
        if (!matcher.find()) {
            throw ApiFacadeException.invalidUserAgent();
        }
        if (matcher.groupCount() != 3) {
            throw ApiFacadeException.invalidUserAgent();
        }
        String brand = matcher.group(1);
        String platform = matcher.group(2);
        String clientVersion = matcher.group(3);
        return new UserAgentInfo(brand, platform, clientVersion);
    }

    /*private*/ String getIfNoneMatch(HttpServletRequest request) {
        String ifNoneMatchValue = request.getHeader(HEADER_IF_NONE_MATCH);
        if (ifNoneMatchValue != null) {
            ifNoneMatchValue = ifNoneMatchValue.replace("-gzip", "");
        }
        return ifNoneMatchValue;
    }

    public String getHostHeader(HttpServletRequest request) {
        String value = request.getHeader(HEADER_X_HOST);
        if (value == null) {
            value = request.getHeader(HEADER_HOST);
        }
        return value;
    }

    public String getLanguage(HttpServletRequest request) {
        String language = request.getHeader(HEADER_ACCEPT_LANGUAGE);
        return language == null ? null : language.replace("-", "_").replaceAll(",.*$", "");
    }

    public String getTrackingId(HttpServletRequest request) {
        String trackingId = request.getHeader(HEADER_TRACKING_ID);
        LogProperties.putProperty(LogProperties.Name.REQUEST_TRACKING_ID, trackingId);
        return trackingId;
    }

    private String getContextId(HttpServletRequest request) {
        String contextId = request.getHeader(HEADER_CONTEXT_ID);
        LogProperties.putProperty(LogProperties.Name.SESSION_CONTEXT_ID, contextId);
        return contextId;
    }

    private String getUserId(HttpServletRequest request) {
        String userId = request.getHeader(HEADER_USER_ID);
        LogProperties.putProperty(LogProperties.Name.SESSION_USER_ID, userId);
        return userId;
    }

    public void checkParam(String name, String value) {
        if (value == null) {
            throw ApiFacadeException.missingParameter(name);
        }
    }

    public void checkParam(String name, List<String> values) {
        if (values == null) {
            throw ApiFacadeException.missingParameter(name);
        }
        if (values.size() == 0) {
            throw ApiFacadeException.emptyArrayNotAllowed(name);
        }
    }

    public void checkParam(String name, Object value) {
        if (value == null) {
            throw ApiFacadeException.missingParameter(name);
        }
    }

    public boolean getBooleanParameter(Boolean currentValue, boolean defaultValue) {
        return currentValue == null ? defaultValue : currentValue.booleanValue();
    }

}
