/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mobile.api.facade.endpoints.responses.ctos;

import java.util.ArrayList;
import java.util.List;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.openexchange.mobile.api.facade.models.Account;
import com.openexchange.mobile.api.facade.models.Folder;
import com.openexchange.mobile.api.facade.models.MailAccountData;
import com.openexchange.mobile.api.facade.utils.ListUtil;
import com.openexchange.mobile.api.facade.utils.MapFunction;

import lombok.Data;
import lombok.NonNull;

@Data
@JsonInclude(JsonInclude.Include.NON_NULL)
public class AccountCto {

    @NonNull
    private final String id;

    @NonNull
    private final String accountName;

    private final String userName;

    private final String firstName;

    private final String lastName;

    @NonNull
    private final String primaryAddress;

    private final List<AliasCto> aliases;

    private final boolean canSend;

    @NonNull
    private final List<FolderCto> folders;

    @NonNull
    private final StandardFoldersCto standardFolders;

    private static final MapFunction<String, AliasCto> aliasesMapFunction = new MapFunction<String, AliasCto>() {
        @Override
        public AliasCto map(String address) {
            return new AliasCto(address);
        }
    };

    public AccountCto(Account account) {
        id = account.getId();
        accountName = account.getAccountName();
        userName = account.getUserName();
        firstName = account.getFirstName();
        lastName = account.getLastName();
        primaryAddress = account.getPrimaryAddress();
        aliases = ListUtil.map(account.getAliases(), aliasesMapFunction);
        canSend = account.isCanSend();
        if (account.getFolders() != null) {
            /* Java8: Use with Java 8.
            folders = account.getFolders().stream()
                    .map(folder -> new FolderCto(folder))
                    .collect(Collectors.toList());
            */
            folders = new ArrayList<>();
            for (Folder folder : account.getFolders()) {
                folders.add(new FolderCto(folder));
            }
            standardFolders = new StandardFoldersCto(account.getStandardFolders());
        } else {
            folders = null;
            standardFolders = null;
        }
    }

    public AccountCto(MailAccountData account) {
        id = account.getId();
        accountName = account.getAccountName();
        firstName = "";
        lastName = "";
        userName = "";
        primaryAddress = account.getPrimaryAddress();
        aliases = ListUtil.map(account.getAliases(), aliasesMapFunction);
        canSend = true;
        folders = null;
        standardFolders = null;
    }

}
