/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.mobile.api.facade.utils;

import com.openexchange.mobile.api.facade.connectors.ConnectorFactory;
import com.openexchange.mobile.api.facade.connectors.ConnectorFactoryFactory;
import com.openexchange.mobile.api.facade.connectors.impl.VersionConnector;
import com.openexchange.mobile.api.facade.connectors.responses.VersionResponseMto;

import lombok.experimental.UtilityClass;
import lombok.extern.slf4j.Slf4j;

@UtilityClass
@Slf4j
public class VersionCheckerUtil {

    static final String SUPPORTED_VERSION_NUMBER = "7.10.0";

    static final int SUPPORTED_MINIMUM_REVISION = 0;

    static final long MILLIS_SLEEP_TIME = 10_000;

    public void check(ConnectorFactoryFactory connectorFactoryFactory) {
        String remoteVersion = retrieveRemoteVersion(connectorFactoryFactory, MILLIS_SLEEP_TIME);
        checkVersion(remoteVersion);
    }

    private String retrieveRemoteVersion(ConnectorFactoryFactory connectorFactoryFactory, long millisSleep) {
        String version = null;
        while (version == null) {
            try {
                version = retrieveRemoteVersion(connectorFactoryFactory);
                log.info("Remote middleware version: " + version);
            } catch (Exception e) {
                log.warn("Failed to retrieve remote middleware version. Waiting for next try");
                try {
                    Thread.sleep(millisSleep);
                } catch (InterruptedException e2) {
                    e2.printStackTrace();
                }
            }
        }
        return version;
    }

    private String retrieveRemoteVersion(ConnectorFactoryFactory connectorFactoryFactory) {
        try {
            ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
            VersionConnector connector = connectorFactory.getVersionConnector();
            VersionResponseMto response = connector.execute();
            return response.getData().getVersion();
        } catch (Exception e) {
            throw new RuntimeException("Failed to retrieve middleware version: " + e.getMessage());
        }
    }

    void checkVersion(String remoteVersion) {
        if (remoteVersion == null) {
            throw new RuntimeException("Remote version is null");
        }
        String message = String.format("Remote version is incompatible: %s. This version of the facade is compatible with version %s-Rev%d or newer", remoteVersion, SUPPORTED_VERSION_NUMBER, SUPPORTED_MINIMUM_REVISION);
        if (!remoteVersion.startsWith(SUPPORTED_VERSION_NUMBER)) {
            throw new RuntimeException(message);
        }
        try {
            String revisionString = remoteVersion.substring(SUPPORTED_VERSION_NUMBER.length() + 4);
            int revision = Integer.parseInt(revisionString);
            if (revision < SUPPORTED_MINIMUM_REVISION) {
                throw new RuntimeException(message);
            }
        } catch (NumberFormatException e) {
            throw new RuntimeException("Remote version is parseable");
        }
    }

}
