/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.connectors.impl;

import java.io.ByteArrayOutputStream;
import java.io.IOException;

import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.openexchange.mobile.api.facade.configuration.HostConfiguration;
import com.openexchange.mobile.api.facade.connectors.AbstractResponseMto;
import com.openexchange.mobile.api.facade.connectors.Connector;
import com.openexchange.mobile.api.facade.exceptions.ApiFacadeException;
import com.openexchange.mobile.api.facade.utils.JacksonUtil;
import com.openexchange.mobile.api.facade.utils.RequestUtil;
import com.openexchange.tools.encoding.Base64;

import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.StatusLine;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.methods.RequestBuilder;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.entity.BufferedHttpEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;

@RequiredArgsConstructor
@Slf4j
public abstract class WeakforceConnector<T> implements Connector<T, T> {

    private static final CloseableHttpClient httpClient = HttpClientBuilder.create().build();

    private final HostConfiguration configuration;

    private final String command;

    private final Class<T> responseBodyType;

    @Override
    public String createUrl() {
        String url = configuration.getWeakforceBaseUrl();
        if (!url.endsWith("/")) {
            url += "/";
        }
        return url + "?command=" + command;
    }

    @SneakyThrows
    protected void addAuthentication(RequestBuilder builder) {
        if (configuration.isWeakforceUseAuthentication()) {
            String authorization = "facade:" + configuration.getWeakforcePassword();
            String encoded = Base64.encode(authorization);
            builder.addHeader(RequestUtil.HEADER_AUTHORIZATION, "Basic " + encoded);
        }
    }

    protected abstract HttpUriRequest createRequest();

    @Override
    public T execute() {
        HttpUriRequest request = createRequest();
        HttpClientContext context = HttpClientContext.create();
        try (CloseableHttpResponse response = httpClient.execute(request, context)) {
            return handleResponse(response);
        } catch (Exception e) {
            log.error("Failed to execute request", e);
            throw ApiFacadeException.internalServerError(e);
        }
    }

    @Override
    public T handleResponse(HttpResponse response) {
        try {
            StatusLine statusLine = response.getStatusLine();
            int statusCode = statusLine.getStatusCode();
            HttpEntity entity = new BufferedHttpEntity(response.getEntity());
            ObjectMapper mapper = JacksonUtil.getMapper();
            T responseBody;
            if (log.isDebugEnabled()) {
                ByteArrayOutputStream baos = new ByteArrayOutputStream();
                entity.writeTo(baos);
                log.debug(new String(baos.toByteArray(), "UTF-8"));
            }
            try {
                responseBody = mapper.readValue(entity.getContent(), responseBodyType);
            } catch (JsonMappingException e) {
                JavaType errorResponseBodyType = mapper.getTypeFactory().constructType(AbstractResponseMto.class);
                AbstractResponseMto errorResponseBody = mapper.readValue(entity.getContent(), errorResponseBodyType);
                throw ApiFacadeException.fromResponseBody(errorResponseBody);
            }
            EntityUtils.consume(entity);
            if (statusCode < 200 || statusCode > 299) {
                throw ApiFacadeException.internalServerError("Request could not be completed successfully. Returned status code: " + statusCode);
            }
            return responseBody;
        } catch (IOException e) {
            throw ApiFacadeException.internalServerError(e);
        }
    }

}
