/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.configuration;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import com.openexchange.config.ConfigurationService;
import com.openexchange.mobile.api.facade.configuration.clientspecific.MatchingHostConfiguration;
import com.openexchange.mobile.api.facade.utils.RequestUtil;
import com.openexchange.mobile.api.facade.utils.UserAgentInfo;

import lombok.experimental.UtilityClass;

@UtilityClass
@SuppressWarnings({"PMD.AvoidDuplicateLiterals"})
public class ConfigurationHelper {

    public static final String CONFIG_MOBILE_API_FACADE_CONFIG_YAML = "mobile-api-facade-config.yml";

    public static final String CONFIG_FACADE_PROPERTIES = "facade.properties";

    private HostConfiguration defaultHostConfiguration;

    private Map<String, List<HostConfiguration>> configurations = Collections.emptyMap();

    public void initialize(ConfigurationService configurationService) {
        GlobalConfiguration.initialize(configurationService);
        defaultHostConfiguration = new HostConfiguration(configurationService);
        loadConfigurations(configurationService, ConfigurationHelper.CONFIG_MOBILE_API_FACADE_CONFIG_YAML);
    }

    public HostConfiguration getConfiguration(HttpServletRequest request) {
        String hostname = RequestUtil.getHostHeader(request);
        String xHostname = RequestUtil.getXHostHeader(request);
        String userAgent = RequestUtil.getUserAgent(request);
        UserAgentInfo userAgentInfo = RequestUtil.parseUserAgent(userAgent);
        HostConfiguration hostConfiguration = getConfiguration(hostname, xHostname, userAgentInfo);
        RequestUtil.checkUserAgent(userAgentInfo, hostConfiguration);
        return hostConfiguration;
    }

    public HostConfiguration getConfiguration(String hostname, String xHostname, UserAgentInfo userAgentInfo) {
        if (xHostname != null) {
            hostname = xHostname;
        }
        List<HostConfiguration> configurationList = configurations.get(hostname);
        if (configurationList == null) {
            return defaultHostConfiguration;
        }
        Iterator<HostConfiguration> it = configurationList.iterator();
        while (it.hasNext()) {
            HostConfiguration configuration = it.next();
            if (configuration instanceof MatchingHostConfiguration) {
                MatchingHostConfiguration matchingConfiguration = (MatchingHostConfiguration) configuration;
                if (matchingConfiguration.matches(userAgentInfo)) {
                    return matchingConfiguration;
                }
            } else {
                return configuration;
            }
        }
        return defaultHostConfiguration;
    }

    public HostConfiguration getDefaultConfiguration() {
        return defaultHostConfiguration;
    }

    public void setDefaultConfiguration(HostConfiguration hostConfiguration) {
        defaultHostConfiguration = hostConfiguration;
    }

    public String getString(ConfigurationService configurationService, String propertyName, String defaultValue) {
        try {
            String value = getStringOrNull(configurationService, propertyName);
            return value != null ? value : defaultValue;
        } catch (Exception e) {
            return defaultValue;
        }
    }

    public int getNumber(ConfigurationService configurationService, String propertyName, int defaultValue) {
        try {
            String value = getStringOrNull(configurationService, propertyName);
            return value != null ? Integer.parseInt(value) : defaultValue;
        } catch (Exception e) {
            return defaultValue;
        }
    }

    public String getNumberString(ConfigurationService configurationService, String propertyName, String defaultValue) {
        try {
            String value = getStringOrNull(configurationService, propertyName);
            if (value == null) {
                return defaultValue;
            }
            Integer.parseInt(value);
            return value;
        } catch (Exception e) {
            return defaultValue;
        }
    }

    public boolean getBoolean(ConfigurationService configurationService, String propertyName, boolean defaultValue) {
        try {
            String value = getStringOrNull(configurationService, propertyName);
            return value != null ? Boolean.parseBoolean(value) : defaultValue;
        } catch (Exception e) {
            return defaultValue;
        }
    }

    public String getStringOrNull(ConfigurationService configurationService, String propertyName) {
        return configurationService.getProperty(propertyName);
    }

    public Map<String, List<HostConfiguration>> loadConfigurations(ConfigurationService configurationService, String filename) {
        Object yaml = configurationService.getYaml(filename);
        if (yaml instanceof Map) {
            @SuppressWarnings("unchecked")
            Map<String, Object> map = (Map<String, Object>) yaml;
            if (!map.isEmpty()) {
                Map<String, List<HostConfiguration>> parsedConfigurations = parseHostConfigurations(map, defaultHostConfiguration);
                if (!parsedConfigurations.isEmpty()) {
                    configurations = parsedConfigurations;
                    return parsedConfigurations;
                }
            }
        }
        configurations = Collections.emptyMap();
        return configurations;
    }

    private Map<String, List<HostConfiguration>> parseHostConfigurations(Map<String, Object> map, HostConfiguration defaultConfiguration) {
        Map<String, List<HostConfiguration>> configurations = new HashMap<>();
        for (Map.Entry<String, Object> entry : map.entrySet()) {
            String name = entry.getKey();
            Object value = entry.getValue();
            if (value instanceof Map) {
                @SuppressWarnings("unchecked")
                Map<String, Object> properties = (Map<String, Object>) value;
                HostConfiguration configuration = new HostConfiguration(defaultConfiguration);
                configuration.loadProperties(properties);
                List<HostConfiguration> configurationList = new ArrayList<>(1);
                configurationList.add(configuration);
                configurations.put(name, configurationList);
            } else if (value instanceof List) {
                List<HostConfiguration> configurationList = new LinkedList<>();
                configurations.put(name, configurationList);
                @SuppressWarnings("unchecked")
                List<Map<String, Object>> list = (List<Map<String, Object>>) value;
                Iterator<Map<String, Object>> it = list.iterator();
                while (it.hasNext()) {
                    Object rawListEntry = it.next();
                    if (rawListEntry instanceof Map) {
                        @SuppressWarnings("unchecked")
                        Map<String, Object> properties = (Map<String, Object>) rawListEntry;
                        MatchingHostConfiguration configuration = new MatchingHostConfiguration(defaultConfiguration);
                        configuration.loadProperties(properties);
                        configurationList.add(configuration);
                    }
                }
            }
        }
        return configurations;
    }

}
