/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.services;

import java.util.ArrayList;
import java.util.List;

import com.openexchange.mobile.api.facade.auth.SessionData;
import com.openexchange.mobile.api.facade.connectors.ConnectorFactory;
import com.openexchange.mobile.api.facade.connectors.ConnectorFactoryFactory;
import com.openexchange.mobile.api.facade.connectors.impl.DeleteMailsConnector;
import com.openexchange.mobile.api.facade.connectors.impl.EnvelopesConnector;
import com.openexchange.mobile.api.facade.connectors.impl.GetMailConnector;
import com.openexchange.mobile.api.facade.connectors.impl.MoveMailsConnector;
import com.openexchange.mobile.api.facade.connectors.impl.SetAndClearFlagsConnector;
import com.openexchange.mobile.api.facade.connectors.impl.SetColorsConnector;
import com.openexchange.mobile.api.facade.connectors.impl.StubsConnector;
import com.openexchange.mobile.api.facade.connectors.impl.ThreadReferencesConnector;
import com.openexchange.mobile.api.facade.connectors.impl.ThreadedAllStubsConnector;
import com.openexchange.mobile.api.facade.connectors.responses.ArrayDataResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.DeleteMailsResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.MailIdResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.MailResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.ThreadReferencesResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.ThreadedAllResponseMto;
import com.openexchange.mobile.api.facade.connectors.responses.mtos.ThreadMto;
import com.openexchange.mobile.api.facade.models.DeletedMails;
import com.openexchange.mobile.api.facade.models.Mail;
import com.openexchange.mobile.api.facade.models.MailFlags;
import com.openexchange.mobile.api.facade.models.MailIds;
import com.openexchange.mobile.api.facade.models.MailThread;
import com.openexchange.mobile.api.facade.utils.AttachmentUtil;
import com.openexchange.mobile.api.facade.utils.FilterFunction;
import com.openexchange.mobile.api.facade.utils.ListUtil;
import com.openexchange.mobile.api.facade.utils.MailTransformations;
import com.openexchange.mobile.api.facade.utils.MailUtil;
import com.openexchange.mobile.api.facade.utils.MapFunction;

import lombok.RequiredArgsConstructor;

@RequiredArgsConstructor
public class MailsService {

    private final ConnectorFactoryFactory connectorFactoryFactory;

    private final MapFunction<ThreadMto, MailThread> threadMapFunction = new MapFunction<ThreadMto, MailThread>() {
        @Override
        public MailThread map(ThreadMto threadMto) {
            return new MailThread(threadMto);
        }
    };

    private final MapFunction<MailThread, String> mailIdMapFunction = new MapFunction<MailThread, String>() {
        @Override
        public String map(MailThread mail) {
            return mail.getRootMailId();
        }
    };

    public List<MailThread> getStubs(SessionData sessionData, String folderId, boolean threaded) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        List<MailThread> mailThreads;
        ThreadingType threadingType = threaded ? sessionData.getConfiguration().getThreadingType() : ThreadingType.DISABLED;
        switch (threadingType) {
        case THREADREFERENCES:
            mailThreads = getStubsWithThreadReferences(connectorFactory, sessionData, folderId);
            break;

        case THREADEDALL:
            mailThreads = getStubsWithThreadedAll(connectorFactory, sessionData, folderId);
            break;

        case ALL:
            mailThreads = getStubsWithAll(connectorFactory, sessionData, folderId);
            break;

        case DISABLED:
        default:
            mailThreads = getStubsWithoutThreading(connectorFactory, sessionData, folderId);
            break;
        }
        return MailUtil.removeDeletedMails(mailThreads);
    }

    private List<MailThread> getStubsWithThreadReferences(ConnectorFactory connectorFactory, SessionData sessionData, String folderId) {
        if (folderId.startsWith("default0")) {
            ThreadReferencesConnector connector = connectorFactory.getThreadReferencesConnector(sessionData, folderId);
            ThreadReferencesResponseMto responseBody = connector.execute();
            return MailUtil.convert(responseBody.getData());
        } else {
            return getStubsWithoutThreading(connectorFactory, sessionData, folderId);
        }
    }

    private List<MailThread> getStubsWithAll(ConnectorFactory connectorFactory, SessionData sessionData, String folderId) {
        StubsConnector connector = connectorFactory.getStubsConnector(sessionData, folderId, true);
        ArrayDataResponseMto responseBody = connector.execute();
        List<Mail> mails = StubsConnector.map(responseBody.getData());
        return MailUtil.calculateThreads(mails);
    }

    private List<MailThread> getStubsWithThreadedAll(ConnectorFactory connectorFactory, SessionData sessionData, String folderId) {
        ThreadedAllStubsConnector connector = connectorFactory.getThreadedAllStubsConnector(sessionData, folderId);
        ThreadedAllResponseMto responseBody = connector.execute();
        return ListUtil.map(responseBody.getData(), threadMapFunction);
    }

    private List<MailThread> getStubsWithoutThreading(ConnectorFactory connectorFactory, SessionData sessionData, String folderId) {
        StubsConnector connector = connectorFactory.getStubsConnector(sessionData, folderId, false);
        ArrayDataResponseMto responseBody = connector.execute();
        List<Mail> mails = StubsConnector.map(responseBody.getData());
        return MailUtil.wrapAsThreads(mails);
    }

    public List<MailThread> getNewStubs(SessionData sessionData, String folderId, long lastReceivedDate, boolean threaded) {
        List<MailThread> allStubs = getStubs(sessionData, folderId, threaded);
        List<MailThread> newStubs = new ArrayList<>();
        for (MailThread thread : allStubs) {
            if (thread.getLastReceivedDate() >= lastReceivedDate) {
                newStubs.add(thread);
            }
        }
        return newStubs;
    }

    public List<Mail> getEnvelopes(SessionData sessionData, String folderId, List<String> mailIds) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        EnvelopesConnector connector = connectorFactory.getEnvelopesConnector(sessionData, folderId, mailIds);
        ArrayDataResponseMto responseBody = connector.execute();
        return EnvelopesConnector.map(responseBody.getData());
    }

    public List<Mail> getNewEnvelopes(SessionData sessionData, String folderId, final long lastReceivedDate) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        List<MailThread> stubs = getStubsWithoutThreading(connectorFactory, sessionData, folderId);
        List<String> mailIds = ListUtil.map(ListUtil.filter(stubs, new FilterFunction<MailThread>() {
            @Override
            public boolean filter(MailThread thread) {
                return thread.getLastReceivedDate() >= lastReceivedDate;
            }
        }), mailIdMapFunction);
        EnvelopesConnector connector = connectorFactory.getEnvelopesConnector(sessionData, folderId, mailIds);
        ArrayDataResponseMto responseBody = connector.execute();
        return EnvelopesConnector.map(responseBody.getData());
    }

    public Mail getMail(SessionData sessionData, String folderId, String mailId, int maximumTeaserLength) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        GetMailConnector connector = connectorFactory.getGetMailConnector(sessionData, folderId, mailId);
        MailResponseMto responseBody = connector.execute();
        Mail mail = GetMailConnector.map(responseBody.getData());
        boolean mailTeasersEnabled = sessionData.getConfiguration().isMailTeasersEnabled();
        mail = MailTransformations.apply(mail, mailTeasersEnabled, maximumTeaserLength);
        AttachmentUtil.initializeDownloadUrls(folderId, mail);
        return mail;
    }

    public DeletedMails delete(SessionData sessionData, String folderId, List<String> mailIds, boolean purge) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        DeleteMailsConnector connector = connectorFactory.getDeleteMailsConnector(sessionData, folderId, mailIds, purge);
        DeleteMailsResponseMto response = connector.execute();
        return DeleteMailsConnector.map(response);
    }

    public MailIds moveMails(SessionData sessionData, String folderId, List<String> mailIds, String targetFolderId) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        MoveMailsConnector connector = connectorFactory.getMoveMailsConnector(sessionData, folderId, mailIds, targetFolderId);
        List<MailIdResponseMto> response = connector.execute();
        List<String> newMailIds = MoveMailsConnector.map(response);
        return new MailIds(newMailIds);
    }

    public void setFlags(SessionData sessionData, String folderId, MailFlags flags, List<String> mailIds) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        SetAndClearFlagsConnector connector = connectorFactory.getSetAndClearFlagsConnector(sessionData, folderId, mailIds, flags, true);
        connector.execute();
    }

    public void clearFlags(SessionData sessionData, String folderId, MailFlags flags, List<String> mailIds) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        SetAndClearFlagsConnector connector = connectorFactory.getSetAndClearFlagsConnector(sessionData, folderId, mailIds, flags, false);
        connector.execute();
    }

    public void setColors(SessionData sessionData, String folderId, Integer color, List<String> mailIds) {
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        SetColorsConnector connector = connectorFactory.getSetColorsConnector(sessionData, folderId, mailIds, color);
        connector.execute();
    }

}
