/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.auth.impl;

import com.openexchange.mobile.api.facade.auth.AuthenticationService;
import com.openexchange.mobile.api.facade.auth.LoginCredentials;
import com.openexchange.mobile.api.facade.auth.SessionData;
import com.openexchange.mobile.api.facade.connectors.ConnectorFactory;
import com.openexchange.mobile.api.facade.connectors.ConnectorFactoryFactory;
import com.openexchange.mobile.api.facade.connectors.impl.PnsUnsubscribeConnector;
import com.openexchange.mobile.api.facade.endpoints.requests.LogoutRequestBody;
import com.openexchange.mobile.api.facade.endpoints.requests.PushRequestBody;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

@RequiredArgsConstructor
@Slf4j
public class DefaultAuthenticationService implements AuthenticationService {

    private final ConnectorFactoryFactory connectorFactoryFactory;

    @Override
    public SessionData authenticate(LoginCredentials credentials) {
        log.debug("Authenticating");
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        LoginConnector connector = connectorFactory.getLoginConnector(credentials);
        return connector.execute();
    }

    @Override
    public void logout(SessionData sessionData, LogoutRequestBody requestBody) {
        log.debug("Deauthenticating");
        ConnectorFactory connectorFactory = connectorFactoryFactory.getConnectorFactory();
        unsubscribeFromPush(connectorFactory, sessionData, requestBody);
        invalidateSession(connectorFactory, sessionData);
    }

    private void invalidateSession(ConnectorFactory connectorFactory, SessionData sessionData) {
        LogoutConnector connector = connectorFactory.getLogoutConnector(sessionData);
        connector.execute();
    }

    private void unsubscribeFromPush(ConnectorFactory connectorFactory, SessionData sessionData, LogoutRequestBody requestBody) {
        PushRequestBody pushInfo = requestBody.getPushInfo();

        if (pushInfo != null) {
            PnsUnsubscribeConnector connector = connectorFactory.getPnsUnsubscribeConnector(sessionData, pushInfo.getTransport(), pushInfo.getToken());
            connector.execute();
        }
    }

}
