/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.models;

import java.util.List;

import com.openexchange.java.Strings;
import com.openexchange.mobile.api.facade.connectors.requests.mtos.MailAccountDataMto;
import com.openexchange.mobile.api.facade.endpoints.requests.AccountRequestBody;
import com.openexchange.mobile.api.facade.endpoints.responses.ctos.AliasCto;
import com.openexchange.mobile.api.facade.utils.AccountUtil;
import com.openexchange.mobile.api.facade.utils.ListUtil;
import com.openexchange.mobile.api.facade.utils.MapFunction;

import lombok.AllArgsConstructor;
import lombok.Setter;
import lombok.Value;
import lombok.experimental.NonFinal;

@Value
@AllArgsConstructor
public class MailAccountData {

    private static final MapFunction<AliasCto, String> aliasCtosMapFunction = new MapFunction<AliasCto, String>() {

        @Override
        public String map(AliasCto value) {
            return value.getAddress();
        }

    };

    // General properties

    String id;

    String accountName;

    String personalName;

    String login;

    String password;

    @NonFinal
    @Setter
    String userName;

    @NonFinal
    @Setter
    String firstName;

    @NonFinal
    @Setter
    String lastName;

    String primaryAddress;

    List<String> aliases;

    // IMAP/POP3 related properties

    String mailProtocol;

    String mailServer;

    Integer mailPort;

    Boolean mailSecure;

    Boolean mailStartTLS;

    Integer mailOAuth;

    Boolean mailDisabled;

    Integer pop3RefreshRate;

    Boolean pop3ExpungeOnQuit;

    Boolean pop3DeleteWriteThrough;

    // Transport related properties

    String transportAuth;

    String transportProtocol;

    String transportServer;

    Integer transportPort;

    Boolean transportSecure;

    Boolean transportStartTLS;

    String transportLogin;

    Integer transportOAuth;

    Boolean transportDisabled;

    // Folder related properties

    String spamHandler;

    String rootFolderId;

    String trashFolder;

    String sentFolder;

    String draftsFolder;

    String spamFolder;

    String confirmedSpamFolder;

    String confirmedHamFolder;

    String inboxFolderId;

    String archiveFolderId;

    Boolean unifiedInboxEnabled;

    Boolean canSend;

    @NonFinal
    List<Folder> folders;

    @NonFinal
    StandardFolders standardFolders;

    public MailAccountData(AccountRequestBody requestBody) {
        id = requestBody.getId();
        accountName = requestBody.getAccountName();
        personalName = requestBody.getPersonalName();
        login = requestBody.getLogin();
        password = requestBody.getPassword();
        userName = null;
        firstName = null;
        lastName = null;
        primaryAddress = requestBody.getPrimaryAddress();
        aliases = ListUtil.map(requestBody.getAliases(), aliasCtosMapFunction);
        mailProtocol = requestBody.getMailProtocol();
        mailServer = requestBody.getMailServer();
        mailPort = requestBody.getMailPort();
        mailSecure = requestBody.getMailSecure();
        mailStartTLS = requestBody.getMailStartTLS();
        mailOAuth = requestBody.getMailOAuth();
        mailDisabled = requestBody.getMailDisabled();
        pop3RefreshRate = requestBody.getPop3RefreshRate();
        pop3ExpungeOnQuit = requestBody.getPop3ExpungeOnQuit();
        pop3DeleteWriteThrough = requestBody.getPop3DeleteWriteThrough();

        transportProtocol = requestBody.getTransportProtocol();
        transportServer = requestBody.getTransportServer();
        transportPort = requestBody.getTransportPort();
        transportLogin = requestBody.getTransportLogin();
        transportAuth = requestBody.getTransportAuth();
        transportOAuth = requestBody.getTransportOAuth();
        transportSecure = requestBody.getTransportSecure();
        transportStartTLS = requestBody.getTransportStartTLS();
        transportDisabled = requestBody.getTransportDisabled();

        spamHandler = requestBody.getSpamHandler();
        rootFolderId = requestBody.getRootFolderId();
        inboxFolderId = requestBody.getInboxFolderId();
        draftsFolder = requestBody.getDraftsFolder();
        sentFolder = requestBody.getSentFolder();
        trashFolder = requestBody.getTrashFolder();
        spamFolder = requestBody.getSpamFolder();
        confirmedSpamFolder = requestBody.getConfirmedSpamFolder();
        confirmedHamFolder = requestBody.getConfirmedHamFolder();
        archiveFolderId = requestBody.getArchiveFolderId();
        unifiedInboxEnabled = requestBody.getUnifiedInboxEnabled();

        canSend = null;

        folders = null;
        standardFolders = null;
    }

    public MailAccountData(MailAccountDataMto accountMto) {
        id = accountMto.getId() != null ? accountMto.getId().toString() : null;
        accountName = accountMto.getAccountName();
        personalName = accountMto.getPersonalName();
        login = accountMto.getLogin();
        password = null;
        userName = null;
        firstName = null;
        lastName = null;
        primaryAddress = accountMto.getPrimaryAddress();
        aliases = AccountUtil.getAliases(primaryAddress, accountMto.getAliases());
        mailProtocol = accountMto.getMailProtocol();
        mailServer = accountMto.getMailServer();
        mailPort = accountMto.getMailPort();
        mailSecure = accountMto.getMailSecure();
        mailStartTLS = accountMto.getMailStartTLS();
        mailOAuth = accountMto.getMailOAuth();
        mailDisabled = accountMto.getMailDisabled();
        pop3RefreshRate = accountMto.getPop3RefreshRate();
        pop3ExpungeOnQuit = accountMto.getPop3ExpungeOnQuit();
        pop3DeleteWriteThrough = accountMto.getPop3DeleteWriteThrough();

        transportProtocol = accountMto.getTransportProtocol();
        transportServer = accountMto.getTransportServer();
        transportPort = accountMto.getTransportPort();
        transportLogin = accountMto.getTransportLogin();
        transportAuth = accountMto.getTransportAuth();
        transportOAuth = accountMto.getTransportOAuth();
        transportSecure = accountMto.getTransportSecure();
        transportStartTLS = accountMto.getTransportStartTLS();
        transportDisabled = accountMto.getTransportDisabled();

        spamHandler = accountMto.getSpamHandler();
        rootFolderId = accountMto.getRootFolderId();
        inboxFolderId = accountMto.getInboxFolderId();
        draftsFolder = accountMto.getDraftsFolder();
        sentFolder = accountMto.getSentFolder();
        trashFolder = accountMto.getTrashFolder();
        spamFolder = accountMto.getSpamFolder();
        confirmedSpamFolder = accountMto.getConfirmedSpamFolder();
        confirmedHamFolder = accountMto.getConfirmedHamFolder();
        archiveFolderId = accountMto.getArchiveFolderId();
        unifiedInboxEnabled = accountMto.getUnifiedInboxEnabled();

        canSend = (accountMto.getId() != null && "0".equals(accountMto.getId().toString())) || !Strings.isEmpty(accountMto.getTransportUrl());

        folders = null;
        standardFolders = null;
    }

    public MailAccountData withFolders(List<Folder> folders) {
        this.folders = folders;
        standardFolders = createStandardFolders(folders);
        return this;
    }

    private StandardFolders createStandardFolders(List<Folder> folders) {
        Folder inboxFolder = null;
        Folder archiveFolder = null;
        Folder draftsFolder = null;
        Folder sentFolder = null;
        Folder spamFolder = null;
        Folder trashFolder = null;

        for (Folder folder : folders) {
            switch (folder.getType()) {
            case INBOX:
                inboxFolder = folder;
                break;
            case ARCHIVE:
                archiveFolder = folder;
                break;
            case DRAFT:
                draftsFolder = folder;
                break;
            case SENT:
                sentFolder = folder;
                break;
            case SPAM:
                spamFolder = folder;
                break;
            case TRASH:
                trashFolder = folder;
                break;
            default:
                // Nothing to do.
                break;
            }
        }

        return new StandardFolders(draftsFolder, inboxFolder, sentFolder, spamFolder, trashFolder, archiveFolder);
    }

    public boolean isPrimaryAccount() {
        return "0".equals(id);
    }

    public boolean isPop3Account() {
        return mailProtocol != null && mailProtocol.startsWith("pop3");
    }

}
