/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.exceptions;

import java.util.Collections;
import java.util.List;

import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;

import com.openexchange.java.Strings;
import com.openexchange.mobile.api.facade.connectors.AbstractResponseMto;
import com.openexchange.mobile.api.facade.endpoints.responses.AbstractResponseBody;
import com.openexchange.mobile.api.facade.utils.JacksonUtil;

import lombok.Getter;

@SuppressWarnings("serial")
public class ApiFacadeException extends WebApplicationException {

    public interface Code {

        int INVALID_CREDENTIALS = 1;

        int MISSING_HEADER = 2;

        int MISSING_PARAMETER = 3;

        int MISSING_COOKIES = 4;

        int INVALID_USER_AGENT = 5;

        int MAXIMUM_NUMBER_OF_SESSIONS_EXCEEDED = 6;

        int ACCOUNT_NOT_FOUND = 7;

        int EMPTY_ARRAY_NOT_ALLOWED = 8;

        int INVALID_ACCOUNT_DATA = 9;

        int REDIRECT = 10;

        int SERVICE_UNAVAILABLE = 11;

        int PASSWORD_OF_SECONDARY_MAIL_ACCOUNT_CHANGED = 12;

        int PERMISSION_DENIED = 13;

        String INVALID_SESSION = "SES-206";

        int CLIENT_UPDATE_MANDATORY = 900;

        int INTERNAL_SERVER_ERROR = 999;

    }

    @Getter
    private String errorCode;

    private ApiFacadeException(String errorCode, Exception cause, String message, Response response) {
        super(message, cause, response);
        this.errorCode = errorCode;
    }

    public static ApiFacadeException fromResponseBody(AbstractResponseMto responseBody) {
        String code = responseBody.getCode();
        String error = responseBody.getError();
        if (isMaximumNumberOfSessionsExceeded(code)) {
            return maximumNumberOfSessionsExceeded();
        } else if (isSessionInvalid(code, error)) {
            return invalidSession();
        } else if (areCredentialsInvalid(code)) {
            return invalidCredentials();
        } else if (isDatabaseUpdate(code)) {
            return serviceUnavailable();
        } else if (isPasswordOfSecondaryAccountChanged(code)) {
            return passwordSecondaryAccountChanged();
        } else if (isPermissionDenied(code)) {
            return permissionDenied(error);
        } else if (isRedirect(code)) {
            return redirect(error);
        }
        List<String> categories = getCategories(responseBody.getCategories());
        return create(Status.BAD_REQUEST, code, null, responseBody.getError(), categories);
    }

    @SuppressWarnings("unchecked")
    private static List<String> getCategories(Object categories) {
        if (categories instanceof String) {
            return Collections.<String>singletonList((String) categories);
        } else if (categories instanceof List) {
            return (List<String>) categories;
        } else {
            return null;
        }
    }

    private static boolean isMaximumNumberOfSessionsExceeded(String code) {
        return "SES-0007".equals(code);
    }

    private static boolean isSessionInvalid(String code, String error) {
        return (!Strings.isEmpty(code) && code.startsWith("SES-"))
                || (!Strings.isEmpty(error) && error.startsWith("SES-"));
    }

    private static boolean areCredentialsInvalid(String code) {
        return "MSG-1000".equals(code) || "LGI-0006".equals(code);
    }

    private static boolean isDatabaseUpdate(String code) {
        return "CTX-0007".equals(code);
    }

    private static boolean isPasswordOfSecondaryAccountChanged(String code) {
        return "MSG-1016".equals(code);
    }

    private static boolean isPermissionDenied(String code) {
        return "LGI-0013".equals(code);
    }

    private static boolean isRedirect(String code) {
        return "LGI-0016".equals(code);
    }

    public static ApiFacadeException badRequest(int errorCode, String message) {
        return create(Status.BAD_REQUEST, errorCode, message);
    }

    public static ApiFacadeException invalidCredentials() {
        return create(Status.UNAUTHORIZED, Code.INVALID_CREDENTIALS, "Username/password invalid");
    }

    public static ApiFacadeException maximumNumberOfSessionsExceeded() {
        return create(Status.UNAUTHORIZED, Code.MAXIMUM_NUMBER_OF_SESSIONS_EXCEEDED, "Maximum number of sessions exceeded for your account. Please logout from other clients and try again.");
    }

    public static ApiFacadeException invalidSession() {
        return create(Status.UNAUTHORIZED, Code.INVALID_SESSION, "Your session was invalidated. Please try again.");
    }

    public static ApiFacadeException redirect(String location) {
        return create(Status.TEMPORARY_REDIRECT, Code.REDIRECT, location);
    }

    public static ApiFacadeException missingHeader(String name) {
        return create(Status.BAD_REQUEST, Code.MISSING_HEADER, "Header missing: " + name);
    }

    public static ApiFacadeException missingParameter(String name) {
        return create(Status.BAD_REQUEST, Code.MISSING_PARAMETER, "Parameter missing: " + name);
    }

    public static ApiFacadeException invalidUserAgent() {
        return create(Status.BAD_REQUEST, Code.INVALID_USER_AGENT, "Invalid User-Agent header");
    }

    public static ApiFacadeException emptyArrayNotAllowed(String name) {
        return create(Status.BAD_REQUEST, Code.EMPTY_ARRAY_NOT_ALLOWED, "Empty array not allowed: " + name);
    }

    public static ApiFacadeException authenticationNotFound() {
        return internalServerError("Authentication mechanism not found");
    }

    public static ApiFacadeException forbidden() {
        return create(Status.FORBIDDEN, null, "You are not allowed to access this server");
    }

    public static ApiFacadeException internalServerError(String message) {
        return create(Status.INTERNAL_SERVER_ERROR, Code.INTERNAL_SERVER_ERROR, message);
    }

    public static ApiFacadeException gatewayTimeoutError(Exception cause) {
        return create(Status.GATEWAY_TIMEOUT, Code.INTERNAL_SERVER_ERROR, cause, "Timeout connecting to middleware");
    }

    public static ApiFacadeException internalServerError(Exception cause) {
        if (cause instanceof ApiFacadeException) {
            return (ApiFacadeException) cause;
        }
        String message = cause.getMessage() != null ? cause.getMessage() : cause.getClass().getName();
        return create(Status.INTERNAL_SERVER_ERROR, Code.INTERNAL_SERVER_ERROR, cause, message);
    }

    public static ApiFacadeException clientUpdateMandatory() {
        return create(Status.GONE, Code.CLIENT_UPDATE_MANDATORY, "Client too old. Please update");
    }

    private static ApiFacadeException create(Status status, int errorCode, String message) {
        return create(status, String.valueOf(errorCode), message);
    }

    private static ApiFacadeException create(Status status, int errorCode, Exception cause, String message) {
        return create(status, String.valueOf(errorCode), cause, message, null);
    }

    private static ApiFacadeException create(Status status, String errorCode, String message) {
        return create(status, errorCode, null, message, null);
    }

    private static ApiFacadeException create(Status status, String errorCode, Exception cause, String message, List<String> categories) {
        String responseBody = JacksonUtil.toString(new AbstractResponseBody(errorCode, message, categories));
        Response response = Response.status(status).entity(responseBody).build();
        return new ApiFacadeException(errorCode, cause, message, response);
    }

    public static ApiFacadeException weakforceInvalidHashAlgorithm(String algorithm) {
        return create(Status.INTERNAL_SERVER_ERROR, Code.INTERNAL_SERVER_ERROR, "Hashing algorithm not supported: " + algorithm);
    }

    public static ApiFacadeException accountNotFound(String accountId) {
        return create(Status.NOT_FOUND, Code.ACCOUNT_NOT_FOUND, "Account not found: " + accountId);
    }

    public static ApiFacadeException invalidAccountData() {
        return create(Status.BAD_REQUEST, Code.INVALID_ACCOUNT_DATA, "Invalid account data");
    }

    public static ApiFacadeException serviceUnavailable() {
        return create(Status.SERVICE_UNAVAILABLE, Code.SERVICE_UNAVAILABLE, "Service unavailable");
    }

    public static ApiFacadeException passwordSecondaryAccountChanged() {
        return create(Status.SERVICE_UNAVAILABLE, Code.PASSWORD_OF_SECONDARY_MAIL_ACCOUNT_CHANGED, "Password of secondary mail account changed");
    }

    public static ApiFacadeException permissionDenied(String errorMessage) {
        return create(Status.FORBIDDEN, Code.PERMISSION_DENIED, errorMessage != null ? errorMessage : "Permission denied");
    }

}
