/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.configuration.clientspecific;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import com.openexchange.mobile.api.facade.configuration.HostConfiguration;
import com.openexchange.mobile.api.facade.utils.UserAgentInfo;

import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.extern.slf4j.Slf4j;

@Data
@EqualsAndHashCode(callSuper = true)
@Slf4j
public class MatchingHostConfiguration extends HostConfiguration {

    private List<MatchCriteria> criteria = new LinkedList<>();

    public MatchingHostConfiguration(HostConfiguration defaultConfiguration) {
        super(defaultConfiguration);
    }

    @Override
    public void loadProperties(Map<String, Object> properties) {
        super.loadProperties(properties);
        Object rawMatches = properties.get(MATCHES);
        if (rawMatches instanceof Map) {
            @SuppressWarnings("unchecked")
            Map<String, Object> matches = (Map<String, Object>) rawMatches;
            Iterator<Map.Entry<String, Object>> it = matches.entrySet().iterator();
            while (it.hasNext()) {
                Entry<String, Object> entry = it.next();
                String matcherName = entry.getKey();
                Object value = entry.getValue();
                switch (matcherName) {
                case "brand":
                    criteria.add(new BrandMatchCriteria((String) value));
                    continue;

                case "device":
                    criteria.add(new DeviceMatchCriteria((String) value));
                    continue;

                case "platform":
                    criteria.add(new PlatformMatchCriteria((String) value));
                    continue;

                case "version":
                    {
                        VersionRange range = new VersionRange((String) value);
                        criteria.add(new AppVersionMatchCriteria(range));
                    }
                    continue;

                case "osVersion":
                    {
                        VersionRange range = new VersionRange((String) value);
                        criteria.add(new OsVersionMatchCriteria(range));
                    }
                    continue;

                default:
                    log.warn("Unrecognized matcher name '" + matcherName + "'");
                    continue;
                }
            }
        }
    }

    public boolean matches(UserAgentInfo userAgentInfo) {
        Iterator<MatchCriteria> it = this.criteria.iterator();
        while (it.hasNext()) {
            MatchCriteria criteria = it.next();
            if (!criteria.matches(userAgentInfo)) {
                return false;
            }
        }
        return true;
    }

    @Override
    public String toString() {
        return "matcher criteria: " + criteria;
    }

}
