/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.utils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.StringTokenizer;

import com.openexchange.java.Strings;
import com.openexchange.mobile.api.facade.connectors.responses.mtos.ThreadReferenceMto;
import com.openexchange.mobile.api.facade.models.Attachment;
import com.openexchange.mobile.api.facade.models.Attachment.Disposition;
import com.openexchange.mobile.api.facade.models.Mail;
import com.openexchange.mobile.api.facade.models.MailContact;
import com.openexchange.mobile.api.facade.models.MailThread;

import lombok.experimental.UtilityClass;
import lombok.extern.slf4j.Slf4j;

@UtilityClass
@Slf4j
public class MailUtil {

    private final MapFunction<ThreadReferenceMto, MailThread> threadMapFunction = new MapFunction<ThreadReferenceMto, MailThread>() {
        @Override
        public MailThread map(ThreadReferenceMto value) {
            return new MailThread(value);
        }
    };

    public Attachment getBody(Mail mail) {
        List<Attachment> parts = mail.getParts();
        Attachment contentPart = findContentPart(parts, "text/html");
        if (contentPart == null) {
            contentPart = findContentPart(parts, "text/plain");
        }
        return contentPart;
    }

    private Attachment findContentPart(List<Attachment> parts, String contentType) {
        if (parts == null) {
            return null;
        }
        for (Attachment part : parts) {
            Disposition partDisposition = part.getDisposition();
            String partContentType = part.getContentType();
            if (!Disposition.ATTACHMENT.equals(partDisposition)
                && !Disposition.NONE.equals(partDisposition)
                && contentType.equals(partContentType)) {
                return part;
            }
        }
        return null;
    }

    public boolean isTextPlain(Attachment part) {
        return part != null && "text/plain".equals(part.getContentType());
    }

    public List<MailThread> calculateThreads(List<Mail> mails) {
        if (mails == null) {
            return Collections.emptyList();
        }

        int index = 0;
        List<MailThread> threads = new ArrayList<>();
        List<Mail> threadMails = new ArrayList<>();
        Mail latestMail = null;

        while (index < mails.size()) {
            Mail mail = mails.get(index);

            if (mail.getLevel() == 0) {
                addThread(threads, threadMails, latestMail);
                threadMails = new ArrayList<>();
                latestMail = mail;
            } else if (latestMail == null || latestMail.getReceivedDate() < mail.getReceivedDate()) {
                latestMail = mail;
            }

            threadMails.add(mail);
            index++;
        }

        addThread(threads, threadMails, latestMail);

        return threads;
    }

    private void addThread(List<MailThread> threads, List<Mail> mails, Mail latestMail) {
        if (mails.size() > 0) {
            Mail rootMail = mails.get(0);
            threads.add(new MailThread(mails, rootMail.getId(), latestMail.getId(), latestMail.getReceivedDate()));
        }
    }

    public List<MailThread> wrapAsThreads(List<Mail> mails) {
        if (mails == null) {
            return Collections.emptyList();
        }
        List<MailThread> threads = new ArrayList<>(mails.size());
        for (Mail mail : mails) {
            List<Mail> threadMails = Collections.singletonList(mail);
            MailThread thread = new MailThread(threadMails, mail.getId(), mail.getId(), mail.getReceivedDate());
            threads.add(thread);
        }
        return threads;
    }

    public List<MailThread> convert(List<ThreadReferenceMto> data) {
        if (data == null) {
            return Collections.emptyList();
        }
        return ListUtil.map(data, threadMapFunction);
    }

    public List<MailContact> parseContacts(String value) {
        if (Strings.isEmpty(value)) {
            return null;
        }
        List<MailContact> contacts = new ArrayList<>(1);
        StringTokenizer tokenizer = new StringTokenizer(value, ";");
        while (tokenizer.hasMoreTokens()) {
            String token = tokenizer.nextToken();
            int indexLessThan = token.lastIndexOf('<');
            int indexGreaterThan = token.lastIndexOf('>');
            String name;
            String address;
            if (indexLessThan != -1 && indexGreaterThan != -1) {
                name = token.substring(0,  indexLessThan).trim();
                name = name.length() != 0 ? name : null;
                address = token.substring(indexLessThan + 1, indexGreaterThan).trim();
            } else {
                name = null;
                address = token.trim();
            }
            contacts.add(new MailContact(name, address));
        }
        return contacts;
    }

    public List<MailThread> removeDeletedMails(List<MailThread> mailThreads) {
        if (ListUtil.isEmpty(mailThreads)) {
            return Collections.emptyList();
        }
        List<MailThread> cleanedThreads = new ArrayList<>(mailThreads.size());
        for (MailThread mailThread : mailThreads) {
            List<Mail> mails = mailThread.getMails();
            List<Mail> nonDeletedMails = new ArrayList<>(mails.size());
            for (Mail mail : mails) {
                if (!mail.getFlags().isDeleted()) {
                    nonDeletedMails.add(mail);
                }
            }
            if (nonDeletedMails.size() > 0) {
                Mail rootMail = nonDeletedMails.get(0);
                Mail latestMail = nonDeletedMails.get(nonDeletedMails.size() - 1);
                cleanedThreads.add(new MailThread(nonDeletedMails, rootMail.getId(), latestMail.getId(), latestMail.getReceivedDate()));
            }
        }
        return cleanedThreads;
    }

    public Long readAsLong(Object value) {
        if (value instanceof Long) {
            return (Long) value;
        } else if (value instanceof Integer) {
            return ((Integer) value).longValue();
        } else if (value == null) {
            return null;
        } else {
            log.error("Cannot read value as Long: " + value.toString());
            return null;
        }
    }

}
