/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.mobile.api.facade.utils;

import java.util.List;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;

import lombok.SneakyThrows;
import lombok.experimental.UtilityClass;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@UtilityClass
public class JacksonUtil {

    private final ObjectMapper mapper;

    static {
        mapper = new ObjectMapper()
                .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false)
                ;
    }

    public ObjectMapper getMapper() {
        return mapper;
    }

    @SneakyThrows
    public <T> T parse(String str, Class<T> type) {
        ObjectMapper mapper = getMapper();
        return mapper.readValue(str, type);
    }

    @SneakyThrows
    public <T> List<T> parseList(String str, Class<T> entityType) {
        ObjectMapper mapper = getMapper();
        JavaType type = mapper.getTypeFactory().constructCollectionType(List.class, entityType);
        return mapper.readValue(str, type);
    }

    public String toString(Object object) {
        if (object instanceof String) {
            return (String) object;
        }
        ObjectMapper mapper = getMapper();
        String result;
        try {
            result = mapper.writeValueAsString(object);
        } catch (JsonProcessingException e) {
            log.error("Failed to parse object to JSON", e);
            result = "";
        }
        return result;
    }

}
